//------------------------------------------------------------------------------
// SilChessRayTracer.cpp
//
// Copyright (C) 2001-2003,2007-2008 Oliver Hamann.
//
// Homepage: http://eaglemode.sourceforge.net/
//
// This program is free software: you can redistribute it and/or modify it under
// the terms of the GNU General Public License version 3 as published by the
// Free Software Foundation.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License version 3 for
// more details.
//
// You should have received a copy of the GNU General Public License version 3
// along with this program. If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#if defined(__WATCOMC__) || defined(__BORLANDC__)
#	define sqrtf(x) ((float)sqrt(x))
#endif
#include <SilChess/SilChessRayTracer.h>


const unsigned char SilChessRayTracer::BoardTexture[16*128]={
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0xC0,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x03,
	0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,
	0x40,0x80,0x00,0x80,0x01,0x00,0x00,0x18,0x00,0x00,0x0F,0x00,0x80,0x01,0x01,0x02,
	0x40,0x80,0xF8,0x81,0x01,0xF8,0x01,0x18,0xF8,0x01,0x03,0xF8,0x81,0x01,0x01,0x02,
	0x40,0x80,0x88,0x81,0x1F,0x18,0x80,0x1F,0x18,0x01,0x03,0x88,0x81,0x1F,0x01,0x02,
	0x40,0x80,0x88,0x81,0x11,0x18,0x80,0x18,0xF8,0x01,0x0F,0xF8,0x81,0x11,0x01,0x02,
	0x40,0x80,0x88,0x81,0x11,0x18,0x80,0x18,0x18,0x00,0x03,0x80,0x81,0x11,0x01,0x02,
	0x40,0x80,0xF8,0x81,0x1F,0xF8,0x81,0x1F,0xF8,0x01,0x03,0xF8,0x81,0x11,0x01,0x02,
	0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,
	0xC0,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x03,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0xB2,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x65,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0xB2,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x65,0x02,
	0x40,0xB2,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x65,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0xBE,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x7D,0x02,
	0x40,0xB2,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x65,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0x86,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x0D,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0xA6,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x4D,0x02,
	0x40,0xA6,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x4D,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0xBE,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x7D,0x02,
	0x40,0x82,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x05,0x02,
	0x40,0xBE,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x7D,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xBE,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x7D,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x82,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x05,0x02,
	0x40,0x9A,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x35,0x02,
	0x40,0x9A,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x35,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0x98,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x31,0x02,
	0x40,0x98,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x31,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0xBE,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x7D,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xBC,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x79,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xB0,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x61,0x02,
	0x40,0xBE,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x7D,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0xA0,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x41,0x02,
	0x40,0xA0,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x41,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0x86,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x0D,0x02,
	0x40,0xBE,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x7D,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0xFF,0x0F,0x00,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x9C,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x39,0x02,
	0x40,0x98,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x31,0x02,
	0x40,0x98,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x31,0x02,
	0x40,0x98,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x31,0x02,
	0x40,0x98,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x31,0x02,
	0x40,0x98,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x31,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0x40,0x80,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x00,0xF0,0xFF,0x01,0x02,
	0xC0,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x03,
	0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,
	0x40,0x80,0x00,0x80,0x01,0x00,0x00,0x18,0x00,0x00,0x0F,0x00,0x80,0x01,0x01,0x02,
	0x40,0x80,0xF8,0x81,0x01,0xF8,0x01,0x18,0xF8,0x01,0x03,0xF8,0x81,0x01,0x01,0x02,
	0x40,0x80,0x88,0x81,0x1F,0x18,0x80,0x1F,0x18,0x01,0x03,0x88,0x81,0x1F,0x01,0x02,
	0x40,0x80,0x88,0x81,0x11,0x18,0x80,0x18,0xF8,0x01,0x0F,0xF8,0x81,0x11,0x01,0x02,
	0x40,0x80,0x88,0x81,0x11,0x18,0x80,0x18,0x18,0x00,0x03,0x80,0x81,0x11,0x01,0x02,
	0x40,0x80,0xF8,0x81,0x1F,0xF8,0x81,0x1F,0xF8,0x01,0x03,0xF8,0x81,0x11,0x01,0x02,
	0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,
	0xC0,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x03,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

const SilChessRayTracer::Sphere SilChessRayTracer::PawnShape[]={
	{ 0.000F, 0.000F, 0.000F, 0.190F },
	{ 0.000F, 0.000F, 0.160F, 0.120F },
	{ 0.000F, 0.000F, 0.370F, 0.120F }
};

const SilChessRayTracer::Sphere SilChessRayTracer::KnightShape[]={
	{ 0.000F, 0.000F, 0.000F, 0.220F },
	{ 0.021F,-0.021F, 0.250F, 0.160F },
	{ 0.042F,-0.042F, 0.450F, 0.180F },
	{-0.014F, 0.014F, 0.430F, 0.160F },
	{-0.071F, 0.071F, 0.400F, 0.140F },
	{-0.113F, 0.113F, 0.370F, 0.120F },
	{ 0.028F,-0.170F, 0.600F, 0.060F },
	{ 0.170F,-0.028F, 0.600F, 0.060F }
};

const SilChessRayTracer::Sphere SilChessRayTracer::BishopShape[]={
	{ 0.000F, 0.000F, 0.000F, 0.220F },
	{ 0.000F, 0.000F, 0.210F, 0.150F },
	{ 0.000F, 0.000F, 0.370F, 0.140F },
	{ 0.000F, 0.000F, 0.530F, 0.170F },
	{ 0.000F, 0.000F, 0.680F, 0.090F }
};

const SilChessRayTracer::Sphere SilChessRayTracer::RookShape[]={
	{ 0.000F, 0.000F, 0.000F, 0.240F },
	{ 0.000F, 0.000F, 0.170F, 0.200F },
	{ 0.000F, 0.000F, 0.350F, 0.190F },
	{ 0.000F, 0.000F, 0.500F, 0.210F },
	{ 0.110F, 0.000F, 0.600F, 0.100F },
	{ 0.078F, 0.078F, 0.600F, 0.100F },
	{ 0.000F, 0.110F, 0.600F, 0.100F },
	{-0.078F, 0.078F, 0.600F, 0.100F },
	{-0.110F, 0.000F, 0.600F, 0.100F },
	{-0.078F,-0.078F, 0.600F, 0.100F },
	{ 0.000F,-0.110F, 0.600F, 0.100F },
	{ 0.078F,-0.078F, 0.600F, 0.100F }
};

const SilChessRayTracer::Sphere SilChessRayTracer::QueenShape[]={
	{ 0.000F, 0.000F, 0.000F, 0.240F },
	{ 0.000F, 0.000F, 0.130F, 0.180F },
	{ 0.000F, 0.000F, 0.370F, 0.170F },
	{ 0.000F, 0.000F, 0.450F, 0.190F },
	{ 0.000F, 0.000F, 0.540F, 0.200F },
	{ 0.000F, 0.000F, 0.750F, 0.120F },
	{ 0.000F, 0.000F, 0.840F, 0.160F },
	{ 0.060F, 0.000F, 0.880F, 0.120F },
	{ 0.042F, 0.042F, 0.880F, 0.120F },
	{ 0.000F, 0.060F, 0.880F, 0.120F },
	{-0.042F, 0.042F, 0.880F, 0.120F },
	{-0.060F, 0.000F, 0.880F, 0.120F },
	{-0.042F,-0.042F, 0.880F, 0.120F },
	{ 0.000F,-0.060F, 0.880F, 0.120F },
	{ 0.042F,-0.042F, 0.880F, 0.120F },
	{ 0.140F, 0.000F, 0.970F, 0.045F },
	{ 0.099F, 0.099F, 0.970F, 0.045F },
	{ 0.000F, 0.140F, 0.970F, 0.045F },
	{-0.099F, 0.099F, 0.970F, 0.045F },
	{-0.140F, 0.000F, 0.970F, 0.045F },
	{-0.099F,-0.099F, 0.970F, 0.045F },
	{ 0.000F,-0.140F, 0.970F, 0.045F },
	{ 0.099F,-0.099F, 0.970F, 0.045F }
};

const SilChessRayTracer::Sphere SilChessRayTracer::KingShape[]={
	{ 0.000F, 0.000F, 0.000F, 0.250F },
	{ 0.000F, 0.000F, 0.250F, 0.180F },
	{ 0.000F, 0.000F, 0.370F, 0.230F },
	{ 0.000F, 0.000F, 0.440F, 0.220F },
	{ 0.000F, 0.000F, 0.510F, 0.200F },
	{ 0.000F, 0.000F, 0.710F, 0.150F },
	{ 0.050F, 0.000F, 0.820F, 0.160F },
	{ 0.000F, 0.050F, 0.820F, 0.160F },
	{-0.050F, 0.000F, 0.820F, 0.160F },
	{ 0.000F,-0.050F, 0.820F, 0.160F },
	{ 0.000F, 0.000F, 0.970F, 0.040F },
	{ 0.050F, 0.000F, 1.020F, 0.040F },
	{ 0.000F, 0.000F, 1.020F, 0.040F },
	{-0.050F, 0.000F, 1.020F, 0.040F },
	{ 0.000F, 0.000F, 1.070F, 0.040F }
};

const SilChessRayTracer::Material SilChessRayTracer::BoardMaterial[2]={
	{ { 102,  77,  51 }, 256, 128,  90,  51 },
	{ { 163, 122,  82 }, 256, 128,  90,  51 }
};

const SilChessRayTracer::Material SilChessRayTracer::PieceMaterial[2]={
	{ {  13,  13,  13 }, 256, 256,  51, 128 },
	{ { 127, 127, 127 }, 256, 256,  51, 128 }
};

const SilChessRayTracer::Color SilChessRayTracer::SkyColor={  77,  77, 102 };
const SilChessRayTracer::Color SilChessRayTracer::GndColor={  77,  51,  25 };

const float SilChessRayTracer::LightX=-3.0F;
const float SilChessRayTracer::LightY=1.0F;
const float SilChessRayTracer::LightZ=6.0F;

const float SilChessRayTracer::CamPosY=-12.0F;
const float SilChessRayTracer::CamPosZ=14.0F;
const float SilChessRayTracer::CamDirY=1.0F;
const float SilChessRayTracer::CamDirZ=-1.253F;
const float SilChessRayTracer::CamZoom=0.5F;
const float SilChessRayTracer::CamAsp=0.85F;


SilChessRayTracer::SilChessRayTracer()
{
	float k;

	k=1.0F/sqrtf(CamDirY*CamDirY+CamDirZ*CamDirZ);
	UCamDirY=CamDirY*k;
	UCamDirZ=CamDirZ*k;
	k=1.0F/sqrtf(LightX*LightX+LightY*LightY+LightZ*LightZ);
	ULightX=LightX*k;
	ULightY=LightY*k;
	ULightZ=LightZ*k;
	memset(Board,0,sizeof(Board));
}


SilChessRayTracer::~SilChessRayTracer()
{
	int i;
	for (i=0; i<64; i++) if (Board[i]) free(Board[i]);
}


void SilChessRayTracer::SetWorld(const SilChessMachine * machine)
{
	const Sphere * s;
	Piece * p;
	float tx,ty,t;
	int x,y,i,cnt;

	IsHumanWhite=machine->IsHumanWhite();
	MaxBndH=0;
	for (y=0; y<8; y++) for (x=0; x<8; x++)  {
		if (Board[y*8+x]) {
			free(Board[y*8+x]);
			Board[y*8+x]=NULL;
		}
		if (IsHumanWhite) i=machine->GetField(x,7-y);
		else i=machine->GetField(7-x,y);
		if (i==0) continue;
		if      (i==1 || i==7 ) { s=PawnShape  ; cnt=sizeof(PawnShape  )/sizeof(Sphere); }
		else if (i==2 || i==8 ) { s=KnightShape; cnt=sizeof(KnightShape)/sizeof(Sphere); }
		else if (i==3 || i==9 ) { s=BishopShape; cnt=sizeof(BishopShape)/sizeof(Sphere); }
		else if (i==4 || i==10) { s=RookShape  ; cnt=sizeof(RookShape  )/sizeof(Sphere); }
		else if (i==5 || i==11) { s=QueenShape ; cnt=sizeof(QueenShape )/sizeof(Sphere); }
		else                    { s=KingShape  ; cnt=sizeof(KingShape  )/sizeof(Sphere); }
		p=(Piece*)malloc(sizeof(Piece)+sizeof(Sphere)*(cnt-1));
		p->Mat=&(PieceMaterial[i<7?1:0]);
		p->BndX=x-3.5F;
		p->BndY=y-3.5F;
		p->BndH=0.0F;
		p->BndRR=0.0F;
		p->SCnt=cnt;
		memcpy(p->S,s,cnt*sizeof(Sphere));
		for (i=0; i<cnt; i++) {
			p->S[i].x+=x-3.5F;
			p->S[i].y+=y-3.5F;
			t=p->S[i].z+p->S[i].r;
			if (p->BndH<t) p->BndH=t;
			if (MaxBndH<t) MaxBndH=t;
			tx=p->S[i].x-p->BndX;
			ty=p->S[i].y-p->BndY;
			t=sqrtf(tx*tx+ty*ty)+p->S[i].r;
			t=t*t;
			if (p->BndRR<t) p->BndRR=t;
		}
		Board[y*8+x]=p;
	}
}


void SilChessRayTracer::SetViewSize(int width, int height)
{
	float tx,ty;

	ViewWidth=width;
	ViewHeight=height;
	ViewCenterX=width/2.0F-0.5F;
	ViewCenterY=height/2.0F-0.5F;
	tx=(float)width;
	ty=(float)height/CamAsp;
	ViewZoom=(tx<ty?tx:ty)/CamZoom;
}


void SilChessRayTracer::RenderScanline(int y, char * buf, int pixsize,
                                       int rmsk, int gmsk, int bmsk) const
{
	float rx,ry,rz,ryry_rzrz,k;
	int ir,ig,ib,ic,x,rsh,gsh,bsh;
	Color col;

	rx=-ViewCenterX;
	ry=UCamDirY*ViewZoom-UCamDirZ*(ViewCenterY-y);
	rz=UCamDirY*(ViewCenterY-y)+UCamDirZ*ViewZoom;
	ryry_rzrz=ry*ry+rz*rz;
	if (pixsize==sizeof(char) && rmsk==0x07 && gmsk==0x38 && bmsk==0xc0) {
		for (x=0; x<ViewWidth; x++, rx+=1.0F, buf+=1) {
			k=1.0F/sqrtf(ryry_rzrz+rx*rx);
			TraceRay(1,0.0F,CamPosY,CamPosZ,rx*k,ry*k,rz*k,&col);
			ir=col.Red; if (ir>255) ir=255;
			ig=col.Green; if (ig>255) ig=255;
			ib=col.Blue; if (ib>255) ib=255;
			*((unsigned char*)buf)=(unsigned char)(
				(ir>>5) |
				((ig>>2)&0x38) |
				(ib&0xC0)
			);
		}
	}
	else if (pixsize==sizeof(short) && rmsk==0xF800 && gmsk==0x07E0 && bmsk==0x001F) {
		for (x=0; x<ViewWidth; x++, rx+=1.0F, buf+=2) {
			k=1.0F/sqrtf(ryry_rzrz+rx*rx);
			TraceRay(1,0.0F,CamPosY,CamPosZ,rx*k,ry*k,rz*k,&col);
			ir=col.Red; if (ir>255) ir=255;
			ig=col.Green; if (ig>255) ig=255;
			ib=col.Blue; if (ib>255) ib=255;
			*((unsigned short*)buf)=(unsigned short)(
				((ir<<8)&0xF800) |
				((ig<<3)&0x07E0) |
				(ib>>3)
			);
		}
	}
	else if (pixsize==sizeof(int) && rmsk==0xFF0000 && gmsk==0xFF00 && bmsk==0xFF) {
		for (x=0; x<ViewWidth; x++, rx+=1.0F, buf+=4) {
			k=1.0F/sqrtf(ryry_rzrz+rx*rx);
			TraceRay(1,0.0F,CamPosY,CamPosZ,rx*k,ry*k,rz*k,&col);
			ir=col.Red; if (ir>255) ir=255;
			ig=col.Green; if (ig>255) ig=255;
			ib=col.Blue; if (ib>255) ib=255;
			*((unsigned int*)buf)=(unsigned int)(
				(ir<<16) |
				(ig<<8)  |
				ib
			);
		}
	}
	else {
		for (rsh=24; rsh>=-7 && (rmsk&(1<<(rsh+7)))==0; rsh--);
		for (gsh=24; gsh>=-7 && (gmsk&(1<<(gsh+7)))==0; gsh--);
		for (bsh=24; bsh>=-7 && (bmsk&(1<<(bsh+7)))==0; bsh--);
		for (x=0; x<ViewWidth; x++, rx+=1.0F, buf+=pixsize) {
			k=1.0F/sqrtf(ryry_rzrz+rx*rx);
			TraceRay(1,0.0F,CamPosY,CamPosZ,rx*k,ry*k,rz*k,&col);
			ir=col.Red; if (ir>255) ir=255;
			ig=col.Green; if (ig>255) ig=255;
			ib=col.Blue; if (ib>255) ib=255;
			if (rsh<0) ir>>=(-rsh); else  ir<<=rsh;
			if (gsh<0) ig>>=(-gsh); else  ig<<=gsh;
			if (bsh<0) ib>>=(-bsh); else  ib<<=bsh;
			ic=(ir&rmsk)|(ig&gmsk)|(ib&bmsk);
			if      (pixsize==sizeof(char) ) *((unsigned char *)buf)=(unsigned char )ic;
			else if (pixsize==sizeof(short)) *((unsigned short*)buf)=(unsigned short)ic;
			else if (pixsize==sizeof(int)  ) *((unsigned int * )buf)=(unsigned int  )ic;
			else if (pixsize==sizeof(long) ) *((unsigned long *)buf)=(unsigned long )ic;
		}
	}
}


void SilChessRayTracer::View2Board(int sx, int sy, int * bx, int * by) const
{
	float rx,ry,rz,k;
	int x,y;

	ry=UCamDirY*ViewZoom-UCamDirZ*(ViewCenterY-sy);
	rz=UCamDirY*(ViewCenterY-sy)+UCamDirZ*ViewZoom;
	rx=sx-ViewCenterX;
	k=1.0F/sqrtf(ry*ry+rz*rz+rx*rx);
	rx*=k; ry*=k; rz*=k;
	*bx=-1; *by=-1;
	if (rz<=-0.0001F) {
		x=(int)(-rx*CamPosZ/rz+6)-2;
		y=(int)(CamPosY-ry*CamPosZ/rz+6)-2;
		if (x>=0 && x<8 && y>=0 && y<8) {
			if (IsHumanWhite) y=7-y; else x=7-x;
			*bx=x; *by=y;
		}
	}
}


void SilChessRayTracer::Board2View(float bx, float by, int * sx, int * sy) const
{
	float fx,fy,fz;

	if (IsHumanWhite) by=7.0F-by; else bx=7.0F-bx;
	fx=bx-3.5F;
	fy=-CamPosZ*UCamDirY-(by-3.5F-CamPosY)*UCamDirZ;
	fz=(by-3.5F-CamPosY)*UCamDirY-CamPosZ*UCamDirZ;
	*sx=(int)(fx/fz*ViewZoom+ViewCenterX);
	*sy=(int)(ViewCenterY-fy/fz*ViewZoom);
}


bool SilChessRayTracer::TraceRay(int depth,
                                 float px, float py, float pz,
                                 float rx, float ry, float rz,
                                 Color * col) const
{
	float dx,dy,dz,k,d,dmin,dmax,l,fx,fy;
	const Piece * p;
	const Sphere * s, * smin;
	int ix,iy,i,j;
	const Material * mat;

	if (rz<0.0F) {
		if (rz>-0.0001F) rz=-0.0001F;
		dmax=-pz/rz;
		dmin=(MaxBndH-pz)/rz;
	}
	else {
		if (rz<0.0001F) rz=0.0001F;
		dmin=-pz/rz;
		dmax=(MaxBndH-pz)/rz;
	}
	if (dmin<0.0F) dmin=0.0F;

	ix=((int)((px+rx*dmin)*2+23.5F))-16;
	iy=((int)((py+ry*dmin)*2+23.5F))-16;
	smin=NULL;
	mat=NULL;
	if (rx>=0.0F) {
		if (ry>=0.0F) {
			fx=ix*0.5F-px-3.25F;
			fy=iy*0.5F-py-3.25F;
			for (;;) {
				if (ix>14 || iy>14) break;
				if (((ix|iy)&~14)==0 && (p=Board[iy*4+ix/2])!=NULL) {
					for (i=0; i<p->SCnt; i++) {
						s=p->S+i;
						dx=s->x-px;
						dy=s->y-py;
						dz=s->z-pz;
						k=rx*dx+ry*dy+rz*dz;
						if (k<0.0001F) continue;
						d=k*k-dx*dx-dy*dy-dz*dz+s->r*s->r;
						if (d<0.0001F) continue;
						d=k-sqrtf(d);
						if (d<0.0001F) continue;
						if (!col) return true;
						if (d>=dmax) continue;
						dmax=d;
						smin=s;
						mat=p->Mat;
					}
				}
				if (fx*ry<=fy*rx) {
					if (dmax*rx<=fx) break;
					ix++;
					fx+=0.5F;
				}
				else {
					if (dmax*ry<=fy) break;
					iy++;
					fy+=0.5F;
				}
			}
		}
		else {
			fx=ix*0.5F-px-3.25F;
			fy=iy*0.5F-py-3.75F;
			for (;;) {
				if (ix>14 || iy<0) break;
				if (((ix|iy)&~14)==0 && (p=Board[iy*4+ix/2])!=NULL) {
					for (i=0; i<p->SCnt; i++) {
						s=p->S+i;
						dx=s->x-px;
						dy=s->y-py;
						dz=s->z-pz;
						k=rx*dx+ry*dy+rz*dz;
						if (k<0.0001F) continue;
						d=k*k-dx*dx-dy*dy-dz*dz+s->r*s->r;
						if (d<0.0001F) continue;
						d=k-sqrtf(d);
						if (d<0.0001F) continue;
						if (!col) return true;
						if (d>=dmax) continue;
						dmax=d;
						smin=s;
						mat=p->Mat;
					}
				}
				if (fx*ry>=fy*rx) {
					if (dmax*rx<=fx) break;
					ix++;
					fx+=0.5F;
				}
				else {
					if (dmax*ry>=fy) break;
					iy--;
					fy-=0.5F;
				}
			}
		}
	}
	else {
		if (ry>=0.0F) {
			fx=ix*0.5F-px-3.75F;
			fy=iy*0.5F-py-3.25F;
			for (;;) {
				if (ix<0 || iy>14) break;
				if (((ix|iy)&~14)==0 && (p=Board[iy*4+ix/2])!=NULL) {
					for (i=0; i<p->SCnt; i++) {
						s=p->S+i;
						dx=s->x-px;
						dy=s->y-py;
						dz=s->z-pz;
						k=rx*dx+ry*dy+rz*dz;
						if (k<0.0001F) continue;
						d=k*k-dx*dx-dy*dy-dz*dz+s->r*s->r;
						if (d<0.0001F) continue;
						d=k-sqrtf(d);
						if (d<0.0001F) continue;
						if (!col) return true;
						if (d>=dmax) continue;
						dmax=d;
						smin=s;
						mat=p->Mat;
					}
				}
				if (fx*ry>=fy*rx) {
					if (dmax*rx>=fx) break;
					ix--;
					fx-=0.5F;
				}
				else {
					if (dmax*ry<=fy) break;
					iy++;
					fy+=0.5F;
				}
			}
		}
		else {
			fx=ix*0.5F-px-3.75F;
			fy=iy*0.5F-py-3.75F;
			for (;;) {
				if (ix<0 || iy<0) break;
				if (((ix|iy)&~14)==0 && (p=Board[iy*4+ix/2])!=NULL) {
					for (i=0; i<p->SCnt; i++) {
						s=p->S+i;
						dx=s->x-px;
						dy=s->y-py;
						dz=s->z-pz;
						k=rx*dx+ry*dy+rz*dz;
						if (k<0.0001F) continue;
						d=k*k-dx*dx-dy*dy-dz*dz+s->r*s->r;
						if (d<0.0001F) continue;
						d=k-sqrtf(d);
						if (d<0.0001F) continue;
						if (!col) return true;
						if (d>=dmax) continue;
						dmax=d;
						smin=s;
						mat=p->Mat;
					}
				}
				if (fx*ry<=fy*rx) {
					if (dmax*rx>=fx) break;
					ix--;
					fx-=0.5F;
				}
				else {
					if (dmax*ry>=fy) break;
					iy--;
					fy-=0.5F;
				}
			}
		}
	}

	if (smin) {
		px=px+rx*dmax; dx=px-smin->x;
		py=py+ry*dmax; dy=py-smin->y;
		pz=pz+rz*dmax; dz=pz-smin->z;
		k=(rx*dx+ry*dy+rz*dz)*2/(dx*dx+dy*dy+dz*dz);
		rx-=dx*k; ry-=dy*k; rz-=dz*k;
		l=(ULightX*dx+ULightY*dy+ULightZ*dz)/smin->r;
	}
	else if (rz<0.0F) {
		px=px+rx*dmax; py=py+ry*dmax;
		ix=(int)(px*12+64); iy=(int)(py*12+64);
		if (ix>=123 || ix<5 || iy>=123 || iy<5) {
			if (col) *col=GndColor;
			return false;
		}
		if (!col) return true;
		if (!IsHumanWhite) { ix=127-ix; iy=127-iy; }
		mat=BoardMaterial;
		if (BoardTexture[(127-iy)*16+(ix>>3)]&(1<<(ix&7))) mat++;
		pz=0.0F; rz=-rz; l=ULightZ;
	}
	else {
		if (col) *col=SkyColor;
		return false;
	}

	if (depth>0) TraceRay(depth-1,px,py,pz,rx,ry,rz,col);
	else if (rz>=0.0F) *col=SkyColor;
	else *col=GndColor;
	i=mat->Ambient;
	j=0;
	if (l>0.01F) {
		if (depth<=0 || !TraceRay(0,px,py,pz,ULightX,ULightY,ULightZ)) {
			i+=(int)(l*mat->Diffuse);
			k=ULightX*rx+ULightY*ry+ULightZ*rz;
			if (k>0.3F) j=(int)((k*k)*(k*k)*mat->Highlight);
		}
	}
	col->Red = ((col->Red*mat->Reflect + mat->Col.Red*i)>>8) + j;
	col->Green = ((col->Green*mat->Reflect + mat->Col.Green*i)>>8) + j;
	col->Blue = ((col->Blue*mat->Reflect + mat->Col.Blue*i)>>8) + j;
	return true;
}
