/*
    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include "imagepluginloader.h"
#include <QProcess>

MainWindow::MainWindow(QWidget *parent) :
        QMainWindow(parent),
        ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    // Set Window title
    this->setWindowTitle(QCoreApplication::applicationName());
    // Set window icon
    this->setWindowIcon(QIcon(":/icon/logo"));

    // Signal Slot
    // GUI
    connect(ui->AddImageButton, SIGNAL(clicked()), this, SLOT(addImage()));
    connect(ui->AddDirButton, SIGNAL(clicked()), this, SLOT(addDir()));
    connect(ui->ClearImageListButton, SIGNAL(clicked()), this, SLOT(clearList()));
    connect(ui->SaveToDirPushButton, SIGNAL(clicked()), this, SLOT(setSaveToDir()));
    connect(ui->SaveButton, SIGNAL(clicked()), this, SLOT(startSave()));
    connect(ui->StopButton, SIGNAL(clicked()), this, SLOT(stopSave()));
    ui->StopButton->setVisible(false);
    connect(ui->CopyImageCheckBox, SIGNAL(clicked()), this, SLOT(changeCopyImagesUse()));
    connect(ui->ShowIconCheckBox, SIGNAL(stateChanged(int)), this, SLOT(changeShowImages()));
    connect(ui->RemoveImageButton, SIGNAL(clicked()), this, SLOT(removeImage()));
    // Toolbar
    connect(ui->actionAdd_Images, SIGNAL(triggered()), this, SLOT(addImage()));
    connect(ui->actionAdd_Dir, SIGNAL(triggered()), this, SLOT(addDir()));
    connect(ui->actionAdd_rDir, SIGNAL(triggered()), this, SLOT(addRDir()));
    connect(ui->actionSave_Images, SIGNAL(triggered()), this, SLOT(startSave()));
    // Quality Tab
    connect(ui->ChangeQualityCheckBox, SIGNAL(stateChanged(int)), this, SLOT(changeQualityActivated(int)));
    // about Boxes
    connect( ui->actionAbout_Qt, SIGNAL(triggered()), this, SLOT(showAboutQt()));
    connect( ui->actionAbout, SIGNAL(triggered()), this, SLOT(showAbout()));


    connect( &ImagePluginLoader::getInstance(), SIGNAL(pluginStateChanged()), this, SLOT(loadPluginWidgets()));

    //
    connect(this, SIGNAL(image_list_Changed(int)), this, SLOT(imageListChanged(int)));

    // init PluginsDialog
    this->pluginsDialog = new PluginsDialog();
    //Plugins dialog
    connect( ui->actionPlugins, SIGNAL(triggered()), pluginsDialog, SLOT(show()));
    // init convert class
    this->convert = new Convert();
    connect(convert, SIGNAL(message(QString,int)), ui->statusBar, SLOT(showMessage(QString,int)));
    connect(convert, SIGNAL(process(int)), ui->fileProcessBar, SLOT(setValue(int)));
    connect(convert, SIGNAL(taskProcess(int)), ui->taskProcessBar, SLOT(setValue(int)));

    // set Icons for Actions
    InitIcons();

    // state of gui
    loadState();
    // check for existing of exiftool command
    checkForExifTool();
}

MainWindow::~MainWindow()
{
    saveState();
    pluginsDialog->~PluginsDialog();
    delete ui;
}

void MainWindow::loadPluginWidgets(){
    // Adds in ImagePluginLoader loaded plugins to tabWidget
    QWidget *resize = ui->tabWidget->widget(0);
    ui->tabWidget->clear();
    ui->tabWidget->addTab(resize, "Resize");
    foreach (ImagePlugin *plugin, ImagePluginLoader::getInstance().getActivatedImagePlugins()) {
        if(plugin)
            ui->tabWidget->addTab(plugin, plugin->getTitle());
    }
}


void MainWindow::addImage()
{
    // Last open dir is last file dir
    if(imagesList.count()>0)
        this->lastOpenFileDir = QDir(imagesList.last()).absolutePath();
    // Create File Dialog and open selected Files
    QStringList files = QFileDialog::getOpenFileNames(this, tr("Please select a file"), this->lastOpenFileDir,
                                                      "All Supported Image Files (*.bmp *.jpg *.jpeg *.png *.tif *.tiff);;"
                                                      "Joint Photographic Experts Group File (*.jpg *.jpeg);;"
                                                      "All Files (*.*)"
                                                      );
    if(files.isEmpty()) return;
    // checks for existing dir and saves last dialog opened dir
    if( QDir(QFileInfo(files.at(0)).absoluteDir()).exists() )
        this->lastOpenFileDir = QFileInfo(files.at(0)).absoluteDir().absolutePath();

    addImages(files);
}

void MainWindow::addDir()
{
    // Last open dir is last file dir
    if(imagesList.count()>0)
        this->lastOpenFileDir = QDir(imagesList.last()).absolutePath();
    // Create dir dialog to choose a directory
    QString dirString = QFileDialog::getExistingDirectory(this, tr("Please select a directory"), this->lastOpenFileDir );
    // return if no Dir is chosen or if Dir is non existent
    if(dirString.isEmpty()) return;
    if(!QDir(dirString).exists() ) return;
    // save chosen dir to show the same dir next File Dialog
    this->lastOpenFileDir = dirString;
    addDir(dirString);
}
// Adds a dir and all its subdirs
void MainWindow::addRDir()
{
    // Last open dir is last file dir
    if(imagesList.count()>0)
        this->lastOpenFileDir = QDir(imagesList.last()).absolutePath();
    // Create dir dialog to choose a directory
    QString dirString = QFileDialog::getExistingDirectory(this, tr("Please select a directory"), this->lastOpenFileDir );
    // return if no Dir is chosen or if Dir is non existent
    if(dirString.isEmpty()) return;
    if(!QDir(dirString).exists() ) return;
    // save chosen dir to show the same dir next File Dialog
    this->lastOpenFileDir = dirString;
    addRDir(dirString);
}

// Add Images to Image List
void MainWindow::addImages(QStringList files)
{
    // Check if a file exists
    if( !files.isEmpty())
    {
        for(int i = 0; i< files.count(); i++){
        //foreach(QString file, files){
            ui->taskProcessBar->setValue(int((i+1)*100/files.count()) );
            if(QFile().exists(files.at(i)) && !fileAlreadyAdded(files.at(i)))
            {
                imagesList.append(files.at(i));
                emit image_list_Changed(this->imagesList.count()-1 );
            }
        }
        ui->taskProcessBar->setValue(0);
    }
}

void MainWindow::addDir(QString dirString){
    if(QDir().exists(dirString)){
        QDir dir = QDir(dirString);
        // Set Filter to only load Image Files
        dir.setNameFilters(QStringList() << QString("*.bmp")
                           << QString("*.jpg")
                           << QString("*.jpeg")
                           << QString("*.png")
                           << QString("*.tif")
                           << QString("*.tiff") );
        QFileInfoList finfo = dir.entryInfoList(QDir::Files );
        QStringList files;
        // generate file list
        for (int i=0; i<finfo.count(); i++){
            files.append( finfo.at(i).absoluteFilePath());
        }
        // adds files to Image List
        this->addImages(files);
    }
}

void MainWindow::addRDir(QString dirString){
    if(QDir().exists(dirString)){
        QDir dir = QDir(dirString);
        QFileInfoList finfo = dir.entryInfoList(QDir::NoDotAndDotDot | QDir::Dirs);
        for(int i=0; i<finfo.count(); i++){
            addRDir( finfo.at(i).absoluteFilePath());
        }
        this->addDir(dirString);
    }
}

bool MainWindow::fileAlreadyAdded(QString file){
    if(this->imagesList.contains(file)) return true;
    return false;
}

// loads preview of an Image in Image List
void MainWindow::imageListChanged(int index){
    QStringList files = imagesList;
    if(index < 0 || index > files.count())
        return;
        // Process Bar
        ui->fileProcessBar->setValue(33);
        // check if file exists
        if(files.at(index)=="") return;
        if( !QFile(files.at(index)).exists() ) return;
        QListWidgetItem *item = new QListWidgetItem(files.at(index));
        if( this->ui->ShowIconCheckBox->isChecked() )
        {
            QIcon icon;
            QPixmap pixmap;
            // make copy of scaled image
            pixmap = QPixmap(files.at(index)).scaledToHeight(48);
            ui->fileProcessBar->setValue(66);
            // open Image to use it as Icon
            icon = QIcon(pixmap);
            // Process Bar
            ui->taskProcessBar->setValue(100);
            // create List Item
            item->setIcon(icon);
        }
        // add Item to List
        ui->ImageListWidget->addItem(item);
        // Process Bar
        ui->fileProcessBar->setValue(0);
    // Process Bar
    ui->fileProcessBar->setValue(0);
}

// Clears List of images
void MainWindow::clearList(){
    this->imagesList.clear();
}

// set Save Dir
void MainWindow::setSaveToDir()
{
    // opens a Dialog to choose save directory
    QString dir = QFileDialog::getExistingDirectory(
            this,
            "Please select a directory.",
            this->lastOpenFileDir );
    // checks for dir exists
    if( QDir(dir).exists() )
    {
        // saves last chosen savedir
        this->lastOpenFileDir = dir;
        // save chosen dir
        ui->SaveToDirLineEdit->setText(dir);
    }
}

// save images in image list
void MainWindow::startSave(){
    ui->StopButton->setVisible(true);

    convert->setImagesList(this->imagesList);
    convert->setSaveTo(ui->SaveToCheckBox->isChecked());
    convert->setSaveToDir(ui->SaveToDirLineEdit->text());
    convert->setCopyImage(ui->CopyImageCheckBox->isChecked());
    convert->setChangeQuality(ui->ChangeQualityCheckBox->isChecked());
    convert->setQuality(ui->QualitySpinBox->value());
    convert->setImageSaveFormat(ui->ImageSaveFormatComboBox->currentText());
    connect(convert,SIGNAL(finished()), this, SLOT(stopSave()));
    convert->start();
}

void MainWindow::stopSave(){
    ui->StopButton->setVisible(false);
    convert->stopConvert();
}

void MainWindow::changeQualityActivated(int state){
    if(state == Qt::Checked){
        ui->QualityHorizontalSlider->setEnabled(true);
        ui->QualitySpinBox->setEnabled(true);
    }else {
        ui->QualityHorizontalSlider->setEnabled(false);
        ui->QualitySpinBox->setEnabled(false);
    }
}

// triggered if CopyImage is clicked
void MainWindow::changeCopyImagesUse()
{
    if(!ui->CopyImageCheckBox->isChecked())
    {
        // QuestionBox to check if the User really wants to convert original Files
        if(QMessageBox::warning(this, tr("Original Image?"),
                                tr("Do you really want use original image for your convertion?"),
                                QMessageBox::Yes | QMessageBox::No )
            == QMessageBox::Yes )
            {
        } else {
            ui->CopyImageCheckBox->setCheckState( Qt::Checked );
        }
    }
}

// Drag and Drop
void MainWindow::dragEnterEvent( QDragEnterEvent* event )
{
    // Drag part
    const QMimeData* md = event->mimeData();
    if( event && ( md->hasUrls() ))
        event->acceptProposedAction();
}

void MainWindow::dropEvent( QDropEvent* event)
{
    // Drop part
    if(event && event->mimeData())
    {
        const QMimeData *mData = event->mimeData();
        // Drop Images from FileManager into ImageList
        if( mData->hasUrls())
        {
            QList<QUrl> urls = mData->urls();
            QStringList files;
            for(int x = 0; x<urls.count(); ++x)
            {
                if(QFileInfo(urls.at(x).toLocalFile()).isDir())
                    this->addRDir(urls.at(x).toLocalFile());
                else
                    this->addImages(QStringList(urls.at(x).toLocalFile()));
            }
        }
    }
}

// Qt About Box
void MainWindow::showAboutQt()
{
    QMessageBox::aboutQt(this, "About Qt");
}

void MainWindow::showAbout()
{
    QMessageBox::about( this, tr("About ") + QApplication::applicationName(),
                        ("<p align=\"right\"><h1>") + QApplication::applicationName() +
                        ("</h1></p>") +
                        ("<p align=\"left\"><font size=\"2\" color=\"grey\">Version ") +
                        QApplication::applicationVersion() +
                        ("</font>") +
                        ("<p align=\"left\">(C)2009-2011 FalseCAM</p>") +
                        ("<p><a href=\"http://easyimagesizer.sourceforge.net/\">") +
                        ("Homepage</a></p>") +
                        ("<br/><br/><p>")+
                        tr("English translation: FalseCAM")+
                        ("</p>"));
}

// show images in ImageListWidget
void MainWindow::changeShowImages()
{
    for(int i = 0; i< this->ui->ImageListWidget->count(); ++i)
    {

        if( this->ui->ShowIconCheckBox->isChecked() )
        {
            // Process Bar
            ui->taskProcessBar->setValue(33);
            ui->fileProcessBar->setValue(int((i+1)*100/this->ui->ImageListWidget->count()) );
            QIcon icon;
            QPixmap pixmap;
            // make copy of scaled image
            pixmap = QPixmap(this->ui->ImageListWidget->item(i)->text()).scaledToHeight(48);
            ui->taskProcessBar->setValue(66);
            // open Image to use it as Icon
            icon = QIcon(pixmap);
            // Process Bar
            ui->taskProcessBar->setValue(100);
            // add Icon to Item
            QListWidgetItem *item = new QListWidgetItem( this->ui->ImageListWidget->item(i)->text() );
            item->setIcon(icon);
            // remove old item and insert new item
            ui->ImageListWidget->takeItem(i);
            ui->ImageListWidget->insertItem(i, item);
            // Process Bar
            ui->taskProcessBar->setValue(0);
        } else
        {
            // create new item without icon
            QListWidgetItem *item = new QListWidgetItem( this->ui->ImageListWidget->item(i)->text() );
            // remove old item and insert new item
            ui->ImageListWidget->takeItem(i);
            ui->ImageListWidget->insertItem(i, item);
        }

    }
    // Process Bar
    ui->taskProcessBar->setValue(0);
    ui->fileProcessBar->setValue(0);
}

// loads last state of gui
void MainWindow::loadState(){
    QSettings settings(QCoreApplication::organizationName(), QCoreApplication::applicationName());
    settings.beginGroup("window");
    this->lastOpenFileDir = settings.value("LastOpenFile", "").toString();
    ui->SaveToDirLineEdit->setText(settings.value("SaveToDir", "").toString());
    ui->ImageSaveFormatComboBox->setCurrentIndex(settings.value("ImageSaveFormat", 0).toInt());
    ui->ChangeQualityCheckBox->setChecked(settings.value("QualityUse", false).toBool());
    ui->QualitySpinBox->setValue(settings.value("Quality", 70).toInt());
}

// saves state of gui
void MainWindow::saveState(){
    QSettings settings(QCoreApplication::organizationName(), QCoreApplication::applicationName());
    settings.beginGroup("window");
    settings.setValue("LastOpenFile", this->lastOpenFileDir);
    settings.setValue("SaveToDir", ui->SaveToDirLineEdit->text());
    settings.setValue("ImageSaveFormat", ui->ImageSaveFormatComboBox->currentIndex());
    settings.setValue("QualityUse", ui->ChangeQualityCheckBox->isChecked());
    settings.setValue("Quality", ui->QualitySpinBox->value());

}

// Set Actions Icons
void MainWindow::InitIcons(){
    // Actions
    ui->actionAdd_Images->setIcon(this->style()->standardIcon(QStyle::SP_FileIcon));
    ui->actionAdd_Dir->setIcon(this->style()->standardIcon(QStyle::SP_DirOpenIcon));
    ui->actionAdd_rDir->setIcon(this->style()->standardIcon(QStyle::SP_DirOpenIcon));
    ui->actionSave_Images->setIcon(this->style()->standardIcon(QStyle::SP_DialogApplyButton));
    // Buttons
    ui->AddImageButton->setIcon(this->style()->standardIcon(QStyle::SP_FileIcon));
    ui->AddDirButton->setIcon(this->style()->standardIcon(QStyle::SP_DirOpenIcon));
    ui->SaveButton->setIcon(this->style()->standardIcon(QStyle::SP_DialogApplyButton));
    ui->StopButton->setIcon(this->style()->standardIcon(QStyle::SP_DialogCancelButton));
    ui->ClearImageListButton->setIcon(this->style()->standardIcon(QStyle::SP_TrashIcon));
}

// remove Image
void MainWindow::removeImage(){
    this->imagesList.removeAt(ui->ImageListWidget->currentRow());
    ui->ImageListWidget->takeItem(ui->ImageListWidget->currentRow());
}

// check for exif tool
void MainWindow::checkForExifTool(){
    QProcess exiftool;
    exiftool.start("exiftool -help");
    if(exiftool.waitForFinished(-1)){

    }else{
        QMessageBox::warning(this , tr("exiftool warning"),
                         tr("Exiftool command is not installed or in workingdirectory.<br />")+
                         tr("Exif data of images wont be copied."));
    }
}
