package IO::Scalar;

use vars qw($VERSION);
$VERSION = "2.110";

1;
__END__


=head1 NAME

IO-stringy - I/O on in-core objects like strings and arrays


=head1 SYNOPSIS

    IO::
    ::AtomicFile   adpO  Write a file which is updated atomically     ERYQ
    ::Lines        bdpO  I/O handle to read/write to array of lines   ERYQ
    ::Scalar       RdpO  I/O handle to read/write to a string         ERYQ
    ::ScalarArray  RdpO  I/O handle to read/write to array of scalars ERYQ
    ::Wrap         RdpO  Wrap old-style FHs in standard OO interface  ERYQ
    ::WrapTie      adpO  Tie your handles & retain full OO interface  ERYQ


=head1 DESCRIPTION

This toolkit primarily provides modules for performing both traditional
and object-oriented i/o) on things I<other> than normal filehandles;
in particular, L<IO::Scalar|IO::Scalar>, L<IO::ScalarArray|IO::ScalarArray>,
and L<IO::Lines|IO::Lines>.

In the more-traditional IO::Handle front, we
have L<IO::AtomicFile|IO::AtomicFile>
which may be used to painlessly create files which are updated
atomically.

And in the "this-may-prove-useful" corner, we have L<IO::Wrap|IO::Wrap>,
whose exported wraphandle() function will clothe anything that's not
a blessed object in an IO::Handle-like wrapper... so you can just
use OO syntax and stop worrying about whether your function's caller
handed you a string, a globref, or a FileHandle.


=head1 WARNINGS

Perl's TIEHANDLE spec was incomplete prior to 5.005_57;
it was missing support for C<seek()>, C<tell()>, and C<eof()>.
Attempting to use these functions with an IO::Scalar, IO::ScalarArray,
IO::Lines, etc. B<will not work> prior to 5.005_57.
None of the relevant methods will be invoked by Perl;
and even worse, this kind of bug can lie dormant for a while.
If you turn warnings on (via C<$^W> or C<perl -w>), and you see
something like this...

    seek() on unopened file

...then you are probably trying to use one of these functions
on one of our IO:: classes with an old Perl.  The remedy is to simply
use the OO version; e.g.:

    $SH->seek(0,0);    ### GOOD: will work on any 5.005
    seek($SH,0,0);     ### WARNING: will only work on 5.005_57 and beyond



=head1 INSTALLATION


=head2 Requirements

As of version 2.x, this toolkit requires Perl 5.005 for
the IO::Handle subclasses, and 5.005_57 or better is
B<strongly> recommended.  See L<"WARNINGS"> for details.


=head2 Directions

Most of you already know the drill...

    perl Makefile.PL
    make
    make test
    make install

For everyone else out there...
if you've never installed Perl code before, or you're trying to use
this in an environment where your sysadmin or ISP won't let you do
interesting things, B<relax:> since this module contains no binary
extensions, you can cheat.  That means copying the directory tree
under my "./lib" directory into someplace where your script can "see"
it.  For example, under Linux:

    cp -r IO-stringy-1.234/lib/* /path/to/my/perl/

Now, in your Perl code, do this:

    use lib "/path/to/my/perl";
    use IO::Scalar;                   ### or whatever

Ok, now you've been told.  At this point, anyone who whines about
not being given enough information gets an unflattering haiku
written about them in the next change log.  I'll do it.
Don't think I won't.



=head1 VERSION

$Id: Stringy.pm,v 1.3 2005/02/10 21:24:05 dfs Exp $



=head1 TO DO

=over 4

=item (2000/08/02)  Finalize $/ support

Graham Barr submitted this patch half a I<year> ago;
Like a moron, I lost his message under a ton of others,
and only now have the experimental implementation done.

Will the sudden sensitivity to $/ hose anyone out there?
I'm worried, so you have to enable it explicitly in 1.x.
It will be on by default in 2.x, though only IO::Scalar
has been implemented.

=item (2001/08/08)  Remove IO::WrapTie from new IO:: classes

It's not needed.  Backwards compatibility could be maintained
by having new_tie() be identical to new().  Heck, I'll bet
that IO::WrapTie should be reimplemented so the returned
object is just like an IO::Scalar in its use of globrefs.


=back



=head1 CHANGE LOG

=over 4


=item Version 2.110   (2005/02/10)

Maintainership taken over by DSKOLL <dfs@roaringpenguin.com>

Closed the following bugs at
https://rt.cpan.org/NoAuth/Bugs.html?Dist=IO-stringy:

=item

2208 IO::ScalarArray->getline does not return undef for EOF if undef($/)

=item

7132 IO-stringy/Makefile.PL bug - name should be module name

=item

11249 IO::Scalar flush shouldn't return undef

=item

2172 $\ (output record separator) not respected

=item

8605 IO::InnerFile::seek() should return 1 on success

=item

4798 *.html in lib/

=item

4369 Improvement: handling of fixed-size reads in IO::Scalar

(Actually, bug 4369 was closed in Version 2.109)

=item Version 2.109   (2003/12/21)

IO::Scalar::getline now works with ref to int.
I<Thanks to Dominique Quatravaux for this patch.>


=item Version 2.108   (2001/08/20)

The terms-of-use have been placed in the distribution file "COPYING".
Also, small documentation tweaks were made.


=item Version 2.105   (2001/08/09)

Added support for various seek() whences to IO::ScalarArray.

Added support for consulting $/ in IO::Scalar and IO::ScalarArray.
The old C<use_RS()> is not even an option.
Unsupported record separators will cause a croak().

Added a lot of regression tests to supoprt the above.

Better on-line docs (hyperlinks to individual functions).


=item Version 2.103   (2001/08/08)

After sober consideration I have reimplemented IO::Scalar::print()
so that it once again always seeks to the end of the string.
Benchmarks show the new implementation to be just as fast as
Juergen's contributed patch; until someone can convince me otherwise,
the current, safer implementation stays.

I thought more about giving IO::Scalar two separate handles,
one for reading and one for writing, as suggested by Binkley.
His points about what tell() and eof() return are, I think,
show-stoppers for this feature.  Even the manpages for stdio's fseek()
seem to imply a I<single> file position indicator, not two.
So I think I will take this off the TO DO list.
B<Remedy:> you can always have two handles open on the same
scalar, one which you only write to, and one which you only read from.
That should give the same effect.


=item Version 2.101   (2001/08/07)

B<Alpha release.>
This is the initial release of the "IO::Scalar and friends are
now subclasses of IO::Handle".  I'm flinging it against the wall.
Please tell me if the banana sticks.  When it does, the banana
will be called 2.2x.

First off, I<many many thanks to Doug Wilson>, who
has provided an I<invaluable> service by patching IO::Scalar
and friends so that they (1) inherit from IO::Handle, (2) automatically
tie themselves so that the C<new()> objects can be used in native i/o
constructs, and (3) doing it so that the whole damn thing passes
its regression tests.  As Doug knows, my globref Kung-Fu was not
up to the task; he graciously provided the patches.  This has earned
him a seat at the L<Co-Authors|"AUTHOR"> table, and the
right to have me address him as I<sensei>.

Performance of IO::Scalar::print() has been improved by as much as 2x
for lots of little prints, with the cost of forcing those
who print-then-seek-then-print to explicitly seek to end-of-string
before printing again.
I<Thanks to Juergen Zeller for this patch.>

Added the COPYING file, which had been missing from prior versions.
I<Thanks to Albert Chin-A-Young for pointing this out.>

IO::Scalar consults $/ by default (1.x ignored it by default).
Yes, I still need to support IO::ScalarArray.


=item Version 1.221   (2001/08/07)

I threatened in L<"INSTALLATION"> to write an unflattering haiku
about anyone who whined that I gave them insufficient information...
but it turns out that I left out a crucial direction.  D'OH!
I<Thanks to David Beroff for the "patch" and the haiku...>

       Enough info there?
	 Here's unflattering haiku:
       Forgot the line, "make"!  ;-)



=item Version 1.220   (2001/04/03)

Added untested SEEK, TELL, and EOF methods to IO::Scalar
and IO::ScalarArray to support corresponding functions for
tied filehandles: untested, because I'm still running 5.00556
and Perl is complaining about "tell() on unopened file".
I<Thanks to Graham Barr for the suggestion.>

Removed not-fully-blank lines from modules; these were causing
lots of POD-related warnings.
I<Thanks to Nicolas Joly for the suggestion.>


=item Version 1.219   (2001/02/23)

IO::Scalar objects can now be made sensitive to $/ .
Pains were taken to keep the fast code fast while adding this feature.
I<Cheers to Graham Barr for submitting his patch;
jeers to me for losing his email for 6 months.>


=item Version 1.218   (2001/02/23)

IO::Scalar has a new sysseek() method.
I<Thanks again to Richard Jones.>

New "TO DO" section, because people who submit patches/ideas should
at least know that they're in the system... and that I won't lose
their stuff.  Please read it.

New entries in L<"AUTHOR">.
Please read those too.



=item Version 1.216   (2000/09/28)

B<IO::Scalar and IO::ScalarArray now inherit from IO::Handle.>
I thought I'd remembered a problem with this ages ago, related to
the fact that these IO:: modules don't have "real" filehandles,
but the problem apparently isn't surfacing now.
If you suddenly encounter Perl warnings during global destruction
(especially if you're using tied filehandles), then please let me know!
I<Thanks to B. K. Oxley (binkley) for this.>

B<Nasty bug fixed in IO::Scalar::write().>
Apparently, the offset and the number-of-bytes arguments were,
for all practical purposes, I<reversed.>  You were okay if
you did all your writing with print(), but boy was I<this> a stupid bug!
I<Thanks to Richard Jones for finding this one.
For you, Rich, a double-length haiku:>

       Newspaper headline
	  typeset by dyslexic man
       loses urgency

       BABY EATS FISH is
	  simply not equivalent
       to FISH EATS BABY

B<New sysread and syswrite methods for IO::Scalar.>
I<Thanks again to Richard Jones for this.>


=item Version 1.215   (2000/09/05)

Added 'bool' overload to '""' overload, so object always evaluates
to true.  (Whew.  Glad I caught this before it went to CPAN.)


=item Version 1.214   (2000/09/03)

Evaluating an IO::Scalar in a string context now yields
the underlying string.
I<Thanks to B. K. Oxley (binkley) for this.>


=item Version 1.213   (2000/08/16)

Minor documentation fixes.


=item Version 1.212   (2000/06/02)

Fixed IO::InnerFile incompatibility with Perl5.004.
I<Thanks to many folks for reporting this.>


=item Version 1.210   (2000/04/17)

Added flush() and other no-op methods.
I<Thanks to Doru Petrescu for suggesting this.>


=item Version 1.209   (2000/03/17)

Small bug fixes.


=item Version 1.208   (2000/03/14)

Incorporated a number of contributed patches and extensions,
mostly related to speed hacks, support for "offset", and
WRITE/CLOSE methods.
I<Thanks to Richard Jones, Doru Petrescu, and many others.>



=item Version 1.206   (1999/04/18)

Added creation of ./testout when Makefile.PL is run.


=item Version 1.205   (1999/01/15)

Verified for Perl5.005.


=item Version 1.202   (1998/04/18)

New IO::WrapTie and IO::AtomicFile added.


=item Version 1.110

Added IO::WrapTie.


=item Version 1.107

Added IO::Lines, and made some bug fixes to IO::ScalarArray.
Also, added getc().


=item Version 1.105

No real changes; just upgraded IO::Wrap to have a $VERSION string.

=back




=head1 AUTHOR

=over 4

=item Primary Maintainer

David F. Skoll (F<dfs@roaringpenguin.com>).

=item Original Author

Eryq (F<eryq@zeegee.com>).
President, ZeeGee Software Inc (F<http://www.zeegee.com>).

=item Co-Authors

For all their bug reports and patch submissions, the following
are officially recognized:

     Richard Jones
     B. K. Oxley (binkley)
     Doru Petrescu
     Doug Wilson (for picking up the ball I dropped, and doing tie() right)


=back

Go to F<http://www.zeegee.com> for the latest downloads
and on-line documentation for this module.

Enjoy.  Yell if it breaks.


=cut
                                                                                                                                                                                       usr/lib/perl5/site_perl/5.8.8/IO/Wrap.pm                                                            0000444 0000000 0000000 00000011623 10202750161 016401  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   package IO::Wrap;

# SEE DOCUMENTATION AT BOTTOM OF FILE

require 5.002;

use strict;
use vars qw(@ISA @EXPORT $VERSION);
@ISA = qw(Exporter);
@EXPORT = qw(wraphandle);

use FileHandle;
use Carp;

# The package version, both in 1.23 style *and* usable by MakeMaker:
$VERSION = "2.110";


#------------------------------
# wraphandle RAW
#------------------------------
sub wraphandle {
    my $raw = shift;
    new IO::Wrap $raw;
}

#------------------------------
# new STREAM
#------------------------------
sub new {
    my ($class, $stream) = @_;
    no strict 'refs';

    ### Convert raw scalar to globref:
    ref($stream) or $stream = \*$stream;

    ### Wrap globref and incomplete objects:
    if ((ref($stream) eq 'GLOB') or      ### globref
	(ref($stream) eq 'FileHandle') && !defined(&FileHandle::read)) {
	return bless \$stream, $class;
    }
    $stream;           ### already okay!
}

#------------------------------
# I/O methods...
#------------------------------
sub close {
    my $self = shift;
    return close($$self);
}
sub getline {
    my $self = shift;
    my $fh = $$self;
    return scalar(<$fh>);
}
sub getlines {
    my $self = shift;
    wantarray or croak("Can't call getlines in scalar context!");
    my $fh = $$self;
    <$fh>;
}
sub print {
    my $self = shift;
    print { $$self } @_;
}
sub read {
    my $self = shift;
    return read($$self, $_[0], $_[1]);
}
sub seek {
    my $self = shift;
    return seek($$self, $_[0], $_[1]);
}
sub tell {
    my $self = shift;
    return tell($$self);
}

#------------------------------
1;
__END__


=head1 NAME

IO::Wrap - wrap raw filehandles in IO::Handle interface


=head1 SYNOPSIS

   use IO::Wrap;
       
   ### Do stuff with any kind of filehandle (including a bare globref), or 
   ### any kind of blessed object that responds to a print() message.
   ###
   sub do_stuff {
       my $fh = shift;         
       
       ### At this point, we have no idea what the user gave us... 
       ### a globref? a FileHandle? a scalar filehandle name?
       
       $fh = wraphandle($fh);  
        
       ### At this point, we know we have an IO::Handle-like object!
       
       $fh->print("Hey there!");
       ...
   }
    

=head1 DESCRIPTION

Let's say you want to write some code which does I/O, but you don't 
want to force the caller to provide you with a FileHandle or IO::Handle
object.  You want them to be able to say:

    do_stuff(\*STDOUT);
    do_stuff('STDERR');
    do_stuff($some_FileHandle_object);
    do_stuff($some_IO_Handle_object);

And even:

    do_stuff($any_object_with_a_print_method);

Sure, one way to do it is to force the caller to use tiehandle().  
But that puts the burden on them.  Another way to do it is to 
use B<IO::Wrap>, which provides you with the following functions:


=over 4

=item wraphandle SCALAR

This function will take a single argument, and "wrap" it based on
what it seems to be...

=over 4

=item *

B<A raw scalar filehandle name,> like C<"STDOUT"> or C<"Class::HANDLE">.
In this case, the filehandle name is wrapped in an IO::Wrap object, 
which is returned.

=item *

B<A raw filehandle glob,> like C<\*STDOUT>.
In this case, the filehandle glob is wrapped in an IO::Wrap object, 
which is returned.

=item *

B<A blessed FileHandle object.>
In this case, the FileHandle is wrapped in an IO::Wrap object if and only
if your FileHandle class does not support the C<read()> method.

=item *

B<Any other kind of blessed object,> which is assumed to be already
conformant to the IO::Handle interface.
In this case, you just get back that object.

=back

=back


If you get back an IO::Wrap object, it will obey a basic subset of
the IO:: interface.  That is, the following methods (note: I said
I<methods>, not named operators) should work on the thing you get back:

    close 
    getline 
    getlines 
    print ARGS...
    read BUFFER,NBYTES
    seek POS,WHENCE
    tell 



=head1 NOTES

Clearly, when wrapping a raw external filehandle (like \*STDOUT), 
I didn't want to close the file descriptor when the "wrapper" object is
destroyed... since the user might not appreciate that!  Hence,
there's no DESTROY method in this class.

When wrapping a FileHandle object, however, I believe that Perl will 
invoke the FileHandle::DESTROY when the last reference goes away,
so in that case, the filehandle is closed if the wrapped FileHandle
really was the last reference to it.


=head1 WARNINGS

This module does not allow you to wrap filehandle names which are given
as strings that lack the package they were opened in. That is, if a user 
opens FOO in package Foo, they must pass it to you either as C<\*FOO> 
or as C<"Foo::FOO">.  However, C<"STDIN"> and friends will work just fine.


=head1 VERSION

$Id: Wrap.pm,v 1.2 2005/02/10 21:21:53 dfs Exp $
    

=head1 AUTHOR

=item Primary Maintainer

David F. Skoll (F<dfs@roaringpenguin.com>).

=item Original Author

Eryq (F<eryq@zeegee.com>).
President, ZeeGee Software Inc (F<http://www.zeegee.com>).

=cut

                                                                                                             usr/lib/perl5/site_perl/5.8.8/IO/AtomicFile.pm                                                      0000444 0000000 0000000 00000011344 10202750161 017504  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   package IO::AtomicFile;

### DOCUMENTATION AT BOTTOM OF FILE

# Be strict:
use strict;

# External modules:
use IO::File;


#------------------------------
#
# GLOBALS...
#
#------------------------------
use vars qw($VERSION @ISA);

# The package version, both in 1.23 style *and* usable by MakeMaker:
$VERSION = "2.110";

# Inheritance:
@ISA = qw(IO::File);


#------------------------------
# new ARGS...
#------------------------------
# Class method, constructor.
# Any arguments are sent to open().
#
sub new {
    my $class = shift;
    my $self = $class->SUPER::new();
    ${*$self}{'io_atomicfile_suffix'} = '';
    $self->open(@_) if @_;
    $self;
}

#------------------------------
# DESTROY 
#------------------------------
# Destructor.
#
sub DESTROY {
    shift->close(1);   ### like close, but raises fatal exception on failure
}

#------------------------------
# open PATH, MODE
#------------------------------
# Class/instance method.
#
sub open {
    my ($self, $path, $mode) = @_;
    ref($self) or $self = $self->new;    ### now we have an instance! 

    ### Create tmp path, and remember this info: 
    my $temp = "${path}..TMP" . ${*$self}{'io_atomicfile_suffix'};
    ${*$self}{'io_atomicfile_temp'} = $temp;
    ${*$self}{'io_atomicfile_path'} = $path;

    ### Open the file!  Returns filehandle on success, for use as a constructor: 
    $self->SUPER::open($temp, $mode) ? $self : undef;
}

#------------------------------
# _closed [YESNO]
#------------------------------
# Instance method, private.
# Are we already closed?  Argument sets new value, returns previous one.
#
sub _closed {
    my $self = shift;
    my $oldval = ${*$self}{'io_atomicfile_closed'};
    ${*$self}{'io_atomicfile_closed'} = shift if @_;
    $oldval;
}

#------------------------------
# close
#------------------------------
# Instance method.
# Close the handle, and rename the temp file to its final name.
#
sub close {
    my ($self, $die) = @_;
    unless ($self->_closed(1)) {             ### sentinel...
        $self->SUPER::close();    
        rename(${*$self}{'io_atomicfile_temp'},
	       ${*$self}{'io_atomicfile_path'}) 
            or ($die ? die "close atomic file: $!\n" : return undef); 
    }
    1;
}

#------------------------------
# delete
#------------------------------
# Instance method.
# Close the handle, and delete the temp file.
#
sub delete {
    my $self = shift;
    unless ($self->_closed(1)) {             ### sentinel...
        $self->SUPER::close();    
        return unlink(${*$self}{'io_atomicfile_temp'});
    }
    1;
}

#------------------------------
# detach
#------------------------------
# Instance method.
# Close the handle, but DO NOT delete the temp file.
#
sub detach {
    my $self = shift;
    $self->SUPER::close() unless ($self->_closed(1));
    1;
}

#------------------------------
1;
__END__


=head1 NAME

IO::AtomicFile - write a file which is updated atomically


=head1 SYNOPSIS

    use IO::AtomicFile;

    ### Write a temp file, and have it install itself when closed:
    my $FH = IO::AtomicFile->open("bar.dat", "w");
    print $FH "Hello!\n";
    $FH->close || die "couldn't install atomic file: $!";    

    ### Write a temp file, but delete it before it gets installed:
    my $FH = IO::AtomicFile->open("bar.dat", "w");
    print $FH "Hello!\n";
    $FH->delete; 

    ### Write a temp file, but neither install it nor delete it:
    my $FH = IO::AtomicFile->open("bar.dat", "w");
    print $FH "Hello!\n";
    $FH->detach;   


=head1 DESCRIPTION

This module is intended for people who need to update files 
reliably in the face of unexpected program termination.  

For example, you generally don't want to be halfway in the middle of
writing I</etc/passwd> and have your program terminate!  Even
the act of writing a single scalar to a filehandle is I<not> atomic.

But this module gives you true atomic updates, via rename().
When you open a file I</foo/bar.dat> via this module, you are I<actually> 
opening a temporary file I</foo/bar.dat..TMP>, and writing your
output there.   The act of closing this file (either explicitly
via close(), or implicitly via the destruction of the object)
will cause rename() to be called... therefore, from the point
of view of the outside world, the file's contents are updated
in a single time quantum.

To ensure that problems do not go undetected, the "close" method
done by the destructor will raise a fatal exception if the rename()
fails.  The explicit close() just returns undef.   

You can also decide at any point to trash the file you've been 
building. 


=head1 AUTHOR

=head2 Primary Maintainer

David F. Skoll (F<dfs@roaringpenguin.com>).

=head2 Original Author

Eryq (F<eryq@zeegee.com>).
President, ZeeGee Software Inc (F<http://www.zeegee.com>).


=head1 REVISION

$Revision: 1.2 $

=cut 
                                                                                                                                                                                                                                                                                            usr/lib/perl5/site_perl/5.8.8/IO/WrapTie.pm                                                         0000444 0000000 0000000 00000034645 10202750161 017054  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   # SEE DOCUMENTATION AT BOTTOM OF FILE


#------------------------------------------------------------
package IO::WrapTie;
#------------------------------------------------------------
require 5.004;              ### for tie
use strict;
use vars qw(@ISA @EXPORT $VERSION);
use Exporter;

# Inheritance, exporting, and package version:
@ISA     = qw(Exporter);
@EXPORT  = qw(wraptie);
$VERSION = "2.110";

# Function, exported.
sub wraptie {
    IO::WrapTie::Master->new(@_);
}

# Class method; BACKWARDS-COMPATIBILITY ONLY!
sub new { 
    shift; 
    IO::WrapTie::Master->new(@_);
}



#------------------------------------------------------------
package IO::WrapTie::Master;
#------------------------------------------------------------

use strict;
use vars qw(@ISA $AUTOLOAD);
use IO::Handle;

# We inherit from IO::Handle to get methods which invoke i/o operators,
# like print(), on our tied handle:
@ISA = qw(IO::Handle);

#------------------------------
# new SLAVE, TIEARGS...
#------------------------------
# Create a new subclass of IO::Handle which...
#
#   (1) Handles i/o OPERATORS because it is tied to an instance of 
#       an i/o-like class, like IO::Scalar.
#
#   (2) Handles i/o METHODS by delegating them to that same tied object!.
#
# Arguments are the slave class (e.g., IO::Scalar), followed by all 
# the arguments normally sent into that class's TIEHANDLE method.
# In other words, much like the arguments to tie().  :-)
#
# NOTE:
# The thing $x we return must be a BLESSED REF, for ($x->print()).
# The underlying symbol must be a FILEHANDLE, for (print $x "foo").
# It has to have a way of getting to the "real" back-end object...
#
sub new {
    my $master = shift;
    my $io = IO::Handle->new;   ### create a new handle
    my $slave = shift;
    tie *$io, $slave, @_;       ### tie: will invoke slave's TIEHANDLE
    bless $io, $master;         ### return a master
}

#------------------------------
# AUTOLOAD
#------------------------------
# Delegate method invocations on the master to the underlying slave.
#
sub AUTOLOAD {
    my $method = $AUTOLOAD;
    $method =~ s/.*:://;
    my $self = shift; tied(*$self)->$method(\@_);
}

#------------------------------
# PRELOAD
#------------------------------
# Utility.
#
# Most methods like print(), getline(), etc. which work on the tied object 
# via Perl's i/o operators (like 'print') are inherited from IO::Handle.
#
# Other methods, like seek() and sref(), we must delegate ourselves.
# AUTOLOAD takes care of these.
#
# However, it may be necessary to preload delegators into your
# own class.  PRELOAD will do this.
#
sub PRELOAD {
    my $class = shift;
    foreach (@_) {
	eval "sub ${class}::$_ { my \$s = shift; tied(*\$s)->$_(\@_) }";
    }    
}

# Preload delegators for some standard methods which we can't simply
# inherit from IO::Handle... for example, some IO::Handle methods 
# assume that there is an underlying file descriptor.
#
PRELOAD IO::WrapTie::Master 
    qw(open opened close read clearerr eof seek tell setpos getpos);



#------------------------------------------------------------
package IO::WrapTie::Slave;
#------------------------------------------------------------
# Teeny private class providing a new_tie constructor...
#
# HOW IT ALL WORKS:
# 
# Slaves inherit from this class.
#
# When you send a new_tie() message to a tie-slave class (like IO::Scalar),
# it first determines what class should provide its master, via TIE_MASTER.
# In this case, IO::Scalar->TIE_MASTER would return IO::Scalar::Master.
# Then, we create a new master (an IO::Scalar::Master) with the same args
# sent to new_tie.
#
# In general, the new() method of the master is inherited directly 
# from IO::WrapTie::Master.
#
sub new_tie {
    my $self = shift;
    $self->TIE_MASTER->new($self,@_);     ### e.g., IO::Scalar::Master->new(@_)
}

# Default class method for new_tie().
# All your tie-slave class (like IO::Scalar) has to do is override this 
# method with a method that returns the name of an appropriate "master"
# class for tying that slave.
#
sub TIE_MASTER { 'IO::WrapTie::Master' }

#------------------------------
1;
__END__


package IO::WrapTie;      ### for doc generator


=head1 NAME

IO::WrapTie - wrap tieable objects in IO::Handle interface

I<This is currently Alpha code, released for comments.  
  Please give me your feedback!>


=head1 SYNOPSIS

First of all, you'll need tie(), so:

   require 5.004;

I<Function interface (experimental).>
Use this with any existing class...

   use IO::WrapTie;
   use FooHandle;                  ### implements TIEHANDLE interface

   ### Suppose we want a "FooHandle->new(&FOO_RDWR, 2)".
   ### We can instead say...

   $FH = wraptie('FooHandle', &FOO_RDWR, 2); 

   ### Now we can use...    
   print $FH "Hello, ";            ### traditional operator syntax...
   $FH->print("world!\n");         ### ...and OO syntax as well!

I<OO interface (preferred).>
You can inherit from the IO::WrapTie::Slave mixin to get a
nifty C<new_tie()> constructor...

   #------------------------------    
   package FooHandle;                        ### a class which can TIEHANDLE

   use IO::WrapTie;  
   @ISA = qw(IO::WrapTie::Slave);            ### inherit new_tie()
   ...


   #------------------------------    
   package main; 

   $FH = FooHandle->new_tie(&FOO_RDWR, 2);   ### $FH is an IO::WrapTie::Master
   print $FH "Hello, ";                      ### traditional operator syntax
   $FH->print("world!\n");                   ### OO syntax

See IO::Scalar as an example.  It also shows you how to create classes
which work both with and without 5.004.


=head1 DESCRIPTION

Suppose you have a class C<FooHandle>, where...

=over 4

=item *

B<FooHandle does not inherit from IO::Handle;> that is, it performs
filehandle-like I/O, but to something other than an underlying
file descriptor.  Good examples are IO::Scalar (for printing to a
string) and IO::Lines (for printing to an array of lines).

=item *

B<FooHandle implements the TIEHANDLE interface> (see L<perltie>);
that is, it provides methods TIEHANDLE, GETC, PRINT, PRINTF,
READ, and READLINE.

=item *

B<FooHandle implements the traditional OO interface> of
FileHandle and IO::Handle; i.e., it contains methods like getline(), 
read(), print(), seek(), tell(), eof(), etc.

=back


Normally, users of your class would have two options:


=over 4

=item *

B<Use only OO syntax,> and forsake named I/O operators like 'print'.

=item * 

B<Use with tie,> and forsake treating it as a first-class object 
(i.e., class-specific methods can only be invoked through the underlying
object via tied()... giving the object a "split personality").

=back


But now with IO::WrapTie, you can say:

    $WT = wraptie('FooHandle', &FOO_RDWR, 2);
    $WT->print("Hello, world\n");   ### OO syntax
    print $WT "Yes!\n";             ### Named operator syntax too!
    $WT->weird_stuff;               ### Other methods!

And if you're authoring a class like FooHandle, just have it inherit 
from C<IO::WrapTie::Slave> and that first line becomes even prettier:

    $WT = FooHandle->new_tie(&FOO_RDWR, 2);

B<The bottom line:> now, almost any class can look and work exactly like
an IO::Handle... and be used both with OO and non-OO filehandle syntax.


=head1 HOW IT ALL WORKS


=head2 The data structures

Consider this example code, using classes in this distribution:

    use IO::Scalar;
    use IO::WrapTie;

    $WT = wraptie('IO::Scalar',\$s);
    print $WT "Hello, ";
    $WT->print("world!\n");

In it, the wraptie() function creates a data structure as follows:

                          * $WT is a blessed reference to a tied filehandle
              $WT           glob; that glob is tied to the "Slave" object.
               |          * You would do all your i/o with $WT directly.
               |       
               |
               |     ,---isa--> IO::WrapTie::Master >--isa--> IO::Handle
               V    /
        .-------------. 
        |             | 
        |             |   * Perl i/o operators work on the tied object,  
        |  "Master"   |     invoking the TIEHANDLE methods.
        |             |   * Method invocations are delegated to the tied 
        |             |     slave.
        `-------------' 
               |    
    tied(*$WT) |     .---isa--> IO::WrapTie::Slave
               V    /   
        .-------------.
        |             |
        |   "Slave"   |   * Instance of FileHandle-like class which doesn't
        |             |     actually use file descriptors, like IO::Scalar.
        |  IO::Scalar |   * The slave can be any kind of object.
        |             |   * Must implement the TIEHANDLE interface.
        `-------------'


I<NOTE:> just as an IO::Handle is really just a blessed reference to a 
I<traditional> filehandle glob... so also, an IO::WrapTie::Master 
is really just a blessed reference to a filehandle 
glob I<which has been tied to some "slave" class.>


=head2 How wraptie() works

=over 4

=item 1.

The call to function C<wraptie(SLAVECLASS, TIEARGS...)> is 
passed onto C<IO::WrapTie::Master::new()>.  
Note that class IO::WrapTie::Master is a subclass of IO::Handle.

=item 2.

The C<IO::WrapTie::Master::new> method creates a new IO::Handle object,
reblessed into class IO::WrapTie::Master.  This object is the I<master>, 
which will be returned from the constructor.  At the same time...

=item 3.

The C<new> method also creates the I<slave>: this is an instance 
of SLAVECLASS which is created by tying the master's IO::Handle 
to SLAVECLASS via C<tie(HANDLE, SLAVECLASS, TIEARGS...)>.  
This call to C<tie()> creates the slave in the following manner:

=item 4.

Class SLAVECLASS is sent the message C<TIEHANDLE(TIEARGS...)>; it 
will usually delegate this to C<SLAVECLASS::new(TIEARGS...)>, resulting
in a new instance of SLAVECLASS being created and returned.

=item 5.

Once both master and slave have been created, the master is returned
to the caller.  

=back


=head2 How I/O operators work (on the master)

Consider using an i/o operator on the master:

    print $WT "Hello, world!\n";   

Since the master ($WT) is really a [blessed] reference to a glob, 
the normal Perl i/o operators like C<print> may be used on it.
They will just operate on the symbol part of the glob.

Since the glob is tied to the slave, the slave's PRINT method 
(part of the TIEHANDLE interface) will be automatically invoked.  

If the slave is an IO::Scalar, that means IO::Scalar::PRINT will be 
invoked, and that method happens to delegate to the C<print()> method 
of the same class.  So the I<real> work is ultimately done by 
IO::Scalar::print().


=head2 How methods work (on the master)

Consider using a method on the master:

    $WT->print("Hello, world!\n");

Since the master ($WT) is blessed into the class IO::WrapTie::Master,
Perl first attempts to find a C<print()> method there.  Failing that,
Perl next attempts to find a C<print()> method in the superclass,
IO::Handle.  It just so happens that there I<is> such a method;
that method merely invokes the C<print> i/o operator on the self object...
and for that, see above!

But let's suppose we're dealing with a method which I<isn't> part
of IO::Handle... for example:

    my $sref = $WT->sref;

In this case, the intuitive behavior is to have the master delegate the
method invocation to the slave (now do you see where the designations
come from?).  This is indeed what happens: IO::WrapTie::Master contains
an AUTOLOAD method which performs the delegation.  

So: when C<sref()> can't be found in IO::Handle, the AUTOLOAD method
of IO::WrapTie::Master is invoked, and the standard behavior of
delegating the method to the underlying slave (here, an IO::Scalar)
is done.

Sometimes, to get this to work properly, you may need to create 
a subclass of IO::WrapTie::Master which is an effective master for
I<your> class, and do the delegation there.




=head1 NOTES

B<Why not simply use the object's OO interface?> 
    Because that means forsaking the use of named operators
like print(), and you may need to pass the object to a subroutine
which will attempt to use those operators:

    $O = FooHandle->new(&FOO_RDWR, 2);
    $O->print("Hello, world\n");  ### OO syntax is okay, BUT....

    sub nope { print $_[0] "Nope!\n" }
 X  nope($O);                     ### ERROR!!! (not a glob ref)


B<Why not simply use tie()?> 
    Because (1) you have to use tied() to invoke methods in the
object's public interface (yuck), and (2) you may need to pass 
the tied symbol to another subroutine which will attempt to treat 
it in an OO-way... and that will break it:

    tie *T, 'FooHandle', &FOO_RDWR, 2; 
    print T "Hello, world\n";   ### Operator is okay, BUT... 

    tied(*T)->other_stuff;      ### yuck! AND...

    sub nope { shift->print("Nope!\n") }
 X  nope(\*T);                  ### ERROR!!! (method "print" on unblessed ref)


B<Why a master and slave? 
  Why not simply write FooHandle to inherit from IO::Handle?>
    I tried this, with an implementation similar to that of IO::Socket.  
The problem is that I<the whole point is to use this with objects
that don't have an underlying file/socket descriptor.>.
Subclassing IO::Handle will work fine for the OO stuff, and fine with 
named operators I<if> you tie()... but if you just attempt to say:

    $IO = FooHandle->new(&FOO_RDWR, 2);
    print $IO "Hello!\n";

you get a warning from Perl like:

    Filehandle GEN001 never opened

because it's trying to do system-level i/o on an (unopened) file 
descriptor.  To avoid this, you apparently have to tie() the handle...
which brings us right back to where we started!  At least the
IO::WrapTie mixin lets us say:

    $IO = FooHandle->new_tie(&FOO_RDWR, 2);
    print $IO "Hello!\n";

and so is not I<too> bad.  C<:-)>


=head1 WARNINGS

Remember: this stuff is for doing FileHandle-like i/o on things
I<without underlying file descriptors>.  If you have an underlying
file descriptor, you're better off just inheriting from IO::Handle.

B<Be aware that new_tie() always returns an instance of a
kind of IO::WrapTie::Master...> it does B<not> return an instance 
of the i/o class you're tying to!  

Invoking some methods on the master object causes AUTOLOAD to delegate
them to the slave object... so it I<looks> like you're manipulating a 
"FooHandle" object directly, but you're not.

I have not explored all the ramifications of this use of tie().
I<Here there be dragons>.


=head1 VERSION

$Id: WrapTie.pm,v 1.2 2005/02/10 21:21:53 dfs Exp $


=head1 AUTHOR

=item Primary Maintainer

David F. Skoll (F<dfs@roaringpenguin.com>).

=item Original Author

Eryq (F<eryq@zeegee.com>).
President, ZeeGee Software Inc (F<http://www.zeegee.com>).

=cut

                                                                                           usr/lib/perl5/site_perl/5.8.8/IO/ScalarArray.pm                                                     0000444 0000000 0000000 00000042265 10202750161 017702  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   package IO::ScalarArray;


=head1 NAME

IO::ScalarArray - IO:: interface for reading/writing an array of scalars


=head1 SYNOPSIS

Perform I/O on strings, using the basic OO interface...

    use IO::ScalarArray;
    @data = ("My mes", "sage:\n");

    ### Open a handle on an array, and append to it:
    $AH = new IO::ScalarArray \@data;
    $AH->print("Hello");       
    $AH->print(", world!\nBye now!\n");  
    print "The array is now: ", @data, "\n";

    ### Open a handle on an array, read it line-by-line, then close it:
    $AH = new IO::ScalarArray \@data;
    while (defined($_ = $AH->getline)) { 
	print "Got line: $_";
    }
    $AH->close;

    ### Open a handle on an array, and slurp in all the lines:
    $AH = new IO::ScalarArray \@data;
    print "All lines:\n", $AH->getlines; 

    ### Get the current position (either of two ways):
    $pos = $AH->getpos;         
    $offset = $AH->tell;  

    ### Set the current position (either of two ways):
    $AH->setpos($pos);        
    $AH->seek($offset, 0);

    ### Open an anonymous temporary array:
    $AH = new IO::ScalarArray;
    $AH->print("Hi there!");
    print "I printed: ", @{$AH->aref}, "\n";      ### get at value


Don't like OO for your I/O?  No problem.  
Thanks to the magic of an invisible tie(), the following now 
works out of the box, just as it does with IO::Handle:
    
    use IO::ScalarArray;
    @data = ("My mes", "sage:\n");

    ### Open a handle on an array, and append to it:
    $AH = new IO::ScalarArray \@data;
    print $AH "Hello";    
    print $AH ", world!\nBye now!\n";
    print "The array is now: ", @data, "\n";

    ### Open a handle on a string, read it line-by-line, then close it:
    $AH = new IO::ScalarArray \@data;
    while (<$AH>) {
	print "Got line: $_";
    }
    close $AH;

    ### Open a handle on a string, and slurp in all the lines:
    $AH = new IO::ScalarArray \@data;
    print "All lines:\n", <$AH>;

    ### Get the current position (WARNING: requires 5.6):
    $offset = tell $AH;

    ### Set the current position (WARNING: requires 5.6):
    seek $AH, $offset, 0;

    ### Open an anonymous temporary scalar:
    $AH = new IO::ScalarArray;
    print $AH "Hi there!";
    print "I printed: ", @{$AH->aref}, "\n";      ### get at value


And for you folks with 1.x code out there: the old tie() style still works,
though this is I<unnecessary and deprecated>:

    use IO::ScalarArray;

    ### Writing to a scalar...
    my @a; 
    tie *OUT, 'IO::ScalarArray', \@a;
    print OUT "line 1\nline 2\n", "line 3\n";
    print "Array is now: ", @a, "\n"

    ### Reading and writing an anonymous scalar... 
    tie *OUT, 'IO::ScalarArray';
    print OUT "line 1\nline 2\n", "line 3\n";
    tied(OUT)->seek(0,0);
    while (<OUT>) { 
        print "Got line: ", $_;
    }



=head1 DESCRIPTION

This class is part of the IO::Stringy distribution;
see L<IO::Stringy> for change log and general information.

The IO::ScalarArray class implements objects which behave just like 
IO::Handle (or FileHandle) objects, except that you may use them 
to write to (or read from) arrays of scalars.  Logically, an
array of scalars defines an in-core "file" whose contents are
the concatenation of the scalars in the array.  The handles created by 
this class are automatically tiehandle'd (though please see L<"WARNINGS">
for information relevant to your Perl version).

For writing large amounts of data with individual print() statements, 
this class is likely to be more efficient than IO::Scalar.

Basically, this:

    my @a;
    $AH = new IO::ScalarArray \@a;
    $AH->print("Hel", "lo, ");         ### OO style
    $AH->print("world!\n");            ### ditto

Or this:

    my @a;
    $AH = new IO::ScalarArray \@a;
    print $AH "Hel", "lo, ";           ### non-OO style
    print $AH "world!\n";              ### ditto

Causes @a to be set to the following array of 3 strings:

    ( "Hel" , 
      "lo, " , 
      "world!\n" )

See L<IO::Scalar> and compare with this class.


=head1 PUBLIC INTERFACE

=cut

use Carp;
use strict;
use vars qw($VERSION @ISA);
use IO::Handle;

# The package version, both in 1.23 style *and* usable by MakeMaker:
$VERSION = "2.110";

# Inheritance:
@ISA = qw(IO::Handle);
require IO::WrapTie and push @ISA, 'IO::WrapTie::Slave' if ($] >= 5.004);


#==============================

=head2 Construction 

=over 4

=cut

#------------------------------

=item new [ARGS...]

I<Class method.>
Return a new, unattached array handle.  
If any arguments are given, they're sent to open().

=cut

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto;
    my $self = bless \do { local *FH }, $class;
    tie *$self, $class, $self;
    $self->open(@_);  ### open on anonymous by default
    $self;
}
sub DESTROY { 
    shift->close;
}


#------------------------------

=item open [ARRAYREF]

I<Instance method.>
Open the array handle on a new array, pointed to by ARRAYREF.
If no ARRAYREF is given, a "private" array is created to hold
the file data.

Returns the self object on success, undefined on error.

=cut

sub open {
    my ($self, $aref) = @_;

    ### Sanity:
    defined($aref) or do {my @a; $aref = \@a};
    (ref($aref) eq "ARRAY") or croak "open needs a ref to a array";

    ### Setup:
    $self->setpos([0,0]);
    *$self->{AR} = $aref;
    $self;
}

#------------------------------

=item opened

I<Instance method.>
Is the array handle opened on something?

=cut

sub opened {
    *{shift()}->{AR};
}

#------------------------------

=item close

I<Instance method.>
Disassociate the array handle from its underlying array.
Done automatically on destroy.

=cut

sub close {
    my $self = shift;
    %{*$self} = ();
    1;
}

=back

=cut



#==============================

=head2 Input and output

=over 4

=cut

#------------------------------

=item flush 

I<Instance method.>
No-op, provided for OO compatibility.

=cut

sub flush { "0 but true" } 

#------------------------------

=item getc

I<Instance method.>
Return the next character, or undef if none remain.
This does a read(1), which is somewhat costly.

=cut

sub getc {
    my $buf = '';
    ($_[0]->read($buf, 1) ? $buf : undef);
}

#------------------------------

=item getline

I<Instance method.>
Return the next line, or undef on end of data.
Can safely be called in an array context.
Currently, lines are delimited by "\n".

=cut

sub getline {
    my $self = shift;
    my ($str, $line) = (undef, '');


    ### Minimal impact implementation!
    ### We do the fast fast thing (no regexps) if using the
    ### classic input record separator.

    ### Case 1: $/ is undef: slurp all...    
    if    (!defined($/)) {

        return undef if ($self->eof);

	### Get the rest of the current string, followed by remaining strings:
	my $ar = *$self->{AR};
	my @slurp = (
		     substr($ar->[*$self->{Str}], *$self->{Pos}),
		     @$ar[(1 + *$self->{Str}) .. $#$ar ] 
		     );
	     	
	### Seek to end:
	$self->_setpos_to_eof;
	return join('', @slurp);
    }

    ### Case 2: $/ is "\n": 
    elsif ($/ eq "\012") {    
	
	### Until we hit EOF (or exitted because of a found line):
	until ($self->eof) {
	    ### If at end of current string, go fwd to next one (won't be EOF):
	    if ($self->_eos) {++*$self->{Str}, *$self->{Pos}=0};

	    ### Get ref to current string in array, and set internal pos mark:
	    $str = \(*$self->{AR}[*$self->{Str}]); ### get current string
	    pos($$str) = *$self->{Pos};            ### start matching from here
	
	    ### Get from here to either \n or end of string, and add to line:
	    $$str =~ m/\G(.*?)((\n)|\Z)/g;         ### match to 1st \n or EOS
	    $line .= $1.$2;                        ### add it
	    *$self->{Pos} += length($1.$2);        ### move fwd by len matched
	    return $line if $3;                    ### done, got line with "\n"
        }
        return ($line eq '') ? undef : $line;  ### return undef if EOF
    }

    ### Case 3: $/ is ref to int.  Bail out.
    elsif (ref($/)) {
        croak '$/ given as a ref to int; currently unsupported';
    }

    ### Case 4: $/ is either "" (paragraphs) or something weird...
    ###         Bail for now.
    else {                
        croak '$/ as given is currently unsupported';
    }
}

#------------------------------

=item getlines

I<Instance method.>
Get all remaining lines.
It will croak() if accidentally called in a scalar context.

=cut

sub getlines {
    my $self = shift;
    wantarray or croak("can't call getlines in scalar context!");
    my ($line, @lines);
    push @lines, $line while (defined($line = $self->getline));
    @lines;
}

#------------------------------

=item print ARGS...

I<Instance method.>
Print ARGS to the underlying array.  

Currently, this always causes a "seek to the end of the array"
and generates a new array entry.  This may change in the future.

=cut

sub print {
    my $self = shift;
    push @{*$self->{AR}}, join('', @_) . (defined($\) ? $\ : "");      ### add the data
    $self->_setpos_to_eof;
    1;
}

#------------------------------

=item read BUF, NBYTES, [OFFSET];

I<Instance method.>
Read some bytes from the array.
Returns the number of bytes actually read, 0 on end-of-file, undef on error.

=cut

sub read {
    my $self = $_[0];
    ### we must use $_[1] as a ref
    my $n    = $_[2];
    my $off  = $_[3] || 0;

    ### print "getline\n";
    my $justread;
    my $len;
    ($off ? substr($_[1], $off) : $_[1]) = '';

    ### Stop when we have zero bytes to go, or when we hit EOF:
    my @got;
    until (!$n or $self->eof) {       
        ### If at end of current string, go forward to next one (won't be EOF):
        if ($self->_eos) {
            ++*$self->{Str};
            *$self->{Pos} = 0;
        }

        ### Get longest possible desired substring of current string:
        $justread = substr(*$self->{AR}[*$self->{Str}], *$self->{Pos}, $n);
        $len = length($justread);
        push @got, $justread;
        $n            -= $len; 
        *$self->{Pos} += $len;
    }
    $_[1] .= join('', @got);
    return length($_[1])-$off;
}

#------------------------------

=item write BUF, NBYTES, [OFFSET];

I<Instance method.>
Write some bytes into the array.

=cut

sub write {
    my $self = $_[0];
    my $n    = $_[2];
    my $off  = $_[3] || 0;

    my $data = substr($_[1], $n, $off);
    $n = length($data);
    $self->print($data);
    return $n;
}


=back

=cut



#==============================

=head2 Seeking/telling and other attributes

=over 4

=cut

#------------------------------

=item autoflush 

I<Instance method.>
No-op, provided for OO compatibility.

=cut

sub autoflush {} 

#------------------------------

=item binmode

I<Instance method.>
No-op, provided for OO compatibility.

=cut

sub binmode {} 

#------------------------------

=item clearerr

I<Instance method.>  Clear the error and EOF flags.  A no-op.

=cut

sub clearerr { 1 }

#------------------------------

=item eof 

I<Instance method.>  Are we at end of file?

=cut

sub eof {
    ### print "checking EOF [*$self->{Str}, *$self->{Pos}]\n";
    ### print "SR = ", $#{*$self->{AR}}, "\n";

    return 0 if (*{$_[0]}->{Str} < $#{*{$_[0]}->{AR}});  ### before EOA
    return 1 if (*{$_[0]}->{Str} > $#{*{$_[0]}->{AR}});  ### after EOA
    ###                                                  ### at EOA, past EOS:
    ((*{$_[0]}->{Str} == $#{*{$_[0]}->{AR}}) && ($_[0]->_eos)); 
}

#------------------------------
#
# _eos
#
# I<Instance method, private.>  Are we at end of the CURRENT string?
#
sub _eos {
    (*{$_[0]}->{Pos} >= length(*{$_[0]}->{AR}[*{$_[0]}->{Str}])); ### past last char
}

#------------------------------

=item seek POS,WHENCE

I<Instance method.>
Seek to a given position in the stream.
Only a WHENCE of 0 (SEEK_SET) is supported.

=cut

sub seek {
    my ($self, $pos, $whence) = @_; 

    ### Seek:
    if    ($whence == 0) { $self->_seek_set($pos); }
    elsif ($whence == 1) { $self->_seek_cur($pos); }
    elsif ($whence == 2) { $self->_seek_end($pos); }
    else                 { croak "bad seek whence ($whence)" }
    return 1;
}

#------------------------------
#
# _seek_set POS
#
# Instance method, private.
# Seek to $pos relative to start:
#
sub _seek_set {
    my ($self, $pos) = @_; 

    ### Advance through array until done:
    my $istr = 0;
    while (($pos >= 0) && ($istr < scalar(@{*$self->{AR}}))) {
	if (length(*$self->{AR}[$istr]) > $pos) {   ### it's in this string! 
	    return $self->setpos([$istr, $pos]);
	}
	else {                                      ### it's in next string
	    $pos -= length(*$self->{AR}[$istr++]);  ### move forward one string
	}
    }
    ### If we reached this point, pos is at or past end; zoom to EOF:
    return $self->_setpos_to_eof;
}

#------------------------------
#
# _seek_cur POS
#
# Instance method, private.
# Seek to $pos relative to current position.
#
sub _seek_cur {
    my ($self, $pos) = @_; 
    $self->_seek_set($self->tell + $pos);
}

#------------------------------
#
# _seek_end POS
#
# Instance method, private.
# Seek to $pos relative to end.
# We actually seek relative to beginning, which is simple.
#
sub _seek_end {
    my ($self, $pos) = @_; 
    $self->_seek_set($self->_tell_eof + $pos);
}

#------------------------------

=item tell

I<Instance method.>
Return the current position in the stream, as a numeric offset.

=cut

sub tell {
    my $self = shift;
    my $off = 0;
    my ($s, $str_s);
    for ($s = 0; $s < *$self->{Str}; $s++) {   ### count all "whole" scalars
	defined($str_s = *$self->{AR}[$s]) or $str_s = '';
	###print STDERR "COUNTING STRING $s (". length($str_s) . ")\n";
	$off += length($str_s);
    }
    ###print STDERR "COUNTING POS ($self->{Pos})\n";
    return ($off += *$self->{Pos});            ### plus the final, partial one
}

#------------------------------
#
# _tell_eof
#
# Instance method, private.
# Get position of EOF, as a numeric offset.
# This is identical to the size of the stream - 1.
#
sub _tell_eof {
    my $self = shift;
    my $len = 0;
    foreach (@{*$self->{AR}}) { $len += length($_) }
    $len;
}

#------------------------------

=item setpos POS

I<Instance method.>
Seek to a given position in the array, using the opaque getpos() value.
Don't expect this to be a number.

=cut

sub setpos { 
    my ($self, $pos) = @_;
    (ref($pos) eq 'ARRAY') or
	die "setpos: only use a value returned by getpos!\n";
    (*$self->{Str}, *$self->{Pos}) = @$pos;
}

#------------------------------
#
# _setpos_to_eof
#
# Fast-forward to EOF.
#
sub _setpos_to_eof {
    my $self = shift;
    $self->setpos([scalar(@{*$self->{AR}}), 0]);
}

#------------------------------

=item getpos

I<Instance method.>
Return the current position in the array, as an opaque value.
Don't expect this to be a number.

=cut

sub getpos {
    [*{$_[0]}->{Str}, *{$_[0]}->{Pos}];
}

#------------------------------

=item aref

I<Instance method.>
Return a reference to the underlying array.

=cut

sub aref {
    *{shift()}->{AR};
}

=back

=cut

#------------------------------
# Tied handle methods...
#------------------------------

### Conventional tiehandle interface:
sub TIEHANDLE { (defined($_[1]) && UNIVERSAL::isa($_[1],"IO::ScalarArray"))
		    ? $_[1] 
		    : shift->new(@_) }
sub GETC      { shift->getc(@_) }
sub PRINT     { shift->print(@_) }
sub PRINTF    { shift->print(sprintf(shift, @_)) }
sub READ      { shift->read(@_) }
sub READLINE  { wantarray ? shift->getlines(@_) : shift->getline(@_) }
sub WRITE     { shift->write(@_); }
sub CLOSE     { shift->close(@_); }
sub SEEK      { shift->seek(@_); }
sub TELL      { shift->tell(@_); }
sub EOF       { shift->eof(@_); }

#------------------------------------------------------------

1;
__END__

# SOME PRIVATE NOTES:
#
#     * The "current position" is the position before the next
#       character to be read/written.
#
#     * Str gives the string index of the current position, 0-based
#
#     * Pos gives the offset within AR[Str], 0-based.
#
#     * Inital pos is [0,0].  After print("Hello"), it is [1,0].



=head1 WARNINGS

Perl's TIEHANDLE spec was incomplete prior to 5.005_57;
it was missing support for C<seek()>, C<tell()>, and C<eof()>.
Attempting to use these functions with an IO::ScalarArray will not work
prior to 5.005_57. IO::ScalarArray will not have the relevant methods 
invoked; and even worse, this kind of bug can lie dormant for a while.
If you turn warnings on (via C<$^W> or C<perl -w>),
and you see something like this...

    attempt to seek on unopened filehandle

...then you are probably trying to use one of these functions
on an IO::ScalarArray with an old Perl.  The remedy is to simply
use the OO version; e.g.:

    $AH->seek(0,0);    ### GOOD: will work on any 5.005
    seek($AH,0,0);     ### WARNING: will only work on 5.005_57 and beyond



=head1 VERSION

$Id: ScalarArray.pm,v 1.7 2005/02/10 21:21:53 dfs Exp $


=head1 AUTHOR

=head2 Primary Maintainer

David F. Skoll (F<dfs@roaringpenguin.com>).

=head2 Principal author

Eryq (F<eryq@zeegee.com>).
President, ZeeGee Software Inc (F<http://www.zeegee.com>).


=head2 Other contributors 

Thanks to the following individuals for their invaluable contributions
(if I've forgotten or misspelled your name, please email me!):

I<Andy Glew,>
for suggesting C<getc()>.

I<Brandon Browning,>
for suggesting C<opened()>.

I<Eric L. Brine,>
for his offset-using read() and write() implementations. 

I<Doug Wilson,>
for the IO::Handle inheritance and automatic tie-ing.

=cut

#------------------------------
1;

                                                                                                                                                                                                                                                                                                                                           usr/lib/perl5/site_perl/5.8.8/IO/InnerFile.pm                                                       0000444 0000000 0000000 00000012643 10202750161 017346  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   package IO::InnerFile;

=head1 NAME

IO::InnerFile - define a file inside another file


=head1 SYNOPSIS


    ### Read a subset of a file:
    $inner = IO::InnerFile->new($fh, $start, $length);
    while (<$inner>) {
	...
    }


=head1 DESCRIPTION

If you have a filehandle that can seek() and tell(), then you 
can open an IO::InnerFile on a range of the underlying file.


=head1 PUBLIC INTERFACE

=over

=cut

use Symbol;

# The package version, both in 1.23 style *and* usable by MakeMaker:
$VERSION = "2.110";

#------------------------------

=item new FILEHANDLE, [START, [LENGTH]]

I<Class method, constructor.>
Create a new inner-file opened on the given FILEHANDLE,
from bytes START to START+LENGTH.  Both START and LENGTH
default to 0; negative values are silently coerced to zero.

Note that FILEHANDLE must be able to seek() and tell(), in addition
to whatever other methods you may desire for reading it.

=cut

sub new {
   my ($class, $fh, $start, $lg) = @_;
   $start = 0 if (!$start or ($start < 0));
   $lg    = 0 if (!$lg    or ($lg    < 0));

   ### Create the underlying "object":
   my $a = {
      FH 	=> 	$fh,
      CRPOS 	=> 	0,
      START	=>	$start,
      LG	=>	$lg,
   };

   ### Create a new filehandle tied to this object:
   $fh = gensym;
   tie(*$fh, $class, $a); 
   return bless($fh, $class);
}

sub TIEHANDLE { 
   my ($class, $data) = @_;
   return bless($data, $class);
}

sub DESTROY { 
   my ($self) = @_;
   $self->close() if (ref($self) eq 'SCALAR'); 
}

#------------------------------

=item set_length LENGTH

=item get_length 

=item add_length NBYTES

I<Instance methods.>
Get/set the virtual length of the inner file.

=cut

sub set_length { tied(${$_[0]})->{LG} = $_[1]; }
sub get_length { tied(${$_[0]})->{LG}; }
sub add_length { tied(${$_[0]})->{LG} += $_[1]; }

#------------------------------

=item set_start START

=item get_start 

=item add_start NBYTES

I<Instance methods.>
Get/set the virtual start position of the inner file.

=cut

sub set_start  { tied(${$_[0]})->{START} = $_[1]; }
sub get_start  { tied(${$_[0]})->{START}; } 
sub set_end    { tied(${$_[0]})->{LG} =  $_[1] - tied(${$_[0]})->{START}; }
sub get_end    { tied(${$_[0]})->{LG} + tied(${$_[0]})->{START}; }


#------------------------------

=item binmode

=item close

=item flush

=item getc

=item getline

=item print LIST

=item printf LIST

=item read BUF, NBYTES

=item readline

=item seek OFFFSET, WHENCE

=item tell

=item write ARGS...

I<Instance methods.>
Standard filehandle methods.

=cut

sub write    { shift->WRITE(@_) }
sub print    { shift->PRINT(@_) }
sub printf   { shift->PRINTF(@_) }
sub flush    { "0 but true"; }
sub binmode  { 1; }
sub getc     { return GETC(tied(${$_[0]}) ); }
sub read     { return READ(     tied(${$_[0]}), @_[1,2,3] ); }
sub readline { return READLINE( tied(${$_[0]}) ); }
sub getline  { return READLINE( tied(${$_[0]}) ); }
sub close    { return CLOSE(tied(${$_[0]}) ); }

sub seek {
   my ($self, $ofs, $whence) = @_;
   $self = tied( $$self );

   $self->{CRPOS} = $ofs if ($whence == 0);
   $self->{CRPOS}+= $ofs if ($whence == 1);
   $self->{CRPOS} = $self->{LG} + $ofs if ($whence == 2);

   $self->{CRPOS} = 0 if ($self->{CRPOS} < 0);
   $self->{CRPOS} = $self->{LG} if ($self->{CRPOS} > $self->{LG});
   return 1;
}

sub tell { 
    return tied(${$_[0]})->{CRPOS}; 
}

sub WRITE  { 
    die "inner files can only open for reading\n";
}

sub PRINT  {
    die "inner files can only open for reading\n";
}

sub PRINTF { 
    die "inner files can only open for reading\n";
}

sub GETC   { 
    my ($self) = @_;
    return 0 if ($self->{CRPOS} >= $self->{LG});

    my $data;

    ### Save and seek...
    my $old_pos = $self->{FH}->tell;
    $self->{FH}->seek($self->{CRPOS}+$self->{START}, 0);

    ### ...read...
    my $lg = $self->{FH}->read($data, 1);
    $self->{CRPOS} += $lg;

    ### ...and restore:
    $self->{FH}->seek($old_pos, 0);

    $self->{LG} = $self->{CRPOS} unless ($lg); 
    return ($lg ? $data : undef);
}

sub READ   { 
    my ($self, $undefined, $lg, $ofs) = @_;
    $undefined = undef;

    return 0 if ($self->{CRPOS} >= $self->{LG});
    $lg = $self->{LG} - $self->{CRPOS} if ($self->{CRPOS} + $lg > $self->{LG});
    return 0 unless ($lg);

    ### Save and seek...
    my $old_pos = $self->{FH}->tell;
    $self->{FH}->seek($self->{CRPOS}+$self->{START}, 0);

    ### ...read...
    $lg = $self->{FH}->read($_[1], $lg, $_[3] );
    $self->{CRPOS} += $lg;

    ### ...and restore:
    $self->{FH}->seek($old_pos, 0);

    $self->{LG} = $self->{CRPOS} unless ($lg); 
    return $lg;
}

sub READLINE { 
    my ($self) = @_;
    return undef if ($self->{CRPOS} >= $self->{LG});

    ### Save and seek...
    my $old_pos = $self->{FH}->tell;
    $self->{FH}->seek($self->{CRPOS}+$self->{START}, 0);

    ### ...read...
    my $text = $self->{FH}->getline;

    ### ...and restore:
    $self->{FH}->seek($old_pos, 0);

    #### If we detected a new EOF ...
    unless (defined $text) {  
       $self->{LG} = $self->{CRPOS};
       return undef;
    }

    my $lg=length($text);

    $lg = $self->{LG} - $self->{CRPOS} if ($self->{CRPOS} + $lg > $self->{LG});
    $self->{CRPOS} += $lg;

    return substr($text, 0,$lg);
}

sub CLOSE { %{$_[0]}=(); }



1;
__END__

=back


=head1 VERSION

$Id: InnerFile.pm,v 1.4 2005/02/10 21:21:53 dfs Exp $


=head1 AUTHOR

Original version by Doru Petrescu (pdoru@kappa.ro).

Documentation and by Eryq (eryq@zeegee.com).

Currently maintained by David F. Skoll (dfs@roaringpenguin.com).

=cut


                                                                                             usr/lib/perl5/site_perl/5.8.8/IO/Lines.pm                                                           0000444 0000000 0000000 00000010334 10202750161 016540  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   package IO::Lines;


=head1 NAME

IO::Lines - IO:: interface for reading/writing an array of lines


=head1 SYNOPSIS

    use IO::Lines;

    ### See IO::ScalarArray for details 


=head1 DESCRIPTION

This class implements objects which behave just like FileHandle
(or IO::Handle) objects, except that you may use them to write to
(or read from) an array of lines.  They can be tiehandle'd as well.  

This is a subclass of L<IO::ScalarArray|IO::ScalarArray> 
in which the underlying
array has its data stored in a line-oriented-format: that is,
every element ends in a C<"\n">, with the possible exception of the
final element.  This makes C<getline()> I<much> more efficient;
if you plan to do line-oriented reading/printing, you want this class.

The C<print()> method will enforce this rule, so you can print
arbitrary data to the line-array: it will break the data at
newlines appropriately.

See L<IO::ScalarArray> for full usage and warnings.

=cut

use Carp;
use strict;
use IO::ScalarArray;
use vars qw($VERSION @ISA);

# The package version, both in 1.23 style *and* usable by MakeMaker:
$VERSION = "2.110";

# Inheritance:
@ISA = qw(IO::ScalarArray);     ### also gets us new_tie  :-)


#------------------------------
#
# getline
#
# Instance method, override.
# Return the next line, or undef on end of data.
# Can safely be called in an array context.
# Currently, lines are delimited by "\n".
#
sub getline {
    my $self = shift;

    if (!defined $/) {
	return join( '', $self->_getlines_for_newlines );
    }
    elsif ($/ eq "\n") {
	if (!*$self->{Pos}) {      ### full line...
	    return *$self->{AR}[*$self->{Str}++];
	}
	else {                     ### partial line...
	    my $partial = substr(*$self->{AR}[*$self->{Str}++], *$self->{Pos});
	    *$self->{Pos} = 0;
	    return $partial;
	}
    }
    else {
	croak 'unsupported $/: must be "\n" or undef';
    }
}

#------------------------------
#
# getlines
#
# Instance method, override.
# Return an array comprised of the remaining lines, or () on end of data.
# Must be called in an array context.
# Currently, lines are delimited by "\n".
#
sub getlines {
    my $self = shift;
    wantarray or croak("can't call getlines in scalar context!");

    if ((defined $/) and ($/ eq "\n")) {
	return $self->_getlines_for_newlines(@_);
    }
    else {         ### slow but steady
	return $self->SUPER::getlines(@_);
    }
}

#------------------------------
#
# _getlines_for_newlines
#
# Instance method, private.
# If $/ is newline, do fast getlines.
# This CAN NOT invoke getline!
#
sub _getlines_for_newlines {
    my $self = shift;
    my ($rArray, $Str, $Pos) = @{*$self}{ qw( AR Str Pos ) };
    my @partial = ();

    if ($Pos) {				### partial line...
	@partial = (substr( $rArray->[ $Str++ ], $Pos ));
	*$self->{Pos} = 0;
    }
    *$self->{Str} = scalar @$rArray;	### about to exhaust @$rArray
    return (@partial,
	    @$rArray[ $Str .. $#$rArray ]);	### remaining full lines...
}

#------------------------------
#
# print ARGS...
#
# Instance method, override.
# Print ARGS to the underlying line array.  
#
sub print {
    if (defined $\ && $\ ne "\n") {
	croak 'unsupported $\: must be "\n" or undef';
    }

    my $self = shift;
    ### print STDERR "\n[[ARRAY WAS...\n", @{*$self->{AR}}, "<<EOF>>\n";
    my @lines = split /^/, join('', @_); @lines or return 1;

    ### Did the previous print not end with a newline?  
    ### If so, append first line:
    if (@{*$self->{AR}} and (*$self->{AR}[-1] !~ /\n\Z/)) {
	*$self->{AR}[-1] .= shift @lines;
    }
    push @{*$self->{AR}}, @lines;       ### add the remainder
    ### print STDERR "\n[[ARRAY IS NOW...\n", @{*$self->{AR}}, "<<EOF>>\n";
    1;
}

#------------------------------
1;

__END__


=head1 VERSION

$Id: Lines.pm,v 1.3 2005/02/10 21:21:53 dfs Exp $


=head1 AUTHORS


=head2 Primary Maintainer

David F. Skoll (F<dfs@roaringpenguin.com>).

=head2 Principal author

Eryq (F<eryq@zeegee.com>).
President, ZeeGee Software Inc (F<http://www.zeegee.com>).


=head2 Other contributors 

Thanks to the following individuals for their invaluable contributions
(if I've forgotten or misspelled your name, please email me!):

I<Morris M. Siegel,>
for his $/ patch and the new C<getlines()>.

I<Doug Wilson,>
for the IO::Handle inheritance and automatic tie-ing.

=cut

                                                                                                                                                                                                                                                                                                    usr/lib/perl5/site_perl/5.8.8/i486-linux-ld/auto/IO/Stringy/.packlist                               0000644 0000000 0000000 00000001216 10436506535 023614  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   /usr/lib/perl5/site_perl/5.8.8/IO/AtomicFile.pm
/usr/lib/perl5/site_perl/5.8.8/IO/InnerFile.pm
/usr/lib/perl5/site_perl/5.8.8/IO/Lines.pm
/usr/lib/perl5/site_perl/5.8.8/IO/Scalar.pm
/usr/lib/perl5/site_perl/5.8.8/IO/ScalarArray.pm
/usr/lib/perl5/site_perl/5.8.8/IO/Stringy.pm
/usr/lib/perl5/site_perl/5.8.8/IO/Wrap.pm
/usr/lib/perl5/site_perl/5.8.8/IO/WrapTie.pm
/usr/share/man/man3/IO::AtomicFile.3pm
/usr/share/man/man3/IO::InnerFile.3pm
/usr/share/man/man3/IO::Lines.3pm
/usr/share/man/man3/IO::Scalar.3pm
/usr/share/man/man3/IO::ScalarArray.3pm
/usr/share/man/man3/IO::Stringy.3pm
/usr/share/man/man3/IO::Wrap.3pm
/usr/share/man/man3/IO::WrapTie.3pm
                                                                                                                                                                                                                                                                                                                                                                                  usr/man/man3/IO::Wrap.3pm                                                                           0000444 0000000 0000000 00000016735 10436506517 013765  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::Wrap 3"
.TH IO::Wrap 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO::Wrap \- wrap raw filehandles in IO::Handle interface
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
.Vb 1
\&   use IO::Wrap;
.Ve
.PP
.Vb 5
\&   ### Do stuff with any kind of filehandle (including a bare globref), or 
\&   ### any kind of blessed object that responds to a print() message.
\&   ###
\&   sub do_stuff {
\&       my $fh = shift;
.Ve
.PP
.Vb 2
\&       ### At this point, we have no idea what the user gave us... 
\&       ### a globref? a FileHandle? a scalar filehandle name?
.Ve
.PP
.Vb 1
\&       $fh = wraphandle($fh);
.Ve
.PP
.Vb 1
\&       ### At this point, we know we have an IO::Handle-like object!
.Ve
.PP
.Vb 3
\&       $fh->print("Hey there!");
\&       ...
\&   }
.Ve
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
Let's say you want to write some code which does I/O, but you don't 
want to force the caller to provide you with a FileHandle or IO::Handle
object.  You want them to be able to say:
.PP
.Vb 4
\&    do_stuff(\e*STDOUT);
\&    do_stuff('STDERR');
\&    do_stuff($some_FileHandle_object);
\&    do_stuff($some_IO_Handle_object);
.Ve
.PP
And even:
.PP
.Vb 1
\&    do_stuff($any_object_with_a_print_method);
.Ve
.PP
Sure, one way to do it is to force the caller to use \fItiehandle()\fR.  
But that puts the burden on them.  Another way to do it is to 
use \fBIO::Wrap\fR, which provides you with the following functions:
.IP "wraphandle \s-1SCALAR\s0" 4
.IX Item "wraphandle SCALAR"
This function will take a single argument, and \*(L"wrap\*(R" it based on
what it seems to be...
.RS 4
.IP "*" 4
\&\fBA raw scalar filehandle name,\fR like \f(CW"STDOUT"\fR or \f(CW"Class::HANDLE"\fR.
In this case, the filehandle name is wrapped in an IO::Wrap object, 
which is returned.
.IP "*" 4
\&\fBA raw filehandle glob,\fR like \f(CW\*(C`\e*STDOUT\*(C'\fR.
In this case, the filehandle glob is wrapped in an IO::Wrap object, 
which is returned.
.IP "*" 4
\&\fBA blessed FileHandle object.\fR
In this case, the FileHandle is wrapped in an IO::Wrap object if and only
if your FileHandle class does not support the \f(CW\*(C`read()\*(C'\fR method.
.IP "*" 4
\&\fBAny other kind of blessed object,\fR which is assumed to be already
conformant to the IO::Handle interface.
In this case, you just get back that object.
.RE
.RS 4
.RE
.PP
If you get back an IO::Wrap object, it will obey a basic subset of
the \s-1IO::\s0 interface.  That is, the following methods (note: I said
\&\fImethods\fR, not named operators) should work on the thing you get back:
.PP
.Vb 7
\&    close 
\&    getline 
\&    getlines 
\&    print ARGS...
\&    read BUFFER,NBYTES
\&    seek POS,WHENCE
\&    tell
.Ve
.SH "NOTES"
.IX Header "NOTES"
Clearly, when wrapping a raw external filehandle (like \e*STDOUT), 
I didn't want to close the file descriptor when the \*(L"wrapper\*(R" object is
destroyed... since the user might not appreciate that!  Hence,
there's no \s-1DESTROY\s0 method in this class.
.PP
When wrapping a FileHandle object, however, I believe that Perl will 
invoke the FileHandle::DESTROY when the last reference goes away,
so in that case, the filehandle is closed if the wrapped FileHandle
really was the last reference to it.
.SH "WARNINGS"
.IX Header "WARNINGS"
This module does not allow you to wrap filehandle names which are given
as strings that lack the package they were opened in. That is, if a user 
opens \s-1FOO\s0 in package Foo, they must pass it to you either as \f(CW\*(C`\e*FOO\*(C'\fR 
or as \f(CW"Foo::FOO"\fR.  However, \f(CW"STDIN"\fR and friends will work just fine.
.SH "VERSION"
.IX Header "VERSION"
$Id: Wrap.pm,v 1.2 2005/02/10 21:21:53 dfs Exp $
.SH "AUTHOR"
.IX Header "AUTHOR"
.RE
.IP "Primary Maintainer"
.IX Item "Primary Maintainer"
David F. Skoll (\fIdfs@roaringpenguin.com\fR).
.RE
.IP "Original Author"
.IX Item "Original Author"
Eryq (\fIeryq@zeegee.com\fR).
President, ZeeGee Software Inc (\fIhttp://www.zeegee.com\fR).
                                   usr/man/man3/IO::AtomicFile.3pm                                                                     0000444 0000000 0000000 00000013605 10436506517 015061  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::AtomicFile 3"
.TH IO::AtomicFile 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO::AtomicFile \- write a file which is updated atomically
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
.Vb 1
\&    use IO::AtomicFile;
.Ve
.PP
.Vb 4
\&    ### Write a temp file, and have it install itself when closed:
\&    my $FH = IO::AtomicFile->open("bar.dat", "w");
\&    print $FH "Hello!\en";
\&    $FH->close || die "couldn't install atomic file: $!";
.Ve
.PP
.Vb 4
\&    ### Write a temp file, but delete it before it gets installed:
\&    my $FH = IO::AtomicFile->open("bar.dat", "w");
\&    print $FH "Hello!\en";
\&    $FH->delete;
.Ve
.PP
.Vb 4
\&    ### Write a temp file, but neither install it nor delete it:
\&    my $FH = IO::AtomicFile->open("bar.dat", "w");
\&    print $FH "Hello!\en";
\&    $FH->detach;
.Ve
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
This module is intended for people who need to update files 
reliably in the face of unexpected program termination.  
.PP
For example, you generally don't want to be halfway in the middle of
writing \fI/etc/passwd\fR and have your program terminate!  Even
the act of writing a single scalar to a filehandle is \fInot\fR atomic.
.PP
But this module gives you true atomic updates, via \fIrename()\fR.
When you open a file \fI/foo/bar.dat\fR via this module, you are \fIactually\fR 
opening a temporary file \fI/foo/bar.dat..TMP\fR, and writing your
output there.   The act of closing this file (either explicitly
via \fIclose()\fR, or implicitly via the destruction of the object)
will cause \fIrename()\fR to be called... therefore, from the point
of view of the outside world, the file's contents are updated
in a single time quantum.
.PP
To ensure that problems do not go undetected, the \*(L"close\*(R" method
done by the destructor will raise a fatal exception if the \fIrename()\fR
fails.  The explicit \fIclose()\fR just returns undef.   
.PP
You can also decide at any point to trash the file you've been 
building. 
.SH "AUTHOR"
.IX Header "AUTHOR"
.Sh "Primary Maintainer"
.IX Subsection "Primary Maintainer"
David F. Skoll (\fIdfs@roaringpenguin.com\fR).
.Sh "Original Author"
.IX Subsection "Original Author"
Eryq (\fIeryq@zeegee.com\fR).
President, ZeeGee Software Inc (\fIhttp://www.zeegee.com\fR).
.SH "REVISION"
.IX Header "REVISION"
$Revision: 1.2 $
                                                                                                                           usr/man/man3/IO::Scalar.3pm                                                                         0000444 0000000 0000000 00000034275 10436506517 014260  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::Scalar 3"
.TH IO::Scalar 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO::Scalar \- IO:: interface for reading/writing a scalar
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
Perform I/O on strings, using the basic \s-1OO\s0 interface...
.PP
.Vb 3
\&    use 5.005;
\&    use IO::Scalar;
\&    $data = "My message:\en";
.Ve
.PP
.Vb 5
\&    ### Open a handle on a string, and append to it:
\&    $SH = new IO::Scalar \e$data;
\&    $SH->print("Hello");
\&    $SH->print(", world!\enBye now!\en");
\&    print "The string is now: ", $data, "\en";
.Ve
.PP
.Vb 6
\&    ### Open a handle on a string, read it line-by-line, then close it:
\&    $SH = new IO::Scalar \e$data;
\&    while (defined($_ = $SH->getline)) {
\&        print "Got line: $_";
\&    }
\&    $SH->close;
.Ve
.PP
.Vb 3
\&    ### Open a handle on a string, and slurp in all the lines:
\&    $SH = new IO::Scalar \e$data;
\&    print "All lines:\en", $SH->getlines;
.Ve
.PP
.Vb 3
\&    ### Get the current position (either of two ways):
\&    $pos = $SH->getpos;
\&    $offset = $SH->tell;
.Ve
.PP
.Vb 3
\&    ### Set the current position (either of two ways):
\&    $SH->setpos($pos);
\&    $SH->seek($offset, 0);
.Ve
.PP
.Vb 4
\&    ### Open an anonymous temporary scalar:
\&    $SH = new IO::Scalar;
\&    $SH->print("Hi there!");
\&    print "I printed: ", ${$SH->sref}, "\en";      ### get at value
.Ve
.PP
Don't like \s-1OO\s0 for your I/O?  No problem.
Thanks to the magic of an invisible \fItie()\fR, the following now
works out of the box, just as it does with IO::Handle:
.PP
.Vb 3
\&    use 5.005;
\&    use IO::Scalar;
\&    $data = "My message:\en";
.Ve
.PP
.Vb 5
\&    ### Open a handle on a string, and append to it:
\&    $SH = new IO::Scalar \e$data;
\&    print $SH "Hello";
\&    print $SH ", world!\enBye now!\en";
\&    print "The string is now: ", $data, "\en";
.Ve
.PP
.Vb 6
\&    ### Open a handle on a string, read it line-by-line, then close it:
\&    $SH = new IO::Scalar \e$data;
\&    while (<$SH>) {
\&        print "Got line: $_";
\&    }
\&    close $SH;
.Ve
.PP
.Vb 3
\&    ### Open a handle on a string, and slurp in all the lines:
\&    $SH = new IO::Scalar \e$data;
\&    print "All lines:\en", <$SH>;
.Ve
.PP
.Vb 2
\&    ### Get the current position (WARNING: requires 5.6):
\&    $offset = tell $SH;
.Ve
.PP
.Vb 2
\&    ### Set the current position (WARNING: requires 5.6):
\&    seek $SH, $offset, 0;
.Ve
.PP
.Vb 4
\&    ### Open an anonymous temporary scalar:
\&    $SH = new IO::Scalar;
\&    print $SH "Hi there!";
\&    print "I printed: ", ${$SH->sref}, "\en";      ### get at value
.Ve
.PP
And for you folks with 1.x code out there: the old \fItie()\fR style still works,
though this is \fIunnecessary and deprecated\fR:
.PP
.Vb 1
\&    use IO::Scalar;
.Ve
.PP
.Vb 5
\&    ### Writing to a scalar...
\&    my $s;
\&    tie *OUT, 'IO::Scalar', \e$s;
\&    print OUT "line 1\enline 2\en", "line 3\en";
\&    print "String is now: $s\en"
.Ve
.PP
.Vb 7
\&    ### Reading and writing an anonymous scalar...
\&    tie *OUT, 'IO::Scalar';
\&    print OUT "line 1\enline 2\en", "line 3\en";
\&    tied(OUT)->seek(0,0);
\&    while (<OUT>) {
\&        print "Got line: ", $_;
\&    }
.Ve
.PP
Stringification works, too!
.PP
.Vb 4
\&    my $SH = new IO::Scalar \e$data;
\&    print $SH "Hello, ";
\&    print $SH "world!";
\&    print "I printed: $SH\en";
.Ve
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
This class is part of the IO::Stringy distribution;
see IO::Stringy for change log and general information.
.PP
The IO::Scalar class implements objects which behave just like
IO::Handle (or FileHandle) objects, except that you may use them
to write to (or read from) scalars.  These handles are
automatically tiehandle'd (though please see \*(L"\s-1WARNINGS\s0\*(R"
for information relevant to your Perl version).
.PP
Basically, this:
.PP
.Vb 4
\&    my $s;
\&    $SH = new IO::Scalar \e$s;
\&    $SH->print("Hel", "lo, ");         ### OO style
\&    $SH->print("world!\en");            ### ditto
.Ve
.PP
Or this:
.PP
.Vb 4
\&    my $s;
\&    $SH = tie *OUT, 'IO::Scalar', \e$s;
\&    print OUT "Hel", "lo, ";           ### non-OO style
\&    print OUT "world!\en";              ### ditto
.Ve
.PP
Causes \f(CW$s\fR to be set to:
.PP
.Vb 1
\&    "Hello, world!\en"
.Ve
.SH "PUBLIC INTERFACE"
.IX Header "PUBLIC INTERFACE"
.Sh "Construction"
.IX Subsection "Construction"
.IP "new [\s-1ARGS\s0...]" 4
.IX Item "new [ARGS...]"
\&\fIClass method.\fR
Return a new, unattached scalar handle.
If any arguments are given, they're sent to \fIopen()\fR.
.IP "open [\s-1SCALARREF\s0]" 4
.IX Item "open [SCALARREF]"
\&\fIInstance method.\fR
Open the scalar handle on a new scalar, pointed to by \s-1SCALARREF\s0.
If no \s-1SCALARREF\s0 is given, a \*(L"private\*(R" scalar is created to hold
the file data.
.Sp
Returns the self object on success, undefined on error.
.IP "opened" 4
.IX Item "opened"
\&\fIInstance method.\fR
Is the scalar handle opened on something?
.IP "close" 4
.IX Item "close"
\&\fIInstance method.\fR
Disassociate the scalar handle from its underlying scalar.
Done automatically on destroy.
.Sh "Input and output"
.IX Subsection "Input and output"
.IP "flush" 4
.IX Item "flush"
\&\fIInstance method.\fR
No\-op, provided for \s-1OO\s0 compatibility.
.IP "getc" 4
.IX Item "getc"
\&\fIInstance method.\fR
Return the next character, or undef if none remain.
.IP "getline" 4
.IX Item "getline"
\&\fIInstance method.\fR
Return the next line, or undef on end of string.
Can safely be called in an array context.
Currently, lines are delimited by \*(L"\en\*(R".
.IP "getlines" 4
.IX Item "getlines"
\&\fIInstance method.\fR
Get all remaining lines.
It will \fIcroak()\fR if accidentally called in a scalar context.
.IP "print \s-1ARGS\s0..." 4
.IX Item "print ARGS..."
\&\fIInstance method.\fR
Print \s-1ARGS\s0 to the underlying scalar.
.Sp
\&\fBWarning:\fR this continues to always cause a seek to the end
of the string, but if you perform \fIseek()\fRs and \fItell()\fRs, it is
still safer to explicitly seek-to-end before subsequent \fIprint()\fRs.
.IP "read \s-1BUF\s0, \s-1NBYTES\s0, [\s-1OFFSET\s0]" 4
.IX Item "read BUF, NBYTES, [OFFSET]"
\&\fIInstance method.\fR
Read some bytes from the scalar.
Returns the number of bytes actually read, 0 on end\-of\-file, undef on error.
.IP "write \s-1BUF\s0, \s-1NBYTES\s0, [\s-1OFFSET\s0]" 4
.IX Item "write BUF, NBYTES, [OFFSET]"
\&\fIInstance method.\fR
Write some bytes to the scalar.
.IP "sysread \s-1BUF\s0, \s-1LEN\s0, [\s-1OFFSET\s0]" 4
.IX Item "sysread BUF, LEN, [OFFSET]"
\&\fIInstance method.\fR
Read some bytes from the scalar.
Returns the number of bytes actually read, 0 on end\-of\-file, undef on error.
.IP "syswrite \s-1BUF\s0, \s-1NBYTES\s0, [\s-1OFFSET\s0]" 4
.IX Item "syswrite BUF, NBYTES, [OFFSET]"
\&\fIInstance method.\fR
Write some bytes to the scalar.
.Sh "Seeking/telling and other attributes"
.IX Subsection "Seeking/telling and other attributes"
.IP "autoflush" 4
.IX Item "autoflush"
\&\fIInstance method.\fR
No\-op, provided for \s-1OO\s0 compatibility.
.IP "binmode" 4
.IX Item "binmode"
\&\fIInstance method.\fR
No\-op, provided for \s-1OO\s0 compatibility.
.IP "clearerr" 4
.IX Item "clearerr"
\&\fIInstance method.\fR  Clear the error and \s-1EOF\s0 flags.  A no\-op.
.IP "eof" 4
.IX Item "eof"
\&\fIInstance method.\fR  Are we at end of file?
.IP "seek \s-1OFFSET\s0, \s-1WHENCE\s0" 4
.IX Item "seek OFFSET, WHENCE"
\&\fIInstance method.\fR  Seek to a given position in the stream.
.IP "sysseek \s-1OFFSET\s0, \s-1WHENCE\s0" 4
.IX Item "sysseek OFFSET, WHENCE"
\&\fIInstance method.\fR Identical to \f(CW\*(C`seek OFFSET, WHENCE\*(C'\fR, \fIq.v.\fR
.IP "tell" 4
.IX Item "tell"
\&\fIInstance method.\fR
Return the current position in the stream, as a numeric offset.
.IP "setpos \s-1POS\s0" 4
.IX Item "setpos POS"
\&\fIInstance method.\fR
Set the current position, using the opaque value returned by \f(CW\*(C`getpos()\*(C'\fR.
.IP "getpos" 4
.IX Item "getpos"
\&\fIInstance method.\fR
Return the current position in the string, as an opaque object.
.IP "sref" 4
.IX Item "sref"
\&\fIInstance method.\fR
Return a reference to the underlying scalar.
.SH "WARNINGS"
.IX Header "WARNINGS"
Perl's \s-1TIEHANDLE\s0 spec was incomplete prior to 5.005_57;
it was missing support for \f(CW\*(C`seek()\*(C'\fR, \f(CW\*(C`tell()\*(C'\fR, and \f(CW\*(C`eof()\*(C'\fR.
Attempting to use these functions with an IO::Scalar will not work
prior to 5.005_57. IO::Scalar will not have the relevant methods
invoked; and even worse, this kind of bug can lie dormant for a while.
If you turn warnings on (via \f(CW$^W\fR or \f(CW\*(C`perl \-w\*(C'\fR),
and you see something like this...
.PP
.Vb 1
\&    attempt to seek on unopened filehandle
.Ve
.PP
\&...then you are probably trying to use one of these functions
on an IO::Scalar with an old Perl.  The remedy is to simply
use the \s-1OO\s0 version; e.g.:
.PP
.Vb 2
\&    $SH->seek(0,0);    ### GOOD: will work on any 5.005
\&    seek($SH,0,0);     ### WARNING: will only work on 5.005_57 and beyond
.Ve
.SH "VERSION"
.IX Header "VERSION"
$Id: Scalar.pm,v 1.6 2005/02/10 21:21:53 dfs Exp $
.SH "AUTHORS"
.IX Header "AUTHORS"
.Sh "Primary Maintainer"
.IX Subsection "Primary Maintainer"
David F. Skoll (\fIdfs@roaringpenguin.com\fR).
.Sh "Principal author"
.IX Subsection "Principal author"
Eryq (\fIeryq@zeegee.com\fR).
President, ZeeGee Software Inc (\fIhttp://www.zeegee.com\fR).
.Sh "Other contributors"
.IX Subsection "Other contributors"
The full set of contributors always includes the folks mentioned
in \*(L"\s-1CHANGE\s0 \s-1LOG\s0\*(R" in IO::Stringy.  But just the same, special
thanks to the following individuals for their invaluable contributions
(if I've forgotten or misspelled your name, please email me!):
.PP
\&\fIAndy Glew,\fR
for contributing \f(CW\*(C`getc()\*(C'\fR.
.PP
\&\fIBrandon Browning,\fR
for suggesting \f(CW\*(C`opened()\*(C'\fR.
.PP
\&\fIDavid Richter,\fR
for finding and fixing the bug in \f(CW\*(C`PRINTF()\*(C'\fR.
.PP
\&\fIEric L. Brine,\fR
for his offset-using \fIread()\fR and \fIwrite()\fR implementations.
.PP
\&\fIRichard Jones,\fR
for his patches to massively improve the performance of \f(CW\*(C`getline()\*(C'\fR
and add \f(CW\*(C`sysread\*(C'\fR and \f(CW\*(C`syswrite\*(C'\fR.
.PP
\&\fIB. K. Oxley (binkley),\fR
for stringification and inheritance improvements,
and sundry good ideas.
.PP
\&\fIDoug Wilson,\fR
for the IO::Handle inheritance and automatic tie\-ing.
.SH "SEE ALSO"
.IX Header "SEE ALSO"
IO::String, which is quite similar but which was designed
more-recently and with an IO::Handle\-like interface in mind,
so you could mix \s-1OO\-\s0 and native-filehandle usage without using \fItied()\fR.
.PP
\&\fINote:\fR as of version 2.x, these classes all work like
their IO::Handle counterparts, so we have comparable
functionality to IO::String.
                                                                                                                                                                                                                                                                                                                                   usr/man/man3/IO::ScalarArray.3pm                                                                    0000444 0000000 0000000 00000032303 10436506517 015245  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::ScalarArray 3"
.TH IO::ScalarArray 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO::ScalarArray \- IO:: interface for reading/writing an array of scalars
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
Perform I/O on strings, using the basic \s-1OO\s0 interface...
.PP
.Vb 2
\&    use IO::ScalarArray;
\&    @data = ("My mes", "sage:\en");
.Ve
.PP
.Vb 5
\&    ### Open a handle on an array, and append to it:
\&    $AH = new IO::ScalarArray \e@data;
\&    $AH->print("Hello");       
\&    $AH->print(", world!\enBye now!\en");  
\&    print "The array is now: ", @data, "\en";
.Ve
.PP
.Vb 6
\&    ### Open a handle on an array, read it line-by-line, then close it:
\&    $AH = new IO::ScalarArray \e@data;
\&    while (defined($_ = $AH->getline)) { 
\&        print "Got line: $_";
\&    }
\&    $AH->close;
.Ve
.PP
.Vb 3
\&    ### Open a handle on an array, and slurp in all the lines:
\&    $AH = new IO::ScalarArray \e@data;
\&    print "All lines:\en", $AH->getlines;
.Ve
.PP
.Vb 3
\&    ### Get the current position (either of two ways):
\&    $pos = $AH->getpos;         
\&    $offset = $AH->tell;
.Ve
.PP
.Vb 3
\&    ### Set the current position (either of two ways):
\&    $AH->setpos($pos);        
\&    $AH->seek($offset, 0);
.Ve
.PP
.Vb 4
\&    ### Open an anonymous temporary array:
\&    $AH = new IO::ScalarArray;
\&    $AH->print("Hi there!");
\&    print "I printed: ", @{$AH->aref}, "\en";      ### get at value
.Ve
.PP
Don't like \s-1OO\s0 for your I/O?  No problem.  
Thanks to the magic of an invisible \fItie()\fR, the following now 
works out of the box, just as it does with IO::Handle:
.PP
.Vb 2
\&    use IO::ScalarArray;
\&    @data = ("My mes", "sage:\en");
.Ve
.PP
.Vb 5
\&    ### Open a handle on an array, and append to it:
\&    $AH = new IO::ScalarArray \e@data;
\&    print $AH "Hello";    
\&    print $AH ", world!\enBye now!\en";
\&    print "The array is now: ", @data, "\en";
.Ve
.PP
.Vb 6
\&    ### Open a handle on a string, read it line-by-line, then close it:
\&    $AH = new IO::ScalarArray \e@data;
\&    while (<$AH>) {
\&        print "Got line: $_";
\&    }
\&    close $AH;
.Ve
.PP
.Vb 3
\&    ### Open a handle on a string, and slurp in all the lines:
\&    $AH = new IO::ScalarArray \e@data;
\&    print "All lines:\en", <$AH>;
.Ve
.PP
.Vb 2
\&    ### Get the current position (WARNING: requires 5.6):
\&    $offset = tell $AH;
.Ve
.PP
.Vb 2
\&    ### Set the current position (WARNING: requires 5.6):
\&    seek $AH, $offset, 0;
.Ve
.PP
.Vb 4
\&    ### Open an anonymous temporary scalar:
\&    $AH = new IO::ScalarArray;
\&    print $AH "Hi there!";
\&    print "I printed: ", @{$AH->aref}, "\en";      ### get at value
.Ve
.PP
And for you folks with 1.x code out there: the old \fItie()\fR style still works,
though this is \fIunnecessary and deprecated\fR:
.PP
.Vb 1
\&    use IO::ScalarArray;
.Ve
.PP
.Vb 5
\&    ### Writing to a scalar...
\&    my @a; 
\&    tie *OUT, 'IO::ScalarArray', \e@a;
\&    print OUT "line 1\enline 2\en", "line 3\en";
\&    print "Array is now: ", @a, "\en"
.Ve
.PP
.Vb 7
\&    ### Reading and writing an anonymous scalar... 
\&    tie *OUT, 'IO::ScalarArray';
\&    print OUT "line 1\enline 2\en", "line 3\en";
\&    tied(OUT)->seek(0,0);
\&    while (<OUT>) { 
\&        print "Got line: ", $_;
\&    }
.Ve
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
This class is part of the IO::Stringy distribution;
see IO::Stringy for change log and general information.
.PP
The IO::ScalarArray class implements objects which behave just like 
IO::Handle (or FileHandle) objects, except that you may use them 
to write to (or read from) arrays of scalars.  Logically, an
array of scalars defines an in-core \*(L"file\*(R" whose contents are
the concatenation of the scalars in the array.  The handles created by 
this class are automatically tiehandle'd (though please see \*(L"\s-1WARNINGS\s0\*(R"
for information relevant to your Perl version).
.PP
For writing large amounts of data with individual \fIprint()\fR statements, 
this class is likely to be more efficient than IO::Scalar.
.PP
Basically, this:
.PP
.Vb 4
\&    my @a;
\&    $AH = new IO::ScalarArray \e@a;
\&    $AH->print("Hel", "lo, ");         ### OO style
\&    $AH->print("world!\en");            ### ditto
.Ve
.PP
Or this:
.PP
.Vb 4
\&    my @a;
\&    $AH = new IO::ScalarArray \e@a;
\&    print $AH "Hel", "lo, ";           ### non-OO style
\&    print $AH "world!\en";              ### ditto
.Ve
.PP
Causes \f(CW@a\fR to be set to the following array of 3 strings:
.PP
.Vb 3
\&    ( "Hel" , 
\&      "lo, " , 
\&      "world!\en" )
.Ve
.PP
See IO::Scalar and compare with this class.
.SH "PUBLIC INTERFACE"
.IX Header "PUBLIC INTERFACE"
.Sh "Construction"
.IX Subsection "Construction"
.IP "new [\s-1ARGS\s0...]" 4
.IX Item "new [ARGS...]"
\&\fIClass method.\fR
Return a new, unattached array handle.  
If any arguments are given, they're sent to \fIopen()\fR.
.IP "open [\s-1ARRAYREF\s0]" 4
.IX Item "open [ARRAYREF]"
\&\fIInstance method.\fR
Open the array handle on a new array, pointed to by \s-1ARRAYREF\s0.
If no \s-1ARRAYREF\s0 is given, a \*(L"private\*(R" array is created to hold
the file data.
.Sp
Returns the self object on success, undefined on error.
.IP "opened" 4
.IX Item "opened"
\&\fIInstance method.\fR
Is the array handle opened on something?
.IP "close" 4
.IX Item "close"
\&\fIInstance method.\fR
Disassociate the array handle from its underlying array.
Done automatically on destroy.
.Sh "Input and output"
.IX Subsection "Input and output"
.IP "flush" 4
.IX Item "flush"
\&\fIInstance method.\fR
No\-op, provided for \s-1OO\s0 compatibility.
.IP "getc" 4
.IX Item "getc"
\&\fIInstance method.\fR
Return the next character, or undef if none remain.
This does a \fIread\fR\|(1), which is somewhat costly.
.IP "getline" 4
.IX Item "getline"
\&\fIInstance method.\fR
Return the next line, or undef on end of data.
Can safely be called in an array context.
Currently, lines are delimited by \*(L"\en\*(R".
.IP "getlines" 4
.IX Item "getlines"
\&\fIInstance method.\fR
Get all remaining lines.
It will \fIcroak()\fR if accidentally called in a scalar context.
.IP "print \s-1ARGS\s0..." 4
.IX Item "print ARGS..."
\&\fIInstance method.\fR
Print \s-1ARGS\s0 to the underlying array.  
.Sp
Currently, this always causes a \*(L"seek to the end of the array\*(R"
and generates a new array entry.  This may change in the future.
.IP "read \s-1BUF\s0, \s-1NBYTES\s0, [\s-1OFFSET\s0];" 4
.IX Item "read BUF, NBYTES, [OFFSET];"
\&\fIInstance method.\fR
Read some bytes from the array.
Returns the number of bytes actually read, 0 on end\-of\-file, undef on error.
.IP "write \s-1BUF\s0, \s-1NBYTES\s0, [\s-1OFFSET\s0];" 4
.IX Item "write BUF, NBYTES, [OFFSET];"
\&\fIInstance method.\fR
Write some bytes into the array.
.Sh "Seeking/telling and other attributes"
.IX Subsection "Seeking/telling and other attributes"
.IP "autoflush" 4
.IX Item "autoflush"
\&\fIInstance method.\fR
No\-op, provided for \s-1OO\s0 compatibility.
.IP "binmode" 4
.IX Item "binmode"
\&\fIInstance method.\fR
No\-op, provided for \s-1OO\s0 compatibility.
.IP "clearerr" 4
.IX Item "clearerr"
\&\fIInstance method.\fR  Clear the error and \s-1EOF\s0 flags.  A no\-op.
.IP "eof" 4
.IX Item "eof"
\&\fIInstance method.\fR  Are we at end of file?
.IP "seek \s-1POS\s0,WHENCE" 4
.IX Item "seek POS,WHENCE"
\&\fIInstance method.\fR
Seek to a given position in the stream.
Only a \s-1WHENCE\s0 of 0 (\s-1SEEK_SET\s0) is supported.
.IP "tell" 4
.IX Item "tell"
\&\fIInstance method.\fR
Return the current position in the stream, as a numeric offset.
.IP "setpos \s-1POS\s0" 4
.IX Item "setpos POS"
\&\fIInstance method.\fR
Seek to a given position in the array, using the opaque \fIgetpos()\fR value.
Don't expect this to be a number.
.IP "getpos" 4
.IX Item "getpos"
\&\fIInstance method.\fR
Return the current position in the array, as an opaque value.
Don't expect this to be a number.
.IP "aref" 4
.IX Item "aref"
\&\fIInstance method.\fR
Return a reference to the underlying array.
.SH "WARNINGS"
.IX Header "WARNINGS"
Perl's \s-1TIEHANDLE\s0 spec was incomplete prior to 5.005_57;
it was missing support for \f(CW\*(C`seek()\*(C'\fR, \f(CW\*(C`tell()\*(C'\fR, and \f(CW\*(C`eof()\*(C'\fR.
Attempting to use these functions with an IO::ScalarArray will not work
prior to 5.005_57. IO::ScalarArray will not have the relevant methods 
invoked; and even worse, this kind of bug can lie dormant for a while.
If you turn warnings on (via \f(CW$^W\fR or \f(CW\*(C`perl \-w\*(C'\fR),
and you see something like this...
.PP
.Vb 1
\&    attempt to seek on unopened filehandle
.Ve
.PP
\&...then you are probably trying to use one of these functions
on an IO::ScalarArray with an old Perl.  The remedy is to simply
use the \s-1OO\s0 version; e.g.:
.PP
.Vb 2
\&    $AH->seek(0,0);    ### GOOD: will work on any 5.005
\&    seek($AH,0,0);     ### WARNING: will only work on 5.005_57 and beyond
.Ve
.SH "VERSION"
.IX Header "VERSION"
$Id: ScalarArray.pm,v 1.7 2005/02/10 21:21:53 dfs Exp $
.SH "AUTHOR"
.IX Header "AUTHOR"
.Sh "Primary Maintainer"
.IX Subsection "Primary Maintainer"
David F. Skoll (\fIdfs@roaringpenguin.com\fR).
.Sh "Principal author"
.IX Subsection "Principal author"
Eryq (\fIeryq@zeegee.com\fR).
President, ZeeGee Software Inc (\fIhttp://www.zeegee.com\fR).
.Sh "Other contributors"
.IX Subsection "Other contributors"
Thanks to the following individuals for their invaluable contributions
(if I've forgotten or misspelled your name, please email me!):
.PP
\&\fIAndy Glew,\fR
for suggesting \f(CW\*(C`getc()\*(C'\fR.
.PP
\&\fIBrandon Browning,\fR
for suggesting \f(CW\*(C`opened()\*(C'\fR.
.PP
\&\fIEric L. Brine,\fR
for his offset-using \fIread()\fR and \fIwrite()\fR implementations. 
.PP
\&\fIDoug Wilson,\fR
for the IO::Handle inheritance and automatic tie\-ing.
                                                                                                                                                                                                                                                                                                                             usr/man/man3/IO::Stringy.3pm                                                                        0000444 0000000 0000000 00000042675 10436506517 014515  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::Stringy 3"
.TH IO::Stringy 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO\-stringy \- I/O on in\-core objects like strings and arrays
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
.Vb 7
\&    IO::
\&    ::AtomicFile   adpO  Write a file which is updated atomically     ERYQ
\&    ::Lines        bdpO  I/O handle to read/write to array of lines   ERYQ
\&    ::Scalar       RdpO  I/O handle to read/write to a string         ERYQ
\&    ::ScalarArray  RdpO  I/O handle to read/write to array of scalars ERYQ
\&    ::Wrap         RdpO  Wrap old-style FHs in standard OO interface  ERYQ
\&    ::WrapTie      adpO  Tie your handles & retain full OO interface  ERYQ
.Ve
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
This toolkit primarily provides modules for performing both traditional
and object-oriented i/o) on things \fIother\fR than normal filehandles;
in particular, IO::Scalar, IO::ScalarArray,
and IO::Lines.
.PP
In the more-traditional IO::Handle front, we
have IO::AtomicFile
which may be used to painlessly create files which are updated
atomically.
.PP
And in the \*(L"this\-may\-prove\-useful\*(R" corner, we have IO::Wrap,
whose exported \fIwraphandle()\fR function will clothe anything that's not
a blessed object in an IO::Handle\-like wrapper... so you can just
use \s-1OO\s0 syntax and stop worrying about whether your function's caller
handed you a string, a globref, or a FileHandle.
.SH "WARNINGS"
.IX Header "WARNINGS"
Perl's \s-1TIEHANDLE\s0 spec was incomplete prior to 5.005_57;
it was missing support for \f(CW\*(C`seek()\*(C'\fR, \f(CW\*(C`tell()\*(C'\fR, and \f(CW\*(C`eof()\*(C'\fR.
Attempting to use these functions with an IO::Scalar, IO::ScalarArray,
IO::Lines, etc. \fBwill not work\fR prior to 5.005_57.
None of the relevant methods will be invoked by Perl;
and even worse, this kind of bug can lie dormant for a while.
If you turn warnings on (via \f(CW$^W\fR or \f(CW\*(C`perl \-w\*(C'\fR), and you see
something like this...
.PP
.Vb 1
\&    seek() on unopened file
.Ve
.PP
\&...then you are probably trying to use one of these functions
on one of our \s-1IO::\s0 classes with an old Perl.  The remedy is to simply
use the \s-1OO\s0 version; e.g.:
.PP
.Vb 2
\&    $SH->seek(0,0);    ### GOOD: will work on any 5.005
\&    seek($SH,0,0);     ### WARNING: will only work on 5.005_57 and beyond
.Ve
.SH "INSTALLATION"
.IX Header "INSTALLATION"
.Sh "Requirements"
.IX Subsection "Requirements"
As of version 2.x, this toolkit requires Perl 5.005 for
the IO::Handle subclasses, and 5.005_57 or better is
\&\fBstrongly\fR recommended.  See \*(L"\s-1WARNINGS\s0\*(R" for details.
.Sh "Directions"
.IX Subsection "Directions"
Most of you already know the drill...
.PP
.Vb 4
\&    perl Makefile.PL
\&    make
\&    make test
\&    make install
.Ve
.PP
For everyone else out there...
if you've never installed Perl code before, or you're trying to use
this in an environment where your sysadmin or \s-1ISP\s0 won't let you do
interesting things, \fBrelax:\fR since this module contains no binary
extensions, you can cheat.  That means copying the directory tree
under my \*(L"./lib\*(R" directory into someplace where your script can \*(L"see\*(R"
it.  For example, under Linux:
.PP
.Vb 1
\&    cp -r IO-stringy-1.234/lib/* /path/to/my/perl/
.Ve
.PP
Now, in your Perl code, do this:
.PP
.Vb 2
\&    use lib "/path/to/my/perl";
\&    use IO::Scalar;                   ### or whatever
.Ve
.PP
Ok, now you've been told.  At this point, anyone who whines about
not being given enough information gets an unflattering haiku
written about them in the next change log.  I'll do it.
Don't think I won't.
.SH "VERSION"
.IX Header "VERSION"
$Id: Stringy.pm,v 1.3 2005/02/10 21:24:05 dfs Exp $
.SH "TO DO"
.IX Header "TO DO"
.IP "(2000/08/02)  Finalize $/ support" 4
.IX Item "(2000/08/02)  Finalize $/ support"
Graham Barr submitted this patch half a \fIyear\fR ago;
Like a moron, I lost his message under a ton of others,
and only now have the experimental implementation done.
.Sp
Will the sudden sensitivity to $/ hose anyone out there?
I'm worried, so you have to enable it explicitly in 1.x.
It will be on by default in 2.x, though only IO::Scalar
has been implemented.
.IP "(2001/08/08)  Remove IO::WrapTie from new \s-1IO::\s0 classes" 4
.IX Item "(2001/08/08)  Remove IO::WrapTie from new IO:: classes"
It's not needed.  Backwards compatibility could be maintained
by having \fInew_tie()\fR be identical to \fInew()\fR.  Heck, I'll bet
that IO::WrapTie should be reimplemented so the returned
object is just like an IO::Scalar in its use of globrefs.
.SH "CHANGE LOG"
.IX Header "CHANGE LOG"
.IP "Version 2.110   (2005/02/10)" 4
.IX Item "Version 2.110   (2005/02/10)"
Maintainership taken over by \s-1DSKOLL\s0 <dfs@roaringpenguin.com>
.Sp
Closed the following bugs at
https://rt.cpan.org/NoAuth/Bugs.html?Dist=IO\-stringy:
.IP "*" 4
2208 IO::ScalarArray\->getline does not return undef for \s-1EOF\s0 if undef($/)
.IP "*" 4
7132 IO\-stringy/Makefile.PL bug \- name should be module name
.IP "*" 4
11249 IO::Scalar flush shouldn't return undef
.IP "*" 4
2172 $\e (output record separator) not respected
.IP "*" 4
8605 \fIIO::InnerFile::seek()\fR should return 1 on success
.IP "*" 4
4798 *.html in lib/
.IP "*" 4
4369 Improvement: handling of fixed-size reads in IO::Scalar
.Sp
(Actually, bug 4369 was closed in Version 2.109)
.IP "Version 2.109   (2003/12/21)" 4
.IX Item "Version 2.109   (2003/12/21)"
IO::Scalar::getline now works with ref to int.
\&\fIThanks to Dominique Quatravaux for this patch.\fR
.IP "Version 2.108   (2001/08/20)" 4
.IX Item "Version 2.108   (2001/08/20)"
The terms-of-use have been placed in the distribution file \*(L"\s-1COPYING\s0\*(R".
Also, small documentation tweaks were made.
.IP "Version 2.105   (2001/08/09)" 4
.IX Item "Version 2.105   (2001/08/09)"
Added support for various \fIseek()\fR whences to IO::ScalarArray.
.Sp
Added support for consulting $/ in IO::Scalar and IO::ScalarArray.
The old \f(CW\*(C`use_RS()\*(C'\fR is not even an option.
Unsupported record separators will cause a \fIcroak()\fR.
.Sp
Added a lot of regression tests to supoprt the above.
.Sp
Better on-line docs (hyperlinks to individual functions).
.IP "Version 2.103   (2001/08/08)" 4
.IX Item "Version 2.103   (2001/08/08)"
After sober consideration I have reimplemented \fIIO::Scalar::print()\fR
so that it once again always seeks to the end of the string.
Benchmarks show the new implementation to be just as fast as
Juergen's contributed patch; until someone can convince me otherwise,
the current, safer implementation stays.
.Sp
I thought more about giving IO::Scalar two separate handles,
one for reading and one for writing, as suggested by Binkley.
His points about what \fItell()\fR and \fIeof()\fR return are, I think,
show-stoppers for this feature.  Even the manpages for stdio's \fIfseek()\fR
seem to imply a \fIsingle\fR file position indicator, not two.
So I think I will take this off the \s-1TO\s0 \s-1DO\s0 list.
\&\fBRemedy:\fR you can always have two handles open on the same
scalar, one which you only write to, and one which you only read from.
That should give the same effect.
.IP "Version 2.101   (2001/08/07)" 4
.IX Item "Version 2.101   (2001/08/07)"
\&\fBAlpha release.\fR
This is the initial release of the \*(L"IO::Scalar and friends are
now subclasses of IO::Handle\*(R".  I'm flinging it against the wall.
Please tell me if the banana sticks.  When it does, the banana
will be called 2.2x.
.Sp
First off, \fImany many thanks to Doug Wilson\fR, who
has provided an \fIinvaluable\fR service by patching IO::Scalar
and friends so that they (1) inherit from IO::Handle, (2) automatically
tie themselves so that the \f(CW\*(C`new()\*(C'\fR objects can be used in native i/o
constructs, and (3) doing it so that the whole damn thing passes
its regression tests.  As Doug knows, my globref Kung-Fu was not
up to the task; he graciously provided the patches.  This has earned
him a seat at the Co-Authors table, and the
right to have me address him as \fIsensei\fR.
.Sp
Performance of \fIIO::Scalar::print()\fR has been improved by as much as 2x
for lots of little prints, with the cost of forcing those
who print-then-seek-then-print to explicitly seek to end-of-string
before printing again.
\&\fIThanks to Juergen Zeller for this patch.\fR
.Sp
Added the \s-1COPYING\s0 file, which had been missing from prior versions.
\&\fIThanks to Albert Chin-A-Young for pointing this out.\fR
.Sp
IO::Scalar consults $/ by default (1.x ignored it by default).
Yes, I still need to support IO::ScalarArray.
.IP "Version 1.221   (2001/08/07)" 4
.IX Item "Version 1.221   (2001/08/07)"
I threatened in \*(L"\s-1INSTALLATION\s0\*(R" to write an unflattering haiku
about anyone who whined that I gave them insufficient information...
but it turns out that I left out a crucial direction.  D'\s-1OH\s0!
\&\fIThanks to David Beroff for the \*(L"patch\*(R" and the haiku...\fR
.Sp
.Vb 3
\&       Enough info there?
\&         Here's unflattering haiku:
\&       Forgot the line, "make"!  ;-)
.Ve
.IP "Version 1.220   (2001/04/03)" 4
.IX Item "Version 1.220   (2001/04/03)"
Added untested \s-1SEEK\s0, \s-1TELL\s0, and \s-1EOF\s0 methods to IO::Scalar
and IO::ScalarArray to support corresponding functions for
tied filehandles: untested, because I'm still running 5.00556
and Perl is complaining about \*(L"\fItell()\fR on unopened file\*(R".
\&\fIThanks to Graham Barr for the suggestion.\fR
.Sp
Removed not-fully-blank lines from modules; these were causing
lots of POD-related warnings.
\&\fIThanks to Nicolas Joly for the suggestion.\fR
.IP "Version 1.219   (2001/02/23)" 4
.IX Item "Version 1.219   (2001/02/23)"
IO::Scalar objects can now be made sensitive to $/ .
Pains were taken to keep the fast code fast while adding this feature.
\&\fICheers to Graham Barr for submitting his patch;
jeers to me for losing his email for 6 months.\fR
.IP "Version 1.218   (2001/02/23)" 4
.IX Item "Version 1.218   (2001/02/23)"
IO::Scalar has a new \fIsysseek()\fR method.
\&\fIThanks again to Richard Jones.\fR
.Sp
New \*(L"\s-1TO\s0 \s-1DO\s0\*(R" section, because people who submit patches/ideas should
at least know that they're in the system... and that I won't lose
their stuff.  Please read it.
.Sp
New entries in \*(L"\s-1AUTHOR\s0\*(R".
Please read those too.
.IP "Version 1.216   (2000/09/28)" 4
.IX Item "Version 1.216   (2000/09/28)"
\&\fBIO::Scalar and IO::ScalarArray now inherit from IO::Handle.\fR
I thought I'd remembered a problem with this ages ago, related to
the fact that these \s-1IO::\s0 modules don't have \*(L"real\*(R" filehandles,
but the problem apparently isn't surfacing now.
If you suddenly encounter Perl warnings during global destruction
(especially if you're using tied filehandles), then please let me know!
\&\fIThanks to B. K. Oxley (binkley) for this.\fR
.Sp
\&\fBNasty bug fixed in \f(BIIO::Scalar::write()\fB.\fR
Apparently, the offset and the number-of-bytes arguments were,
for all practical purposes, \fIreversed.\fR  You were okay if
you did all your writing with \fIprint()\fR, but boy was \fIthis\fR a stupid bug!
\&\fIThanks to Richard Jones for finding this one.
For you, Rich, a double-length haiku:\fR
.Sp
.Vb 3
\&       Newspaper headline
\&          typeset by dyslexic man
\&       loses urgency
.Ve
.Sp
.Vb 3
\&       BABY EATS FISH is
\&          simply not equivalent
\&       to FISH EATS BABY
.Ve
.Sp
\&\fBNew sysread and syswrite methods for IO::Scalar.\fR
\&\fIThanks again to Richard Jones for this.\fR
.IP "Version 1.215   (2000/09/05)" 4
.IX Item "Version 1.215   (2000/09/05)"
Added 'bool' overload to '""' overload, so object always evaluates
to true.  (Whew.  Glad I caught this before it went to \s-1CPAN\s0.)
.IP "Version 1.214   (2000/09/03)" 4
.IX Item "Version 1.214   (2000/09/03)"
Evaluating an IO::Scalar in a string context now yields
the underlying string.
\&\fIThanks to B. K. Oxley (binkley) for this.\fR
.IP "Version 1.213   (2000/08/16)" 4
.IX Item "Version 1.213   (2000/08/16)"
Minor documentation fixes.
.IP "Version 1.212   (2000/06/02)" 4
.IX Item "Version 1.212   (2000/06/02)"
Fixed IO::InnerFile incompatibility with Perl5.004.
\&\fIThanks to many folks for reporting this.\fR
.IP "Version 1.210   (2000/04/17)" 4
.IX Item "Version 1.210   (2000/04/17)"
Added \fIflush()\fR and other no-op methods.
\&\fIThanks to Doru Petrescu for suggesting this.\fR
.IP "Version 1.209   (2000/03/17)" 4
.IX Item "Version 1.209   (2000/03/17)"
Small bug fixes.
.IP "Version 1.208   (2000/03/14)" 4
.IX Item "Version 1.208   (2000/03/14)"
Incorporated a number of contributed patches and extensions,
mostly related to speed hacks, support for \*(L"offset\*(R", and
\&\s-1WRITE/CLOSE\s0 methods.
\&\fIThanks to Richard Jones, Doru Petrescu, and many others.\fR
.IP "Version 1.206   (1999/04/18)" 4
.IX Item "Version 1.206   (1999/04/18)"
Added creation of ./testout when Makefile.PL is run.
.IP "Version 1.205   (1999/01/15)" 4
.IX Item "Version 1.205   (1999/01/15)"
Verified for Perl5.005.
.IP "Version 1.202   (1998/04/18)" 4
.IX Item "Version 1.202   (1998/04/18)"
New IO::WrapTie and IO::AtomicFile added.
.IP "Version 1.110" 4
.IX Item "Version 1.110"
Added IO::WrapTie.
.IP "Version 1.107" 4
.IX Item "Version 1.107"
Added IO::Lines, and made some bug fixes to IO::ScalarArray.
Also, added \fIgetc()\fR.
.IP "Version 1.105" 4
.IX Item "Version 1.105"
No real changes; just upgraded IO::Wrap to have a \f(CW$VERSION\fR string.
.SH "AUTHOR"
.IX Header "AUTHOR"
.IP "Primary Maintainer" 4
.IX Item "Primary Maintainer"
David F. Skoll (\fIdfs@roaringpenguin.com\fR).
.IP "Original Author" 4
.IX Item "Original Author"
Eryq (\fIeryq@zeegee.com\fR).
President, ZeeGee Software Inc (\fIhttp://www.zeegee.com\fR).
.IP "Co-Authors" 4
.IX Item "Co-Authors"
For all their bug reports and patch submissions, the following
are officially recognized:
.Sp
.Vb 4
\&     Richard Jones
\&     B. K. Oxley (binkley)
\&     Doru Petrescu
\&     Doug Wilson (for picking up the ball I dropped, and doing tie() right)
.Ve
.PP
Go to \fIhttp://www.zeegee.com\fR for the latest downloads
and on-line documentation for this module.
.PP
Enjoy.  Yell if it breaks.
                                                                   usr/man/man3/IO::InnerFile.3pm                                                                      0000444 0000000 0000000 00000013757 10436506517 014730  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::InnerFile 3"
.TH IO::InnerFile 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO::InnerFile \- define a file inside another file
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
.Vb 5
\&    ### Read a subset of a file:
\&    $inner = IO::InnerFile->new($fh, $start, $length);
\&    while (<$inner>) {
\&        ...
\&    }
.Ve
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
If you have a filehandle that can \fIseek()\fR and \fItell()\fR, then you 
can open an IO::InnerFile on a range of the underlying file.
.SH "PUBLIC INTERFACE"
.IX Header "PUBLIC INTERFACE"
.IP "new \s-1FILEHANDLE\s0, [\s-1START\s0, [\s-1LENGTH\s0]]" 4
.IX Item "new FILEHANDLE, [START, [LENGTH]]"
\&\fIClass method, constructor.\fR
Create a new inner-file opened on the given \s-1FILEHANDLE\s0,
from bytes \s-1START\s0 to \s-1START+LENGTH\s0.  Both \s-1START\s0 and \s-1LENGTH\s0
default to 0; negative values are silently coerced to zero.
.Sp
Note that \s-1FILEHANDLE\s0 must be able to \fIseek()\fR and \fItell()\fR, in addition
to whatever other methods you may desire for reading it.
.IP "set_length \s-1LENGTH\s0" 4
.IX Item "set_length LENGTH"
.PD 0
.IP "get_length" 4
.IX Item "get_length"
.IP "add_length \s-1NBYTES\s0" 4
.IX Item "add_length NBYTES"
.PD
\&\fIInstance methods.\fR
Get/set the virtual length of the inner file.
.IP "set_start \s-1START\s0" 4
.IX Item "set_start START"
.PD 0
.IP "get_start" 4
.IX Item "get_start"
.IP "add_start \s-1NBYTES\s0" 4
.IX Item "add_start NBYTES"
.PD
\&\fIInstance methods.\fR
Get/set the virtual start position of the inner file.
.IP "binmode" 4
.IX Item "binmode"
.PD 0
.IP "close" 4
.IX Item "close"
.IP "flush" 4
.IX Item "flush"
.IP "getc" 4
.IX Item "getc"
.IP "getline" 4
.IX Item "getline"
.IP "print \s-1LIST\s0" 4
.IX Item "print LIST"
.IP "printf \s-1LIST\s0" 4
.IX Item "printf LIST"
.IP "read \s-1BUF\s0, \s-1NBYTES\s0" 4
.IX Item "read BUF, NBYTES"
.IP "readline" 4
.IX Item "readline"
.IP "seek \s-1OFFFSET\s0, \s-1WHENCE\s0" 4
.IX Item "seek OFFFSET, WHENCE"
.IP "tell" 4
.IX Item "tell"
.IP "write \s-1ARGS\s0..." 4
.IX Item "write ARGS..."
.PD
\&\fIInstance methods.\fR
Standard filehandle methods.
.SH "VERSION"
.IX Header "VERSION"
$Id: InnerFile.pm,v 1.4 2005/02/10 21:21:53 dfs Exp $
.SH "AUTHOR"
.IX Header "AUTHOR"
Original version by Doru Petrescu (pdoru@kappa.ro).
.PP
Documentation and by Eryq (eryq@zeegee.com).
.PP
Currently maintained by David F. Skoll (dfs@roaringpenguin.com).
                 usr/man/man3/IO::Lines.3pm                                                                          0000444 0000000 0000000 00000012633 10436506517 014117  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::Lines 3"
.TH IO::Lines 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO::Lines \- IO:: interface for reading/writing an array of lines
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
.Vb 1
\&    use IO::Lines;
.Ve
.PP
.Vb 1
\&    ### See IO::ScalarArray for details
.Ve
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
This class implements objects which behave just like FileHandle
(or IO::Handle) objects, except that you may use them to write to
(or read from) an array of lines.  They can be tiehandle'd as well.  
.PP
This is a subclass of IO::ScalarArray 
in which the underlying
array has its data stored in a line\-oriented\-format: that is,
every element ends in a \f(CW"\en"\fR, with the possible exception of the
final element.  This makes \f(CW\*(C`getline()\*(C'\fR \fImuch\fR more efficient;
if you plan to do line-oriented reading/printing, you want this class.
.PP
The \f(CW\*(C`print()\*(C'\fR method will enforce this rule, so you can print
arbitrary data to the line\-array: it will break the data at
newlines appropriately.
.PP
See IO::ScalarArray for full usage and warnings.
.SH "VERSION"
.IX Header "VERSION"
$Id: Lines.pm,v 1.3 2005/02/10 21:21:53 dfs Exp $
.SH "AUTHORS"
.IX Header "AUTHORS"
.Sh "Primary Maintainer"
.IX Subsection "Primary Maintainer"
David F. Skoll (\fIdfs@roaringpenguin.com\fR).
.Sh "Principal author"
.IX Subsection "Principal author"
Eryq (\fIeryq@zeegee.com\fR).
President, ZeeGee Software Inc (\fIhttp://www.zeegee.com\fR).
.Sh "Other contributors"
.IX Subsection "Other contributors"
Thanks to the following individuals for their invaluable contributions
(if I've forgotten or misspelled your name, please email me!):
.PP
\&\fIMorris M. Siegel,\fR
for his $/ patch and the new \f(CW\*(C`getlines()\*(C'\fR.
.PP
\&\fIDoug Wilson,\fR
for the IO::Handle inheritance and automatic tie\-ing.
                                                                                                     usr/man/man3/IO::WrapTie.3pm                                                                        0000444 0000000 0000000 00000037033 10436506517 014421  0                                                                                                    ustar   root                            root                                                                                                                                                                                                                   .\" Automatically generated by Pod::Man v1.37, Pod::Parser v1.32
.\"
.\" Standard preamble:
.\" ========================================================================
.de Sh \" Subsection heading
.br
.if t .Sp
.ne 5
.PP
\fB\\$1\fR
.PP
..
.de Sp \" Vertical space (when we can't use .PP)
.if t .sp .5v
.if n .sp
..
.de Vb \" Begin verbatim text
.ft CW
.nf
.ne \\$1
..
.de Ve \" End verbatim text
.ft R
.fi
..
.\" Set up some character translations and predefined strings.  \*(-- will
.\" give an unbreakable dash, \*(PI will give pi, \*(L" will give a left
.\" double quote, and \*(R" will give a right double quote.  | will give a
.\" real vertical bar.  \*(C+ will give a nicer C++.  Capital omega is used to
.\" do unbreakable dashes and therefore won't be available.  \*(C` and \*(C'
.\" expand to `' in nroff, nothing in troff, for use with C<>.
.tr \(*W-|\(bv\*(Tr
.ds C+ C\v'-.1v'\h'-1p'\s-2+\h'-1p'+\s0\v'.1v'\h'-1p'
.ie n \{\
.    ds -- \(*W-
.    ds PI pi
.    if (\n(.H=4u)&(1m=24u) .ds -- \(*W\h'-12u'\(*W\h'-12u'-\" diablo 10 pitch
.    if (\n(.H=4u)&(1m=20u) .ds -- \(*W\h'-12u'\(*W\h'-8u'-\"  diablo 12 pitch
.    ds L" ""
.    ds R" ""
.    ds C` ""
.    ds C' ""
'br\}
.el\{\
.    ds -- \|\(em\|
.    ds PI \(*p
.    ds L" ``
.    ds R" ''
'br\}
.\"
.\" If the F register is turned on, we'll generate index entries on stderr for
.\" titles (.TH), headers (.SH), subsections (.Sh), items (.Ip), and index
.\" entries marked with X<> in POD.  Of course, you'll have to process the
.\" output yourself in some meaningful fashion.
.if \nF \{\
.    de IX
.    tm Index:\\$1\t\\n%\t"\\$2"
..
.    nr % 0
.    rr F
.\}
.\"
.\" For nroff, turn off justification.  Always turn off hyphenation; it makes
.\" way too many mistakes in technical documents.
.hy 0
.if n .na
.\"
.\" Accent mark definitions (@(#)ms.acc 1.5 88/02/08 SMI; from UCB 4.2).
.\" Fear.  Run.  Save yourself.  No user-serviceable parts.
.    \" fudge factors for nroff and troff
.if n \{\
.    ds #H 0
.    ds #V .8m
.    ds #F .3m
.    ds #[ \f1
.    ds #] \fP
.\}
.if t \{\
.    ds #H ((1u-(\\\\n(.fu%2u))*.13m)
.    ds #V .6m
.    ds #F 0
.    ds #[ \&
.    ds #] \&
.\}
.    \" simple accents for nroff and troff
.if n \{\
.    ds ' \&
.    ds ` \&
.    ds ^ \&
.    ds , \&
.    ds ~ ~
.    ds /
.\}
.if t \{\
.    ds ' \\k:\h'-(\\n(.wu*8/10-\*(#H)'\'\h"|\\n:u"
.    ds ` \\k:\h'-(\\n(.wu*8/10-\*(#H)'\`\h'|\\n:u'
.    ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'^\h'|\\n:u'
.    ds , \\k:\h'-(\\n(.wu*8/10)',\h'|\\n:u'
.    ds ~ \\k:\h'-(\\n(.wu-\*(#H-.1m)'~\h'|\\n:u'
.    ds / \\k:\h'-(\\n(.wu*8/10-\*(#H)'\z\(sl\h'|\\n:u'
.\}
.    \" troff and (daisy-wheel) nroff accents
.ds : \\k:\h'-(\\n(.wu*8/10-\*(#H+.1m+\*(#F)'\v'-\*(#V'\z.\h'.2m+\*(#F'.\h'|\\n:u'\v'\*(#V'
.ds 8 \h'\*(#H'\(*b\h'-\*(#H'
.ds o \\k:\h'-(\\n(.wu+\w'\(de'u-\*(#H)/2u'\v'-.3n'\*(#[\z\(de\v'.3n'\h'|\\n:u'\*(#]
.ds d- \h'\*(#H'\(pd\h'-\w'~'u'\v'-.25m'\f2\(hy\fP\v'.25m'\h'-\*(#H'
.ds D- D\\k:\h'-\w'D'u'\v'-.11m'\z\(hy\v'.11m'\h'|\\n:u'
.ds th \*(#[\v'.3m'\s+1I\s-1\v'-.3m'\h'-(\w'I'u*2/3)'\s-1o\s+1\*(#]
.ds Th \*(#[\s+2I\s-2\h'-\w'I'u*3/5'\v'-.3m'o\v'.3m'\*(#]
.ds ae a\h'-(\w'a'u*4/10)'e
.ds Ae A\h'-(\w'A'u*4/10)'E
.    \" corrections for vroff
.if v .ds ~ \\k:\h'-(\\n(.wu*9/10-\*(#H)'\s-2\u~\d\s+2\h'|\\n:u'
.if v .ds ^ \\k:\h'-(\\n(.wu*10/11-\*(#H)'\v'-.4m'^\v'.4m'\h'|\\n:u'
.    \" for low resolution devices (crt and lpr)
.if \n(.H>23 .if \n(.V>19 \
\{\
.    ds : e
.    ds 8 ss
.    ds o a
.    ds d- d\h'-1'\(ga
.    ds D- D\h'-1'\(hy
.    ds th \o'bp'
.    ds Th \o'LP'
.    ds ae ae
.    ds Ae AE
.\}
.rm #[ #] #H #V #F C
.\" ========================================================================
.\"
.IX Title "IO::WrapTie 3"
.TH IO::WrapTie 3 "2005-02-10" "perl v5.8.8" "User Contributed Perl Documentation"
.SH "NAME"
IO::WrapTie \- wrap tieable objects in IO::Handle interface
.PP
\&\fIThis is currently Alpha code, released for comments.  
  Please give me your feedback!\fR
.SH "SYNOPSIS"
.IX Header "SYNOPSIS"
First of all, you'll need \fItie()\fR, so:
.PP
.Vb 1
\&   require 5.004;
.Ve
.PP
\&\fIFunction interface (experimental).\fR
Use this with any existing class...
.PP
.Vb 2
\&   use IO::WrapTie;
\&   use FooHandle;                  ### implements TIEHANDLE interface
.Ve
.PP
.Vb 2
\&   ### Suppose we want a "FooHandle->new(&FOO_RDWR, 2)".
\&   ### We can instead say...
.Ve
.PP
.Vb 1
\&   $FH = wraptie('FooHandle', &FOO_RDWR, 2);
.Ve
.PP
.Vb 3
\&   ### Now we can use...    
\&   print $FH "Hello, ";            ### traditional operator syntax...
\&   $FH->print("world!\en");         ### ...and OO syntax as well!
.Ve
.PP
\&\fI\s-1OO\s0 interface (preferred).\fR
You can inherit from the IO::WrapTie::Slave mixin to get a
nifty \f(CW\*(C`new_tie()\*(C'\fR constructor...
.PP
.Vb 2
\&   #------------------------------    
\&   package FooHandle;                        ### a class which can TIEHANDLE
.Ve
.PP
.Vb 3
\&   use IO::WrapTie;  
\&   @ISA = qw(IO::WrapTie::Slave);            ### inherit new_tie()
\&   ...
.Ve
.PP
.Vb 2
\&   #------------------------------    
\&   package main;
.Ve
.PP
.Vb 3
\&   $FH = FooHandle->new_tie(&FOO_RDWR, 2);   ### $FH is an IO::WrapTie::Master
\&   print $FH "Hello, ";                      ### traditional operator syntax
\&   $FH->print("world!\en");                   ### OO syntax
.Ve
.PP
See IO::Scalar as an example.  It also shows you how to create classes
which work both with and without 5.004.
.SH "DESCRIPTION"
.IX Header "DESCRIPTION"
Suppose you have a class \f(CW\*(C`FooHandle\*(C'\fR, where...
.IP "\(bu" 4
\&\fBFooHandle does not inherit from IO::Handle;\fR that is, it performs
filehandle-like I/O, but to something other than an underlying
file descriptor.  Good examples are IO::Scalar (for printing to a
string) and IO::Lines (for printing to an array of lines).
.IP "\(bu" 4
\&\fBFooHandle implements the \s-1TIEHANDLE\s0 interface\fR (see perltie);
that is, it provides methods \s-1TIEHANDLE\s0, \s-1GETC\s0, \s-1PRINT\s0, \s-1PRINTF\s0,
\&\s-1READ\s0, and \s-1READLINE\s0.
.IP "\(bu" 4
\&\fBFooHandle implements the traditional \s-1OO\s0 interface\fR of
FileHandle and IO::Handle; i.e., it contains methods like \fIgetline()\fR, 
\&\fIread()\fR, \fIprint()\fR, \fIseek()\fR, \fItell()\fR, \fIeof()\fR, etc.
.PP
Normally, users of your class would have two options:
.IP "\(bu" 4
\&\fBUse only \s-1OO\s0 syntax,\fR and forsake named I/O operators like 'print'.
.IP "\(bu" 4
\&\fBUse with tie,\fR and forsake treating it as a first-class object 
(i.e., class-specific methods can only be invoked through the underlying
object via \fItied()\fR... giving the object a \*(L"split personality\*(R").
.PP
But now with IO::WrapTie, you can say:
.PP
.Vb 4
\&    $WT = wraptie('FooHandle', &FOO_RDWR, 2);
\&    $WT->print("Hello, world\en");   ### OO syntax
\&    print $WT "Yes!\en";             ### Named operator syntax too!
\&    $WT->weird_stuff;               ### Other methods!
.Ve
.PP
And if you're authoring a class like FooHandle, just have it inherit 
from \f(CW\*(C`IO::WrapTie::Slave\*(C'\fR and that first line becomes even prettier:
.PP
.Vb 1
\&    $WT = FooHandle->new_tie(&FOO_RDWR, 2);
.Ve
.PP
\&\fBThe bottom line:\fR now, almost any class can look and work exactly like
an IO::Handle... and be used both with \s-1OO\s0 and non-OO filehandle syntax.
.SH "HOW IT ALL WORKS"
.IX Header "HOW IT ALL WORKS"
.Sh "The data structures"
.IX Subsection "The data structures"
Consider this example code, using classes in this distribution:
.PP
.Vb 2
\&    use IO::Scalar;
\&    use IO::WrapTie;
.Ve
.PP
.Vb 3
\&    $WT = wraptie('IO::Scalar',\e$s);
\&    print $WT "Hello, ";
\&    $WT->print("world!\en");
.Ve
.PP
In it, the \fIwraptie()\fR function creates a data structure as follows:
.PP
.Vb 24
\&                          * $WT is a blessed reference to a tied filehandle
\&              $WT           glob; that glob is tied to the "Slave" object.
\&               |          * You would do all your i/o with $WT directly.
\&               |       
\&               |
\&               |     ,---isa--> IO::WrapTie::Master >--isa--> IO::Handle
\&               V    /
\&        .-------------. 
\&        |             | 
\&        |             |   * Perl i/o operators work on the tied object,  
\&        |  "Master"   |     invoking the TIEHANDLE methods.
\&        |             |   * Method invocations are delegated to the tied 
\&        |             |     slave.
\&        `-------------' 
\&               |    
\&    tied(*$WT) |     .---isa--> IO::WrapTie::Slave
\&               V    /   
\&        .-------------.
\&        |             |
\&        |   "Slave"   |   * Instance of FileHandle-like class which doesn't
\&        |             |     actually use file descriptors, like IO::Scalar.
\&        |  IO::Scalar |   * The slave can be any kind of object.
\&        |             |   * Must implement the TIEHANDLE interface.
\&        `-------------'
.Ve
.PP
\&\fI\s-1NOTE:\s0\fR just as an IO::Handle is really just a blessed reference to a 
\&\fItraditional\fR filehandle glob... so also, an IO::WrapTie::Master 
is really just a blessed reference to a filehandle 
glob \fIwhich has been tied to some \*(L"slave\*(R" class.\fR
.Sh "How \fIwraptie()\fP works"
.IX Subsection "How wraptie() works"
.IP "1." 4
The call to function \f(CW\*(C`wraptie(SLAVECLASS, TIEARGS...)\*(C'\fR is 
passed onto \f(CW\*(C`IO::WrapTie::Master::new()\*(C'\fR.  
Note that class IO::WrapTie::Master is a subclass of IO::Handle.
.IP "2." 4
The \f(CW\*(C`IO::WrapTie::Master::new\*(C'\fR method creates a new IO::Handle object,
reblessed into class IO::WrapTie::Master.  This object is the \fImaster\fR, 
which will be returned from the constructor.  At the same time...
.IP "3." 4
The \f(CW\*(C`new\*(C'\fR method also creates the \fIslave\fR: this is an instance 
of \s-1SLAVECLASS\s0 which is created by tying the master's IO::Handle 
to \s-1SLAVECLASS\s0 via \f(CW\*(C`tie(HANDLE, SLAVECLASS, TIEARGS...)\*(C'\fR.  
This call to \f(CW\*(C`tie()\*(C'\fR creates the slave in the following manner:
.IP "4." 4
Class \s-1SLAVECLASS\s0 is sent the message \f(CW\*(C`TIEHANDLE(TIEARGS...)\*(C'\fR; it 
will usually delegate this to \f(CW\*(C`SLAVECLASS::new(TIEARGS...)\*(C'\fR, resulting
in a new instance of \s-1SLAVECLASS\s0 being created and returned.
.IP "5." 4
Once both master and slave have been created, the master is returned
to the caller.  
.Sh "How I/O operators work (on the master)"
.IX Subsection "How I/O operators work (on the master)"
Consider using an i/o operator on the master:
.PP
.Vb 1
\&    print $WT "Hello, world!\en";
.Ve
.PP
Since the master ($WT) is really a [blessed] reference to a glob, 
the normal Perl i/o operators like \f(CW\*(C`print\*(C'\fR may be used on it.
They will just operate on the symbol part of the glob.
.PP
Since the glob is tied to the slave, the slave's \s-1PRINT\s0 method 
(part of the \s-1TIEHANDLE\s0 interface) will be automatically invoked.  
.PP
If the slave is an IO::Scalar, that means IO::Scalar::PRINT will be 
invoked, and that method happens to delegate to the \f(CW\*(C`print()\*(C'\fR method 
of the same class.  So the \fIreal\fR work is ultimately done by 
\&\fIIO::Scalar::print()\fR.
.Sh "How methods work (on the master)"
.IX Subsection "How methods work (on the master)"
Consider using a method on the master:
.PP
.Vb 1
\&    $WT->print("Hello, world!\en");
.Ve
.PP
Since the master ($WT) is blessed into the class IO::WrapTie::Master,
Perl first attempts to find a \f(CW\*(C`print()\*(C'\fR method there.  Failing that,
Perl next attempts to find a \f(CW\*(C`print()\*(C'\fR method in the superclass,
IO::Handle.  It just so happens that there \fIis\fR such a method;
that method merely invokes the \f(CW\*(C`print\*(C'\fR i/o operator on the self object...
and for that, see above!
.PP
But let's suppose we're dealing with a method which \fIisn't\fR part
of IO::Handle... for example:
.PP
.Vb 1
\&    my $sref = $WT->sref;
.Ve
.PP
In this case, the intuitive behavior is to have the master delegate the
method invocation to the slave (now do you see where the designations
come from?).  This is indeed what happens: IO::WrapTie::Master contains
an \s-1AUTOLOAD\s0 method which performs the delegation.  
.PP
So: when \f(CW\*(C`sref()\*(C'\fR can't be found in IO::Handle, the \s-1AUTOLOAD\s0 method
of IO::WrapTie::Master is invoked, and the standard behavior of
delegating the method to the underlying slave (here, an IO::Scalar)
is done.
.PP
Sometimes, to get this to work properly, you may need to create 
a subclass of IO::WrapTie::Master which is an effective master for
\&\fIyour\fR class, and do the delegation there.
.SH "NOTES"
.IX Header "NOTES"
\&\fBWhy not simply use the object's \s-1OO\s0 interface?\fR 
    Because that means forsaking the use of named operators
like \fIprint()\fR, and you may need to pass the object to a subroutine
which will attempt to use those operators:
.PP
.Vb 2
\&    $O = FooHandle->new(&FOO_RDWR, 2);
\&    $O->print("Hello, world\en");  ### OO syntax is okay, BUT....
.Ve
.PP
.Vb 2
\&    sub nope { print $_[0] "Nope!\en" }
\& X  nope($O);                     ### ERROR!!! (not a glob ref)
.Ve
.PP
\&\fBWhy not simply use \f(BItie()\fB?\fR 
    Because (1) you have to use \fItied()\fR to invoke methods in the
object's public interface (yuck), and (2) you may need to pass 
the tied symbol to another subroutine which will attempt to treat 
it in an OO\-way... and that will break it:
.PP
.Vb 2
\&    tie *T, 'FooHandle', &FOO_RDWR, 2; 
\&    print T "Hello, world\en";   ### Operator is okay, BUT...
.Ve
.PP
.Vb 1
\&    tied(*T)->other_stuff;      ### yuck! AND...
.Ve
.PP
.Vb 2
\&    sub nope { shift->print("Nope!\en") }
\& X  nope(\e*T);                  ### ERROR!!! (method "print" on unblessed ref)
.Ve
.PP
\&\fBWhy a master and slave? 
  Why not simply write FooHandle to inherit from IO::Handle?\fR
    I tried this, with an implementation similar to that of IO::Socket.  
The problem is that \fIthe whole point is to use this with objects
that don't have an underlying file/socket descriptor.\fR.
Subclassing IO::Handle will work fine for the \s-1OO\s0 stuff, and fine with 
named operators \fIif\fR you \fItie()\fR... but if you just attempt to say:
.PP
.Vb 2
\&    $IO = FooHandle->new(&FOO_RDWR, 2);
\&    print $IO "Hello!\en";
.Ve
.PP
you get a warning from Perl like:
.PP
.Vb 1
\&    Filehandle GEN001 never opened
.Ve
.PP
because it's trying to do system-level i/o on an (unopened) file 
descriptor.  To avoid this, you apparently have to \fItie()\fR the handle...
which brings us right back to where we started!  At least the
IO::WrapTie mixin lets us say:
.PP
.Vb 2
\&    $IO = FooHandle->new_tie(&FOO_RDWR, 2);
\&    print $IO "Hello!\en";
.Ve
.PP
and so is not \fItoo\fR bad.  \f(CW\*(C`:\-)\*(C'\fR
.SH "WARNINGS"
.IX Header "WARNINGS"
Remember: this stuff is for doing FileHandle-like i/o on things
\&\fIwithout underlying file descriptors\fR.  If you have an underlying
file descriptor, you're better off just inheriting from IO::Handle.
.PP
\&\fBBe aware that \f(BInew_tie()\fB always returns an instance of a
kind of IO::WrapTie::Master...\fR it does \fBnot\fR return an instance 
of the i/o class you're tying to!  
.PP
Invoking some methods on the master object causes \s-1AUTOLOAD\s0 to delegate
them to the slave object... so it \fIlooks\fR like you're manipulating a 
\&\*(L"FooHandle\*(R" object directly, but you're not.
.PP
I have not explored all the ramifications of this use of \fItie()\fR.
\&\fIHere there be dragons\fR.
.SH "VERSION"
.IX Header "VERSION"
$Id: WrapTie.pm,v 1.2 2005/02/10 21:21:53 dfs Exp $
.SH "AUTHOR"
.IX Header "AUTHOR"
.RE
.IP "Primary Maintainer"
.IX Item "Primary Maintainer"
David F. Skoll (\fIdfs@roaringpenguin.com\fR).
.RE
.IP "Original Author"
.IX Item "Original Author"
Eryq (\fIeryq@zeegee.com\fR).
President, ZeeGee Software Inc (\fIhttp://www.zeegee.com\fR).
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     
