/*
 *  linux/fs/supermount/dentry.c
 *
 *  Original version:
 *      Copyright (C) 1995
 *      Stephen Tweedie (sct@dcs.ed.ac.uk)
 *
 *  Rewriten for kernel 2.2 & 2.4. (C) 1999, 2000 Alexis Mikhailov
 *                                    (alexis@abc.cap.ru)
 *  Rewritten for kernel 2.4.21 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@mail.ru)
 *  Rewritten for kernel 2.5.70 (C) 2003 Andrey Borzenkov
 *                                       (arvidjaar@mail.ru)
 *
 *  $Id: dentry.c,v 1.9.2.3 2003/07/13 14:52:43 bor Exp $
 */

#define S_DBG_TRACE_CURRENT S_DBG_TRACE_DENTRY
#include "supermount.h"

int
init_dentry_info(struct dentry *dentry)
{
	struct super_block *sb = dentry->d_sb;
	struct supermount_dentry_info *sdi;
	int rc;

	ENTER(sb, "dentry=%s", dentry->d_name.name);

	rc = 1;
	sdi = kmalloc(sizeof(*sdi), GFP_KERNEL);
	if (!sdi)
		goto out;

	memset(sdi, 0, sizeof(*sdi));

	INIT_LIST_HEAD(&sdi->list);
	sdi->dentry = 0;
	sdi->host = dentry;
	
	dentry->d_fsdata = sdi;
	dentry->d_op = &supermount_dops;

	rc = 0;
out:
	LEAVE(sb, "dentry=%s rc=%d", dentry->d_name.name, rc);

	return rc;

}

void
attach_subfs_dentry(struct dentry *dentry, struct dentry *subdent)
{
	struct super_block *sb = dentry->d_sb;
	struct supermount_sb_info *sbi = supermount_sbi(sb);
	struct supermount_dentry_info *sdi;

	ENTER(sb, "dentry=%s subd=%p", dentry->d_name.name, subdent);

	sdi = supermount_d(dentry);
	SUPERMOUNT_BUG_LOCKED_ON(sb, sdi->dentry);
	sdi->dentry = dget(subdent);
	list_add(&sdi->list, &sbi->s_dentries);

	LEAVE(sb, "dentry=%s", dentry->d_name.name);
}


struct dentry *
get_subfs_dentry(struct dentry *dentry)
{
	struct super_block *sb = dentry->d_sb;
	struct dentry *err;
	struct supermount_dentry_info *sdi = supermount_d(dentry);

	ENTER(sb, "dentry=%s", dentry->d_name.name);

	subfs_lock(sb);

	err = ERR_PTR(-ENOMEDIUM);
	if (!subfs_is_mounted(sb))
		goto out;

	err = ERR_PTR(-ESTALE);
	if (is_dentry_obsolete(dentry))
		goto out;

	err = dget(sdi->dentry);
	SUPERMOUNT_BUG_LOCKED_ON(sb, !err);
	SUPERMOUNT_BUG_LOCKED_ON(sb, (dentry->d_inode == 0) ^ (err->d_inode == 0));

out:
	subfs_unlock(sb);

	LEAVE(sb, "dentry=%s subd=%p", dentry->d_name.name, err);

	return err;
}

static int
supermount_d_revalidate(struct dentry *dentry, struct nameidata *nd)
{
	struct super_block *sb = dentry->d_sb;
	struct dentry *subd;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dentry=%s", dentry->d_name.name);

	rc = 0;
	mnt = subfs_prevent_umount(sb);
	if (!mnt)
		goto out;
	
	subd = get_subfs_dentry(dentry);

	if (IS_ERR(subd))
		goto allow_umount;

	rc = 1;
	/* FIXME do we need to build proper subfs nd? */
	if (subd->d_op && subd->d_op->d_revalidate)
		rc = subd->d_op->d_revalidate(subd, nd);

	dput(subd);
allow_umount:
	subfs_allow_umount(sb, mnt);
out:

	LEAVE(sb, "dentry=%s rc=%d", dentry->d_name.name, rc);

	return rc;
}

static int
supermount_d_hash(struct dentry *dentry, struct qstr *name)
{
	struct super_block *sb = dentry->d_sb;
	struct dentry *subd;
	int rc;
	struct vfsmount *mnt;

	ENTER(sb, "dentry=%s name=%s", dentry->d_name.name, name->name);

	rc = -ENOMEDIUM;
	mnt = subfs_prevent_umount(sb);
	if (!mnt)
		goto out;
	
	subd = get_subfs_dentry(dentry);
	rc = PTR_ERR(subd);
	if (IS_ERR(subd))
		goto allow_umount;

	rc = 0;
	if (subd->d_op && subd->d_op && subd->d_op->d_hash)
		rc = subd->d_op->d_hash(subd, name);

	dput(subd);
allow_umount:
	subfs_allow_umount(sb, mnt);
out:
	LEAVE(sb, "dentry=%s rc=%d", dentry->d_name.name, rc);

	return rc;
}


rwlock_t d_compare_lock = RW_LOCK_UNLOCKED;

static int
supermount_d_compare(struct dentry *dentry, struct qstr *name1, struct qstr *name2)
{
	struct super_block *sb = dentry->d_sb;
	struct supermount_dentry_info *sdi;
	struct dentry *subd;
	int rc;

	ENTER(sb, "dentry=%s name1=%s name2=%s", dentry->d_name.name, name1->name, name2->name);

	rc = 1; /* fail by default */
	sdi = dentry->d_fsdata;
	SUPERMOUNT_BUG_ON(!sdi);

	/*
	 * HACK - FIXME
	 * this protects against races with supermount_clean_dentries
	 * I cannot use blocking calls (i.e. sbi->sem) here and it is not
	 * protected by dcache_lock anymore
	 */
	read_lock(&d_compare_lock);
	subd = sdi->dentry;
	if (!subd) {
		read_unlock(&d_compare_lock);
		goto out;
	}

	if (subd->d_op && subd->d_op->d_compare)
		rc = subd->d_op->d_compare(subd, name1, name2);
	else { /* based on fs/dcache.c:d_lookup */
		if (name1->len == name2->len)
		     rc = memcmp(name1->name, name2->name, name1->len);
	}
	read_unlock(&d_compare_lock);

out:
	LEAVE(sb, "dentry=%s rc=%d", dentry->d_name.name, rc);
	return rc;
}

static inline void
handle_subdent(struct dentry *dentry)
{
	struct super_block *sb = dentry->d_sb;
	struct vfsmount *mnt;
	struct supermount_dentry_info *sdi = supermount_d(dentry);
	struct dentry *subd = 0;

	mnt = subfs_prevent_umount(sb);
	if (!mnt)
		goto out;

	subfs_lock(sb);
	if (sdi) {
		subd = sdi->dentry;
		list_del_init(&sdi->list);
		sdi->dentry = 0;
	}
	subfs_unlock(sb);

	if (subd)
		dput(subd);

	subfs_allow_umount(sb, mnt);
out:
	return;
}

/*
 * in case of active dentry we must be sure subfs dentry is released
 * before subfs inode to correctly maintain write state
 */
static void
supermount_d_iput(struct dentry *dentry, struct inode *inode)
{
	struct super_block *sb = dentry->d_sb;

	ENTER(sb, "dentry=%s inode=%p", dentry->d_name.name, inode);

	handle_subdent(dentry);
	iput(inode);

	LEAVE(sb, "dentry=%s", dentry->d_name.name);
}
	

/*
 * this duplicated code is due to the lack of common "destroy" method
 * for both negative and positive dentries
 */
static void
supermount_d_release(struct dentry *dentry)
{
	struct super_block *sb = dentry->d_sb;
	struct supermount_dentry_info *sdi = supermount_d(dentry);

	ENTER(sb, "dentry=%s", dentry->d_name.name);

	handle_subdent(dentry);
	kfree(sdi);

	LEAVE(sb, "dentry=%s", dentry->d_name.name);
}

struct dentry_operations supermount_dops = {
	.d_revalidate	= supermount_d_revalidate,
	.d_hash		= supermount_d_hash,
	.d_compare	= supermount_d_compare,
	.d_iput		= supermount_d_iput,
	.d_release	= supermount_d_release,
};
