/* RWDOCS */
/* filename 	: arp.c
 * purpose	: resolve IP addresses to hardware address
 *
 * Address Resolution Protocol (ARP) functions. Sits between IP and
 * Level 2, mapping IP to Level 2 addresses for all outgoing datagrams.
 *
 */

#include "config.h"

#if defined(ETHER)
#include "global.h"
#include "mbuf.h"
#include "timer.h"
#include "iface.h"

#ifdef ETHER
#include "enet.h"
#endif

#include "arp.h"



extern int32 ip_addr;		/* Our IP address */

/* ARP entries for particular subnetwork types. The table values
 * are filled in by calls to arp_init() at device attach time
 */

struct arp_type arp_type[NTYPES];

/* Hash table headers */
struct arp_tab *arp_tab[ARPSIZE];

struct arp_stat arp_stat;


/* Functions defined in arp.c 
 */
/* Initialize an entry in the ARP table
 * Called by the device driver at attach time
 */
int 
arp_init(hwtype,hwalen,iptype,arptype,bdcst,format,scan)
unsigned int hwtype;	/* ARP Hardware type */
int hwalen;		/* Hardware address length */
int iptype;		/* Subnet's protocol ID for IP */
int arptype;		/* Subnet's protocol ID for ARP */
char *bdcst;		/* Subnet's broadcast address (if any) */
int (*format)();	/* Function to format hardware addresses */
int (*scan)();		/* Function to scan addresses in ascii */	
{

/* RWDOCE */
	register struct arp_type *at;

	if(hwtype >= NTYPES)
		return -1;	/* Table too small */

	at = &arp_type[hwtype];
	at->hwalen = (int)hwalen;
	at->iptype = (int)iptype;
	at->arptype = (int)arptype;
	at->bdcst = bdcst;
	at->format = format;
	at->scan = scan;
	return 0;
}

/* RWDOCS */
/* Resolve an IP address to a hardware address; if not found,
 * initiate query and return NULLCHAR.  If an address is returned, the
 * interface driver may send the packet; if NULLCHAR is returned,
 * res_arp() will have saved the packet on its pending queue,
 * so no further action (like freeing the packet) is necessary.
 */
char *
res_arp(interface,hardware,target,bp)
struct interface *interface;	/* Pointer to interface block */
int  hardware;		/* Hardware type */
int32 target;		/* Target IP address */
struct mbuf *bp;	/* IP datagram to be queued if unresolved */
{

/* RWDOCE */
	register struct arp_tab *arp;

	if((arp = arp_lookup(hardware,target)) != NULLARP && arp->state == ARP_VALID)
		return arp->hw_addr;
	/* Create an entry and put the datagram on the
	 * queue pending an answer
	 */
	arp = arp_add(target,hardware,NULLCHAR,0,0);
	enqueue(&arp->pending,bp);
	arp_output(interface,hardware,target);
	return NULLCHAR;
}

/* RWDOCS */
/* Handle incoming ARP packets. This is almost a direct implementation of
 * the algorithm on page 5 of RFC 826, except for:
 * 1. Outgoing datagrams to unresolved addresses are kept on a queue
 *    pending a reply to our ARP request.
 * 2. The names of the fields in the ARP packet were made more mnemonic.
 * 3. Requests for IP addresses listed in our table as "published" are
 *    responded to, even if the address is not our own.
 */
void 
arp_input(interface,bp)
struct interface *interface;
struct mbuf *bp;
{

/* RWDOCE */
	struct arp arp;
	struct arp_tab *ap;
	struct arp_type *at;
	struct mbuf *htonarp();
	int ntoharp();

	arp_stat.recv++;
	if(ntoharp(&arp,&bp) == -1)	{
		/* Convert into host format */

		return;
	}
	if(arp.hardware >= NTYPES){
		/* Unknown hardware type, ignore */
		arp_stat.badtype++;

		return;
	}
	at = &arp_type[(int)arp.hardware];
	if( (int)arp.protocol != at->iptype){
		/* Unsupported protocol type, ignore */
		arp_stat.badtype++;

		return;
	}
	if( (int)( arp.hwalen & 0xff ) > (int)MAXHWALEN || 
		   (int)(arp.pralen & 0xff) != sizeof(int32)) {

		/* first condition is always false, leave it in in case someone has
		 * a reason to make MAXHWALEN less than 255 - K5JB
		 */
		/* Incorrect protocol addr length (different hw addr lengths
		 * are OK since AX.25 addresses can be of variable length)
		 */
		arp_stat.badlen++;


		return;
	}
	if(memcmp(arp.shwaddr,at->bdcst,(int)at->hwalen) == 0){
		/* This guy is trying to say he's got the broadcast address! */
		arp_stat.badaddr++;

		return;
	}
	/* If this guy is already in the table, update its entry
	 * unless it's a manual entry (noted by the lack of a timer)
	 */
	ap = NULLARP;	/* ap plays the role of merge_flag in the spec */
	if((ap = arp_lookup( (int)arp.hardware,arp.sprotaddr)) != NULLARP
	 && ap->timer.start != 0){
		ap = arp_add(arp.sprotaddr,(int)arp.hardware,arp.shwaddr,
			(int)(arp.hwalen & 0xff),0);

	}
	/* See if we're the address they're looking for */
	if(arp.tprotaddr == ip_addr){
		if(ap == NULLARP)	/* Only if not already in the table */
			arp_add(arp.sprotaddr,(int)arp.hardware,arp.shwaddr,
					(int)(arp.hwalen & 0xff ),0);

		if( (int)arp.opcode == (int)ARP_REQUEST) {
			/* Swap sender's and target's (us) hardware and protocol
			 * fields, and send the packet back as a reply
			 */
			memcpy(arp.thwaddr,arp.shwaddr,(int)(arp.hwalen & 0xff));

			memcpy(arp.shwaddr,interface->hwaddr,(int)(at->hwalen & 0xff) );
			arp.tprotaddr = arp.sprotaddr;
			arp.sprotaddr = ip_addr;
			arp.opcode = (int16)ARP_REPLY;
			if((bp = htonarp(&arp)) == NULLBUF)
				return;
#ifdef FORWARD
			if(interface->forw != NULLIF)
				(*interface->forw->output)(interface->forw,
				 arp.thwaddr,interface->forw->hwaddr,
					at->arptype,bp);
			else
#endif
				(*interface->output)(interface,arp.thwaddr,
				 interface->hwaddr,at->arptype,bp);
			arp_stat.inreq++;
		} else {
			arp_stat.replies++;

		}
	} else if( (int)arp.opcode == (int) ARP_REQUEST
		&& (ap = arp_lookup(arp.hardware,arp.tprotaddr)) != NULLARP
		&& ap->pub){
		/* Otherwise, respond if the guy he's looking for is
		 * published in our table.
		 */
		memcpy(arp.thwaddr,arp.shwaddr,(int)(arp.hwalen & 0xff));
		memcpy(arp.shwaddr,ap->hw_addr,(int)(at->hwalen & 0xff) );

		arp.tprotaddr = arp.sprotaddr;
		arp.sprotaddr = ap->ip_addr;
		arp.opcode = (int16)ARP_REPLY;
		if((bp = htonarp(&arp)) == NULLBUF) {

			return;
		}
#ifdef FORWARD
		if(interface->forw != NULLIF)
			(*interface->forw->output)(interface->forw,
			 arp.thwaddr,interface->forw->hwaddr,at->arptype,bp);
		else
#endif
			(*interface->output)(interface,arp.thwaddr,
			 interface->hwaddr,at->arptype,bp);
		arp_stat.inreq++;
	}

}

/* RWDOCS */
/* Add an IP-addr / hardware-addr pair to the ARP table */
struct arp_tab *
arp_add(ipaddr,hardware,hw_addr,hw_alen,pub)
int32 ipaddr;		/* IP address, host order */
int hardware;		/* Hardware type */
char *hw_addr;		/* Hardware address, if known; NULLCHAR otherwise */
int hw_alen;		/* Length of hardware address */
int pub;		/* Publish this entry? */
{


/* RWDOCE */
	void arp_drop(),free();
	int ip_route(),start_timer();
	struct mbuf *bp,*dequeue();
	register struct arp_tab *ap;
	unsigned hashval;

	if((ap = arp_lookup(hardware,ipaddr)) == NULLARP){
		/* New entry */
		if((ap = (struct arp_tab *)calloc(1,sizeof(struct arp_tab))) == NULLARP)
			return NULLARP;
		ap->timer.func = arp_drop;
		ap->timer.arg = (char *)ap;
		ap->hardware = hardware;
		ap->ip_addr = ipaddr;

		/* Put on head of hash chain */
		hashval = arp_hash(hardware,ipaddr);
		ap->prev = NULLARP;
		ap->next = arp_tab[hashval];
		arp_tab[hashval] = ap;
		if(ap->next != NULLARP){
			ap->next->prev = ap;
		}
	}
	if(hw_addr == NULLCHAR){
		/* Await response */
		ap->state = ARP_PENDING;
		ap->timer.start = PENDTIME * (1000 / MSPTICK);
	} else {
		/* Response has come in, update entry and run through queue */
		ap->state = ARP_VALID;
		ap->timer.start = ARPLIFE * (1000 / MSPTICK);
		if(ap->hw_addr != NULLCHAR)
			free(ap->hw_addr);
		if((ap->hw_addr = malloc(hw_alen)) == NULLCHAR){
			free((char *)ap);
			return NULLARP;
		}
		memcpy(ap->hw_addr,hw_addr,hw_alen);

		ap->pub = pub;
		while((bp = dequeue(&ap->pending)) != NULLBUF)
			ip_route(bp,0);
	}
	start_timer(&ap->timer);
	return ap;
}

/* RWDOCS */
/* Remove an entry from the ARP table */
void
arp_drop(ap)
register struct arp_tab *ap;
{

/* RWDOCE */
	int stop_timer();
	void free(),free_q();

	if(ap == NULLARP)
		return;
	stop_timer(&ap->timer);	/* Shouldn't be necessary */
	if(ap->next != NULLARP)
		ap->next->prev = ap->prev;
	if(ap->prev != NULLARP)
		ap->prev->next = ap->next;
	else
		arp_tab[arp_hash(ap->hardware,ap->ip_addr)] = ap->next;
	if(ap->hw_addr != NULLCHAR)
		free(ap->hw_addr);
	free_q(&ap->pending);
	free((char *)ap);
}

/* RWDOCS */
/* Look up the given IP address in the ARP table */
struct arp_tab *
arp_lookup(hardware,ipaddr)
int hardware;	/* hardware type */
int32 ipaddr;	/* IP addresses */
{

/* RWDOCE */
	register struct arp_tab *ap;

	for(ap = arp_tab[arp_hash(hardware,ipaddr)]; ap != NULLARP; ap = ap->next){
		if(ap->ip_addr == ipaddr && ap->hardware == hardware)
			break;
	}
	return ap;
}

/* RWDOCS */
/* Send an ARP request to resolve IP address target_ip */
void
arp_output(interface,hardware,target)
struct interface *interface;	/* interface */
int hardware;			/* hardware type */
int32 target;			/* IP address */
{

/* RWDOCE */
	struct arp arp;
	struct mbuf *bp,*htonarp();
	struct arp_type *at;
	char * memset();
	at = &arp_type[hardware];
	if(interface->output == NULLFP)
		return;
	
	arp.hardware = (int16)hardware;
	arp.protocol = (int16)at->iptype;
	arp.hwalen = (unsigned char)at->hwalen;
	arp.pralen = (unsigned char)sizeof(int32);
	arp.opcode = (int16)ARP_REQUEST;
	memcpy(arp.shwaddr,interface->hwaddr,at->hwalen);
	arp.sprotaddr = (unsigned)ip_addr;
	memset(arp.thwaddr,0,(unsigned)at->hwalen);
	arp.tprotaddr = target;
	if((bp = htonarp(&arp)) == NULLBUF)
		return;
	(*interface->output)(interface,at->bdcst,
		interface->hwaddr,at->arptype,bp);
	arp_stat.outreq++;
}

/* RWDOCS */
/* Hash a {hardware type, IP address} pair */
unsigned
arp_hash(hardware,ipaddr)
int hardware;
int32 ipaddr;
{

/* RWDOCE */
	register unsigned hashval;

	hashval = hardware;
	hashval ^= hiword(ipaddr);
	hashval ^= loword(ipaddr);
	hashval %= ARPSIZE;
	return hashval;
}

/* RWDOCS */	
/* Copy a host format arp structure into mbuf for transmission */
struct mbuf *
htonarp(arp)
register struct arp *arp;
{

/* RWDOCE */

	struct mbuf *bp;
	register char *buf;

	if(arp == (struct arp *)0)
		return NULLBUF;
	if((bp = alloc_mbuf(sizeof(struct arp))) == NULLBUF)
		return NULLBUF;

	buf = bp->data;

	buf = put16(buf,arp->hardware);
	buf = put16(buf,arp->protocol);
	*buf++ = arp->hwalen;
	*buf++ = arp->pralen;
	buf = put16(buf,arp->opcode);
	memcpy(buf,arp->shwaddr,(int)(arp->hwalen & 0xff));
	buf += (int)arp->hwalen;
	buf = put32(buf,arp->sprotaddr);
	memcpy(buf,arp->thwaddr,(int)(arp->hwalen & 0xff));
	buf += (int)arp->hwalen;
	buf = put32(buf,arp->tprotaddr);

	bp->cnt = buf - bp->data;
	return bp;
}

/* RWDOCS */
/* Convert an incoming ARP packet into a host-format structure */
int
ntoharp(arp,bpp)
register struct arp *arp;
struct mbuf **bpp;
{

/* RWDOCE */

	if(arp == (struct arp *)0 || bpp == NULLBUFP)
		return -1;

	arp->hardware = (int16)pull16(bpp);
	arp->protocol = (int16)pull16(bpp);
	arp->hwalen = (unsigned char)pullchar(bpp);
	arp->pralen = (unsigned char)pullchar(bpp);
	arp->opcode = (int16)pull16(bpp);
	pullup(bpp,arp->shwaddr,(int)(arp->hwalen & 0xff));
	arp->sprotaddr = pull32(bpp);
	pullup(bpp,arp->thwaddr,(int)(arp->hwalen & 0xff));
	arp->tprotaddr = pull32(bpp);

	/* Get rid of anything left over */
	free_p(*bpp);
	*bpp = NULLBUF;
	return 0;
}

#endif /* ETHER */
/* end of arp.c */
