/*
 * xt_control.c,
 * control panel of xvidcap
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This file contains the main window of the Xt GUI
 *
 */

#include "../config.h"	/* autoconf output */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>		/* PATH_MAX */
#include <ctype.h>		/* isdigit() */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#include <X11/Xmu/WinUtil.h>
#include <Xw/Button.h>
#include <Xw/Box.h>
#include <Xw/Toggle.h>
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif
#include "fallback.h"
#include "job.h"
#include "util.h"
#include "colors.h"
#include "capture.h"
#include "frame.h"
#include "xt_frame.h"
#include "main.h"
//#include "control.h"
#include "xt_control.h"
#include "xt_options.h"
#include "xtoxwd.h"
#include "xtopnm.h"
/* #include "sound.h" */
#ifdef HAVE_LIBPNG
#include "xtopng.h"
#include "xtomng.h"
#endif
#include "xtojpg.h"
#include "xtoqtf.h"
#include "xtoffmpeg.h"
#include "xbm/stop.xbm"
#include "xbm/pause.xbm"
#include "xbm/record.xbm"
#include "xbm/step.xbm"
#include "xbm/prev.xbm"
#include "xbm/next.xbm"
#include "xbm/select.xbm"
#include "xbm/animate.xbm"
/* #include "xbm/mkvideo.xbm" */
#include "xbm/edit.xbm"
/* #include "xbm/help.xbm" */
#include "xbm/move.xbm"

/* global
 */
static Job *jobp;
static Widget wFile, start, stop, wPause, step, sel, animate, label;
Widget toplevel, move, file_menu, file_mbox, prev, next;
XtAppContext context;
Atom wm_delete;

// max_time related timer stuff
static XtIntervalId stop_timer_id;
struct timeval curr_time;
long start_time, pause_time, time_captured;


static void PopupFileMenu(Widget w, XEvent *ev, String *parms, Cardinal *num);
static void PopdownFileMenu(Widget w, XEvent *ev, String *parms, Cardinal *num);
void OptionDoneAction(Widget w, XEvent *ev, String *parms, Cardinal *num);
void OptionOpenAction(Widget w, XEvent *ev, String *parms, Cardinal *num);
static void QuitAppAction(Widget w, XEvent *ev, String *parms, Cardinal *num);
static void TCbStop(XtPointer xtp, XtIntervalId id);
    
static XtActionsRec actionTable[] = {
    {"popup_file_menu", PopupFileMenu},
    {"popdown_file_menu", PopdownFileMenu},
    {"option_done", OptionDoneAction},
    {"option_open", OptionOpenAction},
    {"quit_app",	QuitAppAction	},
};

#	define offset(x) XtOffsetOf(AppData, x)
static XtResource resources [] = {
    {XtNfps, XtCFps, XtRFloat, sizeof(int), offset(fps), XtRImmediate, NULL},
    {XtNfile, XtCFile, XtRString, sizeof(String), offset(file), XtRImmediate, NULL},
    {XtNverbose, XtCVerbose, XtRInt, sizeof(int), offset(verbose), XtRImmediate, NULL},
    {XtNcapWidth, XtCCapWidth, XtRInt, sizeof(int), offset(cap_width), XtRInt, 0},
    {XtNcapHeight, XtCCapHeight, XtRInt, sizeof(int), offset(cap_height), XtRInt, 0},
    {XtNtime, XtCTime, XtRFloat, sizeof(int), offset(time), XtRImmediate, NULL},
    {XtNframes, XtCFrames, XtRInt, sizeof(int), offset(frames), XtRImmediate, NULL},
    {XtNstartNo, XtCStartNo, XtRInt, sizeof(int), offset(start_no), XtRImmediate, NULL},
    {XtNcomp, XtCComp, XtRInt, sizeof(int), offset(compress), XtRImmediate, NULL},
    {XtNquality, XtCQuality, XtRInt, sizeof(int), offset(quality), XtRImmediate, NULL},
    {XtNbpp, XtCBpp, XtRInt, sizeof(int), offset(bpp), XtRImmediate, NULL},
    {XtNsource, XtCSource, XtRString, sizeof(String), offset(source), XtRImmediate, NULL},
    {XtNsndDev, XtCSndDev, XtRString, sizeof(String), offset(snddev), XtRImmediate, NULL},
    {XtNsndRate, XtCSndRate, XtRInt, sizeof(int), offset(sndrate), XtRImmediate, NULL},
    {XtNsndSize, XtCSndSize, XtRInt, sizeof(int), offset(sndsize), XtRImmediate, NULL},
    {XtNanimate, XtCAnimate, XtRString, sizeof(String), offset(play_cmd), XtRString, ""},
    {XtNmkVideo, XtCMkVideo, XtRString, sizeof(String), offset(video_cmd), XtRString, ""},
    {XtNedit, XtCEdit, XtRString, sizeof(String), offset(edit_cmd), XtRImmediate, NULL},
    {XtNhelp, XtCHelp, XtRString, sizeof(String), offset(help_cmd), XtRImmediate, NULL},
};

static void start_recording(Job *job);
static void CreateControl ( Widget, AppData * );
static void InitControl ( AppData * );
static void CbStop (Widget, XtPointer, XtPointer);
static void CbStart (Widget, XtPointer, XtPointer);
static void CbQuit (Widget, XtPointer, XtPointer);

/*
 * XVIDCAP GUI API ( IF THAT WORD IS ALLOWED HERE ;) )
 *
 * XVC_PREINIT IS FOR GUI PREINITIALIZATION LIKE SETTING FALLBACK OPTIONS
 * XVC_CREATEGUI IS FOR GUI CREATION
 * XVC_CREATEFRAME SHOULD CREATE THE SELECTION FRAME
 * XVC_INITGUI IS FOR MISCELLANEOUS INITIALIZATION STUFF
 * XVC_RUNGUI IS A WRAPPER AROUND THE UI'S MAIN LOOP
 *
 * XVC_ADDTIMEOUT IS A WRAPPER AROUND THE GUI'S FUNCTIONALITY FOR TRIGGERING ACTIONS AT A
 * GIVEN TIME
 * XVC_CHANGEGUILABEL IS A WRAPPER AROUND A FUNCTION TO CHANGE THE DISPLAY
 * OF THE CURRENT FRAME NUMBER (NOT NECESSARY IN A LABEL)
 * XVC_CHANGEFRAME IS A HOOK FOR MODIFYING THE SELECTION FRAME
 * XVC_STOPCAPTURE IS A HOOK FOR RESETTING THE GUI TO A STOPPED STATE
 * XVC_STARTCAPTURE IS A HOOK FOR STARTING A CAPTURE SESSION
 * XVC_FRAMEMONITOR IS A HOOK FOR THE CAPTURE FUNCTION TO UPDATE A MONITOR WIDGET
 * 
 */

Boolean XVC_PreInit( int argc, char **argv, AppData *app ) {
    Display *dpy;
    
    // need to have this up here to read the Xt ressources ... later will overwrite user
    // options with defaults ... no GTK2 version needed because GTK2 doesn't have ressources
    toplevel = XtVaAppInitialize( &context, "XVidcap",
    NULL, 0, &argc, argv, fallback, XtNinput, True, NULL);
    
    dpy = XtDisplay(toplevel);
    
    XtVaGetApplicationResources(toplevel, app,
    resources, XtNumber(resources), NULL);
    XtAppAddActions(context,
    actionTable, sizeof(actionTable)/sizeof(XtActionsRec));
    
    #ifdef HasDGA
    if (!XF86DGAQueryExtension(dpy, &dga_evb, &dga_errb))
        app.flags &= ~FLG_USE_DGA;
    else {
        int flag = 0;
        XF86DGAQueryDirectVideo(dpy, XDefaultScreen(dpy), &flag);
        if ((flag & XF86DGADirectPresent) == 0) {
            app->flags &= ~FLG_USE_DGA;
            if (app->verbose) {
                printf("xfdga: no direct present\n");
            }
        }
    }
    #endif //HasDGA
    
    #ifdef HAVE_SHMAT
    if (!XShmQueryExtension(dpy))
        app->flags &= ~FLG_USE_SHM;
    #endif //HAVE_SHMAT
    
    return TRUE;
}


Boolean XVC_CreateGUI( AppData *app ) {
    CreateControl(toplevel, app);
    if ( app->flags & FLG_NOGUI )
        XtSetMappedWhenManaged(toplevel, FALSE);
    //XtVaSetValues(toplevel, XmNmappedWhenManaged, FALSE, NULL )
    return TRUE;
}


Boolean XVC_CreateFrame( AppData *app ) {
    if ( app->cap_width == 0 ) app->cap_width = 10;
    if ( app->cap_height == 0 ) app->cap_height = 10;
    CreateFrame(toplevel, app->cap_width, app->cap_height, app->flags );
    return TRUE;
}


Boolean XVC_InitGUI( AppData *app ) {
    XtRealizeWidget(toplevel);
    
    InitControl(app);
    return TRUE;
}


int XVC_RunGUI() {
    Job *job = job_ptr();
    
    if ( ! ( job->flags & FLG_NOGUI ) ) {
        wm_delete = XInternAtom(XtDisplay(toplevel), "WM_DELETE_WINDOW", False);
        XSetWMProtocols(XtDisplay(toplevel), XtWindow(toplevel), &wm_delete, 1);
        XtAddEventHandler(toplevel, NoEventMask, True, exitEH, &wm_delete);
    } else {
        XtAppAddTimeOut(XtWidgetToApplicationContext(toplevel),
            0, (XtTimerCallbackProc)start_recording, job);
    }
    
    XtAppMainLoop(context);
}


void XVC_AddTimeout(int ms, void *func, Job *job) {
    XtAppAddTimeOut(XtWidgetToApplicationContext(toplevel),
    ms, (XtTimerCallbackProc)func, job);
}


void XVC_ChangeGUILabel(int pic_no) {
    ChangeLabel(pic_no);
}


void XVC_StopCapture(Job *job) {
    CbStop(NULL, NULL, NULL);
}


void XVC_StartCapture(Job *job) {
    if (job->flags & FLG_RUN_VERBOSE) printf("starting again\n");
    CbStart(NULL, NULL, NULL);
}


void XVC_ChangeFrame(int x, int y, int width, int height, Boolean reposition_control) {
    ChangeFrame(x, y, width, height, reposition_control );
}


void XVC_FrameMonitor(Job *job, int measured_time) {
    
}



/*
 *   HELPER FUNCTIONS ...........
 *
 */

/*
 * change value of the label
 */
void
ChangeLabel(int pic_no) {
    static char file[PATH_MAX+1], tmp_buf[PATH_MAX+1];
    int filename_length;
    
    if (jobp->flags & FLG_MULTI_IMAGE) {
        sprintf(tmp_buf, jobp->file, jobp->movie_no);
        sprintf(file, "%s[%04d]", tmp_buf, pic_no);
    } else {
        sprintf(file, jobp->file, pic_no);
    }

    XtVaSetValues(label, XtNlabel, file, NULL);
}


/*
 *   CALLBACKS ......
 *
 */

/*
 */
static void
QuitAppAction(Widget w, XEvent *ev, String *parms, Cardinal *num_parms) {
    CbQuit(w, NULL, NULL);
}

/*
 * popup file menu on Alt-F
 */
static void
PopupFileMenu(Widget w, XEvent *ev, String *parms, Cardinal *num_parms) {
    Position topx, topy, x, y;
    Dimension height;
    
    XtVaGetValues(toplevel, XtNx, &topx, NULL);
    XtVaGetValues(toplevel, XtNy, &topy, NULL);

    XtVaGetValues(wFile, XtNheight, &height, NULL);
    XtVaGetValues(wFile, XtNx, &x, NULL);
    XtVaGetValues(wFile, XtNy, &y, NULL);

    XtVaSetValues(file_menu, XtNx, (x + topx), XtNy, (y + topy + height + 3 ), NULL);
    
    //    XtPopup(file_menu, XtGrabNonexclusive);
    XtPopupSpringLoaded(file_menu);

}

/*
 * popdown file menu on Esc
 */
static void
PopdownFileMenu(Widget w, XEvent *ev, String *parms, Cardinal *num_parms) {
    XtPopdown(file_menu);
}



/*
 * lock or unlock the control panel for moving
 */
static void
CbLock(Widget w, XtPointer client_data, XtPointer call_data) {
    Dimension pwidth, pheight;
    XRectangle *frame_rectangle;
    Position x, y;
    
    int val = XwIsSet(w);

    #ifdef DEBUG
    printf("CbMove (%d) %d \n", (int)w, (int)val);
    #endif
    if (!val) {
        /* toggle released */
        XVC_frame_lock = 0;
    } else {
        /* toggle pressed */
        XVC_frame_lock = 1;
        
        XtVaGetValues(toplevel, XtNheight, &pheight, XtNwidth, &pwidth, XtNx, &x, XtNy, &y, NULL); 

        if (x < 0)
            x = 0;
        y += pheight + FRAME_OFFSET;
        if (y < 0)
            y = 0;
        
        frame_rectangle = GetArea();
        XVC_ChangeFrame(x, y, frame_rectangle->width, frame_rectangle->height, FALSE);
    }
}



/*
 */
static void
CbHelp(Widget w, XtPointer client_data, XtPointer call_data) {
    if (jobp->flags & FLG_RUN_VERBOSE)
        printf("calling '%s'\n", (char*) client_data);
    system((char*)client_data);
}

/*
 * callback for edit button
 */
static void
CbEdit(Widget w, XtPointer client, XtPointer call) {
    char cmd[PATH_MAX*2+4];
    char buf[PATH_MAX+1];
    int n;
    
    if (jobp->pic_no < 1)
        n = 0;
    else
        n = jobp->pic_no - 1;
    sprintf(buf, jobp->file, n);
    sprintf(cmd, (char*)client, buf);
    if (jobp->flags & FLG_RUN_VERBOSE)
        printf("calling: '%s'\n", cmd);
    system(cmd);
}

/*
 */
static void
CbMkVideo(Widget w, XtPointer client, XtPointer call) {
    char *fmt = (char *)client;
    char cmd[PATH_MAX*2+2];
    sprintf(cmd, fmt, jobp->file, jobp->start_no, jobp->pic_no-1,
    jobp->area->width, jobp->area->height, jobp->fps, jobp->time_per_frame);
    if (jobp->flags & FLG_RUN_VERBOSE)
        printf("calling: '%s'\n", cmd);
    system(cmd);
}

/*
 */
static void
CbAnimate(Widget w, XtPointer client, XtPointer call) {
    char file[PATH_MAX+1], buff[PATH_MAX*2+1], *p;
    char *cmd = (char *) client;
    int i = 0, n=0;
    
    p = jobp->file;
    while (*p) {
        if (*p == '%') {
            p++;
            sscanf(p, "%dd", &n);
            while (n) {
                file[i++] = '?';
                n--;
            }
            while (isdigit(*p))
                p++;
            p++;
        } else {
            file[i] = *p;
            p++;
            i++;
        }
    }
    file[i] = '\0';
    sprintf(buff, cmd, file, jobp->time_per_frame);
    if (jobp->flags & FLG_RUN_VERBOSE);
    printf("calling: '%s'\n", buff);
    system(buff);
}


/*
 */
static void
CbPrev(Widget w, XtPointer client, XtPointer call) {
    Job *jobp = (Job *)client;
    if (( jobp->flags & FLG_MULTI_IMAGE ) == 0 ) {
        if (jobp->pic_no >= jobp->step) {
            jobp->pic_no -= jobp->step;
            ChangeLabel(jobp->pic_no);
        } else {
            fprintf(stderr, "previous button active although picture number < step. this should never happen\n");
        }
        if ( jobp->pic_no < jobp->step ) XtSetSensitive(prev, False);
    } else {
        if (jobp->movie_no > 0 ) {
            jobp->movie_no -= 1;
            ChangeLabel(jobp->pic_no);
        } else {
            fprintf(stderr, "previous button active although movie number == 0. this should never happen\n");
        }
        if ( jobp->movie_no == 0 ) XtSetSensitive(prev, False);
    }
}

/*
 */
static void
CbNext(Widget w, XtPointer client, XtPointer call) {
    Job *jobp = (Job *) client;

    if (( jobp->flags & FLG_MULTI_IMAGE ) == 0 ) {
        jobp->pic_no += jobp->step;
        ChangeLabel(jobp->pic_no);
        if (jobp->pic_no == jobp->step ) XtSetSensitive(prev, True);
    } else {
        jobp->movie_no += 1;
        ChangeLabel(jobp->pic_no);
        if (jobp->movie_no == 1) XtSetSensitive(prev, True);
    }
}

/*
 */
static void
CbReset(Widget w, XtPointer client, XtPointer call) {
    if ( ( jobp->flags & FLG_MULTI_IMAGE ) == 0 ) {
        if ( jobp->pic_no != jobp->start_no && ( jobp->state & VC_STOP ) > 0 
                    || ( jobp->state & VC_PAUSE ) > 0 ) {
            jobp->pic_no = jobp->start_no;
            ChangeLabel(jobp->pic_no);
        }
    } else {
        if ( jobp->movie_no != 0 && ( jobp->state & VC_STOP ) > 0) {
            jobp->movie_no = 0;
            ChangeLabel(jobp->pic_no);
        }
    }
}

/*
 */
static void
CbPause(Widget w, XtPointer client, XtPointer call) {
    int state = 0;
    
    XtVaGetValues(w, XtNstate, &state, NULL);
    #ifdef DEBUG
    printf("CbPause(%d)\n", (int) state);
    #endif
    if (state) {
        if (jobp->max_time ) {
            gettimeofday(&curr_time, NULL);
            pause_time = curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000;
            if (( jobp->state & VC_REC ) != 0 ) {
                time_captured = ( pause_time - start_time ) + time_captured;
            } else {
                time_captured = 0;
            }
            if (stop_timer_id) XtRemoveTimeOut(stop_timer_id);
        }
        
        jobp->state |= VC_PAUSE;
        jobp->state &= ~VC_STOP;
        XtVaSetValues(stop, XtNstate, False, NULL);
        XtSetSensitive(stop, True);
        if ((jobp->flags & FLG_MULTI_IMAGE) == 0 && jobp->state & VC_REC) {
            XtSetSensitive(step, True);
            XtSetSensitive(next, True);
            if (jobp->pic_no >= jobp->step) XtSetSensitive(prev, True);
        }
    } else {
        jobp->state &= ~VC_PAUSE;
        jobp->state &= ~VC_STEP;
        // step is always only active if a running recording session is paused
        // so releasing pause can always deactivate it
        XtSetSensitive(step, False);
        // the following only when recording is going on (not when just pressing and
        // releasing pause
        if (jobp->state & VC_REC) {
            XtSetSensitive(next, False);
            XtSetSensitive(prev, False);
        }

        // restart timer handling only if max_time is configured
        if (jobp->max_time) {
            /* install a timer which stops recording
             * we need milli secs ..
             */
            gettimeofday(&curr_time, NULL);
            start_time = curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000;
            stop_timer_id = XtAppAddTimeOut(XtWidgetToApplicationContext(toplevel),
                (int)(jobp->max_time * 1000 - time_captured), (XtTimerCallbackProc)TCbStop, jobp);
        }
    }
}

/*
 * step mode, every click records one frame
 */
static void
CbStep(Widget w, XtPointer client, XtPointer call) {
    Job *jobp = (Job *) client;
    if (( jobp->flags & FLG_MULTI_IMAGE ) == 0 ) {
        if (!(jobp->state & (VC_PAUSE | VC_REC)))
            return;
        jobp->state |= VC_STEP;
        jobp->capture(jobp, NULL);
        if (jobp->pic_no == jobp->step ) XtSetSensitive(prev, True);
    }
}

/*
 */
static void
CbQuit(Widget w, XtPointer client, XtPointer call) {
    /* important for ?NG to sync the end of the file */
    if (jobp->state & VC_REC)
        if (jobp->clean)
            (*jobp->clean)(jobp);
    exit(0);
}


static void stop_recording_nongui_stuff(Job* job) {
    job->state = VC_STOP;
}

static void stop_recording_gui_stuff(Job* job) {
    XtSetSensitive(wFile, True);
    XtSetSensitive(stop, False);
    /* XtSetSensitive (start, True); */
    XtSetSensitive(step, False);
    XtSetSensitive(sel, True);
    if (! ((job->flags & FLG_MULTI_IMAGE) != 0 && is_filename_mutable(job->file) != TRUE )) {
        XtSetSensitive(next, True);    
        if ((job->flags & FLG_MULTI_IMAGE) == 0) {
            if (job->pic_no >= job->step) XtSetSensitive(prev, True);
        } else {
            if (job->movie_no > 0) XtSetSensitive(prev, True);
        }
    }
                
    XtVaSetValues(wPause, XtNstate, False, NULL);
    XtVaSetValues(start, XtNstate, False, XtNsensitive, True, NULL);
    if (!(job->flags & FLG_RUN_VERBOSE))
        ChangeLabel(job->pic_no);
}


/*
 */
static void
CbStop(Widget w, XtPointer client, XtPointer call) {
    // remove max_time timer
    if (stop_timer_id) XtRemoveTimeOut(stop_timer_id);
    // nongui stuff
    stop_recording_nongui_stuff(jobp);
    // GUI stuff
    stop_recording_gui_stuff(jobp);
}

/*
 * timer callback for max_time
 */
static void
TCbStop(XtPointer xtp, XtIntervalId id) {
    if ( jobp->flags & FLG_AUTO_CONTINUE) {
        stop_recording_nongui_stuff(jobp);
        jobp->state |= VC_CONTINUE;
    } else {
        stop_recording_nongui_stuff(jobp);
        stop_recording_gui_stuff(jobp);
    }
}


static void
start_recording(Job *job) {
    job->state = (VC_REC | VC_START) | (job->state & VC_PAUSE);
    
    if (job->flags & FLG_MULTI_IMAGE) {
        job->pic_no = job->start_no;
        ChangeLabel(jobp->pic_no);   
    }
    if (job->max_time && (job->state & VC_PAUSE) == 0 ) {
        /* install a timer which stops recording
         * we need milli secs ..
         */
        gettimeofday(&curr_time, NULL);
        time_captured = 0;
        start_time = curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000;
        stop_timer_id = XtAppAddTimeOut(XtWidgetToApplicationContext(toplevel),
            (int)(job->max_time * 1000), (XtTimerCallbackProc)TCbStop, job);
    }
    /* call the timer function
     */
    XtAppAddTimeOut(XtWidgetToApplicationContext(toplevel),
        0, (XtTimerCallbackProc)job->capture, jobp);
    
}

/*
 * this is the record button
 */
static void
CbStart(Widget w, XtPointer client, XtPointer call) {
    Job *jobp = (Job*) job_ptr();
    /* init some values for the new job
     */
    #ifdef DEBUG
    printf("CbStart() fps=%f time slot = %d ms\n", jobp->fps, jobp->time_per_frame);
    #endif
    XtSetSensitive(start, False);
    XtSetSensitive(wFile, False);
    XtSetSensitive(stop, True);
    XtSetSensitive(sel, False);
    XtSetSensitive(wPause, True);
    
    if (jobp->state & VC_PAUSE) {
        if ( (jobp->flags & FLG_MULTI_IMAGE) == 0 ) {
            XtSetSensitive(step, True);
            XtSetSensitive(next, True);
            if (jobp->pic_no >= jobp->step ) XtSetSensitive(prev, True);
        } else {
            XtSetSensitive(step, False);
            if ( ! is_filename_mutable(jobp->file) ) {
                XtSetSensitive(next, False);    
                XtSetSensitive(prev, False);
            }
        }
    } else {
        XtSetSensitive(next, False);    
        XtSetSensitive(prev, False);
    }
    XtVaSetValues(stop, XtNstate, False, NULL);
    
    start_recording(jobp);
}


/*
 * select a window which should be captured
 */
static void
CbSelect(Widget w, XtPointer client, XtPointer call) {
    Display *display = XtDisplay((Widget)client);
    Cursor cursor;
    Window root, target = None, temp = None;
    XEvent event;
    int buttons = 0;
    int x_down, y_down, x_up, y_up, x, y, pheight = 0;
    int width, height;
    Widget toplevel = (Widget)client;
    XGCValues gcv;
    GC gc;
    
    root = RootWindow(display, DefaultScreen(display));
    cursor = XCreateFontCursor(display, XC_crosshair);
    
    gcv.background = XBlackPixel(display, XDefaultScreen(display));
    gcv.foreground = XWhitePixel(display, XDefaultScreen(display));
    gcv.function = GXinvert;
    gcv.plane_mask = gcv.background ^ gcv.foreground;
    gcv.subwindow_mode = IncludeInferiors;
    gc = XCreateGC(display, root,
    GCBackground | GCForeground | GCFunction | GCPlaneMask | GCSubwindowMode, &gcv);
    /* grab the mouse
     */
    if (XGrabPointer(display, root, False,
    PointerMotionMask|ButtonPressMask|ButtonReleaseMask,
    GrabModeSync, GrabModeAsync, root, cursor, CurrentTime)
    != GrabSuccess) {
        printf("Can't grab mouse!\n");
        return;
    }
    x_down = y_down = x_up = y_up = width = height = 0;
    
    while (buttons < 2) {
        /* allow pointer events */
        XAllowEvents(display, SyncPointer, CurrentTime);
        /* search in the queue for button events */
        XWindowEvent(display, root,
        PointerMotionMask|ButtonPressMask|ButtonReleaseMask, &event);
        switch (event.type) {
            case ButtonPress:
                x_down = event.xbutton.x;
                y_down = event.xbutton.y;
                target = event.xbutton.subwindow; /* window selected */
                if (target == None) {
                    target = root;
                }
                buttons++;
                break;
            case ButtonRelease:
                x_up = event.xbutton.x;
                y_up = event.xbutton.y;
                buttons++;
                break;
            default:
                /* motion notify */
                if (buttons == 1) {
                    if (width) {
                        /* remove old frame */
                        XDrawRectangle(display, root, gc, x, y, width, height);
                    }
                    /* button is pressed */
                    if (x_down > event.xbutton.x) {
                        width = x_down - event.xbutton.x + 1;
                        x = event.xbutton.x;
                    } else {
                        width = event.xbutton.x - x_down + 1;
                        x = x_down;
                    }
                    if (y_down > event.xbutton.y) {
                        height = y_down - event.xbutton.y + 1;
                        y = event.xbutton.y;
                    } else {
                        height = event.xbutton.y - y_down + 1;
                        y = y_down;
                    }
                    XDrawRectangle(display, root, gc, x, y, width, height);
                }
                break;
        }
    }
    if (width) /* remove old frame */
        XDrawRectangle(display, root, gc, x, y, width, height);
    XUngrabPointer(display, CurrentTime);      /* Done with pointer */
    XFreeCursor(display, cursor);
    XFreeGC(display, gc);
    
    if ((x_down != x_up) && (y_down != y_up)) {
        /* an individual frame was selected
         */
        if (x_up < x_down) {
            width = x_down - x_up + 2;
            x = x_up;
        } else {
            width = x_up - x_down + 2;
            x = x_down;
        }
        if (y_up < y_down) {
            height = y_down - y_up + 2;
            y = y_up;
        } else {
            height = y_up - y_down + 2;
            y = y_down;
        }
        if (target != root)
            target = XmuClientWindow(display, target);
        XGetWindowAttributes(display, target, &jobp->win_attr);
        jobp->win_attr.width = width;
        jobp->win_attr.height = height;
    } else {
        /* if (target == root) {
         printf ("Drag your mouse to select an area on the root window.\n");
         return;
         } */
        if (target != root) {
            /* get the real window */
            target = XmuClientWindow(display, target);
        }
        XGetWindowAttributes(display, target, &jobp->win_attr);
        XTranslateCoordinates(display, target, root, 0, 0, &x, &y, &temp);
    }
    
    jobp->win_attr.x = x;
    jobp->win_attr.y = y;
    
    /* we must move the panel a little bit, so get the height of
     the panel to adjust the y value */
    XtVaGetValues(toplevel, XtNheight, &pheight, NULL);
    
    /* add the space between the panel and the frame */
    // FIXME: find out how to determine the title bar height, if possible
    // this is just to get the main control out of the way, because
    // if we have the frame locked, the move will trigger the move callback
    // however, this will not leave the selection at the right coordinates
    // because we're not moving the control enough, because we don*t know
    // the height of the title bar ...
    pheight = pheight + FRAME_OFFSET;
    
    if (IsFrameLocked()) {
        XtVaSetValues(toplevel, XtNx, x, XtNy, y - pheight, NULL);
    }
    
    if (jobp->flags & FLG_RUN_VERBOSE) {
        printf("Original Selection geometry: %dx%d+%d+%d\n",
        jobp->win_attr.width, jobp->win_attr.height, x, y);
    }
    #ifdef HAVE_LIBAVCODEC
    #ifdef HAVE_LIBAVFORMAT
    /*
     * make sure we have even width and height for ffmpeg
     */
    if (jobp->target >= CAP_FFM) {
        Boolean changed = FALSE;
        int w, h;
        
        if ((jobp->win_attr.width % 2) > 0 ) {
            jobp->win_attr.width--;
            changed = TRUE;
        }
        if ((jobp->win_attr.height % 2) > 0) {
            jobp->win_attr.height--;
            changed = TRUE;
        }
/*        if ( jobp->win_attr.width < 26 ) {
            jobp->win_attr.width = 26;
            changed = TRUE;
        }
        if ( jobp->win_attr.height < 26 ) {
            jobp->win_attr.height = 26;
            changed = TRUE;
        } */

        if (changed) {
            if (jobp->flags & FLG_RUN_VERBOSE) {
                printf("Modified Selection geometry: %dx%d+%d+%d\n",
                jobp->win_attr.width, jobp->win_attr.height, x, y);
            }
        }
    }
    #endif
    #endif
    ChangeFrame(x, y, jobp->win_attr.width, jobp->win_attr.height, FALSE);
    
    jobp->ncolors = GetColors(display, &jobp->win_attr, &jobp->colors);
    if (jobp->get_colors) {
        if (jobp->color_table)
            free(jobp->color_table);
        jobp->color_table = (*jobp->get_colors)(jobp->colors, jobp->ncolors);
    }
    if (jobp->flags & FLG_RUN_VERBOSE) {
        fprintf(stderr, "color_table first entry: 0x%.8X\n", *(u_int32_t*)jobp->color_table);
    }
    job_set_save_function(jobp->win_attr.visual, jobp->target, jobp);
    #ifdef DEBUG
    printf("new visual: %d\n", jobp->win_attr.visual->class);
    #endif
    return;
}

/*
 */
void
MenuEH(Widget w, XtPointer xtp, XEvent *ev, Boolean *bo) {
    Widget mw = (Widget) xtp;
    
    #ifdef DEBUG
    printf("MenuEH(): %s %d\n", XtName(w), ev->type);
    #endif
    if (ev->type == ButtonRelease) {
        XtPopdown(mw);
        XUngrabPointer(XtDisplay(w), CurrentTime);
    }
}


/*
 * exit event handler
 */
void
exitEH(Widget w, XtPointer xtp, XEvent *ev, Boolean *boo) {
    if ((ev->type == ClientMessage) &&
    (ev->xclient.format == 32) &&
    (ev->xclient.data.l[0] == *((Atom*)xtp)))
        exit(0);
}



/*
 * create control panel and make some inits..
 */
static void
CreateControl( Widget toplevel, AppData *app) {
    Widget box, options, quit, help, edit;
    Widget mkvideo;
    Display *display;
    Window root;
    static char info[PATH_MAX+1];
    Pixmap stop_bm, pause_bm, rec_bm, step_bm, prev_bm, next_bm;
    Pixmap select_bm, animate_bm, edit_bm, move_bm;
    
    char *file;
    int flags, bpp;
    char *animate_cmd, *mkvideo_cmd, *edit_cmd, *help_cmd;
    
    file = app->file;
    flags = app->flags;
    bpp = app->bpp;
    animate_cmd = app->play_cmd;
    mkvideo_cmd = app->video_cmd;
    edit_cmd = app->edit_cmd;
    help_cmd = app->help_cmd;
    
    display = XtDisplay(toplevel);
    root = RootWindow(display, DefaultScreen(display));
    
    jobp = job_init(flags);
    
    jobp->area = GetArea();
    jobp->bpp = bpp;
    if (!XGetWindowAttributes(display, root, &jobp->win_attr))
        perror("Can't get window attributes!\n");
    /* the default is to use the colormap of the root window
     */
    jobp->ncolors = GetColors(display, &jobp->win_attr, &jobp->colors);
    
    sprintf(info, " %s     ", file);
    stop_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    stop_bits, stop_width, stop_height);
    pause_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    pause_bits, pause_width, pause_height);
    rec_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    record_bits, record_width, record_height);
    step_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    step_bits, step_width, step_height);
    prev_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    prev_bits, prev_width, prev_height);
    next_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    next_bits, next_width, next_height);
    select_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    select_bits, select_width, select_height);
    animate_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    animate_bits, animate_width, animate_height);
    edit_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    edit_bits, edit_width, edit_height);
    move_bm = XCreateBitmapFromData(XtDisplay(toplevel), root,
    move_bits, move_width, move_height);
    
    box =  XtVaCreateManagedWidget("panel", xwBoxWidgetClass, toplevel,
    XtNorientation, XtorientHorizontal, NULL);
    
    /* file menu
     */
    wFile = XtVaCreateManagedWidget("file_mb", xwButtonWidgetClass, box,
    XtNmenuName, "file_menu",
    NULL);
    file_menu = XtVaCreatePopupShell("file_menu", overrideShellWidgetClass,
    wFile, NULL);
    
    file_mbox = XtVaCreateManagedWidget("file_mbox", xwBoxWidgetClass,
    file_menu, NULL);
    XtAddEventHandler(file_menu, ButtonReleaseMask|ButtonPressMask, False,
    MenuEH, (XtPointer)file_menu);
    
    options = XtVaCreateManagedWidget("options", xwButtonWidgetClass,
    file_mbox, NULL);
    XtAddCallback(options, XtNcallback, CbOptions, (XtPointer)jobp);
    
    mkvideo = XtVaCreateManagedWidget("mkvideo", xwButtonWidgetClass,
    file_mbox,
    NULL);
    XtAddCallback(mkvideo, XtNcallback, CbMkVideo, (XtPointer)mkvideo_cmd);
    
    help = XtVaCreateManagedWidget("help", xwButtonWidgetClass, file_mbox,
    NULL);
    XtAddCallback(help, XtNcallback, CbHelp, (XtPointer)help_cmd);
    
    quit = XtVaCreateManagedWidget("quit", xwButtonWidgetClass,
    file_mbox, NULL);
    XtAddCallback(quit, XtNcallback, CbQuit, NULL);
    //XtAddEventHandler (quit, ButtonRelease, True, MenuEH, file_menu);
    
    /* labels and bitmap buttons
     */
    label = XtVaCreateManagedWidget("label", xwButtonWidgetClass, box,
    XtNlabel, info,
    NULL);
    stop = XtVaCreateManagedWidget("stop", xwToggleWidgetClass, box,
    XtNsensitive, False,
    XtNstate, True,
    XtNbitmap, stop_bm, NULL);
    wPause = XtVaCreateManagedWidget("pause", xwToggleWidgetClass, box,
    XtNbitmap, pause_bm, NULL);
    start= XtVaCreateManagedWidget("start", xwToggleWidgetClass, box,
    XtNbitmap, rec_bm,
    NULL);
    step = XtVaCreateManagedWidget("step", xwButtonWidgetClass, box,
    XtNsensitive, False,
    XtNbitmap, step_bm,
    NULL);
    prev = XtVaCreateManagedWidget("prev", xwButtonWidgetClass, box,
    XtNbitmap, prev_bm,
    NULL);
    next = XtVaCreateManagedWidget("next", xwButtonWidgetClass, box,
    XtNbitmap, next_bm,
    NULL);
    move = XtVaCreateManagedWidget("move", xwToggleWidgetClass, box,
    XtNbitmap, move_bm, XtNstate, True,
    NULL);
    sel = XtVaCreateManagedWidget("select", xwButtonWidgetClass, box,
    XtNbitmap, select_bm,
    NULL);
    animate = XtVaCreateManagedWidget("animate", xwButtonWidgetClass, box,
    XtNbitmap, animate_bm,
    NULL);
    edit = XtVaCreateManagedWidget("edit", xwButtonWidgetClass, box,
    XtNbitmap, edit_bm,
    NULL);
    
    XtVaSetValues(edit, XtNnext, wFile, XtNprev, animate, NULL);
    XtVaSetValues(animate, XtNnext, edit, XtNprev, sel, NULL);
    XtVaSetValues(sel, XtNnext, animate, XtNprev, next, NULL);
    XtVaSetValues(next, XtNnext, sel, XtNprev, prev, NULL);
    XtVaSetValues(prev, XtNnext, next, XtNprev, step, NULL);
    XtVaSetValues(step, XtNnext, prev, XtNprev, start, NULL);
    XtVaSetValues(start, XtNnext, step, XtNprev, wPause, NULL);
    XtVaSetValues(wPause, XtNnext, start, XtNprev, stop, NULL);
    XtVaSetValues(stop, XtNnext, wPause, XtNprev, label, NULL);
    XtVaSetValues(label, XtNnext, stop, XtNprev, move, NULL);
    XtVaSetValues(move, XtNnext, label, XtNprev, wFile, NULL);
    XtVaSetValues(wFile, XtNnext, move, XtNprev, edit, NULL);
    
    XtAddCallback(move, XtNcallback, CbLock, (XtPointer)toplevel);
    XtAddCallback(label, XtNcallback, CbReset, (XtPointer)toplevel);
    XtAddCallback(stop, XtNcallback, CbStop, (XtPointer)toplevel);
    XtAddCallback(wPause, XtNcallback, CbPause, (XtPointer)toplevel);
    XtAddCallback(start, XtNcallback, CbStart, (XtPointer)jobp);
    XtAddCallback(step, XtNcallback, CbStep, (XtPointer)jobp);
    XtAddCallback(prev, XtNcallback, CbPrev, (XtPointer)jobp);
    XtAddCallback(next, XtNcallback, CbNext, (XtPointer)jobp);
    XtAddCallback(sel, XtNcallback, CbSelect, (XtPointer)toplevel);
    XtAddCallback(animate, XtNcallback, CbAnimate, (XtPointer)animate_cmd);
    XtAddCallback(edit, XtNcallback, CbEdit, (XtPointer)edit_cmd);
}

/*
 * this is called after realizing the widgets above
 */
static void
InitControl(AppData *app) {
    /* 'jobp' is global */
    char *file;
    int codec, compress;
    float fps, max_time, max_frames;
    int start_at_no, step, quality, mouseWanted;
    char *dev, *snd;
    int rate, size, channels;
    extern XVC_frame_lock;
    
    file = app->file;
    codec = app->targetCodec;
    compress = app->compress;
    fps = app->fps;
    max_time = app->time;
    max_frames = app->frames;
    start_at_no = app->start_no;
    step = app->step;
    quality = app->quality;
    mouseWanted = app->mouseWanted;
    dev = app->device;
    snd = app->snddev;
    rate = app->sndrate;
    size = app->sndsize;
    channels = app->sndchannels;
    
    if ( app->cap_pos_x >= 0 || app->cap_pos_y >=0 ) {
        XtVaSetValues(move, XtNstate, False, NULL);
        XVC_frame_lock = 0;
    }
    
    #ifdef DEBUG
    printf("InitControl() fps=%f\n", fps);
    #endif
    /* find capture function */
    job_set_capture();
    
    /* set frames per second */
    job_set_fps(fps);
    
    /* jpeg quality parameter */
    job_set_quality(quality);
    
    /* png compression */
    job_set_compression(compress);
    
#ifdef HAVE_FFMPEG_AUDIO
    /* initialize sound recording parameters */
    job_set_sound_dev(snd, rate, size, channels);
#endif // HAVE_FFMPEG_AUDIO
    
    /* */
    job_set_file(file);
    jobp->targetCodec = codec;
    
    jobp->max_time = max_time;
    jobp->max_frames = max_frames;
    jobp->start_no = start_at_no;
    jobp->pic_no = start_at_no;
    jobp->step = step;
    jobp->video_dev = dev;
    jobp->mouseWanted = mouseWanted;
    
    if (jobp->get_colors)
        jobp->color_table = (*jobp->get_colors)(jobp->colors, jobp->ncolors);
    
    job_set_save_function(jobp->win_attr.visual, jobp->target, jobp);
    ChangeLabel(jobp->pic_no);
    
    #ifdef HAVE_LIBAVCODEC
    #ifdef HAVE_LIBAVFORMAT
    /*
     * make sure we have even width and height for ffmpeg
     */
    if (jobp->target >= CAP_FFM) {
        Boolean changed = FALSE;
        
        if (jobp->flags & FLG_RUN_VERBOSE) {
            fprintf(stderr, "control.InitControl(): Original dimensions: %i * %i\n", jobp->area->width, jobp->area->height);
        }
        if ((jobp->area->width % 2) > 0 ) {
            jobp->area->width--;
            changed = TRUE;
        }
        if ((jobp->area->height % 2) > 0) {
            jobp->area->height--;
            changed = TRUE;
        }
        if ( jobp->win_attr.width < 20 ) {
            jobp->win_attr.width = 20;
            changed = TRUE;
        }
        if ( jobp->win_attr.height < 20 ) {
            jobp->win_attr.height = 20;
            changed = TRUE;
        } 
        
        if (changed) {
            ChangeFrame(jobp->area->x, jobp->area->y, jobp->area->width, jobp->area->height, FALSE);
            if (jobp->flags & FLG_RUN_VERBOSE) {
                fprintf(stderr, "control.InitControl(): Modified dimensions: %i * %i\n", jobp->area->width, jobp->area->height);
            }
        }
    }
    #endif
    #endif

    // unset autocontinue unless we capture to movie and file is mutable
    if ( jobp->flags & FLG_AUTO_CONTINUE && ( (! is_filename_mutable(jobp->file)) ||
                (jobp->flags & FLG_MULTI_IMAGE) == 0 ) ) {
        jobp->flags &= ~FLG_AUTO_CONTINUE;
        printf("Output not a video file or no counter in filename\nDisabling autocontinue!\n");
    }

    /* previous and next buttons have different meanings for on-the-fly encoding
     * and individual frame capture */
    if (( jobp->flags & FLG_MULTI_IMAGE ) == 0 ) {
        if (jobp->pic_no >= jobp->step ) XtSetSensitive(prev, True);
        else XtSetSensitive(prev, False);
    } else {
        if ( is_filename_mutable(jobp->file) ) {
            XtSetSensitive(next, True);    
            if (jobp->movie_no > 0 ) XtSetSensitive(prev, True);    
            else XtSetSensitive(prev, False);    
        } else {
            XtSetSensitive(next, False);    
            XtSetSensitive(prev, False);    
        }
    }

}

