/*
 * gtvr.c
 * GnuTime VR module
 *
 * Copyright (C) 1998,99 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "gt.h"
#include "gtvr.h"

/*
 * rotate the image counterclockwise 90 degree and dice (split into
 * smaller pices) the image
 * return the number of diced images or 0 on error
 *
 * TODO: fixes which will catch images with "rest"
 */
int
gt_vr_dice (gt_image *org, int frac, gt_image ***ret)
{
	int width, height, rest, no, i, x, y, b, xoff;
	char *p;
	gt_image **imgs;

	if (!org || !frac)
		return 0;
	width = org->height;
	height = org->width / frac; 
	rest = org->width - (height * frac);
#ifdef DEBUG
	fprintf (stderr, "gt_vr_dice() width=%d height=%d rest=%d\n",
				width, height, rest);
#endif
	if (rest)
		no = frac + 1;
	else
		no = frac;

	imgs = *ret = malloc (sizeof (gt_image *) * no);
	for (i = 0; i < no; i++ ) {
		imgs[i] = malloc (sizeof (gt_image) + width * height * org->depth);
		if (!ret[i])
			return 0;
		imgs[i]->width = width;
		imgs[i]->height= height;
		imgs[i]->depth = org->depth;
		p = imgs[i]->img = (char *)imgs[i] + sizeof (gt_image);
		xoff = (org->width * org->depth) - org->depth;
		for ( y = 0; y < height; y++) {
			for ( x = 0 ; x < width; x++ ) {
				for (b = 0; b < org->depth; b++ ) {
					*p = org->img [ xoff + b
							+ x * org->width * org->depth
							- (y * org->depth) - (i * height * org->depth)];
					p++;
				}
			}
		}
	}
	return frac;
}

/*
 * create a new qtvr with the given diced images and return
 * it in a buffer ready to write out
 * a temporary file is used to write down the data
 *
 * returns the length of the buffer or 0 on error
 */
int
gt_vr_new_from_diced (gt_image **diced, int num, char **buf)
{
	gt_movie *movie;
	gt_udta_atom *udta;
	gt_atom *vtrak, *ptrak;
	gt_minf_atom *vminf, *pminf;
	gt_mdia_atom *vmdia, *pmdia;
	gt_dinf_atom *vdinf, *pdinf;
	gt_stbl_atom *vstbl, *pstbl;
	gt_stts_atom *vstts;
	gt_stsc_atom *vstsc;
	gt_stsz_atom *vstsz;
	gt_stco_atom *vstco;
	char file[PATH_MAX+1];
	FILE *fp;
	int size, i;

	if (!diced)
		return 0;
	tmpnam (file);
	strcat (file, ".mov");
	/* we need two track and a udta atom -> 3 subatoms
	 */
	movie = gt_movie_new (GT_QTVR, 3, file);
	if (!movie) return 0;

	/* we need only two table entries in the user data atom
	 */
	udta = gt_alloc_atom (GTA_user_data, 2);
	if (!udta) {
		gt_movie_free (movie);
		return 0;
	}
	udta->tab[0]->type = GT_UD_CONTR_TYPE;
	udta->tab[0]->size = 8 + 4 ;	/* type, size, text */
	/* udta->tab[0]->text = (char *)GT_UD_CONTR_DATA; */
	udta->tab[0]->text = (char *)GT_N_QTVR;

	udta->tab[1]->type = GT_UD_INFO;
	udta->tab[1]->tlen = strlen ("gnutime");
	udta->tab[1]->size = 8 + 4 + udta->tab[1]->tlen;
	udta->tab[1]->text = "gnutime";
	udta->size = 8 + udta->tab[0]->size + udta->tab[1]->size;

	/* fill in the user data
	 */
	gt_movie_add_atom (movie, (gt_atom *)udta);
#ifdef DEBUG
	fprintf (stderr, "gt_vr_new_from_diced() size=%d\n", movie->moov->size);
#endif
	/* we need two tracks
	 * the first track is the video track and the second one the
	 * pano track
	 */
	vstbl  = gt_sample_new (GT_VID_FMT_JPEG,
					diced[0]->width, diced[0]->height, num, 4);

	/* time to sample atom */
	vstts  = gt_alloc_atom (GTA_time_to_sample, 1);
	vstts->count = 1;
	vstts->tab->num_samples = num;
	vstts->tab->duration = 500;
	gt_atom_add ((gt_atom *)vstbl, (gt_atom *)vstts);

	/* sample to chunk atom */
	vstsc  = gt_alloc_atom (GTA_sample_to_chunk, 1);
	vstsc->count = 1;
	vstsc->tab->first_chunk = 1;
	vstsc->tab->samples_per_chunk = num;
	vstsc->tab->sample_id = 1;
	gt_atom_add ((gt_atom *)vstbl, (gt_atom *)vstsc);

	/* sample size atom */
	vstsz  = gt_alloc_atom (GTA_sample_size, num);
	vstsz->sample_size = 0;
	vstsz->count = num;
	for (i = 0; i < num; i++) {
		vstsz->tab[i].size = diced[i]->size;
	}
	gt_atom_add ((gt_atom *)vstbl, (gt_atom *)vstsz);

	/* sample chunk offset atom */
	vstco  = gt_alloc_atom (GTA_chunk_offset, 1);
	vstco->count = 1;
	vstco->tab->offset = 0; /* we have to define this later .. */
	gt_atom_add ((gt_atom *)vstbl, (gt_atom *)vstco);

	vdinf  = gt_data_info_new (0);
	vminf  = gt_media_info_new (GT_VIDEO, 2);
	gt_atom_add ((gt_atom *)vminf, (gt_atom *)vdinf);
	gt_atom_add ((gt_atom *)vminf, (gt_atom *)vstbl);

	vmdia = gt_media_new (GT_VIDEO, 1000, 2);
	gt_atom_add ((gt_atom *)vmdia, (gt_atom *)vminf);

	vtrak = gt_track_new (1, diced[0]->width, diced[0]->height, GT_VIDEO,  1);
	gt_atom_add ((gt_atom *)vtrak, (gt_atom *)vmdia);
	gt_movie_add_atom (movie, (gt_atom *)vtrak);

	/* pano track */
	pdinf  = gt_data_info_new (0);
	pstbl  = gt_sample_new (GT_VID_FMT_PANO,
					diced[0]->width, diced[0]->height, num, 3);
	pminf  = gt_media_info_new (GT_N_PANO, 2);
	gt_atom_add ((gt_atom *)pminf, (gt_atom *)pdinf);
	gt_atom_add ((gt_atom *)pminf, (gt_atom *)pstbl);

	pmdia = gt_media_new (GT_N_PANO, 1000, 2);
	gt_atom_add ((gt_atom *)pmdia, (gt_atom *)pminf);

	ptrak = gt_track_new (2, diced[0]->width, diced[0]->height,GT_PANORAMA,1);
	gt_atom_add ((gt_atom *)ptrak, (gt_atom *)pmdia);
	gt_movie_add_atom (movie, (gt_atom *)ptrak);

	vstco->tab->offset = movie->moov->size + 8;
	gt_movie_write_moov (movie, NULL);

	/* write out the diced images
	 */
	for (i = 0; i < num; i++) {
		movie->mdat->size += diced[i]->size;
	}
	fp = fopen (movie->file, "ab");
	if (!fp) { /* todo */
		perror (movie->file);
		gt_movie_free (movie);
		return (0);
	}
	/* fseek (fp, movie->moov->size, SEEK_SET); */
	gt_write4byte (movie->mdat->size, fp);
	gt_write4byte (movie->mdat->type, fp);
	for (i = 0; i < num; i++) {
		fwrite (diced[i]->img, 1, diced[i]->size, fp);
	}
	fclose (fp);

	/* read it back into the buffer
	 */
	*buf = malloc (movie->moov->size + movie->mdat->size);
	fp = fopen (movie->file, "rb");
	if (!fp || !*buf) {
		gt_movie_free (movie);
		return 0;
	}
	/* fprintf (stderr, " gt_vr_new.. size=%d\n", movie->moov->size); */
	size = fread (*buf, 1, movie->moov->size + movie->mdat->size, fp);
	fclose (fp);
	unlink (movie->file);
	gt_movie_free (movie);
	return size;
}

