/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "../game/q_shared.h"
#include "gs_public.h"

//
// gs_gametypes.c
//
typedef struct 
{
	char		*name;
	char		*shortname;
	char		*statusbar;

	//some rules toogles
	qboolean	isTeamBased;
	int			maxWallJumps;
	int			spawnableItemsMask;
	int			respawnableItemsMask;
	int			dropableItemsMask;

}gs_gametype_t;

//==========================================================
//				Gametypes definitions
//note that the enum in gs_public must follow the same order
//==========================================================
gs_gametype_t	gsGametypes[] = 
{
	//GAMETYPE_DM
	{
		"deathmatch",										//name
		"dm",												//short name
		"dm",												//hud
		0,													//is team based
		1,													//max walljumps
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH),	//spawnable item types
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH),	//item that will respawn after being picked up
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH),	//items that can be dropped (by user or death)
	},

	//GAMETYPE_DUEL
	{
		"1v1",												//name
		"duel",												//short name
		"dm",												//hud
		1,													//is team based
		1,													//max walljumps
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_HEALTH),				//spawnable item types
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_HEALTH),				//item that will respawn after being picked up
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_HEALTH),				//items that can be dropped (by user or death)
	},

	//GAMETYPE_TDM
	{
		"team deathmatch",									//name
		"tdm",												//short name
		"tdm",												//hud
		1,													//is team based
		1,													//max walljumps
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH),	//spawnable item types
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH),	//item that will respawn after being picked up
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH),	//items that can be dropped (by user or death)
	},

	//GAMETYPE_CTF
	{
		"capture the flag",									//name
		"ctf",												//short name
		"ctf",												//hud
		1,													//is team based
		1,													//max walljumps
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH|IT_FLAG),	//spawnable item types
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH),	//item that will respawn after being picked up
		(IT_WEAPON|IT_AMMO|IT_ARMOR|IT_POWERUP|IT_HEALTH|IT_FLAG),	//items that can be dropped (by user or death)
	},

	//GAMETYPE_RACE
	{
		"race",												//name
		"race",												//short name
		"dm",												//hud
		0,													//is team based
		1,													//max walljumps
		(IT_WEAPON|IT_AMMO|IT_POWERUP),						//spawnable item types
		(IT_WEAPON|IT_AMMO|IT_POWERUP),						//item that will respawn after being picked up
		(0),												//items that can be dropped (by user or death)
	},

#ifdef MIDAIR
	//GAMETYPE_MIDAIR
	{
		"midair",											//name
		"midair",											//short name
		"dm",												//hud
		1,													//is team based
		1,													//max walljumps
		(0),												//spawnable item types
		(0),												//item that will respawn after being picked up
		(IT_HEALTH),										//items that can be dropped (by user or death)
	},
#endif

	//GAMETYPE_CA
	{
		"clan arena",									//name
		"ca",												//short name
		"tdm",												//hud
		1,													//is team based
		1,													//max walljumps
		(0),	//spawnable item types
		(0),	//item that will respawn after being picked up
		(0),	//items that can be dropped (by user or death)
	},

	{NULL}
};

#define NUM_GAMETYPES ( (int)(sizeof(gsGametypes)/sizeof(gs_gametype_t)) )

//======================================================
//		Game types
//======================================================


//=================
//GS_Gametype_FindByName
//=================
int GS_Gametype_FindByName( char *name )
{
	int				i;
	gs_gametype_t	*gtype;

	if( !name || !name[0] )
		return -1;

	for( i = 0, gtype = gsGametypes; gtype->name; gtype++, i++ ) {
		if( !Q_stricmp( gtype->name, name ) )
			return i;
	}

	return -1; //invalid name
}

//=================
//GS_Gametype_FindByShortName
//=================
int GS_Gametype_FindByShortName( char *name )
{
	int				i;
	gs_gametype_t	*gtype;

	if( !name || !name[0] )
		return -1;

	for( i = 0, gtype = gsGametypes; gtype->name; gtype++, i++ ) {
		if( !Q_stricmp( gtype->shortname, name ) )
			return i;
	}

	return -1; //invalid name
}

//=================
//GS_Gametype_StatusBar
//=================
char *GS_Gametype_StatusBar( int gametype ) 
{
	return gsGametypes[gametype].statusbar;
}


//=================
//GS_Gametype_Name
//=================
char *GS_Gametype_Name( int gametype )
{
	if( gametype < 0 || gametype >= NUM_GAMETYPES )
		return NULL;

	return gsGametypes[gametype].name;
}

//=================
//GS_Gametype_ShortName
//=================
char *GS_Gametype_ShortName( int gametype )
{
	if( gametype < 0 || gametype >= NUM_GAMETYPES )
		return NULL;

	return gsGametypes[gametype].shortname;
}


//=================
//GS_Gametype_IsTeamBased
//=================
qboolean GS_Gametype_IsTeamBased( int gametype )
{
	if( gametype < 0 || gametype >= NUM_GAMETYPES )
		return qfalse;

	return gsGametypes[gametype].isTeamBased;
}

//=================
//GS_Gametype_SpawnableItemMask
//=================
int GS_Gametype_SpawnableItemMask( int gametype )
{
	if( gametype < 0 || gametype >= NUM_GAMETYPES )
		return 0;

	return gsGametypes[gametype].spawnableItemsMask;
}


//=================
//GS_Gametype_RespawnableItemMask
//=================
int GS_Gametype_RespawnableItemMask( int gametype )
{
	if( gametype < 0 || gametype >= NUM_GAMETYPES )
		return 0;
	
	return gsGametypes[gametype].respawnableItemsMask;
}

//=================
//GS_Gametype_DropableItemMask
//=================
int GS_Gametype_DropableItemMask( int gametype )
{
	if( gametype < 0 || gametype >= NUM_GAMETYPES )
		return 0;

	return gsGametypes[gametype].dropableItemsMask;
}

//=================
//GS_GameType_MaxWallJumps
//=================
int	GS_GameType_MaxWallJumps( int gametype )
{
	if( gametype < 0 || gametype >= NUM_GAMETYPES )
		return 1;

	return gsGametypes[gametype].maxWallJumps;
}



//==================================================
//
//		TEAMS
//
//==================================================

static char *gs_teamNames[] =
{
	"SPECTATOR",
	"PLAYERS",
	"ALPHA",
	"BETA",
	"GAMMA",
	"DELTA",
	NULL
};

static char *gs_teamSkinsNames[] =
{
	NULL,		//null means user defined skin
	NULL,
	"default",
	"default",
	"default",
	"default",
	NULL
};

//=================
//GS_TeamName
//=================
char *GS_TeamName( int team )
{
	if( team < 0 && team >= GS_MAX_TEAMS )
		return NULL;

	return gs_teamNames[team];
}

//=================
//GS_TeamSkinName
//=================
char *GS_TeamSkinName( int team )
{
	if( team < 0 && team >= GS_MAX_TEAMS )
		return NULL;

	return gs_teamSkinsNames[team];
}

//=================
//GS_Teams_TeamFromName
//=================
int GS_Teams_TeamFromName( char *teamname )
{
	char *s;
	int		i;

	if( !teamname || !teamname[0] )
		return -1;	//invalid

	for( i = 0; i < GS_MAX_TEAMS; i++ ) 
	{
		s = gs_teamNames[i];
		if( !Q_stricmp( s, teamname ) ) 
			return i;
	}
	return -1;	//invalid
}

