/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.connector.rmi;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.rmi.MarshalledObject;
import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.RuntimeOperationsException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.naming.Context;

import mx4j.tools.adaptor.interceptor.Invocation;
import mx4j.tools.adaptor.interceptor.InvocationContext;
import mx4j.tools.adaptor.interceptor.InvocationResult;
import mx4j.tools.adaptor.rmi.RemoteAdaptor;
import mx4j.tools.connector.JMXAddress;
import mx4j.tools.connector.JMXConnector;
import mx4j.tools.connector.RemoteMBeanServer;

/**
 * Client-side connector for Remote Method Invocation. <p>
 * This is a base class for RMI/JRMP and RMI/IIOP connectors.
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.1 $
 */
public abstract class RMIConnector implements JMXConnector, Serializable
{
	private RemoteAdaptor remoteAdaptor;
	private transient RemoteMBeanServer m_server;
	private transient ThreadLocal m_context = new ThreadLocal();
	private transient ThreadLocal m_classloaderKey = new ThreadLocal();
	private transient ThreadLocal m_objectName = new ThreadLocal();

	/**
	 * Connect to a server-side adaptor using the given address.
	 * Keep it private for now, as it is not strictly needed
	 */
	private void connect(JMXAddress address) throws NamingException
	{
		String proto = address.getProtocol();
		if (!proto.equalsIgnoreCase("rmi") && !proto.equalsIgnoreCase("jrmp") && !proto.equalsIgnoreCase("iiop"))
		{
			throw new IllegalArgumentException("Bad protocol in address: " + proto);
		}

		// JNDI informations
		String provider = proto + "://" + address.getHost();
		int port = address.getPort();
		if (port > 0)
		{
			provider += ":" + port;
		}

		Map map = address.getProperties();
		map.put(Context.PROVIDER_URL, provider);

		connect(address.getPath(), new Hashtable(map));
	}

	/**
	 * Connect to a server-side adaptor using the given JNDI name and JNDI properties.
	 */
	public void connect(String jndiName, Hashtable properties) throws NamingException
	{
		if (jndiName == null)
		{
			throw new NamingException("Null JNDI name");
		}

		InitialContext ctx = new InitialContext(properties);
		Object object = ctx.lookup(jndiName);

		remoteAdaptor = narrow(object);
	}

	public RemoteMBeanServer getRemoteMBeanServer()
	{
		if (remoteAdaptor == null) throw new IllegalStateException("connect() has not been called");

		if (m_server == null) m_server = new RemoteMBeanServerImpl();
		return m_server;
	}

	protected abstract RemoteAdaptor narrow(Object object);

	public void setInvocationContext(InvocationContext context)
	{
		m_context.set(context);
	}

	private InvocationContext getInvocationContext()
	{
		return (InvocationContext)m_context.get();
	}

	private void setClassLoaderKey(String key)
	{
		m_classloaderKey.set(key);
	}

	private String getClassLoaderKey()
	{
		return (String)m_classloaderKey.get();
	}

	private void setObjectNameForClassLoader(ObjectName name)
	{
		m_objectName.set(name);
	}

	private ObjectName getObjectNameForClassLoader()
	{
		return (ObjectName)m_objectName.get();
	}

	public String getRemoteHostName() throws RemoteException
	{
		return remoteAdaptor.getHostName();
	}

	public String getRemoteHostAddress() throws RemoteException
	{
		return remoteAdaptor.getHostAddress();
	}

	public void close()
	{
		remoteAdaptor = null;
	}

	protected abstract NotificationListener createRemoteNotificationListener(NotificationListener listener) throws RemoteException;

	protected abstract NotificationFilter createRemoteNotificationFilter(NotificationFilter listener) throws RemoteException;

	protected Object invoke(String key, ObjectName objectName, String methodName, String[] signature, Object[] arguments) throws Exception
	{
		if (remoteAdaptor == null) throw new IllegalStateException("The connector has been closed");

		Invocation invocation = new Invocation();
		invocation.setClassLoaderKey(key);
		invocation.setObjectNameForClassLoader(objectName);
		invocation.setMethodName(methodName);
		invocation.setSignature(signature);
		invocation.setArguments(arguments);
		invocation.setInvocationContext(getInvocationContext());

		InvocationResult result = remoteAdaptor.invoke(invocation);
		return result.getResult();
	}

	private Object[] marshalObjects(Object[] args) throws IOException
	{
		if (args == null) return null;

		Object[] marshalled = new Object[args.length];
		for (int i = 0;i < marshalled.length;++i)
		{
			Object arg = args[i];
			marshalled[i] = marshalObject(arg);
		}
		return marshalled;
	}

	private Object marshalObject(Object obj) throws IOException
	{
		if (obj == null) return null;

		if (obj.getClass().getClassLoader() == null && !(obj instanceof MarshalledObject))
		{
			// This is a Java Object belonging to the JDK, no need to wrap it
			return obj;
		}
		else
		{
			return new MarshalledObject(obj);
		}
	}

	private class RemoteMBeanServerImpl implements RemoteMBeanServer
	{
		private RemoteMBeanServer m_proxy;
		private HashMap m_listeners = new HashMap();
		private HashMap m_filters = new HashMap();
		private HashMap m_handbacks = new HashMap();

		private RemoteMBeanServerImpl()
		{
			m_proxy = (RemoteMBeanServer)Proxy.newProxyInstance(RemoteMBeanServer.class.getClassLoader(), new Class[]{RemoteMBeanServer.class}, new Handler());
		}

        /**
         * @see javax.management.MBeanServerConnection#addNotificationListener(ObjectName, NotificationListener, NotificationFilter, Object)
         */
		public void addNotificationListener(ObjectName observed, NotificationListener listener, NotificationFilter filter, Object handback)
				throws InstanceNotFoundException, IOException
		{
			if (listener == null) throw new RuntimeOperationsException(new IllegalArgumentException("NotificationListener cannot be null"));

			NotificationListener l = null;
			NotificationFilter f = null;
			Object h = null;
			synchronized (this)
			{
				// Handle the listener
				l = findListener(listener);
				// Handle the filter
				f = findFilter(filter);
				// Handle the handback
				h = findHandback(handback);
			}

			// Invoke the remote server
			m_proxy.addNotificationListener(observed, l, f, h);

			synchronized (this)
			{
				// Everything went ok, register this listener-filter-handback trio
				// Overwrite in case of concurrent threads

				m_listeners.put(listener, l);
				if (filter != null) m_filters.put(filter, f);
				if (handback != null) m_handbacks.put(handback, h);
			}
		}

        /**
         * @see javax.management.MBeanServerConnection#addNotificationListener(ObjectName, ObjectName, NotificationFilter, Object)
         */
        public void addNotificationListener(
            ObjectName observed,
            ObjectName listener,
            NotificationFilter filter,
            Object handback)
            throws InstanceNotFoundException, IOException
        {
            if (listener == null) throw new RuntimeOperationsException(new IllegalArgumentException("NotificationListener cannot be null"));

            // Invoke the remote server
            m_proxy.addNotificationListener(observed, listener, filter, handback);
        }

        /**
         * @see javax.management.MBeanServerConnection#removeNotificationListener(ObjectName, NotificationListener, NotificationFilter, Object)
         */
		public void removeNotificationListener(ObjectName observed, NotificationListener listener, NotificationFilter filter, Object handback)
				throws InstanceNotFoundException, ListenerNotFoundException, IOException
		{
			if (listener == null)
			{
				throw new RuntimeOperationsException(new IllegalArgumentException("NotificationListener cannot be null"));
			}

			NotificationListener l = null;
			NotificationFilter f = null;
			Object h = null;
			synchronized (this)
			{
				l = (NotificationListener)m_listeners.get(listener);
				f = (NotificationFilter)m_filters.get(filter);
				h = m_handbacks.get(handback);
			}

			// Invoke the remote server
			m_proxy.removeNotificationListener(observed, l, f, h);

			synchronized (this)
			{
				// Everything went ok, remove this listener-filter-handback trio
				m_listeners.remove(listener);
				if (filter != null) m_filters.remove(filter);
				if (handback != null) m_handbacks.remove(handback);
			}
		}

        /**
         * @see javax.management.MBeanServerConnection#removeNotificationListener(ObjectName, NotificationListener)
         */
        public void removeNotificationListener(ObjectName observed, NotificationListener listener)
            throws InstanceNotFoundException, ListenerNotFoundException, IOException
        {
            if (listener == null)
            {
                throw new RuntimeOperationsException(new IllegalArgumentException("NotificationListener cannot be null"));
            }

            NotificationListener l = null;
            synchronized (this)
            {
                l = (NotificationListener)m_listeners.get(listener);
            }

            // Invoke the remote server
            m_proxy.removeNotificationListener(observed, l);

            synchronized (this)
            {
                // Everything went ok, remove this listener
                m_listeners.remove(listener);
            }
        }

        /**
         * @see javax.management.MBeanServerConnection#removeNotificationListener(ObjectName, ObjectName, NotificationFilter, Object)
         */
        public void removeNotificationListener(
            ObjectName observed,
            ObjectName listener,
            NotificationFilter filter,
            Object handback)
            throws InstanceNotFoundException, ListenerNotFoundException, IOException
        {
            if (listener == null)
            {
                throw new RuntimeOperationsException(new IllegalArgumentException("NotificationListener cannot be null"));
            }

            // Invoke the remote server
            m_proxy.removeNotificationListener(observed, listener, filter, handback);
        }

        /**
         * @see javax.management.MBeanServerConnection#removeNotificationListener(ObjectName, ObjectName)
         */
        public void removeNotificationListener(ObjectName observed, ObjectName listener)
            throws InstanceNotFoundException, ListenerNotFoundException, IOException
        {
            if (listener == null)
            {
                throw new RuntimeOperationsException(new IllegalArgumentException("NotificationListener cannot be null"));
            }

            // Invoke the remote server
            m_proxy.removeNotificationListener(observed, listener);
        }

        /**
         * @see javax.management.MBeanServerConnection#createMBean(String, ObjectName)
         */
		public ObjectInstance createMBean(String className, ObjectName objectName)
				throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException, IOException
		{
			return createMBean(className, objectName, null, null);
		}

        /**
         * @see javax.management.MBeanServerConnection#createMBean(String, ObjectName, ObjectName)
         */
		public ObjectInstance createMBean(String className, ObjectName objectName, ObjectName loaderName)
				throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException, InstanceNotFoundException, IOException
		{
			return createMBean(className, objectName, loaderName, null, null);
		}

        /**
         * @see javax.management.MBeanServerConnection#createMBean(String, ObjectName, Object[], String[])
         */
		public ObjectInstance createMBean(String className, ObjectName objectName, Object[] args, String[] parameters)
				throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException, IOException
		{
			try
			{
				setClassLoaderKey(Invocation.CLASSLOADER_REPOSITORY);
				Object[] marshalled = marshalObjects(args);
				return m_proxy.createMBean(className, objectName, marshalled, parameters);
			}
			catch (RemoteException x)
			{
				throw x;
			}
		}

        /**
         * @see javax.management.MBeanServerConnection#createMBean(String, ObjectName, ObjectName, Object[], String[])
         */
		public ObjectInstance createMBean(String className, ObjectName objectName, ObjectName loaderName, Object[] args, String[] parameters)
				throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException, InstanceNotFoundException, IOException
		{
			try
			{
				setClassLoaderKey(loaderName == null ? Invocation.CLASSLOADER_MBEANSERVER : Invocation.CLASSLOADER_LOADER);
				setObjectNameForClassLoader(loaderName);
				Object[] marshalled = marshalObjects(args);
				return m_proxy.createMBean(className, objectName, loaderName, marshalled, parameters);
			}
			catch (RemoteException x)
			{
				throw x;
			}
		}

        /**
         * @see javax.management.MBeanServerConnection#unregisterMBean(ObjectName)
         */
		public void unregisterMBean(ObjectName objectName)
				throws InstanceNotFoundException, MBeanRegistrationException, IOException
		{
			m_proxy.unregisterMBean(objectName);
		}

        /**
         * @see javax.management.MBeanServerConnection#getAttribute(ObjectName, String)
         */
		public Object getAttribute(ObjectName objectName, String attribute)
				throws MBeanException, AttributeNotFoundException, InstanceNotFoundException, ReflectionException, IOException
		{
			return m_proxy.getAttribute(objectName, attribute);
		}

        /**
         * @see javax.management.MBeanServerConnection#setAttribute(ObjectName, Attribute)
         */
		public void setAttribute(ObjectName objectName, Attribute attribute)
				throws InstanceNotFoundException, AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException, IOException
		{
			try
			{
				setClassLoaderKey(Invocation.CLASSLOADER_MBEAN);
				setObjectNameForClassLoader(objectName);
				attribute = marshalAttribute(attribute);
				m_proxy.setAttribute(objectName, attribute);
			}
			catch (RemoteException x)
			{
				throw x;
			}
		}

        /**
         * @see javax.management.MBeanServerConnection#getAttributes(ObjectName, String[])
         */
		public AttributeList getAttributes(ObjectName objectName, String[] attributes)
				throws InstanceNotFoundException, ReflectionException, IOException
		{
			return m_proxy.getAttributes(objectName, attributes);
		}

        /**
         * @see javax.management.MBeanServerConnection#setAttributes(ObjectName, AttributeList)
         */
		public AttributeList setAttributes(ObjectName objectName, AttributeList attributes)
				throws InstanceNotFoundException, ReflectionException, IOException
		{
			try
			{
				setClassLoaderKey(Invocation.CLASSLOADER_MBEAN);
				setObjectNameForClassLoader(objectName);
				AttributeList list = new AttributeList();
				for (int i = 0;i < attributes.size();++i)
				{
					Attribute attribute = (Attribute)attributes.get(i);
					attribute = marshalAttribute(attribute);
					list.add(attribute);
				}
				return m_proxy.setAttributes(objectName, list);
			}
			catch (IOException x)
			{
				throw new ReflectionException(x);
			}
		}

        /**
         * @see javax.management.MBeanServerConnection#invoke(ObjectName, String, Object[], String[])
         */
		public Object invoke(ObjectName objectName, String methodName, Object[] args, String[] parameters)
				throws InstanceNotFoundException, MBeanException, ReflectionException, IOException
		{
			try
			{
				setClassLoaderKey(Invocation.CLASSLOADER_MBEAN);
				setObjectNameForClassLoader(objectName);
				Object[] marshalled = marshalObjects(args);
				return m_proxy.invoke(objectName, methodName, marshalled, parameters);
			}
			catch (RemoteException x)
			{
				throw x;
			}
			catch (IOException x)
			{
				throw new ReflectionException(x);
			}
		}

        /**
         * @see javax.management.MBeanServerConnection#getDefaultDomain()
         */
		public String getDefaultDomain() throws IOException
		{
			return m_proxy.getDefaultDomain();
		}

        /**
         * @see javax.management.MBeanServerConnection#getDomains()
         */
        public String[] getDomains() throws IOException
        {
            return m_proxy.getDomains();
        }

        /**
         * @see javax.management.MBeanServerConnection#getMBeanCount()
         */
		public Integer getMBeanCount() throws IOException
		{
			return m_proxy.getMBeanCount();
		}

        /**
         * @see javax.management.MBeanServerConnection#isRegistered(ObjectName)
         */
		public boolean isRegistered(ObjectName objectname) throws IOException
		{
			return m_proxy.isRegistered(objectname);
		}

        /**
         * @see javax.management.MBeanServerConnection#getMBeanInfo(ObjectName)
         */
		public MBeanInfo getMBeanInfo(ObjectName objectName)
				throws InstanceNotFoundException, IntrospectionException, ReflectionException, IOException
		{
			return m_proxy.getMBeanInfo(objectName);
		}

        /**
         * @see javax.management.MBeanServerConnection#getObjectInstance(ObjectName)
         */
		public ObjectInstance getObjectInstance(ObjectName objectName)
				throws InstanceNotFoundException, IOException
		{
			return m_proxy.getObjectInstance(objectName);
		}

        /**
         * @see javax.management.MBeanServerConnection#isInstanceOf(ObjectName, String)
         */
		public boolean isInstanceOf(ObjectName objectName, String className)
				throws InstanceNotFoundException, IOException
		{
			return m_proxy.isInstanceOf(objectName, className);
		}

        /**
         * @see javax.management.MBeanServerConnection#queryMBeans(ObjectName, QueryExp)
         */
		public Set queryMBeans(ObjectName patternName, QueryExp filter) throws IOException
		{
			return m_proxy.queryMBeans(patternName, filter);
		}

        /**
         * @see javax.management.MBeanServerConnection#queryNames(ObjectName, QueryExp)
         */
		public Set queryNames(ObjectName patternName, QueryExp filter) throws IOException
		{
			return m_proxy.queryNames(patternName, filter);
		}

		private NotificationListener findListener(NotificationListener listener) throws RemoteException
		{
			// First, lookup in the map
			NotificationListener l = (NotificationListener)m_listeners.get(listener);
			if (l != null) return l;

			// Check if user provided a specific remote listener, otherwise go default
			if (listener instanceof RemoteNotificationListenerSupport)
			{
				return listener;
			}
			else
			{
				return createRemoteNotificationListener(listener);
			}
		}

		private NotificationFilter findFilter(NotificationFilter filter) throws RemoteException
		{
			if (filter == null) return null;

			NotificationFilter f = (NotificationFilter)m_filters.get(filter);
			if (f != null) return f;

			// Check if user provided a specific remote filter, otherwise go default
			if (filter instanceof RemoteNotificationFilterSupport)
			{
				return filter;
			}
			else
			{
				return createRemoteNotificationFilter(filter);
			}
		}

		private Object findHandback(Object handback)
		{
			if (handback == null) return null;

			Object h = m_handbacks.get(handback);
			if (h != null) return h;

			// Replace the handback with a serializable object
			try
			{
				return marshalObject(handback);
			}
			catch (IOException x)
			{
				throw new RuntimeOperationsException(new IllegalArgumentException("Handback is not serializable"));
			}
		}

		private Attribute marshalAttribute(Attribute attribute) throws IOException
		{
			Object value = attribute.getValue();
			Object marshalledValue = marshalObject(value);
			if (value != marshalledValue)
			{
				attribute = new Attribute(attribute.getName(), marshalledValue);
			}
			return attribute;
		}

		private class Handler implements InvocationHandler
		{
			public Object invoke(Object proxy, Method method, Object[] args) throws Throwable
			{
				// Needed for no-parameter methods
				if (args == null) args = new Object[0];

				int length = method.getParameterTypes().length;
				String[] signature = new String[length];
				for (int i = 0;i < length;++i)
				{
					signature[i] = method.getParameterTypes()[i].getName();
				}

				String key = getClassLoaderKey();
				setClassLoaderKey(null);
				ObjectName name = getObjectNameForClassLoader();
				setObjectNameForClassLoader(null);
				Object retValue = RMIConnector.this.invoke(key, name, method.getName(), signature, args);
				return retValue;
			}
		}
	}
}
