/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.connector;

import java.io.IOException;
import java.lang.reflect.Proxy;

import javax.management.Attribute;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import mx4j.util.StandardMBeanProxy;

/**
 * A utility class that creates proxies for invocation on standard MBeans (does not work for DynamicMBeans) on remote MBeanServers. <br>
 * It is recommended that a new interface is used instead of the management interface of the MBean. This new interface would be a
 * subset of the management interface and the methods would declare an IOException (or a subclass such as RemoteException) in the
 * throws clause of each method. If this suggestion is not followed, remote exception will be thrown as
 * {@link java.lang.reflect.UndeclaredThrowableException UndeclaredThrowableException}s. <br>
 * Usage example:
 * <pre>
 * public interface MyServiceMBean
 * {
 *    public void sayHello(String user) throws UnknownUserException;
 *    public void doesNotMakeSenseRemotely();
 * }
 * public class MyService implements MyServiceMBean {...}
 * public interface MyRemoteService
 * {
 *    public void sayHello(String user) throws UnknownUserException, RemoteException;
 * }
 * pulic class Main
 * {
 *    public static void main(String[] args) throws Exception
 *    {
 *       RemoteMBeanServer server = ...;
 *       ObjectName myServiceObjectName = ...;
 *
 *       MyRemoteService mbean = (MyRemoteService)RemoteStandardMBeanProxy.create(MyRemoteService.class, server, myServiceObjectName);
 *
 *       mbean.sayHello("World");
 *    }
 * }
 * </pre>
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.1 $
 * @deprecated Replaced by {@link javax.management.MBeanServerInvocationHandler}
 */
public class RemoteStandardMBeanProxy extends StandardMBeanProxy
{
	/**
	 * Creates a proxy with the given MBean interface for an MBean with the specified ObjectName
	 * living in the specified remote MBeanServer. <br>
	 * If the MBean is unregistered after the proxy has been created, an attempt to call any method
	 * on the proxy will result in a {@link java.lang.reflect.UndeclaredThrowableException UndeclaredThrowableException}
	 * being thrown. MBeanServer's behavior would be to throw an InstanceNotFoundException, but this exception
	 * is normally not declared in the throws clause of MBean's management interface, thus resulting
	 * in the UndeclaredThrowableException being thrown instead.
	 */
	public static Object create(Class mbeanInterface, MBeanServerConnection server, ObjectName name)
	{
		if (mbeanInterface == null) throw new IllegalArgumentException("MBean interface cannot be null");
		if (!mbeanInterface.isInterface()) throw new IllegalArgumentException("Class parameter must be an interface");
		if (server == null) throw new IllegalArgumentException("Remote MBeanServer cannot be null");
		if (name == null) throw new IllegalArgumentException("MBean name cannot be null");

		// The client must be able to cast the returned object to the mbeanInterface it passes,
		// so the classloader must be the same
		ClassLoader loader = mbeanInterface.getClassLoader();
		return Proxy.newProxyInstance(loader, new Class[]{mbeanInterface}, new RemoteHandler(server, name));
	}

	private static class RemoteHandler extends Handler
	{
		private MBeanServerConnection m_server;
		private ObjectName m_name;

		private RemoteHandler(MBeanServerConnection server, ObjectName name)
		{
			m_server = server;
			m_name = name;
		}

		protected void setAttribute(Attribute attribute) throws InstanceNotFoundException, AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException, IOException
		{
			m_server.setAttribute(m_name, attribute);
		}

		protected Object getAttribute(String attribute) throws InstanceNotFoundException, AttributeNotFoundException, MBeanException, ReflectionException, IOException
		{
			return m_server.getAttribute(m_name, attribute);
		}

		protected Object invokeOperation(String method, Object[] args, String[] params) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException
		{
			return m_server.invoke(m_name, method, args, params);
		}
	}
}
