/**
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */
package javax.management.openmbean;

import java.io.Serializable;
import java.io.ObjectInputStream;
import java.io.IOException;
import java.io.StreamCorruptedException;
import java.util.List;
import java.util.Iterator;
import java.util.ArrayList;
import java.util.Collections;

/**
 *
 * @author <a href="mailto:shadow12@users.sourceforge.net">Bronwen Cassidy</a>
 * @version $Revision: 1.13 $
 */
public class TabularType extends OpenType implements Serializable
{
	private static final long serialVersionUID = 6554071860220659261L;

	private CompositeType rowType = null;
	private List indexNames = null;

	private transient int m_hashcode = 0;
	private transient String m_classDescription = null;

	/**
	 * Constructs a TabularType instance
	 * <p><b>Parameters:</b></p>
	 * <p>typeName    - the name given to this tabularType instance, cannot be null or empty</p>
	 * <p>description - the human readable description of the tabularType this instance refers to cannot be null or empty</p>
	 * <p>rowType     - the type of the row elements of tabular data values described by this tabular type instance cannot be null</p>
	 * <p>indexNames  - the names of the items which are used to uniquely index each row element in the tabular data values, cannot be null or empty<br/>
	 *                - each element should be an item name defined in rowType.
	 * 				  - NOTE: The order of the itemNames is used by the methods <tt>get</tt> and <tt>remove</tt> of class TabularData to match their array of values to items</p>
	 * <p><b>Throws:</b></p>
	 * <p>IllegalArgumentException - if rowType is null, indexNames or any item in indexNames are null or empty, typeName or description are null or empty</p>
	 * <p>OpenDataException - if an element of indexNames is not an item defined in RowType</p>
	 */
	public TabularType(String typeName, String description, CompositeType rowType, String[] indexNames) throws OpenDataException
	{
		super(TabularData.class.getName(), typeName, description);
		if (typeName.trim().length() == 0) throw new IllegalArgumentException("TabularType name can't be empty");
		if (description.trim().length() == 0) throw new IllegalArgumentException("TabularType description can't be empty");
		validate(rowType, indexNames);
		this.rowType = rowType;
		ArrayList temp = new ArrayList();
		for (int i = 0; i < indexNames.length; ++i) temp.add(indexNames[i]);
		this.indexNames = Collections.unmodifiableList(temp);
	}

	/**
	 * checks if all of the values of indexNames match items defined in rowType, that the item is not null or zero length
	 * we then create an unmodifiable list from the "valid" string array. If any validity checks fail an OpenDataException is thrown
	 */
	private void validate(CompositeType rowType, String[] indexNames) throws OpenDataException
	{
		if (rowType == null) throw new IllegalArgumentException("The CompositeType passed in cannot be null");
		if (indexNames == null || indexNames.length == 0) throw new IllegalArgumentException("The String[] indexNames cannot be null or empty");

		for(int i = 0; i < indexNames.length; i++)
		{
			String item = indexNames[i];
			if(item == null || item.length() == 0)
			{
				throw new IllegalArgumentException("An Item in the indexNames[] cannot be null or of zero length");
			}
            if(!(rowType.containsKey(item)))
			{
				throw new OpenDataException("Element value: " + indexNames[i] + " at index: " + i + " is not a valid item name for RowType");
			}
		}
	}

	/**
	 * @return CompositeType - the type of the row elements of tabular data values described by this TabularType instance
	 */
	public CompositeType getRowType()
	{
		return rowType;
	}

	/**
	 * @return - Returns, in the same order as was given to this instance's constructor,
	 * 			an unmodifiable List of the names of the items the values of which are used to uniquely index each row element
	 * 			of tabular data values described by this TabularType instance
	 */
	public List getIndexNames()
	{
		return indexNames;
	}

	/**
	 * checks if the typeName of the object is the same as the typeName of this TabularType instance
	 * @param object - the tabularType instance to check if the typeName is a value of this instance
	 * @return true if the objects typeName is the same as that of this instance, false if either the object is not an instance of this class or the typeNames do not match
	 */
	public boolean isValue(Object object)
	{
		if(!(object instanceof TabularData)) return false;
		TabularData tabularData = (TabularData)object;
        return equals(tabularData.getTabularType());
	}

	/**
	 * determines if the object passed in is equal to this class instance
	 * @param object the object to test if equal to this class
	 * @return true if the object is equal to this class, false if the object is not an instance of this class or if the data
	 * 			contained in the object is not the same as the data contained in this class instance
	 */
	public boolean equals(Object object)
	{
		if(object == this) return true;
		if(!(object instanceof TabularType)) return false;
		TabularType tabularType = (TabularType)object;
		return (rowType.equals(tabularType.rowType) && indexNames.equals(tabularType.indexNames) && getTypeName().equals(tabularType.getTypeName()));
	}

	/**
	 * computes a hashcode for this instance
	 * @return the int value of the hashcode.
	 * 			As TabularTypeBackUP instances are immutable, the hash code for this instance is calculated once, on the first call to hashCode, and then the same value is returned for subsequent calls
	 */
	public int hashCode()
	{
		if(m_hashcode == 0)
		{
			int result = getTypeName().hashCode();
			result += rowType.hashCode();
			Iterator i = indexNames.iterator();
			while (i.hasNext())
			{
				result += i.next().hashCode();
			}
			m_hashcode = result;
		}
		return m_hashcode;
	}

	/**
	 * creates a user friendly string value of the class
	 * @return the string representation of the class
	 * 			As TabularTypeBackUP instances are immutable, the string representation for this instance is calculated once,
	 * 			on the first call to toString, and then the same value is returned for subsequent calls ie. we use lazy initialization
	 */
	public String toString()
	{
		if(m_classDescription == null)
		{
			StringBuffer classString = new StringBuffer();
			classString.append("TabularType name: ").append(getTypeName()).append(" rowType: ").append(rowType.toString()).append("indexNames: (");
			for(Iterator i = indexNames.iterator(); i.hasNext();)
			{
				classString.append(i.next().toString()).append(", ");
			}
			classString.delete(classString.length() -2, classString.length());
			classString.append(")");
			m_classDescription = classString.toString();
		}
		return m_classDescription;
	}
}
