/* BenzeneFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs.chemical;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import org.grinvin.Embedding;
import org.grinvin.Graph;
import org.grinvin.Vertex;
import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.factories.graphs.AbstractGraphFactory;
import org.grinvin.preferences.GrinvinPreferences;

/**
 * Factory for benzenoids.
 */
public class BenzeneFactory extends AbstractGraphFactory {
    
    //
    @Override
    protected void createGraph(Graph graph, Embedding embedding) throws FactoryException {
        numberOfHexagons = ((Integer)values[0]).intValue();
        int rank = ((Integer)values[1]).intValue();

        try {
            File path = GrinvinPreferences.INSTANCE.getProgramPath("benzene");
            ProcessBuilder pb = new ProcessBuilder(path.toString(), "" + numberOfHexagons, "b", "p");
            Process generator = pb.start();
            in = new BufferedInputStream(generator.getInputStream());
        } catch (IOException ex) {
            throw new FactoryException("The program benzene was not found.", ex);
        }
        
        
        int[] planarcode = getNextGraph();
        int graphCount = 1;
        while(planarcode != null && graphCount != rank) {
            planarcode = getNextGraph();
            graphCount++;
        }
        
        if(planarcode != null && graphCount == rank) {
            int nrOfVertices = planarcode[1];
            int[][] benzene = new int[nrOfVertices][];
            int pos = 2;
            int firstDegreeThree = -1;
            for(int vertex=0; vertex < nrOfVertices; vertex++){
                if(planarcode[pos+3]==0){ //degree 3
                    if(firstDegreeThree == -1)
                        firstDegreeThree = vertex;
                    benzene[vertex] = new int[3];
                    benzene[vertex][0] = planarcode[pos++]-1;
                    benzene[vertex][1] = planarcode[pos++]-1;
                    benzene[vertex][2] = planarcode[pos++]-1;
                } else { //degree 2
                    benzene[vertex] = new int[2];
                    benzene[vertex][0] = planarcode[pos++]-1;
                    benzene[vertex][1] = planarcode[pos++]-1;
                }
                pos++;
            }
            int[][] coordinates = new int[nrOfVertices][];
            coordinates[firstDegreeThree] = new int[] {0,0};
            coordinates[benzene[firstDegreeThree][0]] = new int[] {1,0};
            coordinates[benzene[firstDegreeThree][1]] = new int[] {0,1};
            coordinates[benzene[firstDegreeThree][2]] = new int[] {-1,-1};
            //if(benzene[benzene[firstDegreeThree][0]].length == 2)
            //    ;
            //else
            //    ;
            giveCoordinates(firstDegreeThree, benzene[firstDegreeThree][0], 0, coordinates, benzene);
            giveCoordinates(firstDegreeThree, benzene[firstDegreeThree][1], 2, coordinates, benzene);
            giveCoordinates(firstDegreeThree, benzene[firstDegreeThree][2], 4, coordinates, benzene);
            //for(int i= 0; i<coordinates.length; i++)
            //    System.out.println(Arrays.toString(coordinates[i]));
            
            Vertex[] vertices = new Vertex[benzene.length];
            for(int j=0; j < benzene.length; j++)
                vertices[j] = graph.addNewVertex("C");
            for(int j=0; j < benzene.length; j++){
                for(int i = 0; i<benzene[j].length; i++)
                    if(!graph.areAdjacent(graph.getVertex(j),graph.getVertex(benzene[j][i])))
                        // maybe more efficient to just check: graph.getVertex(j) < graph.getVertex(benzene[j][i])
                        graph.addNewEdge(graph.getVertex(j),graph.getVertex(benzene[j][i]),null);
            }
            
            //TODO: find correct rotation for display
            embedding.setDimension(2);
            double[][] realCoordinates = new double[nrOfVertices][2];
            double minX = Double.POSITIVE_INFINITY;
            double maxX = Double.NEGATIVE_INFINITY;
            double minY = Double.POSITIVE_INFINITY;
            double maxY = Double.NEGATIVE_INFINITY;
            for(int i=0; i<nrOfVertices; i++){
                realCoordinates[i][0]=coordinates[i][1]*Math.cos(Math.PI/6);
                if(realCoordinates[i][0]<minX)
                    minX=realCoordinates[i][0];
                if(realCoordinates[i][0]>maxX)
                    maxX=realCoordinates[i][0];
                realCoordinates[i][1]=coordinates[i][0] - coordinates[i][1]*Math.sin(Math.PI/6);
                if(realCoordinates[i][1]<minY)
                    minY=realCoordinates[i][1];
                if(realCoordinates[i][1]>maxY)
                    maxY=realCoordinates[i][1];
            }
            double transX = - (minX+maxX)/2;
            double transY = - (minY+maxY)/2;
            double scale = (maxX - minX) > (maxY - minY) ? 2/(maxX - minX) : 2/(maxY - minY);
            for(int i=0; i<nrOfVertices; i++){
                realCoordinates[i][0]=(realCoordinates[i][0]+transX)*scale;
                realCoordinates[i][1]=(realCoordinates[i][1]+transY)*scale;
                //System.out.println(Arrays.toString(c));
                embedding.setCoordinates(vertices[i],realCoordinates[i]);
            }
        } else {
            throw new FactoryException("There are only " + (graphCount - 1) + " benzenoids with " + numberOfHexagons + " hexagons.");
        }
        try {
            in.close();
        } catch(IOException ex) {
            // TODO: handle this exception
        }
    }
    
    //
    @Override
    protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        if (((Integer)values[0]).intValue() < 2 || ((Integer)values[0]).intValue() > 30)
            throw new FactoryParameterException("Number of hexagons must be in the range 2 to 30.");
    }
    
    private InputStream in;
    private int numberOfHexagons; // NOPMD
    
    private static final int[][] VECTORS = {{1,0},{1,1},{0,1},{-1,0},{-1,-1},{0,-1}}; //run through a hexagon clockwise
    
    private int[] getNextGraph() throws FactoryException {
        int planarcode[] = null;
        try {
            int c = in.read();
            //debug
            //InputStream errorIn = generator.getErrorStream();
            //int error = errorIn.read();
            //while(error!=-1){
            //    System.out.print((char)error);
            //    error = errorIn.read();
            //}
            
            if(c==-1)
                return null; // EOF
            else if(c==0){
                throw new FactoryException("Graph is too big to be translated from planar code.");
            } else {
                int numberOfZeroes=0;
                int numberOfVertices = c;
                int pos=1;
                if(c=='>'){
                    //possibly a header
                    int c1 = in.read();
                    if(c1==0)
                        numberOfZeroes++;
                    int c2 = in.read();
                    if(c2==0)
                        numberOfZeroes++;
                    if(c1=='>' && c2=='p') {
                        //definitely a header
                        while(c!='<')
                            c=in.read();
                        c=in.read();
                        if(c!='<')
                            return null; // error in header
                        c=in.read();
                        if(!(c>0))
                            return null; //error in file or EOF
                        numberOfVertices = c;
                        planarcode = new int[(numberOfVertices*numberOfVertices) + 1];
                        planarcode[pos++]=c;
                    } else {
                        //no header
                        planarcode = new int[(numberOfVertices*numberOfVertices) + 1];
                        planarcode[pos++]=c;
                        planarcode[pos++]=c1;
                        planarcode[pos++]=c2;
                    }
                } else {
                    planarcode = new int[(numberOfVertices*numberOfVertices) + 1];
                    planarcode[pos++]=c;
                }
                while(numberOfZeroes < numberOfVertices){
                    c=in.read();
                    if(c==0)
                        numberOfZeroes++;
                    planarcode[pos++]=c;
                }
                planarcode[0]=pos-1;
                return planarcode;
            }
        } catch (IOException e) {
            return null;
        }
    }
    
    
    //vertex[previous] has degree 2
    private boolean checkClockwise(int previouspreviousVertex, int previousVertex, int previousStep, int[][] coordinates, int[][] benzene) {
        int step = (previousStep+1)%6;
        int[] tempCoords = addVector(coordinates[previousVertex], step);
        int vertex;
        if(benzene[previousVertex][0]==previouspreviousVertex)
            vertex=benzene[previousVertex][1];
        else
            vertex = benzene[previousVertex][0];
        int nodes = 2; //make sure we only check a hexagon
        while(coordinates[vertex]==null && nodes < 6){
            previouspreviousVertex = previousVertex;
            previousVertex = vertex;
            int previousIndex = -1;
            for(int i=0; i < benzene[previousVertex].length; i++)
                if(benzene[previousVertex][i]==previouspreviousVertex)
                    previousIndex = i;
            vertex = benzene[previousVertex][(previousIndex-1+benzene[previousVertex].length)%benzene[previousVertex].length];
            step = (step + 1)%6;
            tempCoords = addVector(tempCoords, step);
            nodes++;
        }
        
        return (coordinates[vertex] != null) && (coordinates[vertex][0] == tempCoords[0]) && (coordinates[vertex][1] == tempCoords[1]);
    }
    
    private int[] addVector(int[] vector, int vectorToAdd){
        int[] newVector = new int[2];
        newVector[0] = vector[0] + VECTORS[vectorToAdd][0];
        newVector[1] = vector[1] + VECTORS[vectorToAdd][1];
        return newVector;
    }
    
    private void giveCoordinates(int previousVertex, int vertex, int step, int[][] coordinates, int[][] benzene){
        coordinates[vertex] = addVector(coordinates[previousVertex], step);
        if(benzene[vertex].length == 2) {
            int index = (benzene[vertex][0]==previousVertex) ? 1 : 0;
            if(coordinates[benzene[vertex][index]]==null)
                if(this.checkClockwise(previousVertex, vertex, step, coordinates, benzene))
                    giveCoordinates(vertex, benzene[vertex][index], (step+1)%6, coordinates, benzene);
                else
                    giveCoordinates(vertex, benzene[vertex][index], (step-1+6)%6, coordinates, benzene);
        } else {
            int previousIndex = -1;
            for(int i=0; i < 3; i++)
                if(benzene[vertex][i]==previousVertex)
                    previousIndex = i;
            if(coordinates[benzene[vertex][(previousIndex-1+3)%3]]==null)
                giveCoordinates(vertex, benzene[vertex][(previousIndex-1+3)%3], (step+1)%6, coordinates, benzene);
            if(coordinates[benzene[vertex][(previousIndex+1)%3]]==null)
                giveCoordinates(vertex, benzene[vertex][(previousIndex+1)%3], (step-1+6)%6, coordinates, benzene);
        }
    }
}
