/* GraphListElementSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.list;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.graphs.GraphURI;
import org.grinvin.io.InvariantValuesSaver;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionSaver;
import org.grinvin.io.graphs.GraphBundleSaver;
import org.grinvin.list.graphs.GraphListElement;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;

/**
 * Saves {@link GraphListElement}s.
 */
public class GraphListElementSaver {
    
    //
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    /**
     * Save a (session) graph list element into a (workspace) directory. Uses the uri
     * of the graph to determine the destination file name. If no URI was
     * assigned, a new URI is created.<p>
     * Only saves graphs with a session URI and which are dirty or for which the
     * corresponding file does not yet exist. The 'dirty' flag of the graph list
     * element is cleared after it is saved.
     */
    public static void saveIntoWorkspace(GraphListElement gle,
            SectionSaver ssaver) throws IOException {
        URI uri = gle.getURI();
        if (!GraphURI.isSession(uri)) {
            if (GraphURI.isGlobal(uri)) {
                cacheInvariants(gle);
            }
            return;
        }
        if (!gle.gotGraph())
            return;
        
        String name = LoaderSaverHelper.prepareSessionURI(gle, "graphs", "gph");
        
        if (!ssaver.containsSection(name)) 
            GraphBundleSaver.save(gle.getBundle(), ssaver, name);
    }

    private static void cacheInvariants(GraphListElement gle) {
        URI uri = gle.getURI();
        GraphBundleView bundle = gle.getBundle();
        if (bundle == null)
            return;
        
        checkCacheDir();
        
        File cachedFile = getCachedFile(uri);
        try {
            InvariantValuesSaver.save(bundle.getInvariantValues(), new FileOutputStream(cachedFile));
        } catch (FileNotFoundException ex) {
            LOGGER.log(Level.FINE, "Unable to save cached invariant values", ex);
            //TODO: handle, it is ok to just ignore this for now, the values will be recomputed
        } catch (IOException ex) {
            LOGGER.log(Level.FINE, "Unable to save cached invariant values", ex);
            //TODO: handle, it is ok to just ignore this for now, the values will be recomputed
        }
    }
    
    private static String getCacheDir() {
        return GrinvinPreferences.getInstance().getStringPreference(Preference.GRINVIN_CACHE_DIR)+"/invariantvalues";
    }
    
    private static String getCachedFilename(URI uri) {
        String cachedFilename;
        String raw = uri.getRawSchemeSpecificPart();
        int pos = raw.indexOf('?');
        if (pos >= 0)
            cachedFilename = raw.substring(0, pos) + "-" + raw.substring(pos+1).replace('&', '_').replace('=','_') + ".xml";
        else
            cachedFilename = raw + "-";
        return cachedFilename;
    }
    
    private static File getCachedFile(URI uri) {
        return new File (getCacheDir() + "/" + getCachedFilename(uri));
    }

    
    //
    private static void checkCacheDir() {
        File cacheDir = new File(getCacheDir());
        if (cacheDir.exists()) {
            if (!cacheDir.isDirectory()) {
                throw new RuntimeException("Cache location is not a directory: " + cacheDir);
            }
        } else {
            if (!cacheDir.mkdir()) {
                LOGGER.log(Level.WARNING, "Could not create cache directory: " + cacheDir);
            }
        }
    }
    
}
