/* GradientBarChartPanel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.details;

import be.ugent.caagt.swirl.lists.TypedListModel;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.List;
import javax.swing.DefaultListSelectionModel;
import javax.swing.JPanel;
import javax.swing.ListSelectionModel;
import javax.swing.ToolTipManager;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

/**
 * Panel which displays a list of real numbers in a bar chart.<p>
 * Data for the bar chart is taken from a list model of real numbers.
 * Selection information comes from a list selection model.
 */
@SuppressWarnings("PMD.TooManyFields")
public class GradientBarChartPanel extends JPanel 
        implements ListSelectionListener, ListDataListener, MouseListener {
    //TODO: factor out common code with BarChartPanel
    
    //
    private final TypedListModel<Double> dataModel;
    
    //
    private  ListSelectionModel selectionModel;
    
    //
    private final Color selectColor = new Color(0,127,255);
    
    //
    private final Color lineColor = Color.BLACK;
    
    //
    private final int preferredHeight = 250;
    
    //
    private final int barMargin = 2;
    
    //
    private final int horMargin = 7;
    
    //
    private final int verMargin = 10;
    
    //
    private final int minBarWidth = 2*barMargin+10;
    
    //
    private final int maxBarWidth = 2*barMargin+30;
    
    //
    private final int preferredBarWidth = 2*barMargin+20;
    
    
    /**
     * Return selection model.
     */
    public ListSelectionModel getSelectionModel() {
        return selectionModel;
    }
    
    //
    private final double referenceValue;
    
    /**
     * Return the reference value which is used to determine
     * the height of the bars. This value corresponds to a
     * bar of full height.
     */
    public double getReferenceValue() {
        return referenceValue;
    }
    
    /**
     * Change the reference value which is used to determine the
     * height of the bars.
     */
    public void setSelectionModel(ListSelectionModel selectionModel) {
        if (selectionModel != null)
            selectionModel.removeListSelectionListener(this);
        this.selectionModel = selectionModel;
        if (selectionModel != null)
            selectionModel.addListSelectionListener(this);
    }
    
    /**
     * Create a bar chart panel with given data model, selection model and
     * reference value.
     */
    public GradientBarChartPanel(TypedListModel<Double> dataModel,
            ListSelectionModel selectionModel, double referenceValue) {
        super(null);
        setBackground(Color.WHITE);
        this.dataModel = dataModel;
        dataModel.addListDataListener(this);
        this.selectionModel = selectionModel;
        if (selectionModel != null)
            selectionModel.addListSelectionListener(this);
        this.referenceValue = referenceValue;
        setPreferredSize(new Dimension(dataModel.size() * preferredBarWidth, preferredHeight));
        
        addComponentListener(new ComponentAdapter() {
            public void componentResized(ComponentEvent e) {
                computeDimensions();
                repaint();
            }
        });
        
        computeDimensions ();
        assignColors();
        
        addMouseListener (this);
        
        ToolTipManager.sharedInstance().registerComponent(this);
    }
    
    /**
     * Create a bar chart panel with given data model and reference value. Creates
     * a new selection model.
     */
    public GradientBarChartPanel(TypedListModel<Double> dataModel,double referenceValue) {
        this(dataModel, new DefaultListSelectionModel(), referenceValue);
    }
    
    //
    private static double maxAbsoluteValue(List<Double> list) {
        double max = 0.0;
        for (double el: list)
            if (el > max)
                max = el;
            else if (-el > max)
                max = el;
        return max;
    }
    
    /**
     * Create a bar chart panel with given data model and selection model.
     * Uses the currently largest absolute value of the data model as reference
     * value.
     */
    public GradientBarChartPanel(TypedListModel<Double> dataModel,
            ListSelectionModel selectionModel) {
        this(dataModel, selectionModel, maxAbsoluteValue(dataModel));
    }
    
    /**
     * Create a bar chart panel with given data model.
     * Creates a new selection model and uses the currently largest absolute
     * value of the data model as reference value.
     */
    public GradientBarChartPanel(TypedListModel<Double> dataModel) {
        this(dataModel, maxAbsoluteValue(dataModel));
    }
    
    //
    private int yOffset;
    
    //
    private double factor;
    
    //
    private int actualWidth;
    
    //
    private int xOffset;
    
    //
    private void computeDimensions() {
        int nr = dataModel.getSize();
        
        // vertical dimensions
        this.yOffset = getHeight() / 2;
        this.factor = (yOffset - verMargin) / referenceValue;
        
        // horizontal dimensions
        this.actualWidth = getWidth() - 2*horMargin;
        this.xOffset = horMargin;
        if (actualWidth > nr * maxBarWidth) {
            actualWidth = nr * maxBarWidth;
            xOffset = (getWidth() - actualWidth)/2;
        } else if (actualWidth < nr * minBarWidth) {
            actualWidth = nr * minBarWidth;
        }
    }
    
    //
    private static final double TOLERANCE = 1E-06;
    
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        
        // reference line
        g.setColor(lineColor);
        g.drawLine(0, yOffset, getWidth(), yOffset);
        
        // bars
        int colorIndex = 1;
        int size = dataModel.getSize();
        int barWidth = actualWidth /size - 2*barMargin;
        for (int i=0; i < size; i++) {
            double value = dataModel.get(i);
            Color drawColor = selectionModel.isSelectedIndex(i) ? selectColor : lineColor;
            
            int x = xOffset + actualWidth * i / size;
            if (value < - TOLERANCE) {
                // negative bar
                int height = (int)(-value * factor);
                g.setColor(drawColor);
                g.fillRect(x+barMargin, yOffset-2, barWidth, height);
                if (selectionModel.isSelectedIndex(i))
                    g.setColor(negatedColors[i]);
                else
                    g.setColor(colors[i]);
                g.fillRect(x+barMargin+2, yOffset, barWidth-4, height-4);
                    // TODO: avoid this trick to draw thick lines, use Graphics2D?
            } else if (value > TOLERANCE) {
                // postive bar
                int height = (int)(value * factor);
                g.setColor(drawColor);
                g.fillRect(x+barMargin, yOffset - height + 2, barWidth, height);
                if (selectionModel.isSelectedIndex(i))
                    g.setColor(negatedColors[i]);
                else
                    g.setColor(colors[i]);
                g.fillRect(x+barMargin+2, yOffset - height+4, barWidth-4, height-4);
                    // TODO: avoid this trick to draw thick lines, use Graphics2D?
            } else {
                // zero
                g.setColor(drawColor);
                g.fillRect(x+barMargin, yOffset - 2, barWidth, 4);
            }
        }
    }
    
    //
    private double max;
    
    //
    private double min;
    
    //
    private Color[] colors;
    
    //
    private Color[] negatedColors;
    
    private void assignColors(){
        max = Double.MIN_VALUE;
        min = Double.MAX_VALUE;
        for(double d : dataModel){
            if(max < d)
                max = d;
            if(min > d)
                min = d;
        }
        double mid = (max + min)/2;
        colors = new Color[dataModel.getSize()];
        negatedColors = new Color[dataModel.getSize()];
        for(int i = 0; i<dataModel.getSize(); i++){
            if(dataModel.get(i)>mid)
                colors[i]=new Color(0,(float)((dataModel.get(i)-mid)/(max-mid)),1f-(float)((dataModel.get(i)-mid)/(max-mid)));
            else
                colors[i]=new Color((float)((mid-dataModel.get(i))/(mid-min)),0,1f-(float)((mid-dataModel.get(i))/(mid-min)));
            negatedColors[i] = new Color(255-colors[i].getRed(), 255-colors[i].getGreen(), 255-colors[i].getBlue());
        }
    }
    
    /**
     * Repaint when selection changes.
     */
    public void valueChanged(ListSelectionEvent e) {
        assignColors();
        repaint();
    }
    
    /**
     * Repaint when data model changes.
     */
    public void intervalRemoved(ListDataEvent e) {
        assignColors();
        repaint();
    }

    public void intervalAdded(ListDataEvent e) {
        assignColors();
        repaint();
    }

    public void contentsChanged(ListDataEvent e) {
        assignColors();
        repaint();
    }

    public void mouseReleased(MouseEvent e) {
        // intentionally empty
    }

    public void mousePressed(MouseEvent e) {
        // intentionally empty
    }

    public void mouseExited(MouseEvent e) {
        // intentionally empty
    }

    public void mouseEntered(MouseEvent e) {
        // intentionally empty
    }

    public void mouseClicked(MouseEvent e) {
        int size = dataModel.getSize();
        int index = (e.getX() - xOffset) * size / actualWidth;
        if (index >= 0 && index < size)
            selectionModel.setSelectionInterval(index, index);
    }

    public String getToolTipText(MouseEvent event) {
        int size = dataModel.getSize();
        int index = (event.getX() - xOffset) * size / actualWidth;
        if (index >= 0 && index < size)
            return dataModel.get(index).toString();
        else
            return null;
    }
    
}
