/* AbstractInequalityEngine.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.engine;

import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JPanel;

import org.grinvin.expr.Expression;
import org.grinvin.worksheet.ConjecturingComboBoxModel;
import org.grinvin.worksheet.WorksheetModel;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.invariants.UnknownInvariantException;
import org.grinvin.invariants.values.NumericValue;
import org.jdom.Element;

public abstract class AbstractInequalityEngine extends AbstractEngine implements InequalityEngine {
    
    //
    private ConjecturingComboBoxModel comboBoxModel;
    
    //
    private Invariant mainInvariant;
    
    //
    public void setMainInvariant(Invariant mainInvariant) {
        this.mainInvariant = mainInvariant;
    }
    
    //
    public Invariant getMainInvariant() {
        return mainInvariant;
    }
    
    //
    public abstract Expression run(InvariantValue [][] values);
    
    /**
     * @inheritDoc <p>
     * This implementation checks whether the type of the invariant value is
     * numeric.
     */
    public boolean allows(Invariant invariant) {
        return NumericValue.class.isAssignableFrom(invariant.getType());
    }
    
    public JPanel getConfigurationPanel(WorksheetModel worksheetModel) {
        if(comboBoxModel==null)
            comboBoxModel = new ConjecturingComboBoxModel(this, worksheetModel);
        return new MainInvariantConfigurationPanel(this, comboBoxModel);
    }

    public boolean isReadyForRun() {
        return comboBoxModel != null && comboBoxModel.getSize() > 1 && comboBoxModel.getSelectedItem() != null;
    }
       
    public void load(Element element, WorksheetModel worksheetModel) {
        if(comboBoxModel==null)
            comboBoxModel = new ConjecturingComboBoxModel(this, worksheetModel);

        Element main = element.getChild("mainInvariant");
        if (main != null) {
            String invariantId = main.getAttributeValue("invariantId");
            if (invariantId.length() > 0) {
                try {
                    mainInvariant = InvariantManager.getInstance().getInvariant(invariantId);
                    if(worksheetModel.getInvariantList().contains(mainInvariant))
                        comboBoxModel.setSelectedItem(mainInvariant);
                    else
                        comboBoxModel.setSelectedItem(comboBoxModel.getElementAt(0));
                } catch (UnknownInvariantException ex) {
                    Logger.getLogger("org.grinvin.workspace").log(
                            Level.WARNING, "Ignored unknown invariant {0}", ex.getId());
                }
            }
        }
    }

    public void save(Element element) {
        Element main = new Element("mainInvariant");
        main.setAttribute("invariantId", mainInvariant==null ? "" : mainInvariant.getId());
        element.addContent(main);
    }

}
