/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.ui;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.team.internal.core.target.TargetManager;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;
import org.eclipse.team.internal.ftp.target.FTPSite;
import org.eclipse.team.internal.ui.UIConstants;
import org.eclipse.team.internal.ui.target.ConfigureTargetWizard;
import org.eclipse.team.internal.ui.target.MappingSelectionPage;
import org.eclipse.team.ui.IConfigurationWizard;
import org.eclipse.team.ui.TeamImages;
import org.eclipse.ui.IWorkbench;

public class FTPConfigurationWizard extends Wizard implements IConfigurationWizard {
			
	IProject project;
	FTPMainPage mainPage = null;
	
	public FTPConfigurationWizard() {
		// initializes the dialogs settings store
		IDialogSettings workbenchSettings = FTPPlugin.getPlugin().getDialogSettings();
		IDialogSettings section = workbenchSettings.getSection("FTPConfigurationWizard");//$NON-NLS-1$
		if (section == null) {
			section = workbenchSettings.addNewSection("FTPConfigurationWizard");//$NON-NLS-1$
		}
		setDialogSettings(section);
	}
	
	protected String getMainPageTitle() {
		return Policy.bind("FTPMainPage.name"); //$NON-NLS-1$
	}
	
	protected String getMainPageDescription() {
		return Policy.bind("FTPMainPage.description"); //$NON-NLS-1$
	}
	
	public void addPages() {
		mainPage = new FTPMainPage("ftp-site-creation", getMainPageTitle(), getMainPageDescription(), TeamImages.getImageDescriptor(UIConstants.IMG_WIZBAN_SHARE)); //$NON-NLS-1$
		mainPage.setDialogSettings(getDialogSettings());
		addPage(mainPage);
	}
	
	/**
	 * Finished pressed while on the target pages. If finished is pressed while on the mapping
	 * page, the target configuration wizard will handle this case.
	 */
	public boolean performFinish() {
		mainPage.finish(null);
		FTPSite site = mainPage.getSite();
		if(ConfigureTargetWizard.validateSite(site, getContainer())) {
			TargetManager.addSite(site);
			return true;
		} else {
			try {
				site.dispose();
			} catch (TeamException e) {
				FTPPlugin.logError(Policy.bind("FTPSite.Site_disposal_error"), e);
			} 
			return false;
		}
	}
	
	public void init(IWorkbench workbench, IProject project) {
		this.project = project;
	}
	
	/*
	 * Convenience method for running an operation with progress and
	 * error feedback.
	 * 
	 * @param runnable  the runnable to run
	 * @param problemMessage an optional message to display in case of errors
	 */
	protected void run(IRunnableWithProgress runnable, String problemMessage) {
		IStatus errors = null;
		boolean internalError = false;
		try {
			new ProgressMonitorDialog(getShell()).run(false, false, runnable);
		} catch (InvocationTargetException e) {
			Throwable t = e.getTargetException();
			if (t instanceof CoreException) {
				errors = ((CoreException)t).getStatus();
			} else {
				errors = new Status(IStatus.ERROR, FTPPlugin.ID, 1, problemMessage, t);
				internalError = true;
			}
		} catch (InterruptedException e) {
			errors = null;
		}
		if (errors != null) {
			String msg = internalError
				? Policy.bind("FTPMainPage.internalError") //$NON-NLS-1$
				: problemMessage;
			ErrorDialog.openError(getShell(), msg, null, errors);
		}
	}
	
	/**
	 * @see IWizard#canFinish()
	 */
	public boolean canFinish() {
		IWizardPage page = getContainer().getCurrentPage();
		if(page == mainPage && project != null) {
			return false;
		}
		return super.canFinish();
	}
	
	/**
	 * @see IWizard#getNextPage(IWizardPage)
	 */
	public IWizardPage getNextPage(IWizardPage page) {
		if(page == mainPage) {
			Site site = mainPage.getSite();
			if(site != null && project != null) {
				MappingSelectionPage mappingPage = ConfigureTargetWizard.getMappingPage();
				mappingPage.setSite(site);
				mappingPage.setPreviousPage(page);
				return mappingPage;
			}				
		}
		return super.getNextPage(page);
	}
	
	/**
	 * @see org.eclipse.jface.wizard.IWizard#getWindowTitle()
	 */
	public String getWindowTitle() {
		return Policy.bind("FTPMainPage.windowTitle"); //$NON-NLS-1$
	}
}
