/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.client;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.core.Assert;
import org.eclipse.team.internal.core.streams.CRLFtoLFInputStream;
import org.eclipse.team.internal.core.streams.LFtoCRLFInputStream;
import org.eclipse.team.internal.core.streams.ProgressMonitorInputStream;
import org.eclipse.team.internal.ftp.FTPException;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;

/**
 * Simple FTP client class.
 * Based on RFC854, RFC959, RFC1123, RFC2640
 * 
 * The public methods of this class all throw FTPExceptions to indicate errors.
 * Recoverable errors: (subsequent operations will be processed normally)
 *   FTPServerException and subclasses
 *     FTPAuthenticationException
 *     FTPFileNotAvailableException
 *     FTPServiceNotAvailableException
 * 
 * Non-recoverable errors: (subsequent operations will probably fail)
 *   FTPCommunicationException and subclasses
 *   ... anything else ...
 * 
 * Note: The DataTransferWorker is responsible for providing a mechanism
 *       for cancelling data transfers in progress.  (eg. via the progress monitor)
 *       This is accomplished simply by dropping the data connection rather
 *       than by sending an ABOR request (tricky to get right and not supported by all servers).
 */
public class FTPClient {
	private FTPServerLocation location;
	private FTPProxyLocation proxy;
	private IFTPClientListener listener;

	private ControlConnection controlConnection = null;
	private byte[] buffer = new byte[256];
	private int responseCode;
	private String responseText;
	private String previousTYPE = null;
	private long previousREST = 0;
	private int timeout;
	
	boolean expectDataTransferCompletion;
	
	public static final String PARENT_DIRECTORY = ".."; //$NON-NLS-1$
	public static final int USE_DEFAULT_TIMEOUT = -1;
	
	/**
	 * Creates an FTPClient for the specified location.
	 * @param location the FTP server location
	 * @param proxy the FTP proxy location, or null
	 * @param listener the FTP listener
	 * @param timeout the communications timeout in milliseconds
	 */
	public FTPClient(FTPServerLocation location, FTPProxyLocation proxy, IFTPClientListener listener, int timeout) {
		this.location = location;
		this.proxy = proxy;
		this.listener = listener;
		if (timeout == USE_DEFAULT_TIMEOUT) {
			this.timeout = FTPPlugin.getPlugin().getTimeout();
		} else {
			this.timeout = timeout;
		}
	}
	
	/**
	 * Execute the given runnable in a context that has access to an open ftp connection
	 */
	public void run(IFTPRunnable runnable, IProgressMonitor monitor) throws TeamException {
		
		// Determine if we are nested or not
		boolean isOuterRun = false;
		
		try {
			monitor = Policy.monitorFor(monitor);
			monitor.beginTask(null, 100 + (controlConnection == null ? 20 : 0));
			if (controlConnection == null) {
				open(Policy.subMonitorFor(monitor, 10));
				isOuterRun = true;
			}
			runnable.run(Policy.subMonitorFor(monitor, 100));
			
			// Check for any responses that may be left due to data transfer completion
			handleDataTransferCompletion();
		} finally {
			if (isOuterRun) {
				close(Policy.subMonitorFor(monitor, 10));
			}
			monitor.done();
		}
	}
	
	/**
	 * Opens and authenticates a connection to the server, if not already open.
	 * @param monitor the progress monitor, or null
	 */
	public void open(IProgressMonitor monitor) throws FTPException {
		if (controlConnection != null) return;
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		monitor.subTask(Policy.bind("FTPClient.openConnection", location.getHostname())); //$NON-NLS-1$
		try {
			// handle non-transparent FTP proxy
			String hostname;
			int port;
			if (proxy != null) {
				hostname = proxy.getHostname();
				port = proxy.getPort();
			} else {
				hostname = location.getHostname();
				port = location.getPort();
			}
				
			// resolve the host address and open the socket (10% )
			InetAddress address = resolveHostname(hostname);
			monitor.worked(10);
			
			// create the control connection (50%)
			controlConnection = new ControlConnection(address, port);
			controlConnection.open(timeout, Policy.subMonitorFor(monitor, 60));
	
			// handle initial connection response (10%)
			switch (readResponseSkip100()) {
				case 220: // service ready
					break;
				default:
					defaultResponseHandler();
			}
			monitor.worked(10);
			
			// authenticate (30%)
			authenticate(location.getUsername(), location.getPassword(), Policy.subMonitorFor(monitor, 30));
		} catch (FTPException e) {
			// There was a problem. Close the connection!
			if (controlConnection != null) {
				controlConnection.close(Policy.subMonitorFor(monitor, 5));
				controlConnection = null;
			}
			throw e;
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * Closes a connection to the server, if not already closed.
	 * @param monitor the progress monitor, or null
	 */
	public void close(IProgressMonitor monitor) throws FTPException {
		if (controlConnection == null) return;
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		monitor.subTask(Policy.bind("FTPClient.closeConnection")); //$NON-NLS-1$
		try {
			// send the QUIT request (50%)
			sendCommand("QUIT", null); //$NON-NLS-1$
			monitor.worked(25);
			int response;
			try {
				response=readResponseSkip100();
				switch (response) {
					case 221: // service closing
						break;
					default:
						defaultResponseHandler();
				}
			} catch (FTPCommunicationException e) {
				if (e.getStatus().getCode()!=FTPException.CONNECTION_LOST) throw e;
			}
			monitor.worked(25);
		} finally {
			try {
				// close the socket (50%)
				controlConnection.close(Policy.subMonitorFor(monitor, 50));
			} finally {
				controlConnection = null;
				expectDataTransferCompletion = false;
				monitor.done();
			}
		}
	}

	public boolean isConnected() {
		return controlConnection != null;
	}
	
	/**
	 * Changes the current remote working directory.
	 * @param directoryPath the absolute or relative path of the directory
	 * @param monitor the progress monitor, or null
	 */
	public void changeDirectory(String directoryPath, IProgressMonitor monitor) throws FTPException {
		Assert.isNotNull(directoryPath);
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		monitor.subTask(Policy.bind("FTPClient.changeDirectory", directoryPath)); //$NON-NLS-1$
		try {
			sendCommand("CWD", directoryPath); //$NON-NLS-1$
			monitor.worked(50);
			switch (readResponseSkip100()) {
				case 250: // requested file action ok
					return;
				default:
					defaultResponseHandler();
			}
			monitor.worked(50);
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * Deletes a remote directory.
	 * @param directoryPath the absolute or relative path of the directory
	 * @param monitor the progress monitor, or null
	 */
	public void deleteDirectory(String directoryPath, IProgressMonitor monitor) throws FTPException {
		Assert.isNotNull(directoryPath);
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		monitor.subTask(Policy.bind("FTPClient.deleteDirectory", directoryPath)); //$NON-NLS-1$
		try {
			sendCommand("RMD", directoryPath); //$NON-NLS-1$
			monitor.worked(50);
			switch (readResponseSkip100()) {
				case 250: // requested file action ok
					return;
				default:
					defaultResponseHandler();
			}
			monitor.worked(50);
		} finally {
			monitor.done();
		}
	}

	/**
	 * Creates a remote directory.
	 * @param directoryPath the absolute or relative path of the directory
	 * @param monitor the progress monitor, or null
	 */
	public void createDirectory(String directoryPath, IProgressMonitor monitor) throws FTPException {
		Assert.isNotNull(directoryPath);
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		monitor.subTask(Policy.bind("FTPClient.createDirectory", directoryPath)); //$NON-NLS-1$
		try {
			sendCommand("MKD", directoryPath); //$NON-NLS-1$
			monitor.worked(50);
			switch (readResponseSkip100()) {
				case 257: // "PATHNAME" created.
					// According to RFC959, the response should contain the name of the 
					// directory just created in specially formatted text string.
					// However some FTP servers disregard this, so we will not
					// attempt to parse the string.
					return;
				default:
					defaultResponseHandler();
			}
			monitor.worked(50);
		} finally {
			monitor.done();
		}
	}

	/**
	 * Deletes a remote file.
	 * @param filePath the absolute or relative path of the file
	 * @param monitor the progress monitor, or null
	 */
	public void deleteFile(String filePath, IProgressMonitor monitor) throws FTPException {
		Assert.isNotNull(filePath);
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		monitor.subTask(Policy.bind("FTPClient.deleteFile", filePath)); //$NON-NLS-1$
		try {
			sendCommand("DELE", filePath); //$NON-NLS-1$
			monitor.worked(50);
			switch (readResponseSkip100()) {
				case 250: // requested file action ok
					return;
				default:
					defaultResponseHandler();
			}
			monitor.worked(50);
		} finally {
			monitor.done();
		}
	}

	/**
	 * Retrieves the contents of a remote file.
	 * 
	 * The return input stream has a direct connection to the server. Therefore,
	 * the contents should be read before the ftp connection is closed.
	 * 
	 * @param filePath the absolute or relative path of the file
	 * @param binary if true, uses binary transfer type
	 * @param resumeAt the position in the remote file to start at (or 0 for the whole file)
	 * @param monitor the progress monitor, or null
	 */
	public InputStream getContents(String filePath, boolean binary, long resumeAt, IProgressMonitor monitor) throws FTPException {
		Assert.isNotNull(filePath);

		// send the request to the server
		DataConnection manager = initiateDataTransfer("RETR", filePath, binary, resumeAt, Policy.monitorFor(monitor)); //$NON-NLS-1$
		InputStream in = manager.getDataTransferInputStream();

		// setup line terminator conversion
		if (! binary && ! Constants.IS_CRLF_PLATFORM) in = new CRLFtoLFInputStream(in);
		return in;
	}
	
	/**
	 * Retrieves a remote file.
	 * @param filePath the absolute or relative path of the file
	 * @param localFile the local file to create
	 * @param binary if true, uses binary transfer type
	 * @param resume if true, attempts to resume a partial transfer, else overwrites
	 * @param monitor the progress monitor, or null
	 */
	public void getFile(String filePath, IFile localFile,
		boolean binary, boolean resume, IProgressMonitor monitor) throws FTPException {

		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		final String title = Policy.bind("FTPClient.getFile", filePath); //$NON-NLS-1$
		monitor.subTask(title);
		try {
			long resumeAt = 0;
			if (resume) {
				resumeAt = Util.getFileSize(localFile);
			}
			InputStream in = getContents(filePath, binary, resumeAt, Policy.subMonitorFor(monitor, 10));
			// setup progress monitoring
			monitor.subTask(Policy.bind("FTPClient.transferNoSize", title)); //$NON-NLS-1$
			in = new ProgressMonitorInputStream(in, 0, Constants.TRANSFER_PROGRESS_INCREMENT, monitor) {
				protected void updateMonitor(long bytesRead, long bytesTotal, IProgressMonitor monitor) {
					if (bytesRead == 0) return;
					monitor.subTask(Policy.bind("FTPClient.transferUnknownSize", //$NON-NLS-1$
						new Object[] { title, Long.toString(bytesRead >> 10) }));
				}
			};

			// transfer the file
			try {
				if (localFile.exists()) {
					if (resumeAt != 0) {
						// don't bother remembering the previous (incomplete) generation of the file
						localFile.appendContents(in, false /*force*/, false /*keepHistory*/, null);
					} else {
						localFile.setContents(in, false /*force*/, true /*keepHistory*/, null);
					}
				} else {
					localFile.create(in, false /*force*/, null);
				}
			} catch (CoreException e) {
				throw FTPPlugin.wrapException(e);
			}
		} finally {
			monitor.done();
		}
	}

	/**
	 * Stores a local file on the remote system.
	 * @param filePath the absolute or relative path of the file
	 * @param localFile the local file to send
	 * @param binary if true, uses binary transfer type
	 * @param monitor the progress monitor, or null
	 */
	public void putFile(String filePath, IFile localFile, boolean binary,
		IProgressMonitor monitor) throws FTPException {
		
		Assert.isNotNull(filePath);
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		final String title = Policy.bind("FTPClient.putFile", filePath); //$NON-NLS-1$
		monitor.subTask(title);
		InputStream in = null;
		DataConnection manager = null;
		try {
			try {
				// get the output stream from the data conenction
				manager = initiateDataTransfer("STOR", filePath, binary, 0, monitor); //$NON-NLS-1$
				OutputStream out = manager.getSocketOutputStream();
				
				// Get the input stream from the file and warp for non-binary line terminator
				// conversion and progress monitoring
				in = new BufferedInputStream(localFile.getContents());
				if (! binary && ! Constants.IS_CRLF_PLATFORM) in = new LFtoCRLFInputStream(in);
				monitor.subTask(Policy.bind("FTPClient.transferNoSize", title)); //$NON-NLS-1$
				in = new ProgressMonitorInputStream(in, Util.getFileSize(localFile), Constants.TRANSFER_PROGRESS_INCREMENT, monitor) {
					protected void updateMonitor(long bytesRead, long bytesTotal, IProgressMonitor monitor) {
						if (bytesRead == 0) return;
						monitor.subTask(Policy.bind("FTPClient.transferSize", //$NON-NLS-1$
							new Object[] { title, Long.toString(bytesRead >> 10), Long.toString(bytesTotal >> 10) }));
					}
				};
			
				// copy file to output stream		
				byte[] buffer = new byte[FTPPlugin.getPlugin().getSendBufferSize()];
				for (int count; (count = in.read(buffer)) != -1;) {
					out.write(buffer, 0, count);
				}
			} catch (CoreException e) {
				throw FTPException.wrapException(Policy.bind("FTPClient.ErrorSendingFile"), e); //$NON-NLS-1$
			} finally {
				try {
					if (in != null) in.close();
				} finally {
					if (manager != null) {
						manager.closeSocket();
						handleDataTransferCompletion();
					}
				}
				monitor.done();
			}
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorSendingFile"), e); //$NON-NLS-1$
		}
	}
	
	/**
	 * Lists the files stored in a remote directory.
	 * @param directoryPath the absolute or relative path of the directory, or null (or empty string) for current
	 * @param monitor the progress monitor, or null
	 * @return an array of directory entries, or an empty array if none
	 */
	public FTPDirectoryEntry[] listFiles(String directoryPath, IProgressMonitor monitor) throws FTPException {
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		final String title = Policy.bind("FTPClient.listFiles", directoryPath == null ? "." : directoryPath); //$NON-NLS-1$ //$NON-NLS-2$
		monitor.subTask(title);
		DataConnection manager = null;
		try {
			// Initiate the data transfer
			manager = initiateDataTransfer("LIST -a", directoryPath, false, 0, Policy.monitorFor(monitor)); //$NON-NLS-1$
			// Get the input stream from the data connection
			monitor.subTask(Policy.bind("FTPClient.transferNoSize", title)); //$NON-NLS-1$
			InputStream in = new ProgressMonitorInputStream(manager.getSocketInputStream(), 0, Constants.TRANSFER_PROGRESS_INCREMENT, monitor) {
				protected void updateMonitor(long bytesRead, long bytesTotal, IProgressMonitor monitor) {
					if (bytesRead == 0) return;
					monitor.subTask(Policy.bind("FTPClient.transferUnknownSize", //$NON-NLS-1$
						new Object[] { title, Long.toString(bytesRead >> 10) }));
				}
			};
			// Read the list entries from the data connection
			Map /* from String to FTPDirectoryEntry */ dirlist = new HashMap();
			String line;
			while ((line = Util.readListLine(in)) != null) {
				if (Policy.DEBUG_LIST) System.out.println(line);
				if (line.length() == 0) continue; // skip blank lines
				boolean reliable = true;
				FTPDirectoryEntry entry = Util.parseEPLFLine(line);
				if (entry == null) {
					reliable = false; // LS is much less reliable than EPLF so we prefer EPLF over LS
					entry = Util.parseLSLine(line);
					if (entry == null) continue;
				}
				String name = entry.getName();
				if (!reliable && dirlist.containsKey(name)) continue;
				// ignore Unix . and .. that are sometimes included in listing
				if (".".equals(name) || "..".equals(name)) continue; //$NON-NLS-1$ //$NON-NLS-2$
				dirlist.put(name, entry);
			}
			return (FTPDirectoryEntry[]) dirlist.values().toArray(new FTPDirectoryEntry[dirlist.size()]);
		} finally {
			if (manager != null) {
				manager.closeSocket();
				handleDataTransferCompletion();
			}
			monitor.done();
		}
	}
	
	/**
	 * Processes a data transfer command.
	 * @param command the command to send
	 * @param argument the argument for the command
	 * @param binary if true, uses binary transfer type
	 * @param resumeAt the restart position as a byte offset from start position
	 * @param worker the data transfer worker to run in the background
	 * @param monitor the progress monitor
	 */
	private DataConnection initiateDataTransfer(String command, String argument, boolean binary, long resumeAt, IProgressMonitor monitor) throws FTPException {
		
		// set the data transfer type
		changeTransferType(binary);
		
		// open data transfer connection
		boolean passive = location.getUsePassive();
		DataConnection manager;
		if (passive) {
			manager = preparePassiveDataTransfer(monitor);
		} else {
			manager = preparePortDataTransfer(monitor);
		}
		boolean success = false;
		try {
			// set the resume position
			resumeAt = changeRestartPosition(binary, resumeAt);
			// XXX I don't know what problem this fixed but I'll leave it for now.
			controlConnection.setTimeoutEnabled(false); // FIXME: need a better solution
			sendCommand(command, argument);
			success = waitForDataTransferInitiation(manager);
		} finally {
			controlConnection.setTimeoutEnabled(true);
			//manager.close(! success);
		}
		if (!success) manager = null;
		return manager;
	}

	private boolean waitForDataTransferInitiation(DataConnection manager) throws FTPException {
		// Wait until the data connection is made. 
		boolean isReady = false;
		while (! isReady) {
			// check to see if the connection is ready (this call will yield to other threads if it's not)
			try {
				isReady = manager.isReady();
			} catch (FTPException e) {
				// The data connection could not be made for some reason (e.g. conenction timeout)
				// XXX Do we need to take additional action?
				// XXX Should we return a status and let the caller decide if they want to try again?
				throw e;
			}
		}
		// The connection was made so we should have a response indicating that
		boolean success = readResponse() < 200;
		if (success) {
			// Set receiver state to indicate that we whould get a transfer completion message at some point
			expectDataTransferCompletion = true;
		} else {
			// Odd that we didn't get the response we expected.
			// Close the connection and throw an exception 
			try {
				manager.close(true);
			} finally {
				defaultResponseHandler();
			}
		}
		return success;
	}
	
	private boolean handleDataTransferCompletion() throws FTPException {
		boolean success = false;
		if (expectDataTransferCompletion) {
			// set the flag to false so the call to readResponseSkip100() will not recurse infinitely
			expectDataTransferCompletion = false;
			switch (readResponseSkip100()) {
				case 226: // closing data connection (file transfer or abort)
				case 250: // requested file action okay, completed
					success = true;
					break;
				default:
					defaultResponseHandler();
			}
		}
		return success;
	}

	/**
	 * Authenticates with the specified username and password.
	 * @param username the user name
	 * @param password the password
	 * @param monitor the progress monitor
	 */
	private void authenticate(String username, String password,
		IProgressMonitor monitor) throws FTPException {
		Assert.isNotNull(username);
		Assert.isNotNull(password);
		monitor = Policy.monitorFor(monitor);
		monitor.beginTask(null, 100);
		monitor.subTask(Policy.bind("FTPClient.authenticate")); //$NON-NLS-1$
		try {
			// handle non-transparent proxy (requires a constructed username)
			if (proxy != null) {
				StringBuffer buf = new StringBuffer();
				if (proxy.getUsername() != null) {
					buf.append(proxy.getUsername());
					buf.append(':');
					if (proxy.getPassword() != null) {
						buf.append(proxy.getPassword());
						buf.append(':');
					}
				}
				buf.append(username);
				buf.append(username.indexOf('@') == -1 ? '@' : '%');
				buf.append(location.getHostname());
				buf.append(':');
				buf.append(Integer.toString(location.getPort()));
				username = buf.toString();
			}
			// send user name (40%)
			boolean needPassword = false, needAccount = false;
			sendCommand("USER", username); //$NON-NLS-1$
			monitor.worked(20);
			switch (readResponseSkip100()) {
				case 202: case 230: // user logged in, proceed
					monitor.worked(80);
					return;
				case 331: // user name okay, need password
					needPassword = true;
					break;
				case 332: // user name okay, need account
					needAccount = true;
					break;
				default:
					defaultResponseHandler();
			}
			// send password (40%)
			monitor.worked(20);
			if (needPassword) {
				sendCommand("PASS", password); //$NON-NLS-1$
				monitor.worked(20);
				switch (readResponseSkip100()) {
					case 202: case 230: // user logged in, proceed
						monitor.worked(40);
						return;
					case 332: // user name okay, need account
						needAccount = true;
						break;
					default:
						defaultResponseHandler();
				}
				monitor.worked(20);
			}
			// send account (20%)
			if (needAccount) {
				// some servers apparently still ask for an account
				sendCommand("ACCT", "noaccount"); //$NON-NLS-1$ //$NON-NLS-2$
				monitor.worked(10);
				switch (readResponseSkip100()) {
					case 202: case 230: // user logged in, proceed
						monitor.worked(10);
						return;
					default:
						defaultResponseHandler();
				}
			}
		} finally {
			monitor.done();
		}
	}

	/**
	 * Changes the transfer type to ASCII or binary
	 * @param isBinary if true, transfers binary data, else transfers ASCII
	 */
	private void changeTransferType(boolean isBinary) throws FTPException {
		String type = isBinary ? "I" : "A"; //$NON-NLS-1$ //$NON-NLS-2$
		if (previousTYPE == null || ! previousTYPE.equals(type)) {
			sendCommand("TYPE", type); //$NON-NLS-1$
			switch (readResponseSkip100()) {
				case 200: // command ok
					previousTYPE = type;
					return;
				default:
					defaultResponseHandler();
			}
		}
	}
	
	/**
	 * Changes the restart position if possible
	 * Must call this method before any data transfer occurs since some servers
	 * do not correctly reset the restart position.
	 * 
	 * @param isBinary if true, assumes restart for binary files, else ASCII
	 * @param resumeAt the restart position as a byte offset from start position
	 * @return the actual restart position
	 */
	private long changeRestartPosition(boolean isBinary, long resumeAt) throws FTPException {
		if (! isBinary) {
			// some servers do not handle restart on ASCII files properly
			resumeAt = 0;
		}
		if (resumeAt == 0 && previousREST == 0) return 0;
		sendCommand("REST", Long.toString(resumeAt)); //$NON-NLS-1$
		switch (readResponseSkip100()) {
			case 350: // file action pending further information
				previousREST = resumeAt;
				return resumeAt;
			case 500: case 501: case 502: // syntax error or command not supported
				return 0;
			default:
				defaultResponseHandler();
		}
		return 0;
	}

	/**
	 * Creates a server socket, asks the server to enter Port mode,
	 * then returns a suitable DataTransferManager.
	 * The caller is reponsible for closing the manager.
	 */
	private DataConnection preparePortDataTransfer(IProgressMonitor monitor) throws FTPException {
		PortDataConnection manager = new PortDataConnection();
		manager.open(monitor);
		boolean success = false;
		try {
			InetAddress address = manager.getLocalAddress();
			int port = manager.getLocalPort();
			byte[] rawAddress = address.getAddress();
			StringBuffer buf = new StringBuffer();
			for (int i = 0; i < 4; ++i) {
				buf.append(Integer.toString(rawAddress[i] & 255));
				buf.append(","); //$NON-NLS-1$
			}
			buf.append(Integer.toString(port >> 8));
			buf.append(","); //$NON-NLS-1$
			buf.append(Integer.toString(port & 255));
			// send port command
			sendCommand("PORT", buf.toString()); //$NON-NLS-1$
			switch (readResponseSkip100()) {
				case 200: // command successful
					break;
				default:
					defaultResponseHandler();
			}
			success = true;
			return manager;
		} finally {
			if (! success) manager.close(true);
		}
	}

	/**
	 * Asks the server to enter Passive mode, then returns a suitable DataTransferManager.
	 * The caller is reponsible for closing the manager.
	 */
	private DataConnection preparePassiveDataTransfer(IProgressMonitor monitor) throws FTPException {
		// send passive command
		sendCommand("PASV", null); //$NON-NLS-1$
		switch (readResponseSkip100()) {
			case 227: // entering passive mode X1,X2,X3,X4,p1,p2
				break;
			default:
				defaultResponseHandler();
		}
		// parse out the address and port
		int[] fields = new int[6];
		int f = 0;
		for (int i = 0; i < responseText.length(); ++i) {
			char c = responseText.charAt(i);
			int digit = c - '0';
			if (digit < 0 || digit > 9) {
				if (f == 5) {
					f = 6;
					break;
				}
				if (c == ',') f += 1; else f = 0;
			} else {
				fields[f] = fields[f] * 10 + digit;
				if (fields[f] < 256) continue;
			}
			fields[f] = 0;
		}
		if (f != 6) throw new FTPException(Policy.bind("FTPClient.MalformedServerResponse", responseText)); //$NON-NLS-1$
		// create the connection
		StringBuffer buf = new StringBuffer(Integer.toString(fields[0]));
		for (int i = 1; i < 4; ++i) {
			buf.append('.');
			buf.append(Integer.toString(fields[i]));
		}
		String dottedIP = buf.toString();
		InetAddress address = resolveHostname(dottedIP);
		int port = (fields[4] << 8) + fields[5];
		DataConnection manager = new PassiveDataConnection(address, port);
		manager.open(monitor);
		return manager;
	}

	/**
	 * Reads responses from the connection until a non-informational
	 * or preliminary reply response (100-series) is found, then returns it.
	 * 
	 * @return the response code
	 */
	private int readResponseSkip100() throws FTPException {
		// Check for any responses that may be left due to data transfer completion
		handleDataTransferCompletion();
		for (;;) {
			int code = readResponse();
			if (code / 100 != 1) return code;
		}
	}

	/**
	 * Reads a single responses from the connection, notifies the listener,
	 * and returns the response code.
	 * 
	 * Treats server response text as UTF-8 encoded data.  Consequences:
	 *   - the text string cannot be converted one character at a time
	 *   - stray CR and LF look-alikes must be preserved
	 *   - CR and LF only have special meaning when paired as CR/LF
	 *     which cannot be an intermediate sequence of a multi-byte
	 *     UTF-8 sequence (though the CR might belong to end of one
	 *     sequence, but that would leave a stray LF in the output
	 *     and it is safe for us to be lenient here)
	 *   - may have trouble with older servers that use a full 8bit
	 *     charset in lieu of UTF-8 (could try to detect invalid
	 *     UTF-8 sequences and fallback... ???)
	 * 
	 * @return the response code
	 */
	private int readResponse() throws FTPException {
		responseCode = 0;
		InputStream is = controlConnection.getInputStream();
		int index = 0; // index into buffer
		boolean longResponse = false; // if true, this is a multi-line response
		boolean seenCR = false; // if true, the last char was a CR
		int lastEOL = 0; // index after last CR/LF
		try {
			for (;;) {
				// read a byte
				int c = is.read();
				if (c == -1) 
					// server dropped connection?
					throw new FTPCommunicationException(Policy.bind("FTPClient.DroppedConnection"), FTPException.CONNECTION_LOST); //$NON-NLS-1$
				if (index == buffer.length) {
					byte[] oldBuffer = buffer;
					buffer = new byte[oldBuffer.length * 2];
					System.arraycopy(oldBuffer, 0, buffer, 0, oldBuffer.length);
				}
				buffer[index++] = (byte) c;
	
				// incrementally process the response
				if (index <= 3) {
					int digit = c - Constants.ZERO;
					if (digit < 0 || digit > 9) // ERROR
						throw new FTPCommunicationException(Policy.bind("FTPClient.InvalidResponseCode",Integer.toString(responseCode)), FTPException.BAD_RESPONSE_CODE);
					responseCode = responseCode * 10 + digit;
				} else if (index == 4) {
					if (c == Constants.HYPHEN) {
						longResponse = true;
					} else if (c == Constants.SPACE) {
						longResponse = false;
					} else {//error
						throw new FTPCommunicationException(Policy.bind("FTPClient.InvalidDelimiter", Integer.toString(responseCode),new String(new char[] {(char)c})), FTPException.BAD_RESPONSE_CODE);
					}
				} else {
					if (c == Constants.CR) {
						seenCR = true;
					} else {
						if (c == Constants.LF && seenCR) {
							buffer[--index - 1] = (byte)Constants.LF; // replace CR/LF with LF
							if (longResponse && lastEOL != 0 && index - lastEOL > 4) {
								// long response, check if this line started with the same response code followed by a SPACE
								int i, prefixCode = 0;
								for (i = 0; i < 3; ++i) {
									int digit = buffer[lastEOL + i] - Constants.ZERO;
									if (digit < 0 || digit > 9) 
										throw new FTPCommunicationException(Policy.bind("FTPClient.InvalidResponseCode",Integer.toString(prefixCode)), FTPException.BAD_RESPONSE_CODE);
									prefixCode = prefixCode * 10 + digit;
								}
								if (i == 3 && prefixCode == responseCode && buffer[lastEOL + 3] == Constants.SPACE) {
									longResponse = false;
								}
							}
							if (! longResponse) {
								// we have finished the response, notify everyone
								responseText = new String(buffer, 0, index);
								listener.responseReceived(responseCode, responseText);
								return responseCode;
							} else {
								// remember position after the last CR/LF sequence seen
								lastEOL = index;
							}
						}
						seenCR = false;
					}
				}
			}
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorReceivingServerResponses"), e); //$NON-NLS-1$
		}
	}
	
	/**
	 * Handles a response by throwing an appropriate exception.
	 * @throws FTPException if the response code is not 100-series
	 */
	private void defaultResponseHandler() throws FTPException {
		switch (responseCode / 100) {
			case 1: // positive preliminary reply
				// it is usually harmless if we fail to recognize one of these messages
				break;
			case 2: // positive completion reply
				// even if the action completed successfully, if we did not recognize the
				// completion response, then we cannot tell whether the expected sequence of
				// actions took place
			case 3: // positive intermediate reply
				// the server expects us to follow up on this reply with additional commands,
				// yet we did not expect this, so we cannot do anything about it
			case 4: 
				// transient negative completion reply
			case 5: 
				// permanent negative completion reply
			default:
				throw new FTPServerException(responseText, responseCode); //$NON-NLS-1$
		}
	}
	
	/**
	 * Sends a properly formatted command to the FTP server and
	 * notifies the listener.
	 * 
	 * @param command the command string
	 * @param argument the argument string, or null if none
	 */
	private void sendCommand(String command, String argument) throws FTPException {
		// Check for any responses that may be left due to data transfer completion
		handleDataTransferCompletion();
		try {
			OutputStream os = controlConnection.getOutputStream();
			listener.requestSent(command, argument);
			os.write(command.getBytes());
			if (argument != null && argument.length() > 0) {
				os.write(Constants.SPACE); // SPACE
				argument = argument.replace('\n', '\0'); // encode LF's as nulls
				os.write(argument.getBytes());
			}
			os.write(Constants.CR); // CR
			os.write(Constants.LF); // LF
			os.flush();
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorSendingCommands"), e); //$NON-NLS-1$
		}
	}

	private InetAddress resolveHostname(String hostname) throws FTPException {
		try {
			return InetAddress.getByName(hostname);
		} catch (UnknownHostException e) {
			throw new FTPException(Policy.bind("FTPClient.CannotResolveHostname", hostname)); //$NON-NLS-1$
		}
	}
}
