/*
 * $Header:  $
 * $Revision:  $
 * $Date:  $
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import java.io.IOException;
import junit.framework.*;
import org.apache.commons.httpclient.methods.*;

/**
 * 
 * Simple tests of {@link TraceMethod} hitting
 * a local HTTP server.
 * <p>
 * This test suite assumes a webserver is running on
 * port 8080 on the 127.0.0.1 (localhost) host.  It
 * further assumes that this webserver will respond
 * to an HTTP TRACE of <tt>/</tt> with a 200 response.
 * <p>
 * You can change the assumed port by setting the
 * "httpclient.test.localPort" property.
 * You can change the assumed host by setting the
 * "httpclient.test.localHost" property.
 *
 * @author Sean C. Sullivan
 * 
 * @version $Id: TestGetMethodLocal.java,v 1.3 2002/02/04 15:26:43 dion Exp $
 * 
 */
public class TestTraceMethodLocal extends TestCase {


    // -------------------------------------------------------------- Constants

    private static final String host = System.getProperty("httpclient.test.localHost","127.0.0.1");
    private static final int port;
    
    static {
        String portString = System.getProperty("httpclient.test.localPort","8080");
        int tempPort = 8080;
        try {
            tempPort = Integer.parseInt(portString);
        } catch(Exception e) {
            tempPort = 8080;
        }
        port = tempPort;
    }

    // ------------------------------------------------------------ Constructor


    public TestTraceMethodLocal(String testName) {
        super(testName);
    }


    // ------------------------------------------------------- TestCase Methods


    public static Test suite() {
        return new TestSuite(TestTraceMethodLocal.class);
    }


    // ------------------------------------------------------------------ Tests



    public void testExecute() {
    	
        HttpClient client = new HttpClient();
        client.startSession(host, port);

        TraceMethod method = new TraceMethod("/");

		final String strTestHeaderName = "MyTestHeader";
		
		final String strTestHeaderValue = "This-is-a-test-value.";
		
		method.setRequestHeader(
					strTestHeaderName, 
					strTestHeaderValue);
		
        try {
            client.executeMethod(method);

			final int iResponseStatusCode = method.getStatusCode();
			assertEquals(200, iResponseStatusCode);
			
            Header[] requestHeaders = method.getRequestHeaders();
            assertTrue( requestHeaders.length > 0);

            Header[] responseHeaders = method.getResponseHeaders();
            assertNotNull(responseHeaders);
            
            //
            // note:  the reason that we convert the String's to lowercase is
            //        because some HTTP servers send a response body that contains 
            //        lower request headers
            //
            final String strResponseBody_lowercase = method.getResponseBodyAsString().toLowerCase();
            assertNotNull(strResponseBody_lowercase);
            assertTrue( strResponseBody_lowercase.length() > 0);
            
            assertTrue( strResponseBody_lowercase.indexOf(strTestHeaderName.toLowerCase()) != -1);
            assertTrue( strResponseBody_lowercase.indexOf(strTestHeaderValue.toLowerCase()) != -1);
            
        } catch (Throwable t) {
            t.printStackTrace();
            fail("Unable to execute method : " + t.toString());
        }
    }
    
    public void testRecycle() {
        HttpClient client = new HttpClient();
        client.startSession(host, port);

        TraceMethod method = new TraceMethod("/");
        
        try {
            client.executeMethod(method);
        } catch (Throwable t) {
            t.printStackTrace();
            fail("Unable to execute method : " + t.toString());
        }

        try {
            String data = method.getResponseBodyAsString();
            assertTrue("No data returned.",(data.length() > 0));
        } catch (Throwable t) {
            t.printStackTrace();
            fail("Unable to execute method : " + t.toString());
        }

        method.recycle();
        method.setPath("/");

        try {
            client.executeMethod(method);
        } catch (Throwable t) {
            t.printStackTrace();
            fail("Unable to execute method : " + t.toString());
        }

        try {
            String data = method.getResponseBodyAsString();
            assertTrue("No data returned.",(data.length() > 0));
        } catch (Throwable t) {
            t.printStackTrace();
            fail("Unable to execute method : " + t.toString());
        }
    }

    public static void main(String args[]) {
        String[] testCaseName = { TestTraceMethodLocal.class.getName() };
        junit.textui.TestRunner.main(testCaseName);
    }
    

}
