/*
 * Copyright(C) 2011 Canonical Ltd.
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

#include <glib.h>
#include <gio/gdesktopappinfo.h>

#include <gdk/gdkx.h>

#include <libindicate/server.h>
#include <libindicate/indicator.h>
#include <libindicate-gtk/indicator.h>
#include <telepathy-glib/telepathy-glib.h>
#include <telepathy-glib/simple-approver.h>

#include <unity.h>

static GHashTable *indicators = NULL;
UnityLauncherEntry *launcher = NULL;

static void
dispatch_op_finished_cb (TpProxy *self, guint domain, gint code, gchar *message, IndicateIndicator *indicator)
{
	if (g_hash_table_lookup (indicators, indicator) != NULL)
	{
		g_debug ("DispatchOperation invalidated, removing indicator");
		indicate_indicator_hide (INDICATE_INDICATOR (indicator));
		g_hash_table_remove (indicators, indicator);
		g_object_unref (indicator);
		if (launcher != NULL)
		{
			gint count = g_hash_table_size (indicators);
			g_debug ("unity launcher: count is %d", count);
			if (count > 0)
			{
				g_debug ("unity launcher: setting count to %d", count);
				unity_launcher_entry_set_count (launcher, count);
				unity_launcher_entry_set_count_visible (launcher, TRUE);
			} else {
				unity_launcher_entry_set_count (launcher, count);
				g_debug ("unity launcher: hiding count");
				unity_launcher_entry_set_count_visible (launcher, FALSE);
			}
		}
	}
}

static void
handle_with_cb (GObject *self, GAsyncResult *result, gpointer user_data)
{
	TpChannelDispatchOperation *dispatch_op = TP_CHANNEL_DISPATCH_OPERATION (self);
	IndicateIndicator *indicator = INDICATE_INDICATOR (user_data);
	GError *error = NULL;

	if (indicator != NULL)
	{
		indicate_indicator_hide (INDICATE_INDICATOR (indicator));
	}
	if (g_hash_table_lookup (indicators, indicator) != NULL)
	{
		g_debug ("Activated, removing indicator");
		g_hash_table_remove (indicators, indicator);
		g_object_unref (indicator);
	}
	if (launcher != NULL)
	{
		gint count = g_hash_table_size (indicators);
		g_debug ("unity launcher: count is %d", count);
		if (count > 0)
		{
			g_debug ("unity launcher: setting count to %d", count);
			unity_launcher_entry_set_count (launcher, count);
			unity_launcher_entry_set_count_visible (launcher, TRUE);
		} else {
			unity_launcher_entry_set_count (launcher, count);
			g_debug ("unity launcher: hiding count");
			unity_launcher_entry_set_count_visible (launcher, FALSE);
		}
	}

	if (!tp_channel_dispatch_operation_handle_with_time_finish (dispatch_op, result, &error))
	{
                if (error)
		{
			g_warning ("Failed to handle operations: %s\n", error->message);
			g_error_free (error);
		}
	}
}

static void
indicator_display (IndicateIndicator *indicator, guint timestamp, TpChannelDispatchOperation * dispatch_op)
{
	g_debug ("indicator_display");
	tp_channel_dispatch_operation_handle_with_time_async (dispatch_op, NULL, timestamp, handle_with_cb, indicator);
}

static void
handle_contacts_add_indicator_cb (TpConnection *connection, guint n_contacts, TpContact * const *contacts, guint n_failed, const TpHandle *failed, const GError *err, gpointer user_data, GObject *weak_object)
{
	IndicateIndicator *indicator = NULL;
	const gchar *alias = NULL;
	GTimeVal time;
	GFile *avatar;
	GdkPixbuf *pixbuf;
	GError *error = NULL;
	
	TpChannelDispatchOperation *dispatch_op = TP_CHANNEL_DISPATCH_OPERATION (user_data);

	alias = tp_contact_get_alias (contacts[0]);
	avatar = tp_contact_get_avatar_file (contacts[0]);

	indicator = indicate_indicator_new ();
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					 "subtype", "im");
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					 "sender", alias);
	indicate_indicator_set_property (INDICATE_INDICATOR (indicator),
					 "draw-attention", "true");

	pixbuf = gdk_pixbuf_new_from_file_at_scale (g_file_get_path (avatar), 22, 22, TRUE, &error);
	if (error)
	{
		g_warning ("Failed to create pixbuf: %s", error->message);
		g_error_free (error);
	}

	if (pixbuf != NULL)
	{
		indicate_gtk_indicator_set_property_icon (indicator, "icon", pixbuf);
		g_object_unref(G_OBJECT(pixbuf));
	}

	g_signal_connect (G_OBJECT (indicator),
			INDICATE_INDICATOR_SIGNAL_DISPLAY,
			G_CALLBACK (indicator_display), dispatch_op);

	g_get_current_time (&time);
	indicate_indicator_set_property_time (INDICATE_INDICATOR (indicator),
					"time", &time);
	indicate_indicator_show (INDICATE_INDICATOR (indicator));
	g_hash_table_insert(indicators, indicator, dispatch_op);
	if (launcher != NULL)
	{
		gint count = g_hash_table_size (indicators);
		g_debug ("unity launcher: count is now %d", count);
		if (count > 0)
		{
			unity_launcher_entry_set_count (launcher, count);
			unity_launcher_entry_set_count_visible (launcher, TRUE);
		}
	}
	g_signal_connect (dispatch_op, "invalidated", G_CALLBACK (dispatch_op_finished_cb), indicator);
}

static void
add_dispatch_operation_cb (TpSimpleApprover *self, TpAccount *account, TpConnection *connection,
    GList *channels, TpChannelDispatchOperation *dispatch_op, TpAddDispatchOperationContext *context,
    gpointer user_data)
{
	GList *itr;
	TpChannel *channel;
	TpHandle handle;

	static TpContactFeature features[] = {
		TP_CONTACT_FEATURE_ALIAS,
		TP_CONTACT_FEATURE_AVATAR_DATA,
	};

	g_object_ref (dispatch_op);
	for (itr = channels; itr != NULL; itr = g_list_next (itr))
	{
		TpChannel *channel = itr->data;
		handle = tp_channel_get_initiator_handle (channel);
		tp_connection_get_contacts_by_handle (connection, 1, &handle, 2, features, handle_contacts_add_indicator_cb, dispatch_op, NULL, NULL);
	}
	g_list_free(itr);
	tp_add_dispatch_operation_context_accept (context);
}

static void
server_display (IndicateServer * indicate_server, guint timestamp, gpointer user_data)
{
	g_debug ("server_display at %u", timestamp);
	GDesktopAppInfo *app;
	GError *error = NULL;

	app = g_desktop_app_info_new ("empathy.desktop");
	g_app_info_launch ((GAppInfo*)app, NULL, NULL, &error);

	if (error)
	{
		g_warning ("Failed to launch empathy: %s", error->message);
		g_error_free (error);
	}

	g_object_unref (app);
}

static IndicateServer *
indicate_server_setup ()
{
	IndicateServer *indicate_server = NULL;

	const gchar *desktop_file = "/usr/share/applications/empathy.desktop";
	indicate_server = indicate_server_ref_default ();
	indicate_server_set_type (indicate_server, "message.im");

	indicate_server_set_desktop_file (indicate_server, desktop_file);
	indicate_server_show (indicate_server);
	indicators = g_hash_table_new(g_str_hash, g_str_equal);
	launcher = unity_launcher_entry_get_for_desktop_id ("empathy.desktop");

	g_signal_connect (G_OBJECT (indicate_server),
		INDICATE_SERVER_SIGNAL_SERVER_DISPLAY,
		G_CALLBACK (server_display), NULL);
	return indicate_server;
}

static TpBaseClient *
approver_setup ()
{
	TpBaseClient *approver;
	TpAccountManager *tp_am;
	GError *error = NULL;

	g_type_init ();

	tp_am = tp_account_manager_dup ();

	approver = tp_simple_approver_new_with_am (tp_am, "IndicatorApprover",
				FALSE, add_dispatch_operation_cb, NULL, NULL);

	/* contact text chat */
	tp_base_client_take_approver_filter (approver, tp_asv_new (
		TP_PROP_CHANNEL_CHANNEL_TYPE, G_TYPE_STRING, TP_IFACE_CHANNEL_TYPE_TEXT,
		TP_PROP_CHANNEL_TARGET_HANDLE_TYPE, G_TYPE_UINT, TP_HANDLE_TYPE_CONTACT,
		NULL));

	/* room text chat */
	tp_base_client_take_approver_filter (approver, tp_asv_new (
		TP_PROP_CHANNEL_CHANNEL_TYPE, G_TYPE_STRING, TP_IFACE_CHANNEL_TYPE_TEXT,
		TP_PROP_CHANNEL_TARGET_HANDLE_TYPE, G_TYPE_UINT, TP_HANDLE_TYPE_ROOM,
		NULL));

	if (!tp_base_client_register (approver, &error))
	{
                if (error)
		{
			g_warning ("Failed to register the approver: %s", error->message);
			g_error_free (error);
		}
		g_object_unref (tp_am);
		g_object_unref (approver);
		return NULL;
	}
	return approver;
}

int
main (int argc, char **argv)
{
	GMainLoop *loop = NULL;

	TpBaseClient *approver;
	IndicateServer *indicate_server;

	approver = approver_setup ();
	if (approver == NULL)
	{
		g_error ("Approver registration failed, exiting");
	}

	indicate_server = indicate_server_setup ();

	g_debug ("Telepathy Indicator started");
	loop = g_main_loop_new (NULL, FALSE);
	g_main_loop_run (loop);
}
