package com.sun.star.wizards.tutorial.creator.frame.objectFinder;

import java.awt.Component;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;

import com.sun.star.accessibility.AccessibleRole;
import com.sun.star.accessibility.XAccessible;
import com.sun.star.accessibility.XAccessibleContext;
import com.sun.star.awt.XExtendedToolkit;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.wizards.tutorial.executer.util.NameProvider;

/**
 * Factor for creating JMenuBar objects for accessible object trees that represent menus
 * in OpenOffice applications.
 */
class MenuFactory {
	private IAccessibleObjectDisplay mDisplay;
	private static final int mnMenuBarRole = AccessibleRole.MENU_BAR;
	private static final int mnMenuRole = AccessibleRole.MENU;
	private static final int mnMenuItemRole = AccessibleRole.MENU_ITEM;
	private static final int mnSeparator = AccessibleRole.SEPARATOR;

	/**
	 * Create a new factory.  When ever one of the menu items in a menu created by the
	 * new object is selected (mouse button released) then the given display is set to
	 * the associated accessible object.
	 *
	 * @param aDisplay
	 */
	public MenuFactory(IAccessibleObjectDisplay aDisplay) {
		mDisplay = aDisplay;
	}

	/**
	 * Create new JMenuBar
	 *
	 * @param xToolkit object
	 *
	 * @return the current staroffice-JMenuBar
	 */
	public JMenuBar GetJMenuBar(XExtendedToolkit xToolkit) {
		JMenuBar aMenuBar = null;

		// Iterate over all toplevel application windows.
		if (xToolkit != null) {
			int nTopWindowCount = xToolkit.getTopWindowCount();

			try {
				for (int i = 0; i < nTopWindowCount; i++) {
					XAccessible xAccessible = (XAccessible) UnoRuntime.queryInterface(XAccessible.class, xToolkit.getTopWindow(i));
					XAccessible xMenu = GetMenu(xAccessible);

					if (xMenu != null) {
						aMenuBar = CreateJMenuBar(xMenu);

						break;
					}
				}
			} catch (com.sun.star.lang.IndexOutOfBoundsException e) {
			}
		}

		return aMenuBar;
	}

	/**
	 * Find the first descendant which represents a menu.
	 *
	 * @param xAccessible object
	 *
	 * @return When there is no descendant that represents a menu then return null.
	 */
	private XAccessible GetMenu(XAccessible xAccessible) {
		XAccessible xMenu = null;
		int nMenuRole = mnMenuBarRole;

		if (xAccessible != null) {
			XAccessibleContext xContext = xAccessible.getAccessibleContext();

			if (xContext != null) {
				if (xContext.getAccessibleRole() == nMenuRole) {
					xMenu = xAccessible;
				} else {
					// Iterate over all children and find the menu among
					// their sub trees.
					try {
						for (int i = 0; i < xContext.getAccessibleChildCount(); i++) {
							xMenu = GetMenu(xContext.getAccessibleChild(i));

							if (xMenu != null) {
								break;
							}
						}
					} catch (com.sun.star.lang.IndexOutOfBoundsException e) {
					}
				}
			}
		}

		return xMenu;
	}

	/**
	 * Create a JMenuBar object for the given accessibility tree.
	 *
	 * @param xMenu object
	 *
	 * @return JMenuBar
	 */
	private JMenuBar CreateJMenuBar(XAccessible xMenu) {
		JMenuBar aMenuBar = new JMenuBar();

		if (xMenu != null) {
			XAccessibleContext xContext = xMenu.getAccessibleContext();

			if (xContext != null) {
				try {
					for (int i = 0; i < xContext.getAccessibleChildCount(); i++) {
						JMenu aMenu = CreateJMenu(xContext.getAccessibleChild(i));

						if (aMenu != null) {
							aMenuBar.add(aMenu);
						}
					}
				} catch (com.sun.star.lang.IndexOutOfBoundsException e) {
				}
			}
		}

		return aMenuBar;
	}

	/**
	 * Create a menu for the given accessible object.  It is either a top level menu or a
	 * sub menu of arbitrary depth.  Children of the accessible object may represent
	 * either a sub menu, a menu item, or a separator.
	 *
	 * @param xMenu object
	 *
	 * @return JMenu
	 */
	private JMenu CreateJMenu(XAccessible xMenu) {
		JMenu aMenu = null;

		if (xMenu != null) {
			final XAccessibleContext xContext = xMenu.getAccessibleContext();

			if (xContext != null) {
				aMenu = new JMenu(xContext.getAccessibleName());
				aMenu.addMouseListener(new java.awt.event.MouseListener() {
					public void mousePressed(java.awt.event.MouseEvent e) {
					}

					public void mouseReleased(java.awt.event.MouseEvent e) {
					}

					public void mouseClicked(java.awt.event.MouseEvent e) {
						mDisplay.setAccessibleObject(xContext);
					}

					public void mouseEntered(java.awt.event.MouseEvent e) {
					}

					public void mouseExited(java.awt.event.MouseEvent e) {
					}
				});

				try {
					for (int i = 0; i < xContext.getAccessibleChildCount(); i++) {
						XAccessible xAccessible = xContext.getAccessibleChild(i);

						if (xAccessible != null) {
							XAccessibleContext xChild = xAccessible.getAccessibleContext();

							if (xChild != null) {
								Component aMenuItem = null;

								switch (xChild.getAccessibleRole()) {
									case mnMenuRole :
										aMenuItem = CreateJMenu(xAccessible);

										break;

									case mnMenuItemRole :
										aMenuItem = CreateJMenuItem(xAccessible);

										break;

									case mnSeparator :
										aMenuItem = null;
										aMenu.addSeparator();

										break;

									default :
										aMenuItem = new JMenuItem(xChild.getAccessibleName() + " / " + NameProvider.getRoleName(xChild.getAccessibleRole()));

										break;
								}

								if (aMenuItem != null) {
									aMenu.add(aMenuItem);
								}
							}
						}
					}
				} catch (com.sun.star.lang.IndexOutOfBoundsException e) {
				}
			}
		}

		return aMenu;
	}

	/**
	 * Create a menu item for the given accessible object.  The menu item is created with
	 * an action that sets that accessible object at the display that was given to this
	 * class' constructor.
	 *
	 * @param xAccessible object
	 *
	 * @return JMenuItem
	 */
	private JMenuItem CreateJMenuItem(XAccessible xAccessible) {
		JMenuItem aMenuItem = null;

		if (xAccessible != null) {
			final XAccessibleContext xContext = xAccessible.getAccessibleContext();
			aMenuItem = new JMenuItem(
				// Create the menu item with an action that, when called,
		// sets the accessible object given to this method at the
		// display.
	new AbstractAction(xContext.getAccessibleName()) {
				public void actionPerformed(ActionEvent e) {
					mDisplay.setAccessibleObject(xContext);
				}
			});
		}

		return aMenuItem;
	}
}
