/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: userpaintoverlay.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 20:32:10 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <canvas/debug.hxx>
#include <userpaintoverlay.hxx>

#ifndef _COM_SUN_STAR_AWT_MOUSEBUTTON_HPP_
#include <com/sun/star/awt/MouseButton.hpp>
#endif
#ifndef _COM_SUN_STAR_PRESENTATION_XSLIDESHOWVIEW_HPP_
#include <com/sun/star/presentation/XSlideShowView.hpp>
#endif

#ifndef _BGFX_POLYGON_B2DPOLYGON_HXX
#include <basegfx/polygon/b2dpolygon.hxx>
#endif

#ifndef _CPPCANVAS_BASEGFXFACTORY_HXX
#include <cppcanvas/basegfxfactory.hxx>
#endif

#ifndef _OSL_MUTEX_HXX_
#include <osl/mutex.hxx>
#endif

#include <mouseeventhandler.hxx>


using namespace ::com::sun::star;

namespace presentation
{
    namespace internal
    {
        class ScreenUpdateActivity : public Activity
        {
        public:
            virtual bool perform() { return isActive(); };
            virtual double calcTimeLag() const { return 0.0; }
            virtual bool isActive() const { return false; }
            virtual bool needsScreenUpdate() const { return true; }
            virtual void dequeued() {}
            // Disposable:
            virtual void dispose() {}
        };
    
        class PaintOverlayHandler : public MouseEventHandler
        {
        public:
            PaintOverlayHandler( const RGBColor&	rStrokeColor,
                                 double		  		nStrokeWidth,
                                 ActivitiesQueue&   rActivitiesQueue ) :
                maViews(),
                mrActivitiesQueue( rActivitiesQueue ),
                mpScreenUpdateActivity( new ScreenUpdateActivity ),
                maStrokeColor( rStrokeColor ),
                mnStrokeWidth( nStrokeWidth ),
                maLastPoint(),
                maLastMouseDownPos(),
                mbIsLastPointValid( false ),
                mbIsLastMouseDownPosValid( false )
            {
            }

            bool addView( const UnoViewSharedPtr& rView )
            {
                // TODO(Q2): Factor out the dispersed storing of Views
                // into a ViewContainer object (other places are
                // Slide, Presentation_Impl, LayerManager etc).

                ::osl::MutexGuard aGuard( maMutex );

                maViews.push_back( rView );

                return true;
            }

			bool removeView( const UnoViewSharedPtr& rView )
            {
                // TODO(Q2): Factor out the dispersed storing of Views
                // into a ViewContainer object (other places are
                // Slide, Presentation_Impl, LayerManager etc).

                ::osl::MutexGuard aGuard( maMutex );

                maViews.erase( ::std::remove( maViews.begin(),
                                              maViews.end(),
                                              rView ) );

                return true;
            }

            // MouseEventHandler methods
            virtual void dispose()
            {
                ::osl::MutexGuard aGuard( maMutex );

                maViews.clear();
            }

            virtual bool handleMousePressed( const awt::MouseEvent& e )
            {
                maLastMouseDownPos.setX( e.X );
                maLastMouseDownPos.setY( e.Y );
                mbIsLastMouseDownPosValid = true;

                // eat mouse click (though we don't process it
                // _directly_, it enables the drag mode
                return true;
            }

            virtual bool handleMouseReleased( const awt::MouseEvent& e )
            {
                // check, whether up- and down press are on exactly
                // the same pixel. If that's the case, ignore the
                // click, and pass on the event to low-prio
                // handlers. This effectively permits effect
                // advancements via clicks also when user paint is
                // enabled.
                if( mbIsLastMouseDownPosValid &&
                    ::basegfx::B2DPoint( e.X,
                                         e.Y ) == maLastMouseDownPos )
                {
                    mbIsLastMouseDownPosValid = false;
                    return false;
                }

                // invalidate, next downpress will have to start a new
                // polygon.
                mbIsLastPointValid = false;

                // eat mouse click (though we don't process it
                // _directly_, it enables the drag mode
                return true;
            }

            virtual bool handleMouseEntered( const awt::MouseEvent& e )
            {
                ::osl::MutexGuard aGuard( maMutex );

                mbIsLastPointValid = true;
                maLastPoint.setX( e.X );
                maLastPoint.setY( e.Y );

                return true;
            }

            virtual bool handleMouseExited( const awt::MouseEvent& )
            {
                ::osl::MutexGuard aGuard( maMutex );

                mbIsLastPointValid = false;
                mbIsLastMouseDownPosValid = false;

                return true;
            }

            virtual bool handleMouseDragged( const awt::MouseEvent& e )
            {
                ::osl::MutexGuard aGuard( maMutex );

                if( !mbIsLastPointValid )
                {
                    mbIsLastPointValid = true;
                    maLastPoint.setX( e.X );
                    maLastPoint.setY( e.Y );
                }
                else
                {
                    ::basegfx::B2DPolygon aPoly;
                    aPoly.append( maLastPoint );

                    maLastPoint.setX( e.X );
                    maLastPoint.setY( e.Y );

                    aPoly.append( maLastPoint );

                    // paint to all views
                    for( UnoViewVector::iterator aIter=maViews.begin(), aEnd=maViews.end();
                         aIter!=aEnd;
                         ++aIter )
                    {
                        ::cppcanvas::PolyPolygonSharedPtr pPolyPoly( 
                            ::cppcanvas::BaseGfxFactory::getInstance().createPolyPolygon( (*aIter)->getCanvas(), 
                                                                                          aPoly ) );
                        
                        if( pPolyPoly.get() )
                        {
                            pPolyPoly->setRGBALineColor( maStrokeColor.getIntegerColor() );
                            pPolyPoly->draw();
                        }
                    }

                    // update screen, force update (layer manager is
                    // bypassed in this implementation)
                    mrActivitiesQueue.addActivity( mpScreenUpdateActivity );
                }

                // mouse events captured
                return true;
            }

            virtual bool handleMouseMoved( const awt::MouseEvent& e )
            {
                // not used here
                return false; // did not handle the event
            }

        private:
            ::osl::Mutex		maMutex;	
            UnoViewVector		maViews;			
            ActivitiesQueue& 	mrActivitiesQueue;
            ActivitySharedPtr   mpScreenUpdateActivity;
			RGBColor			maStrokeColor;
            double				mnStrokeWidth;
            ::basegfx::B2DPoint	maLastPoint;
            ::basegfx::B2DPoint	maLastMouseDownPos;
            bool				mbIsLastPointValid;
            bool				mbIsLastMouseDownPosValid;
        };

        UserPaintOverlay::UserPaintOverlay(
            const RGBColor& 	rStrokeColor,
            double		  		nStrokeWidth,
            EventMultiplexer& 	rMultiplexer,
            ActivitiesQueue&    rActivitiesQueue )
            : mpHandler( new PaintOverlayHandler( rStrokeColor, 
                                                  nStrokeWidth,
                                                  rActivitiesQueue ) ),
            mrMultiplexer( rMultiplexer )
        {
            mrMultiplexer.addClickHandler( mpHandler, 3.0 );
            mrMultiplexer.addMouseMoveHandler( mpHandler, 3.0 );
        }
        
        UserPaintOverlay::~UserPaintOverlay()
        {
            mrMultiplexer.removeMouseMoveHandler( mpHandler );
            mrMultiplexer.removeClickHandler( mpHandler );
            mpHandler->dispose();
        }

        bool UserPaintOverlay::addView( const UnoViewSharedPtr& rView )
        {
            return mpHandler->addView( rView );
        }

        bool UserPaintOverlay::removeView( const UnoViewSharedPtr& rView )
        {
            return mpHandler->removeView( rView );
        }
    }
}
