/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: dx_spritecanvas.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 23:31:00 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <canvas/debug.hxx>
#include <canvas/verbosetrace.hxx>

#ifndef _OSL_MUTEX_HXX_
#include <osl/mutex.hxx>
#endif

#ifndef _COM_SUN_STAR_REGISTRY_XREGISTRYKEY_HPP_
#include <com/sun/star/registry/XRegistryKey.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XINITIALIZATION_HPP_
#include <com/sun/star/lang/XInitialization.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XSERVICEINFO_HPP_
#include <com/sun/star/lang/XServiceInfo.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XSERVICENAME_HPP_
#include <com/sun/star/lang/XServiceName.hpp>
#endif
#include <com/sun/star/lang/NoSupportException.hpp>
#ifndef _COM_SUN_STAR_LANG_XSINGLESERVICEFACTORY_HPP_
#include <com/sun/star/lang/XSingleServiceFactory.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XCOMPONENTCONTEXT_HPP_
#include <com/sun/star/uno/XComponentContext.hpp>
#endif
#ifndef _COM_SUN_STAR_AWT_RECTANGLE_HPP_
#include <com/sun/star/awt/Rectangle.hpp>
#endif

#ifndef _CPPUHELPER_FACTORY_HXX_
#include <cppuhelper/factory.hxx>
#endif
#ifndef _CPPUHELPER_IMPLEMENTATIONENTRY_HXX_
#include <cppuhelper/implementationentry.hxx>
#endif

#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#include <basegfx/matrix/b2dhommatrix.hxx>
#endif
#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_TOOLS_CANVASTOOLS_HXX
#include <basegfx/tools/canvastools.hxx>
#endif
#ifndef _BGFX_NUMERIC_FTOOLS_HXX
#include <basegfx/numeric/ftools.hxx>
#endif

#ifndef BOOST_BIND_HPP_INCLUDED
#include <boost/bind.hpp>
#endif

#include <canvas/elapsedtime.hxx>
#include <canvas/verbosetrace.hxx>

#include <dx_winstuff.hxx>
#include <dx_spritecanvas.hxx>
#include <dx_canvascustomsprite.hxx>
#include <dx_surfacemanager.hxx>
#include <dx_gdisurface.hxx>

using namespace ::com::sun::star;


#define IMPLEMENTATION_NAME "DXCanvas::SpriteCanvas"
#define SERVICE_NAME "com.sun.star.rendering.DXCanvas"

namespace
{
    static ::rtl::OUString SAL_CALL getImplementationName_SpriteCanvas()
    {
        return ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( IMPLEMENTATION_NAME ) );
    }

    static uno::Sequence< ::rtl::OUString > SAL_CALL getSupportedServiceNames_SpriteCanvas()
    {
        uno::Sequence< ::rtl::OUString > aRet(1);
        aRet[0] = ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM ( SERVICE_NAME ) );
        
        return aRet;
    }

}

namespace dxcanvas
{
    SpriteCanvas::SpriteCanvas( const uno::Reference< uno::XComponentContext >& rxContext ) :
        mxComponentContext( rxContext ),
        mxRefDevice(),
        mpDXDevice(),
        mpBackBuffer(),
        mpDXBuffer(),
        maSprites(),
        maChangeRecords(),
        maBounds(),
        mbIsFullscreen( false ),
        mbIsVisible( false ),
        mbDXSurfaceStale( true )
    {
    }

    SpriteCanvas::~SpriteCanvas()
    {
    }

    // this function is called upon disposing the component
    void SAL_CALL SpriteCanvas::disposing()
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        dispose();

        // forward to parent
        SpriteCanvas_Base::disposing();
    }

    uno::Reference< rendering::XAnimatedSprite > SAL_CALL SpriteCanvas::createSpriteFromAnimation( const uno::Reference< rendering::XAnimation >& animation ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        // TODO
        return uno::Reference< rendering::XAnimatedSprite >();
    }

    uno::Reference< rendering::XAnimatedSprite > SAL_CALL SpriteCanvas::createSpriteFromBitmaps( const uno::Sequence< uno::Reference< rendering::XBitmap > >& 	animationBitmaps, 
                                                                                                 sal_Int8 														interpolationMode ) throw (lang::IllegalArgumentException, rendering::VolatileContentDestroyedException, uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        // TODO
        return uno::Reference< rendering::XAnimatedSprite >();
    }

    uno::Reference< rendering::XCustomSprite > SAL_CALL SpriteCanvas::createCustomSprite( const geometry::RealSize2D& spriteSize ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        return uno::Reference< rendering::XCustomSprite >( 
            new CanvasCustomSprite( spriteSize,
                                    mxRefDevice,
                                    ImplRef( this ),
                                    mpDXDevice ) );
    }

    uno::Reference< rendering::XSprite > SAL_CALL SpriteCanvas::createClonedSprite( const uno::Reference< rendering::XSprite >& original ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        // TODO
        return uno::Reference< rendering::XSprite >();
    }

    namespace
    {
        class SpritePainter
        {
        public:
            SpritePainter( const DeviceSharedPtr& rDevice ) :
                mrDevice( rDevice )
            {
            }

            void operator()( const Sprite::ImplRef& rSprite )
            {
                rSprite->redraw( mrDevice );
            }

        private:
            const DeviceSharedPtr& mrDevice;
        };

        /** Collector struct, needed for SpriteUpdater

            This is a cheap version of the
            ::basegfx::b2dconnectedranges template, which simply puts
            all updates into a single bounding rect
         */
        struct UpdateCollector
        {
            UpdateCollector() :
                maUpdateArea(),
                maUpdateSprites()
            {
            }

            void addRange( const ::basegfx::B2DRectangle& 	rRect,
                           const Sprite::ImplRef& 			rSprite )
            {
                if( rSprite.is() )
                    maUpdateSprites.push_back( rSprite );

                maUpdateArea.expand( rRect );
            }

            ::basegfx::B2DRectangle 			maUpdateArea;
            ::std::vector< Sprite::ImplRef > 	maUpdateSprites;
        };
    }

    sal_Bool SAL_CALL SpriteCanvas::updateScreen( sal_Bool bUpdateAll ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        // hidden windows must not paint anything, thus prevent
        // screen updates, then
        if( !mbIsVisible )
            return sal_False;

        // Early exit: if backbuffer didn't change, and sprites,
        // neither, we got exactly _nothing_ to paint here. Bail out.
        if( !bUpdateAll && !mbSurfaceDirty && maChangeRecords.empty() )
            return sal_True;


        // (Re)alloc DirectDraw surface
        // ============================

        if( mpDXBuffer.get() == NULL ||
            mpDXBuffer->contentLost()	)
        {
            // (re-)create DirectX surface
            mpDXBuffer.reset();
            mpDXBuffer = SurfaceManager::getInstance().createSurface( mpDXDevice->getSize(),
                                                                      0.5, 
                                                                      mpDXDevice );
            mbDXSurfaceStale = true;
        }

        // make sure surfaces are up-to-date (nothing queued anymore
        // etc.)
        maCanvasHelper.flush();

        
        // Determine area of screen update
        // ===============================

        const bool bNeedFullScreenUpdate( bUpdateAll			// force complete update
                                          || mbSurfaceDirty		// don't know where and what 
        											  			// changed -> update everything. 
                                          || mbIsFullscreen );	// For a fullscreen canvas, we got
													    		// no choice, either, since backbuffer
													    		// lags at least one frame behind, or
													    		// worse, is trashed by the system.

        // collect sprites and areas that need updates into this
        UpdateCollector aUpdateCollector;
            
        // determine area that needs an update
        if( !bNeedFullScreenUpdate )
        {
            // background has not changed, so we're free to optimize
            // repaint to areas where a sprite has changed

            // TODO(F2): Unchanged sprites, which happen to be in the 
            // area-to-be-updated are currently not repainted.

            // extract all referenced sprites from the maChangeRecords
            // (copy sprites, make the list unique, regarding the
            // sprite pointer). This assumes that until this scope
            // ends, nobody changes the maChangeRecords vector!
            typedef ::std::vector< Sprite::ImplRef > UpdateSprites;
            UpdateSprites aUpdateSprites;
            ::std::transform( maChangeRecords.begin(),
                              maChangeRecords.end(),
                              ::std::back_insert_iterator< UpdateSprites >(aUpdateSprites),
                              ::std::mem_fun_ref( &SpriteChangeRecord::getSprite ) );
            ::std::sort( aUpdateSprites.begin(),
                         aUpdateSprites.end() );

            UpdateSprites::iterator aBegin( aUpdateSprites.begin() );
            UpdateSprites::iterator aEnd  ( aUpdateSprites.end() );
            aEnd = ::std::unique( aBegin, aEnd );

            // check for unique sprites in the change event vector,
            // calculate the update operation from that, and add the
            // result to the aUpdateArea.
            ::std::for_each( aBegin, aEnd,
                             ::canvas::internal::SpriteUpdater< Sprite::ImplRef, 
                             									ChangeRecords, 
                             									UpdateCollector >(aUpdateCollector,
                                                                                  maChangeRecords) );

            if( aUpdateCollector.maUpdateArea.isEmpty() )
            {
                // TODO(E1): Check why this actually happens
                VERBOSE_TRACE( "SpriteCanvas::updateScreen(): Empty update area after SpriteChecker call" );
                return sal_True;
            }

            // Note: we blow up the rectangle by a few pixel in every
            // direction, as we might use subpixel positioning of
            // sprites. This might touch one more pixel on every side
            // of the objects. Furthermore, the DX blit operations
            // seem to have varying opinions about
            // inclusiveness/exclusiveness of the bottom/right edge of
            // rects.
            aUpdateCollector.maUpdateArea = ::basegfx::B2DRectangle( aUpdateCollector.maUpdateArea.getMinX() - 1.0,
                                                                     aUpdateCollector.maUpdateArea.getMinY() - 1.0,
                                                                     aUpdateCollector.maUpdateArea.getMaxX() + 4.0 ,
                                                                     aUpdateCollector.maUpdateArea.getMaxY() + 4.0 );
        }


        // render background content to screen
        // ===================================

        if( mpDXBuffer.get() != NULL )
        {
            // does the DirectX surface need an update?
            if( mbDXSurfaceStale ||
                mbSurfaceDirty )
            {
                // Cache Graphics for destination surface.
                SurfaceGraphicsSharedPtr pGraphics( mpDXBuffer->getGraphics() );

                // need to copy as-is to backbuffer, not blend with the previous content
                pGraphics->get()->SetCompositingMode( Gdiplus::CompositingModeSourceCopy );

                // push backbuffer to DX surface
                pGraphics->get()->DrawImage( mpBackBuffer.get(), 
                                             0, 
                                             0 );
                mbDXSurfaceStale = false;
            }

            // push backbuffer surface to primary surface
            if( !bNeedFullScreenUpdate )
            {
                // only repaint aUpdateCollector's update rect
                mpDXDevice->render( mpDXBuffer, 
                                    ::basegfx::B2IPoint( ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinX()),
                                                         ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinY()) ),
                                    ::basegfx::B2IRectangle( ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinX()),
                                                             ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinY()),
                                                             ::basegfx::fround(aUpdateCollector.maUpdateArea.getMaxX()),
                                                             ::basegfx::fround(aUpdateCollector.maUpdateArea.getMaxY()) ) );
            }
            else
            {
                // need to repaint the whole screen
                mpDXDevice->render( mpDXBuffer, 
                                    ::basegfx::B2IPoint() );
            }
        }
        else
        {
            VERBOSE_TRACE("SpriteCanvas::updateScreen(): emulating redraw via GdiPlus");

            // DirectX seems to be non-operational, push backbuffer
            // directly to primary surface
            if( !bNeedFullScreenUpdate )
            {
                // only repaint aUpdateCollector's update rect
                mpDXDevice->getPrimarySurface()->getGraphics()->get()->DrawImage( mpBackBuffer.get(), 
                                                                                  ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinX()), 
                                                                                  ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinY()),
                                                                                  ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinX()), 
                                                                                  ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinY()),
                                                                                  ::basegfx::fround(aUpdateCollector.maUpdateArea.getWidth()), 
                                                                                  ::basegfx::fround(aUpdateCollector.maUpdateArea.getHeight()),
                                                                                  Gdiplus::UnitPixel );
            }
            else
            {
                // need to repaint the whole screen
                mpDXDevice->getPrimarySurface()->getGraphics()->get()->DrawImage( mpBackBuffer.get(), 
                                                                                  0, 
                                                                                  0 );
            }
        }

        // render the sprites on top of that
        if( !bNeedFullScreenUpdate )
        {
            // only render sprites that really need update
            ::std::for_each( aUpdateCollector.maUpdateSprites.begin(), 
                             aUpdateCollector.maUpdateSprites.end(), 
                             SpritePainter( mpDXDevice ) );
        }
        else
        {
            // render all active sprites
            ::std::for_each( maSprites.begin(), 
                             maSprites.end(), 
                             SpritePainter( mpDXDevice ) );
        }

        // flip primary surface to screen
        // ==============================

        // limit update to parent window area (ignored for fullscreen)
        ::basegfx::B2IRectangle aUpdateArea( 0,0,
                                             maBounds.Width,
                                             maBounds.Height );

        if( !bNeedFullScreenUpdate )
        {
            // only update changed area, clip with update collector area
            aUpdateArea.intersect( ::basegfx::B2IRectangle( ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinX()),
                                                            ::basegfx::fround(aUpdateCollector.maUpdateArea.getMinY()),
                                                            ::basegfx::fround(aUpdateCollector.maUpdateArea.getMaxX()),
                                                            ::basegfx::fround(aUpdateCollector.maUpdateArea.getMaxY()) ) );
        }

        // perform buffer flipping
        mpDXDevice->flip( aUpdateArea,
                          ::basegfx::B2IPoint( maBounds.X,
                                               maBounds.Y ) );

#if defined(VERBOSE) && defined(DBG_UTIL)
        static ::canvas::tools::ElapsedTime aElapsedTime;

        // log time immediately after surface flip
        OSL_TRACE( "SpriteCanvas::updateScreen(): flip done at %f", 
                   aElapsedTime.getElapsedTime() );
#endif

        // change record vector must be cleared, for the next turn of
        // rendering and sprite changing
        maChangeRecords.clear();

        // screen is now up-to-date
        mbSurfaceDirty = false;

        return sal_True;
    }

    void SAL_CALL SpriteCanvas::initialize( const uno::Sequence< uno::Any >& aArguments ) throw( uno::Exception, 
                                                                                                 uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        VERBOSE_TRACE( "SpriteCanvas::initialize called" );
        CHECK_AND_THROW( aArguments.getLength() >= 1, 
                         "SpriteCanvas::initialize: wrong number of arguments" );

        // At index 1, we expect a HWND handle here, containing a
        // pointer to a valid window, on which to output
        if( aArguments.getLength() >= 1 &&
            aArguments[1].getValueTypeClass() == uno::TypeClass_LONG )
        {
            HWND aHWnd = *(HWND*)(aArguments[1].getValue());

            CHECK_AND_THROW( IsWindow( aHWnd ), 
                             "SpriteCanvas::initialize(): No valid HWND given." );

            awt::Rectangle aRect;
            aArguments[2] >>= aRect;
            const ::basegfx::B2ISize aSize(aRect.Width,aRect.Height);

            sal_Bool bIsFullscreen;
            aArguments[3] >>= bIsFullscreen;

            mbIsFullscreen = bIsFullscreen;


            // create reference XGraphicDevice
            // -------------------------------
            if( mbIsFullscreen )
                mxRefDevice = WindowGraphicDevice::ImplRef( 
                    new WindowGraphicDevice( aHWnd, aSize ) );
            else
                mxRefDevice = WindowGraphicDevice::ImplRef( 
                    new WindowGraphicDevice( aHWnd ) );


            // setup DirectX device
            // --------------------

            if( mbIsFullscreen )
                mpDXDevice.reset( new Device( aHWnd, aSize  ) );
            else
                mpDXDevice.reset( new Device( aHWnd ) );

            if (mpDXDevice.get() == 0 ||
                mpDXDevice->getPrimarySurface().get() == 0)
                throw lang::NoSupportException(
                    ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM(
                                  "Could not create DirectX device!") ),
                    static_cast<OWeakObject *>(this) );
            
            // setup back buffer
            // -----------------
            const ::basegfx::B2ISize aScreenSize( mpDXDevice->getSize() );
            mpBackBuffer.reset( 
                new Gdiplus::Bitmap( 
                    aScreenSize.getX(), 
                    aScreenSize.getY(), 
                    mpDXDevice->getPrimarySurface()->getGraphics()->get() ) );

            SurfaceGraphicsSharedPtr pSurface(
                new SurfaceGraphics(
                    Gdiplus::Graphics::FromImage(
                        mpBackBuffer.get() ) ) );
                
            // setup backbuffer graphics
            tools::setupGraphics( *pSurface->get() );

            // clear backbuffer
            Gdiplus::Color whiteColor(255, 255, 255, 255);
            (*pSurface.get())->Clear( whiteColor );


            // setup CanvasHelper
            // ------------------
            maCanvasHelper.setGraphicDevice( mxRefDevice );
            maCanvasHelper.setBitmap( mpBackBuffer );
        }
    }

    void SAL_CALL SpriteCanvas::dispose(  ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        mxComponentContext.clear();
        mxRefDevice.reset();
        mpDXDevice.reset();
        mpBackBuffer.reset();
        mpDXBuffer.reset();
        maSprites.clear();
        maChangeRecords.clear();
    }

    void SAL_CALL SpriteCanvas::addEventListener( const uno::Reference< lang::XEventListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::removeEventListener( const uno::Reference< lang::XEventListener >& aListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::setPosSize( sal_Int32 nX, 
                                            sal_Int32 nY, 
                                            sal_Int32 nWidth, 
                                            sal_Int32 nHeight, 
                                            sal_Int16 nFlags ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        maBounds.X = nX;
        maBounds.Y = nY;
        maBounds.Width = nWidth;
        maBounds.Height = nHeight;
    }

    awt::Rectangle SAL_CALL SpriteCanvas::getPosSize(  ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        return maBounds;
    }

    void SAL_CALL SpriteCanvas::setVisible( ::sal_Bool bVisible ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        mbIsVisible = bVisible;
    }

    void SAL_CALL SpriteCanvas::setEnable( ::sal_Bool Enable ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::setFocus(  ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::addWindowListener( const uno::Reference< awt::XWindowListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::removeWindowListener( const uno::Reference< awt::XWindowListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::addFocusListener( const uno::Reference< awt::XFocusListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::removeFocusListener( const uno::Reference< awt::XFocusListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::addKeyListener( const uno::Reference< awt::XKeyListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::removeKeyListener( const uno::Reference< awt::XKeyListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::addMouseListener( const uno::Reference< awt::XMouseListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::removeMouseListener( const uno::Reference< awt::XMouseListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::addMouseMotionListener( const uno::Reference< awt::XMouseMotionListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::removeMouseMotionListener( const uno::Reference< awt::XMouseMotionListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::addPaintListener( const uno::Reference< awt::XPaintListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    void SAL_CALL SpriteCanvas::removePaintListener( const uno::Reference< awt::XPaintListener >& xListener ) throw (uno::RuntimeException)
    {
        // Ignored
    }

    ::rtl::OUString SAL_CALL SpriteCanvas::getImplementationName() throw( uno::RuntimeException )
    {
        return getImplementationName_SpriteCanvas();
    }

    sal_Bool SAL_CALL SpriteCanvas::supportsService( const ::rtl::OUString& ServiceName ) throw( uno::RuntimeException )
    {
        return ServiceName.equalsAsciiL( RTL_CONSTASCII_STRINGPARAM ( SERVICE_NAME ) );
    }

    uno::Sequence< ::rtl::OUString > SAL_CALL SpriteCanvas::getSupportedServiceNames()  throw( uno::RuntimeException )
    {
        return getSupportedServiceNames_SpriteCanvas();
    }

    ::rtl::OUString SAL_CALL SpriteCanvas::getServiceName(  ) throw (uno::RuntimeException)
    {
        return ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( SERVICE_NAME ) );
    }

    uno::Reference< uno::XInterface > SAL_CALL SpriteCanvas::createInstance( const uno::Reference< uno::XComponentContext >& xContext ) throw ( uno::Exception )
    {
        return uno::Reference< uno::XInterface >( static_cast<cppu::OWeakObject*>( new SpriteCanvas( xContext ) ) );
    }

    void SpriteCanvas::showSprite( const Sprite::ImplRef& rSprite )
    {
        maSprites.push_back( rSprite );
    }

    void SpriteCanvas::hideSprite( const Sprite::ImplRef& rSprite )
    {
        maSprites.remove( rSprite );
    }

    void SpriteCanvas::moveSprite( const Sprite::ImplRef& 		sprite, 
                                   const ::basegfx::B2DPoint& 	rOldPos,
                                   const ::basegfx::B2DPoint&	rNewPos )
    {
        maChangeRecords.push_back( SpriteChangeRecord( sprite,
                                                       rOldPos,
                                                       rNewPos ) );
    }

    void SpriteCanvas::updateSprite( const Sprite::ImplRef& 			sprite, 
                                     const ::basegfx::B2DPoint& 		rPos,
                                     const ::basegfx::B2DRectangle& 	rUpdateArea )
    {
        maChangeRecords.push_back( SpriteChangeRecord( sprite,
                                                       rPos,
                                                       rUpdateArea ) );
    }

}

namespace
{
	/* shared lib exports implemented with helpers */
    static struct ::cppu::ImplementationEntry s_component_entries [] =
    {
        {
            dxcanvas::SpriteCanvas::createInstance, getImplementationName_SpriteCanvas,
            getSupportedServiceNames_SpriteCanvas, ::cppu::createSingleComponentFactory,
            0, 0
        },
        { 0, 0, 0, 0, 0, 0 }
    };
}


/* Exported UNO methods for registration and object creation.
   ==========================================================
 */
extern "C"
{
    void SAL_CALL component_getImplementationEnvironment( const sal_Char** 	ppEnvTypeName, 
                                                          uno_Environment** ppEnv )
    {
        *ppEnvTypeName = CPPU_CURRENT_LANGUAGE_BINDING_NAME;
    }

    sal_Bool SAL_CALL component_writeInfo( lang::XMultiServiceFactory* 	xMgr, 
                                           registry::XRegistryKey* 		xRegistry )
    {
        return ::cppu::component_writeInfoHelper(
            xMgr, xRegistry, s_component_entries );
    }

    void * SAL_CALL component_getFactory( sal_Char const* 				implName, 
                                          lang::XMultiServiceFactory* 	xMgr,
                                          registry::XRegistryKey* 		xRegistry )
    {
        return ::cppu::component_getFactoryHelper(
            implName, xMgr, xRegistry, s_component_entries );
    }
}
