/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

/*
 * The private implementation of the Scope. This makes sure that none of the 
 * implementation details leak out into the public interface.
 */
private class ScopeImpl : GLib.Object, ScopeService
{
  private Scope _owner;
  private uint _dbus_id;
  private uint _info_changed_id;
  private uint _filters_changed_id;

  public Dee.SharedModel _results_model;
  public Dee.SharedModel _global_results_model;
  public Dee.SharedModel _filters_model;

  public ScopeImpl (Scope owner)
  {
    /* NOTE: Vala isn't allowing me to make Owner a construct variable so our
     * construction happens here instead of in construct {}
     */
    _owner = owner;
    _owner.notify["search-in-global"].connect (queue_info_changed);
    _owner.notify["visible"].connect (queue_info_changed);

    create_models (create_dbus_name ());
  }
  
  /* Create usable name prefix for the models */
  private string create_dbus_name ()
  {
    /* We randomize the names to avoid conflicts to ensure that we always
     * have a clean start (no processes hanging around that would cause our models
     * to not be the leaders)
     */
    uint t = (uint)time_t ();
    var dbus_path = _owner.dbus_path;
    var dbus_name = "com.canonical.Unity.Scope";
    dbus_name += "." + Path.get_basename (dbus_path);
    dbus_name += ".T%u".printf (t);
    return dbus_name;
  }

  private void create_models (string dbus_name)
  {
    /* Schema definitions come from the Lens specification */
    _results_model = new Dee.SharedModel (dbus_name + ".Results");
    _results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _global_results_model = new Dee.SharedModel (dbus_name + ".GlobalResults");
    _global_results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _filters_model = new Dee.SharedModel (dbus_name + ".Filters");
    _filters_model.set_schema ("s", "s", "s", "s", "a{sv}", "b", "b", "b");
    _filters_model.row_added.connect (on_filter_added);
    _filters_model.row_changed.connect (on_filter_changed);
    _filters_model.row_removed.connect (on_filter_removed);
  }

  public void export () throws IOError
  {
    var conn = Bus.get_sync (BusType.SESSION);
    _dbus_id = conn.register_object (_owner.dbus_path, this as ScopeService);

    queue_info_changed ();
  }

  /* Queue up info-changed requests as we don't want to be spamming Unity with
   * them.
   */
  private void queue_info_changed ()
  {
    if (_info_changed_id == 0)
    {
      _info_changed_id = Idle.add (emit_info_changed);
    }
  }

  private void queue_filters_changed ()
  {
    if (_filters_changed_id == 0)
    {
      _filters_changed_id = Timeout.add(0, emit_filters_changed);
    }
  }

  private bool emit_filters_changed ()
  {
    _owner.filters_changed ();
    _filters_changed_id = 0;
    return false;
  }

  private bool emit_info_changed ()
  {
    var info = ScopeInfo ();
    info.dbus_path = _owner.dbus_path;
    info.sources = _owner.sources;
    info.search_in_global = _owner.search_in_global;
    info.private_connection_name = "<not implemented>";
    info.results_model_name = _results_model.get_swarm_name ();
    info.global_results_model_name = _global_results_model.get_swarm_name ();
    info.filters_model_name = _filters_model.get_swarm_name ();
    info.hints = new HashTable<string, Variant> (null, null);

    changed (info);

    _info_changed_id = 0;
    return false;
  }

  private void on_filter_added (Dee.Model model, Dee.ModelIter iter)
  {
    string renderer_name = model.get_string (iter, FilterColumn.RENDERER_NAME);
    string icon_hint_s = model.get_string (iter, FilterColumn.ICON_HINT);
    
    Icon? icon_hint = null;
    try
      {
        if (icon_hint_s != "")
          icon_hint = Icon.new_for_string (icon_hint_s);
      }
    catch (Error e)
      {
        warning ("Error parsing GIcon data '%s': %s", icon_hint_s, e.message);
      }
    
    Filter? filter = null;
    if (renderer_name == "filter-ratings")
    {
      filter = new RatingsFilter (model.get_string (iter, FilterColumn.ID),
                                  model.get_string (iter, FilterColumn.NAME),
                                  icon_hint);
    }
    else if (renderer_name == "filter-radiooption")
    {
      filter = new RadioOptionFilter (model.get_string (iter, FilterColumn.ID),
                                      model.get_string (iter, FilterColumn.NAME),
                                      icon_hint);
                                      
    }
    else if (renderer_name == "filter-checkoption")
    {
      filter = new CheckOptionFilter (model.get_string (iter, FilterColumn.ID),
                                      model.get_string (iter, FilterColumn.NAME),
                                      icon_hint);
    }
    else if (renderer_name == "filter-multirange")
    {
      filter = new MultiRangeFilter (model.get_string (iter, FilterColumn.ID),
                                     model.get_string (iter, FilterColumn.NAME),
                                     icon_hint);
    }

    if (filter is Filter)
    {
      filter.set_model_and_iter (model, iter);
      _owner._filters.append (filter);

      queue_filters_changed ();
    }
  }

  private void on_filter_changed (Dee.Model model, Dee.ModelIter iter)
  {
    queue_filters_changed ();
  }

  private void on_filter_removed (Dee.Model model, Dee.ModelIter iter)
  {
    foreach (Filter filter in _owner._filters)
    {
      if (filter.id == model.get_string (iter, FilterColumn.ID))
      {
        _owner._filters.remove (filter);
        break;
      }
    }

    queue_filters_changed ();
  }


  /*
   * DBus Interface Implementation
   */
  public async void info_request ()
  {
    queue_info_changed ();
  }

  public async ActivationReplyRaw activate (string uri,
                                            uint action_type) throws IOError
  {
    var reply = ActivationReplyRaw ();
    
    ActivationResponse? response = _owner.activate_uri (uri);
    if (response == null)
      response = new ActivationResponse(HandledType.NOT_HANDLED);

    reply.uri = uri;
    reply.handled = response.handled;
    reply.hints = response.get_hints ();

    return reply;
  }

  public async void search (string search_string,
                            HashTable<string, Variant> hints) throws IOError
  {
    var s = new LensSearch (search_string, hints);
    s.finished.connect (() =>
    {
      search_finished (s.search_string, s.hints);
    });

    if (!s.equals (_owner.active_search))
      _owner.active_search = s;
  }

  public async void global_search (string search_string,
                                   HashTable<string, Variant> hints) throws IOError
  {
    var s = new LensSearch (search_string, hints);
    s.finished.connect (() =>
    {
      global_search_finished (s.search_string, s.hints);
    });

    if (!s.equals (_owner.active_global_search))
      _owner.active_global_search = s;
  }

  public async PreviewReplyRaw preview (string uri) throws IOError
  {
    var reply = PreviewReplyRaw ();
    
    Preview? response = _owner.preview_uri (uri);
    if (response == null)
      response = new NoPreview();

    reply.uri = uri;
    reply.renderer_name = response.get_renderer_name ();
    reply.properties = response.get_properties ();

    return reply;
  }

  public async void set_active (bool is_active) throws IOError
  {
    _owner.set_active_internal (is_active);
  }

  public async void set_active_sources (string[] sources) throws IOError
  {
    _owner.set_active_sources_internal (sources);
    foreach (string s in sources)
    {
      debug ("Source: %s", s);
    }
  }
}

} /* namespace */
