/*
 * unity-webapps-desktop-infos.h
 * Copyright (C) Canonical LTD 2013
 * 
 * Author: Alexandre Abreu <alexandre.abreu@canonical.com>
 * 
unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <glib.h>
#include <glib/gstdio.h>

#include "unity-webapps-desktop-infos.h"
#include "unity-webapps-string-utils.h"

static void
uwa_makedir_if_necessary (gchar *path)
{
  if (FALSE == g_file_test(path, G_FILE_TEST_EXISTS))
    {
      g_mkdir(path, 0700);
    }
}

gchar*
unity_webapps_desktop_infos_build_desktop_basename(const gchar* name,
                                                   const gchar* domain)
{
#define ADD_CANONICALIZED_STRING_TO_BASENAME(s) \
  { \
    const gchar *canonicalized_identifier; \
    canonicalized_identifier =                                      \
      unity_webapps_string_utils_canonicalize_string ((s), FALSE);  \
    basename = g_string_append(basename, canonicalized_identifier); \
    g_free((gpointer) canonicalized_identifier); \
  }

  GString *basename;

  g_return_val_if_fail(NULL != name, NULL);
  g_return_val_if_fail(NULL != domain, NULL);

  basename = g_string_new("");
  ADD_CANONICALIZED_STRING_TO_BASENAME(name);
  ADD_CANONICALIZED_STRING_TO_BASENAME(domain);

  return g_string_free (basename, FALSE);
}

void
unity_webapps_desktop_infos_generate_desktop_for_url_launch(const gchar* name,
							    const gchar* domain,
							    const gchar* icon_url,
							    const gchar* url)
{
  const gchar *const DESKTOP_FILE_CONTENT_FORMAT =
    "[Desktop Entry]\n"
    "Name=%s\n"
    "Type=Application\n"
    "Icon=%s\n"
    "Actions=S0;S1;S2;S3;S4;S5;S6;S7;S8;S9;S10;\n"
    "Exec=webbrowser-app --webapp --chromeless --app-id='%s' %s\n";
  gchar *desktop_id;
  gchar *app_id;
  gchar *applications_path;
  gchar *desktop_filename;
  gchar *desktop_file_content;
  GError *error;

  g_return_val_if_fail(NULL != name, NULL);
  g_return_val_if_fail(NULL != domain, NULL);
  g_return_val_if_fail(NULL != url, NULL);

  app_id = unity_webapps_desktop_infos_build_desktop_basename(name, domain);
  desktop_id = g_strdup_printf("%s.desktop", app_id);

  applications_path =
    g_build_filename(g_get_user_data_dir(),
		     "applications",
		     NULL);

  uwa_makedir_if_necessary(applications_path);

  desktop_filename =
    g_strdup_printf("%s/%s",
		    applications_path,
		    desktop_id);

  desktop_file_content =
    g_strdup_printf(DESKTOP_FILE_CONTENT_FORMAT,
		    name,
		    icon_url ? icon_url : "",
		    app_id,
		    url);

  error = NULL;
  g_file_set_contents (desktop_filename,
		       desktop_file_content,
		       -1,
		       &error);
  if (error != NULL)
    {
      g_critical ("Error generating desktop file: %s", error->message);
      g_error_free (error);
    }

  g_free(desktop_filename);
  g_free(desktop_file_content);
  g_free(app_id);
  g_free(desktop_id);
  g_free(applications_path);
}

