/*
 * Copyright 1999-2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.jxpath.ri;

import java.io.StringReader;

import org.apache.commons.jxpath.JXPathException;
import org.apache.commons.jxpath.ri.parser.ParseException;
import org.apache.commons.jxpath.ri.parser.TokenMgrError;
import org.apache.commons.jxpath.ri.parser.XPathParser;

/**
 * XPath parser
 *
 * @author Dmitri Plotnikov
 * @version $Revision: 1.8 $ $Date: 2004/02/29 14:17:45 $
 */
public class Parser {

    private static XPathParser parser = new XPathParser(new StringReader(""));

    /**
     * Parses the XPath expression. Throws a JXPathException in case
     * of a syntax error.
     */
    public static Object parseExpression(
        String expression,
        Compiler compiler) 
    {
        synchronized (parser) {
            parser.setCompiler(compiler);
            Object expr = null;
            try {
                parser.ReInit(new StringReader(expression));
                expr = parser.parseExpression();
            }
            catch (TokenMgrError e) {
                throw new JXPathException(
                    "Invalid XPath: '"
                        + addEscapes(expression)
                        + "'. Invalid symbol '"
                        + addEscapes(String.valueOf(e.getCharacter()))
                        + "' "
                        + describePosition(expression, e.getPosition()));
            }
            catch (ParseException e) {
                throw new JXPathException(
                    "Invalid XPath: '"
                        + addEscapes(expression)
                        + "'. Syntax error "
                        + describePosition(
                            expression,
                            e.currentToken.beginColumn));
            }
            return expr;
        }
    }

    private static String describePosition(String expression, int position) {
        if (position <= 0) {
            return "at the beginning of the expression";
        }
        else if (position >= expression.length()) {
            return "- expression incomplete";
        }
        else {
            return "after: '"
                + addEscapes(expression.substring(0, position)) + "'";
        }
    }

    private static String addEscapes(String string) {
        // Piggy-back on the code generated by JavaCC
        return TokenMgrError.addEscapes(string);
    }
}