
import os
from ctypes import *
from copy import copy

class XrandrException(Exception):
    pass

# some global datatypes
RRCrtc = c_long
RROutput = c_long
RRMode = c_long
Connection = c_ushort
SubpixelOrder = c_ushort
Time = c_ulong
Rotation = c_ushort
Status = c_int

# FIXME: move those definitions into a private python file
class _XRRModeInfo(Structure):
    " internal C representation "
    _fields_ = [
        ("id", RRMode), # XID is c_long
        ("width", c_int),
        ("height", c_int),
        ("dotClock", c_long),
        ("hSyncStart", c_int),
        ("hSyncEnd", c_int),
        ("hTotal", c_int),
        ("hSkew", c_int),
        ("vSyncStart", c_int),
        ("vSyncEnd", c_int),
        ("vTotal", c_int),
        ("name", c_char_p),
        ("nameLength", c_int),
        ("modeFlags", c_long),
        ]

class _XRRScreenResources(Structure):
    " internal C representation "
    _fields_ = [
        ("timestamp", Time),
        ("configTimestamp", Time),
        ("ncrtc", c_int),
        ("crtcs", POINTER(RRCrtc*100)),
        ("noutput", c_int),
        ("outputs", POINTER(RROutput*100)),
        ("nmode", c_int),
        ("modes", POINTER(_XRRModeInfo*100)),
        ]

class _XRRScreenConfiguration(Structure):
    " private to Xrandr "
    pass

class _XRROutputInfo(Structure):
    " internal C representation "
    _fields_ = [
        ("timestamp", Time),
        ("crtc", RRCrtc),
        ("name", c_char_p),
        ("nameLen", c_int),
        ("mm_width", c_ulong),
        ("mm_height", c_ulong),
        ("connection", Connection),
        ("subpixel_order", SubpixelOrder),
        ("ncrtc", c_int),
        ("crtcs", POINTER(RRCrtc*100)),
        ("nclone", c_int),
        ("clones", POINTER(RROutput*100)),
        ("nmode", c_int),
        ("npreferred", c_int),
        ("modes", POINTER(RRMode*100))
    ]




class ScreenResources(object):
    def __init__(self):
        self.timestamp = 0
        self.configTimestamp = 0
        self.crtcs = []
        self.outputs = []
        self.modes = []
    def __repr__(self):
        return "Timestamp: %s ConfigTimestamp: %s\n" \
               "crtcs: %s\n" \
               "outputs: %s\n" \
               "modes: %s" % (self.timestamp, self.configTimestamp,
                               self.crtcs, self.outputs, self.modes)

class ModeInfo(object):
    def __init__(self):
        self.id = 0
        self.width = 0
        self.height = 0
        self.dotClock = 0
        self.hSyncStart = 0
        self.hSyncEnd = 0
        self.hTotal = 0
        self.hSkew = 0
        self.vSyncStart = 0
        self.vSyncEnd = 0
        self.vTotal = 0
        self.name = ""
        self.modeFlags = 0
    def __repr__(self):
        return "id: %s name: %s width: %s height: %s" % (self.id, self.name, self.width, self.height)

class OutputInfo(object):
    def __init__(self):
        self.timestamp = 0
        self.name = ""
        self.crtc = RRCrtc
        self.mm_width = 0
        self.mm_height = 0
        self.connection = 0
        self.subpixel_order = 0
        self.npreferred = 0
        # modes, clones, crtcs gets initlaized by the client
    def __repr__(self):
        return "name: %s, current_crtc: %s, mm_width: %s, mm_height: %s crtcs: %s, modes: %s, clones: %s" % (self.name, self.crtc, self.mm_width, self.mm_height, self.crtcs, self.modes, self.clones)

class Xrandr(object):
    def __init__(self):
        self.xlib = cdll.LoadLibrary("libX11.so")
        self.rr = cdll.LoadLibrary("libXrandr.so")
        self.dpy = self.xlib.XOpenDisplay(os.environ["DISPLAY"])
        self.default_screen = self.xlib.XDefaultScreen(self.dpy)
        self.rootwin = self.xlib.XDefaultRootWindow(self.dpy, self.default_screen)
        major = c_int()
        minor = c_int()
        # check if we have it
        res = self.rr.XRRQueryExtension(self.dpy, byref(major), byref(minor))
        if not res:
            raise XrandrException
        # check version
        res = self.rr.XRRQueryVersion(self.dpy, byref(major), byref(minor))
        self.version = (major.value, minor.value)

    def getScreenResources(self):
        gsr = self.rr.XRRGetScreenResources
        gsr.restype = POINTER(_XRRScreenResources)
        rp = gsr(self.dpy, self.rootwin)
        # copy the c-structure to something python-ish
        sr = ScreenResources()
        sr.timestamp = rp.contents.timestamp
        sr.configTimestamp = rp.contents.configTimestamp
        # crtcs info
        for i in range(rp.contents.ncrtc):
            sr.crtcs.append(rp.contents.crtcs.contents[i])
        # output info
        goi = self.rr.XRRGetOutputInfo
        goi.restype = POINTER(_XRROutputInfo)
        for i in range(rp.contents.noutput):
            output = OutputInfo()
            xrroutputinfo = goi(self.dpy, rp, rp.contents.outputs.contents[i])
            for (fieldname, var) in _XRROutputInfo._fields_:
                if hasattr(output,fieldname):
                    setattr(output, fieldname, getattr(xrroutputinfo.contents,fieldname))
            for (sizevar, array) in [("ncrtc","crtcs"),
                                     ("nclone","clones"),
                                     ("nmode","modes")]:
                setattr(output, array, [])
                l = getattr(output, array)
                for i in range(getattr(xrroutputinfo.contents,sizevar)):
                    l.append(getattr(xrroutputinfo.contents,array).contents[i])
            sr.outputs.append(output)
        for i in range(rp.contents.nmode):
            mode = ModeInfo()
            for (fieldname, var) in _XRRModeInfo._fields_:
                if hasattr(mode,fieldname):
                    setattr(mode, fieldname, getattr(rp.contents.modes.contents[i],fieldname))
            sr.modes.append(mode)
        return sr

    def getCurrentRefreshRate(self):
        " get the current refresh rate "
        gsi = self.rr.XRRGetScreenInfo
        gsi.restype = POINTER(_XRRScreenConfiguration)
        xrrscreenconfiguration = gsi(self.dpy, self.rootwin)
        self.rr.XRRFreeScreenConfigInfo(xrrscreenconfiguration)
        return self.rr.XRRConfigCurrentRate(xrrscreenconfiguration)


if __name__ == "__main__":
    randr = Xrandr()
    print randr.version
    screen_resources = randr.getScreenResources()
    print screen_resources
    refresh = randr.getCurrentRefreshRate()
    print "Current refresh: ", refresh

