/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <pthread.h>
#include <sys/sysctl.h>
#include <sys/iostat.h>
#include <string.h>

#include "../OpenBSD/utils.h"

pthread_key_t buffer_key;
pthread_key_t buffer_size_key;
pthread_key_t number_disks_key;
int mib[] = { CTL_HW, HW_IOSTATS, sizeof(struct io_sysctl) };

static void create_keys();

int sa_open_disk() {
	pthread_once_t key_once = PTHREAD_ONCE_INIT;
	if (pthread_once(&key_once, create_keys))
		return ENOSYS;

	return 0;
}

int sa_count_disks(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	size_t size;
	if (sysctl(mib, 3, NULL, &size, NULL, 0) < 0)
		return ENOSYS;

	*number = size / sizeof(struct io_sysctl);
	return 0;
}

int sa_reset_disks() {
	size_t size;
	struct io_sysctl* buffer = pthread_getspecific(buffer_key);
	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);

	do {
		if (sysctl(mib, 3, NULL, &size, NULL, 0) < 0)
			return ENOSYS;

		if (buffer_size < size) {
			safe_realloc((void*) &buffer, &size);
			if (pthread_setspecific(buffer_key, buffer))
				return ENOSYS;
			buffer_size = size;
			if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
				return ENOSYS;
			if (buffer == NULL)
				return ENOMEM;
		}

		errno = 0;
		if (sysctl(mib, 3, buffer, &size, NULL, 0) < 0 && errno != ENOMEM)
			return ENOSYS;
	} while (errno == ENOMEM);

	if (pthread_setspecific(number_disks_key, (void *) (size / sizeof(struct io_sysctl))))
		return ENOSYS;

	return 0;
}

int sa_get_disks_ids(char* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	struct io_sysctl* buffer = pthread_getspecific(buffer_key);
	size_t number_disks = (size_t) pthread_getspecific(number_disks_key);

	*written = 0;
	size_t i;
	for (i = 0; i < dst_size; ++i) {
		if (i == number_disks)
			return ENOMEM;

		strlcpy(&dst[i * SA_DISK_NAME], buffer[i].name, IOSTATNAMELEN);
		++(*written);
	}
	return 0;
}

int sa_get_disk(char* name, struct sa_disk* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	struct io_sysctl* buffer = pthread_getspecific(buffer_key);
	size_t number_disks = (size_t) pthread_getspecific(number_disks_key);

	size_t i;
	for (i = 0; i < number_disks; ++i) {
		if (strncmp(buffer[i].name, name, IOSTATNAMELEN) != 0)
			continue;

		strlcpy(dst->name, buffer[i].name, IOSTATNAMELEN);
		dst->reads = buffer[i].rxfer;
		dst->writes = buffer[i].wxfer;
		dst->bytes_read = buffer[i].rbytes;
		dst->bytes_written = buffer[i].wbytes;
		return 0;
	}

	return ENODEV;
}

int sa_get_disks(struct sa_disk* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	struct io_sysctl* buffer = pthread_getspecific(buffer_key);
	size_t number_disks = (size_t) pthread_getspecific(number_disks_key);

	*written = 0;
	size_t i;
	for (i = 0; i < dst_size; ++i) {
		if (i == number_disks)
			return ENOMEM;

		strlcpy(dst[i].name, buffer[i].name, IOSTATNAMELEN);
		dst[i].reads = buffer[i].rxfer;
		dst[i].writes = buffer[i].wxfer;
		dst[i].bytes_read = buffer[i].rbytes;
		dst[i].bytes_written = buffer[i].wbytes;
		++(*written);
	}
	return 0;
}

static void create_keys() {
	static short keys_created = 0;
	if (keys_created) {
		return;
	}

	pthread_key_create(&buffer_key, free_keys);
	pthread_key_create(&buffer_size_key, NULL);
	pthread_key_create(&number_disks_key, NULL);

	keys_created = 1;
}
