/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
 * \defgroup swap Swap devices interface
 * @{
 */
#ifndef SA_SWAP_H_
#define SA_SWAP_H_

/** \struct sa_swap swap.h
 * This structure gathers the details about a swap device. Each value is measured in bytes.
 */
SA_EXPORT struct sa_swap {
#ifdef SA_SWAP_NAME
	char name[SA_SWAP_NAME]; //!< The name of the swap device.
#endif
#ifdef SA_SWAP_TOTAL
	uint64_t total; //!< Total amount of swap in this device.
#endif
#ifdef SA_SWAP_FREE
	uint64_t free; //!< Available amount of swap in this device.
#endif
#ifdef SA_SWAP_TYPE
	int8_t type; //!< The type of swap media. If it's in a device the value is 1. If it's based in a file the value is 2.
#endif
};

#ifdef SA_OPEN_SWAP
/**
 * Prepares the resources needed for retrieving swap statistics. This function exists (and is needed) only when SA_OPEN_SWAP is defined.
 * @return If successful 0 is returned, otherwise an error code is returned.  If the operating system is not supported the return value will be ENOTSUP.
 * @see sa_close_swap()
 */
int sa_open_swap(void) SA_EXPORT;
#endif

#ifdef SA_CLOSE_SWAP
/**
 * This function closes the resources used for retrieving swap statistics. You should call it even when there was a previous error in another function of this API. This function exists (and is needed) only when SA_CLOSE_SWAP is defined.
 * @return If successful 0 is returned, otherwise an error code is returned.
 * @see sa_open_swap()
 */
int sa_close_swap(void) SA_EXPORT;
#endif

/**
 * Gives the total number of swap file systems.
 * You don't need to call sa_reset_swaps() before this function.
 * @param number The number will be stored here
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
int sa_count_swaps(uint16_t* number) SA_EXPORT SA_NONNULL;

/**
 * Refreshes the underlying operating system cache.
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
int sa_reset_swaps() SA_EXPORT;

/**
 * Retrieves statistics from a given swap index.
 * sa_reset_swaps() should be called at least once before this function and everytime you need fresh values.
 * @param index The swap index. It starts from 0.
 * @param dst Where the statistics will be stored.
 * @return If successful 0 is returned, otherwise an error code is returned. ENODEV is returned when the requested swap index was out of range.
 */
int sa_get_swap(uint16_t index, struct sa_swap* dst) SA_EXPORT SA_NONNULL;

/**
 * Retrieves statistics from as many swap fs as possible.
 * sa_reset_swaps() should be called at least once before this function and everytime you need fresh values.
 * @param dst A buffer where the statistics will be stored.
 * @param dst_size The number of swap fs that fits in the dst buffer. If it's not big enough dst will be filled but ENOMEM will be returned.
 * @param written The amount of swap fs statistics written.
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) SA_EXPORT SA_NONNULL;

/*@}*/
#endif /* SA_SWAP_H_ */
