/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.eviction;

import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.util.TestingUtil;
import org.jboss.cache.util.internals.EvictionController;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.EvictionConfig;
import org.jboss.cache.config.EvictionRegionConfig;
import org.jboss.cache.factories.UnitTestCacheConfigurationFactory;
import org.jboss.cache.lock.IsolationLevel;
import static org.testng.AssertJUnit.*;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.util.List;
import java.util.ArrayList;

/**
 * Unit tests for FIFOPolicy.
 *
 * @author Daniel Huang (dhuang@jboss.org)
 * @version $Revision: 5919 $
 */
@Test(groups = {"functional"})
public class FIFOPolicyTest
{
   CacheSPI<Object, Object> cache;
   int wakeupIntervalMillis = 0;
   final String ROOT_STR = "/test";
   Throwable t1_ex, t2_ex;
   final long DURATION = 10000;
   boolean isTrue;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      initCaches();
      wakeupIntervalMillis = cache.getConfiguration().getEvictionConfig().getWakeupIntervalSeconds() * 1000;
      log("wakeupInterval is " + wakeupIntervalMillis);
      if (wakeupIntervalMillis < 0)
      {
         fail("testEviction(): eviction thread wake up interval is illegal " + wakeupIntervalMillis);
      }

      t1_ex = t2_ex = null;
      isTrue = true;
   }

   void initCaches() throws Exception
   {
      Configuration config = UnitTestCacheConfigurationFactory.createConfiguration(Configuration.CacheMode.LOCAL, true);
      EvictionConfig evConfig = config.getEvictionConfig();
      evConfig.setWakeupIntervalSeconds(3);
      evConfig.setDefaultEventQueueSize(20000);
      evConfig.setDefaultEvictionPolicyClass("org.jboss.cache.eviction.FIFOPolicy");
      List<EvictionRegionConfig> erConfigs = new ArrayList<EvictionRegionConfig>();
      erConfigs.add(createEvictionRegionConfig("/_default_",5000));
      erConfigs.add(createEvictionRegionConfig("/org/jboss/test/data",5));
      evConfig.setEvictionRegionConfigs(erConfigs);
      config.setTransactionManagerLookupClass("org.jboss.cache.transaction.DummyTransactionManagerLookup");
      config.setIsolationLevel(IsolationLevel.SERIALIZABLE);
      cache = (CacheSPI<Object, Object>) new DefaultCacheFactory().createCache(config, true);// read in generic local xml
   }


   private EvictionRegionConfig createEvictionRegionConfig(String regionName, int maxNodes)
   {
      EvictionRegionConfig ercDefault = new EvictionRegionConfig();
      ercDefault.setRegionName(regionName);
      FIFOConfiguration esConfig = new FIFOConfiguration();
      if (maxNodes >= 0) esConfig.setMaxNodes(maxNodes);
      ercDefault.setEvictionPolicyConfig(esConfig);
      return ercDefault;
   }


   @AfterMethod(alwaysRun = true)
   public void tearDown() throws Exception
   {
      cache.stop();
   }

   public void testEviction()
   {
      String rootStr = "/org/jboss/test/data/";
      for (int i = 0; i < 10; i++)
      {
         String str = rootStr + i;
         Fqn fqn = Fqn.fromString(str);
         try
         {
            cache.put(fqn, str, str);
         }
         catch (Exception e)
         {
            fail("Failed to insert data" + e);
            e.printStackTrace();
         }
      }
      EvictionController evictionController = new EvictionController(cache);
      evictionController.startEviction();

      try
      {
         String val = (String) cache.get(rootStr + "3", rootStr + "3");
         assertNull("DataNode should be empty ", val);
         assertNull(cache.get(rootStr + "1", rootStr + "1"));
         assertNull(cache.get(rootStr + "2", rootStr + "2"));
         assertNull(cache.get(rootStr + "0", rootStr + "0"));
         assertNull(cache.get(rootStr + "4", rootStr + "4"));

         assertNotNull(cache.get(rootStr + "5", rootStr + "5"));
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Failed to get" + e);
      }
   }

   public void testEviction2() throws Exception
   {
      String rootStr = "/org/jboss/data";
      int maxNodes = 5000;
      for (int i = 0; i < maxNodes * 2; i++)
      {
         String str = rootStr + i;
         Fqn fqn = Fqn.fromString(str);
         cache.put(fqn, str, str);
      }

      EvictionController evictionController = new EvictionController(cache);
      evictionController.startEviction();
      assertEquals("Number of nodes", maxNodes + 2, cache.getNumberOfNodes());
      for (int i = 0; i < maxNodes; i++)
      {
         String n = rootStr + i;
         assertNull("DataNode should be empty " + cache.getNode(n) + " " + n,
               cache.get(n, n));
      }

      for (int i = maxNodes; i < maxNodes * 2; i++)
      {
         assertNotNull(cache.get(rootStr + Integer.toString(i), rootStr + Integer.toString(i)));
      }

      // add another node to cache. this should push node 5000 out of cache.
      cache.put(rootStr + "a", rootStr + "a", rootStr + "a");
      for (int i = maxNodes + 1; i < maxNodes; i++)
      {
         assertNotNull(cache.get(rootStr + Integer.toString(i), rootStr + Integer.toString(i)));
      }

      assertNotNull(cache.get(rootStr + "a", rootStr + "a"));
   }

   public void testNodeVisited()
   {
      String rootStr = "/org/jboss/test/data/";
      for (int i = 0; i < 10; i++)
      {
         String str = rootStr + i;
         Fqn fqn = Fqn.fromString(str);
         try
         {
            cache.put(fqn, str, str);
         }
         catch (Exception e)
         {
            fail("Failed to insert data" + e);
            e.printStackTrace();
         }
      }

      int period = wakeupIntervalMillis + 500;

      log("sleeping for " + period + "ms");
      EvictionController evictionController = new EvictionController(cache);
      evictionController.startEviction();
      try
      {
         for (int i = 0; i < 5; i++)
         {
            String str = rootStr + Integer.toString(i);
            Fqn fqn = Fqn.fromString(str);
            assertNull(cache.get(fqn, str));
         }
         for (int i = 5; i < 10; i++)
         {
            String str = rootStr + Integer.toString(i);
            Fqn fqn = Fqn.fromString(str);
            assertNotNull(cache.get(fqn, str));
         }

         evictionController.startEviction();
         // since it is FIFO if we leave it alone and revisit, cache should remain the same.
         for (int i = 5; i < 10; i++)
         {
            String str = rootStr + Integer.toString(i);
            Fqn fqn = Fqn.fromString(str);
            cache.get(fqn, str);// just to keep it fresh
            System.out.println("-- sleeping for " + period + "ms");
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Failed to evict" + e);
      }
   }

   public void testNodeRemoved()
   {
      String rootStr = "/org/jboss/test";
      for (int i = 0; i < 10; i++)
      {
         String str = rootStr + i + "/" + i;
         Fqn fqn = Fqn.fromString(str);
         String str2 = rootStr + i;
         Fqn fqn2 = Fqn.fromString(str2);
         try
         {
            cache.put(fqn, str, str);
            cache.put(fqn2, str2, str2);
         }
         catch (Exception e)
         {
            fail("Failed to insert data" + e);
            e.printStackTrace();
         }
      }

      EvictionController evictionController = new EvictionController(cache);
      evictionController.startEviction();

      String str1 = rootStr + "7";
      Fqn fqn1 = Fqn.fromString(str1);
      String str2 = rootStr + "7/7";
      Fqn fqn2 = Fqn.fromString(str2);
      try
      {
         assertNotNull(cache.get(fqn2, str2));
         assertNotNull(cache.get(fqn1, str1));
         cache.removeNode(fqn2);
         evictionController.startEviction();
         assertNull(cache.get(fqn2, str2));
         assertNotNull(cache.get(fqn1, str1));
         cache.removeNode(fqn1);
         evictionController.startEviction();
         assertNull(cache.get(fqn1, str1));
         assertNull(cache.get(fqn2, str2));

         String str3 = rootStr + "5/5";
         String str4 = rootStr + "5";
         Fqn fqn3 = Fqn.fromString(str3);
         Fqn fqn4 = Fqn.fromString(str4);
         assertNotNull(cache.get(fqn3, str3));
         assertNotNull(cache.get(fqn4, str4));

         evictionController.startEviction();

         // remove the node above fqn4 /org/jboss/test/5 will cascade the delete into /org/jboss/test/5/5
         cache.removeNode(fqn4);

         evictionController.startEviction();
         assertNull(cache.get(fqn3, str3));
         assertNull(cache.get(fqn4, str4));
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Failed to evict" + e);
      }
   }


   class MyPutter extends Thread
   {

      public MyPutter(String name)
      {
         super(name);
      }

      public void run()
      {
         int i = 0;
         final String myName = ROOT_STR + "/test1/node" + getName();
         while (isTrue)
         {
            try
            {
               cache.put(myName + i++, "value", i);
               sleep(1);
            }
            catch (Throwable e)
            {
               e.printStackTrace();
               if (t1_ex == null)
               {
                  t1_ex = e;
               }
            }
         }
      }
   }


   public void testConcurrentPutAndEvict() throws Exception
   {
      cache.stop();
      cache.destroy();
      cache.getConfiguration().setIsolationLevel(IsolationLevel.REPEATABLE_READ);
      cache.start();
      cache.put(ROOT_STR + "/concurrentPutAndEvict", "value", 1);

      for (int i = 0; i < 10; i++)
      {
         new MyPutter("Putter" + i).start();
      }

      int counter = 0;
      while (true)
      {
         counter++;
         if (t1_ex != null)
         {
            fail("Exception generated in put() " + t1_ex);
         }
         log("nodes/locks: " + cache.getNumberOfNodes() + "/" + cache.getNumberOfLocksHeld());
         TestingUtil.sleepThread(1000);
         if (counter > 10)
         {// run for 10 seconds
            isTrue = false;
            break;
         }
      }
   }

   void log(String msg)
   {
      System.out.println("-- " + msg);
   }

}
