/** Copyright  2003 by Jean-Hugues de Raigniac <jhraigniac@workingfrog.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.workingfrog.i18n.swing;

import javax.swing.Icon;
import javax.swing.JLabel;

import org.workingfrog.i18n.util.LocaleEvent;
import org.workingfrog.i18n.util.LocaleListener;
import org.workingfrog.i18n.util.Translator;

/**
 * i18n version of javax.swing.JLabel
 *
 * @author Jean-Hugues de Raigniac
 */
public class I18NJLabel extends JLabel implements LocaleListener {

    /** i18n key */
    private String i18nKey = null;

    /**
     * Override javax.swing.JLabel.JLabel ().
     */
    public I18NJLabel () {
        this("", null, LEADING);
    }

    /**
     * Override javax.swing.JLabel.JLabel (String).
     *
     * @param i18nKey i18n bundle key
     */
    public I18NJLabel (String i18nKey) {
        this(i18nKey, null, LEADING);
    }

    /**
     * Override javax.swing.JLabel.JLabel (javax.swing.Icon).
     *
     * @param icon icon
     */
    public I18NJLabel (Icon icon) {
        this(null, icon, CENTER);
    }

    /**
     * Override javax.swing.JLabel.JLabel (javax.swing.Icon, int).
     *
     * @param icon icon
     * @param horizontalAlignment a constant defined in SwingConstants
     */
    public I18NJLabel (Icon icon, int horizontalAlignment) {
        this(null, icon, horizontalAlignment);
    }

    /**
     * Override
     * javax.swing.JLabel.JLabel (java.lang.String, javax.swing.Icon, int).
     *
     * @param i18nKey i18n bundle key
     * @param icon icon
     * @param horizontalAlignment a constant defined in SwingConstants
     */
    public I18NJLabel (String i18nKey, Icon icon, int horizontalAlignment) {
        super(null, icon, horizontalAlignment);

        setText(i18nKey);
    }

    /**
     * Override javax.swing.JLabel.JLabel (java.lang.String, int).
     *
     * @param i18nKey i18n bundle key
     * @param horizontalAlignment a constant defined in SwingConstants
     */
    public I18NJLabel (String i18nKey, int horizontalAlignment) {
        this(i18nKey, null, horizontalAlignment);
    }

    /**
     * Override javax.swing.JLabel.setText(java.lang.String).
     *
     * @param i18nKey i18n bundle key
     */
    public void setText (String i18nKey) {
        if (this.i18nKey == null || this.i18nKey.equals("")) {
            this.i18nKey = i18nKey;
            Translator.checkKey(i18nKey, this);
        }

        if (i18nKey != null && ! i18nKey.equals("")) {
            super.setText(Translator.checkValue(i18nKey, this));
        } else {
            super.setText("");
        }
    }

    /**
     * Update the GUI with a new translation of i18nKey.
     *
     * @param event contains the new Locale
     */
    public void localeChanged (LocaleEvent event) {
        setText(i18nKey);
    }
}
