// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "cvisual.h"
#include <stdexcept>
#include <exception>

#include <boost/python/exception_translator.hpp>
#include <boost/python/to_python_converter.hpp>
#include <boost/python/module.hpp>
#include <boost/python/def.hpp>
#include <boost/python/class.hpp>
#include <boost/python/list.hpp>
#include <boost/python/tuple.hpp>
#include <boost/python/numeric.hpp>
#include <boost/python/extract.hpp>

#include "num_util.h"

#include "axial.h"
#include "arrow.h"
#include "exceptions.h"
#include "cone.h"
#include "sphere.h"
#include "cylinder.h"
#include "ring.h"
#include "box.h"
#include "rate.h"
#include "pyramid.h"
#include "ellipsoid.h"

// Forward declarations for initialization functions defined elsewhere
namespace visual {
	void vector_init_type();
	void vector_array_init_type();
	void scalar_array_init_type();
	void display_init_type();
	void ui_object_init_type();
	void primitive_traits_init_type();	
	void tmatrix_init_type();
	void frame_init_type();
	void curve_init_type();
	void label_init_type();
	void convex_init_type();
	void faces_init_type();
}

// Exception translators
namespace visual {
	namespace py = boost::python;
	
void 
translate_std_out_of_range( std::out_of_range e)
{
	PyErr_SetString( PyExc_IndexError, e.what());
}

void
translate_std_invalid_argument( std::invalid_argument e)
{
	PyErr_SetString( PyExc_ValueError, e.what());
}

void
translate_std_runtime_error( std::runtime_error e)
{
	PyErr_SetString( PyExc_RuntimeError, e.what());
}


struct rgb_from_seq
{
	rgb_from_seq()
	{
		py::converter::registry::push_back( 
			&convertible,
			&construct,
			py::type_id<rgb>());
	}
	
	static void* convertible( PyObject* obj)
	{
		using py::handle;
		using py::allow_null;
		if (PyInt_Check(obj) || PyFloat_Check(obj))
			return obj;
		
		handle<> obj_iter( allow_null( PyObject_GetIter(obj)));
		if (!obj_iter.get()) {
			PyErr_Clear();
			return 0;
		}
		int obj_size = PyObject_Length(obj);
		if (obj_size < 0) {
			PyErr_Clear();
			return 0;
		}
		if (obj_size != 3)
			return 0;
		return obj;
	}
	
	static void construct( 
		PyObject* _obj, 
		py::converter::rvalue_from_python_stage1_data* data)
	{
		py::object obj = py::object(py::handle<>(py::borrowed(_obj)));
		void* storage = (
			(py::converter::rvalue_from_python_storage<rgb>*)
			data)->storage.bytes;
		if (PyFloat_Check(_obj) || PyInt_Check(_obj)) {
			new (storage) rgb( py::extract<float>(obj));
			data->convertible = storage;
			return;
		}
		new (storage) rgb( 
			py::extract<float>(obj[0]), 
			py::extract<float>(obj[1]),
			py::extract<float>(obj[2]));
		data->convertible = storage;
	}
};

struct rgb_to_tuple
{
	static PyObject* convert( const rgb& color)
	{
		py::tuple ret = py::make_tuple( color.r, color.g, color.b);
		Py_INCREF(ret.ptr());
		return ret.ptr();
	}
};

} // !namespace visual

using namespace boost::python;
using namespace visual;

/* Entry point function for the module.  This function either directly or indirectly
 * provides all of the exports to python.  Remember that parents must be exported before
 * children, even if defined elsewhere.  Compiling this function with all of the exports
 * directly consumes a *lot* of memory, especially for optimizing builds,
 * therefore, most exports are provided in their own files.
 */
BOOST_PYTHON_MODULE( cvisual)
{	
	// Platform-specific initialization defined in plat{linux,win}.cpp
	init_platform();
	
	// Initialize the Python thread system
	PyEval_InitThreads();
	
	// Private functions for initializing and choosing the numeric backend
	def( "_init_numeric_impl", init_numeric_impl);
	def( "_init_numarray_impl", init_numarray_impl);
	def( "_use_numeric_impl", use_numeric_impl);
	def( "_use_numarray_impl", use_numarray_impl);
	
	// Some compiler-specific initialization
#if __GNUG__
#if __GNUC__ == 3
#if __GNUCMINOR__ >= 1 && __GNUCMINOR__ < 4
	std::set_terminate( __gnu_cxx::__verbose_terminate_handler);
#endif
#endif
#endif
	
	// A subset of the python standard exceptions may be thrown from visual
	register_exception_translator<visual::python::Exception>( 
		&visual::python::translate_exception);
	register_exception_translator<std::out_of_range>( 
		&visual::translate_std_out_of_range);
	register_exception_translator<std::invalid_argument>( 
		&visual::translate_std_invalid_argument);
	register_exception_translator<std::runtime_error>( 
		&visual::translate_std_runtime_error);
	
	// Register to- and from-python converters for the rgb type.
	rgb_from_seq();	
	to_python_converter< rgb, rgb_to_tuple>();	
	
	// Defined in vector.cpp
	vector_init_type();
	
	// Defined in vector_array.cpp
	vector_array_init_type();
	
	// Defined in scalar_array.cpp
	scalar_array_init_type();
	
	// Free functions for exiting the system.  These are undocumented at the moment.
	def( "shutdown", Display::shutdown, "Close all Displays and shutdown visual.");
	def( "allclosed", Display::allclosed, "Returns true if all of the Displays are closed.");
	def( "waitclose", Display::waitclose, "Blocks until all of the Displays are closed by the user.");

	def( "rate", rate_timer::py_rate, "Pause animation execution for 1/arg seconds.");

	// Defined in display.cpp
	display_init_type();
	
	// Defined in mouseobject.cpp
	ui_object_init_type();

	const char* py_complete_init_docstring =
		"Register the new object with the visual internals.\n"
		"Client code should not call this function.";

	class_< DisplayObject, boost::shared_ptr<DisplayObject>, boost::noncopyable>( "displayobject", no_init)
		.add_property( "display", &DisplayObject::py_get_display, &DisplayObject::set_display)
		.add_property( "frame", &DisplayObject::py_get_parent, &DisplayObject::setParent)
		.add_property( "visible", &DisplayObject::get_visible, &DisplayObject::set_visible)
		.def( "_complete_init", &DisplayObject::py_complete_init, py_complete_init_docstring)
		;
	
	// Defined in prim.cpp
	primitive_init_type();
	
	class_< axialSymmetry, bases<Primitive> , boost::noncopyable>( "axial_symmetry", no_init)
		.add_property( "length", &axialSymmetry::get_length, &axialSymmetry::set_length)
		.add_property( "radius", &axialSymmetry::get_radius, &axialSymmetry::set_radius)
		;


	// Technically, this inherits from axialSymmetry.  We inherit from Primitive to prevent having
	// to deal with the nullification of the "radius" property.
	class_< visual::arrow, bases<Primitive>, boost::shared_ptr<visual::arrow> >("arrow")
		.def( init<const visual::arrow&>())
		.add_property( "length", &axialSymmetry::get_length, &axialSymmetry::set_length)
		.add_property( "shaftwidth", &arrow::get_shaftwidth, &arrow::set_shaftwidth)
		.add_property( "headlength", &arrow::get_headlength, &arrow::set_headlength)
		.add_property( "headwidth", &arrow::get_headwidth, &arrow::set_headwidth)
		.add_property( "fixedwidth", &arrow::get_fixedwidth, &arrow::set_fixedwidth)
		;

	class_< sphere, bases<Primitive>, boost::shared_ptr<sphere> >( "sphere")
		.def( init<const sphere&>())
		.add_property( "radius", &sphere::get_radius, &sphere::set_radius)
		;

	class_< cylinder, bases<axialSymmetry>, boost::shared_ptr<cylinder> >( "cylinder")
		.def( init<const cylinder&>())
		;
	
	class_< cone, bases<axialSymmetry>, boost::shared_ptr<cone> >( "cone")
		.def( init<const cone&>())
		;
		
	class_< ring, bases<Primitive>, boost::shared_ptr<ring> >( "ring")
		.def( init<const ring&>())
		.add_property( "thickness", &ring::get_thickness, &ring::set_thickness)
		.add_property( "radius", &ring::get_radius, &ring::set_radius)
		;

	class_< box, bases<Primitive>, boost::shared_ptr<box> >( "box")
		.def( init<const box&>())
		.add_property( "width", &box::get_width, &box::set_width)
		.add_property( "height", &box::get_height, &box::set_height)
		.add_property( "length", &box::get_length, &box::set_length)
		.add_property( "size", &box::get_size, &box::set_size)
		;
	
	class_< pyramid, bases<Primitive>, boost::shared_ptr<pyramid> >( "pyramid")
		.def( init<const pyramid&>())
		.add_property( "width", &pyramid::get_width, &pyramid::set_width)
		.add_property( "height", &pyramid::get_height, &pyramid::set_height)
		.add_property( "length", &pyramid::get_length, &pyramid::set_length)
		.add_property( "size", &pyramid::get_size, &pyramid::set_size)
		;
	
	class_< ellipsoid, bases<Primitive>, boost::shared_ptr<ellipsoid> >( "ellipsoid")
		.def( init<const ellipsoid&>())
		.add_property( "width", &ellipsoid::get_width, &ellipsoid::set_width)
		.add_property( "height", &ellipsoid::get_height, &ellipsoid::set_height)
		.add_property( "length", &ellipsoid::get_length, &ellipsoid::set_length)
		.add_property( "size", &ellipsoid::get_size, &ellipsoid::set_size)
		;
	
	// Defined in label.cpp
	label_init_type();
	
	// Defined in curve.cpp		
	curve_init_type();

	// Defined in convex.cpp
	convex_init_type();
	
	// Defined in faceset.cpp
	faces_init_type();
	
	// Defined in frame.cpp
	frame_init_type();
}
