/* Calculate coefficients of finite difference formulas using
   algorithm by Fornberg, 1998.
   Reference: http://epubs.siam.org/sam-bin/dbq/article/32250

   Author: Pearu Peterson, September 2002
   Compilation:
     c++ genpfdf.cc -o genpfdf -lcln -lginac
   or
     c++ `ginac-config --cppflags --libs` genpfdf.cc -o genpfdf
   Usage:
     ./genpfdf n [m] > output_file.c
   where
     n -- max. number of grid intervals.
     m -- max. order of derivative [default n].
*/

#include <iostream>
#include <string>
#include <ginac/ginac.h>

using namespace std;
using namespace GiNaC;

static int gen_apply_periodic_fd(void);
static int gen_apply_periodic_fd2(void);
static int gen_fd_coeffs(const numeric &,const numeric &,const numeric &);

int main(int argc, const char *const *argv)
{
    numeric n,m;
    switch (argc) {
    case 2:
      n = numeric(argv[1]);
      m = n;
      break;
    case 3:
      n = numeric(argv[1]);
      m = numeric(argv[2]);
      break;
    default:
      cout << "Usage:" << endl;
      cout << "    " << *argv << " n [m [s]]" << endl;
      cout << "  where" << endl;
      cout << "  n -- max. number of grid intervals" << endl;
      cout << "  m -- max. order of derivative [default n]" << endl;
      return 1;
    }
    cout <<"/*****************************************************/"<<endl;
    cout <<"/******** Periodic Finite Difference Formulae ********/"<<endl;
    cout <<"/*****************************************************/"<<endl;
    cout <<"/* Author: Pearu Peterson, September 2002            */"<<endl;
    cout <<"/*                                                   */"<<endl;
    cout <<"/* This file is automatically generated using        */"<<endl;
    cout <<"/* genpfdf.cc utility from fdf package and it        */"<<endl;
    cout <<"/* contains the following C function                 */"<<endl;
    cout <<"/*                                                   */"<<endl;
    cout <<"/*  void periodic_finite_difference(int n,           */"<<endl;
    cout <<"/*    double *x, double *y, double h, int k, int m); */"<<endl;
    cout <<"/* where                                             */"<<endl;
    cout <<"/*  n -- length of arrays x and y                    */"<<endl;
    cout <<"/*  x -- input array                                 */"<<endl;
    cout <<"/*  y -- output array                                */"<<endl;
    cout <<"/*  h -- discretization step of arrays x and y       */"<<endl;
    cout <<"/*  k -- the order of derivative                     */"<<endl;
    cout <<"/*  m -- number of grid intervals used to evaluate   */"<<endl;
    cout <<"/*       finite differences at the center point. The */"<<endl;
    cout <<"/*       error is O(h^(2*(m-1))) within numerical    */"<<endl;
    cout <<"/*       accuracy.                                   */"<<endl;
    cout <<"/*****************************************************/"<<endl;
    cout <<"/* Reference:                                        */"<<endl;
    cout <<"/*   http://epubs.siam.org/sam-bin/dbq/article/32250 */"<<endl;
    cout <<"/*****************************************************/"<<endl;

    cout <<"/* Parameters to genpfdf n="<<n<<" m="<<m<<" */"<<endl;
    cout <<"#include <stdio.h>"<<endl;
    cout <<"#include <math.h>"<<endl;
    for (int i=1;i<=n;++i)
      for (int j=1;j<=m && j<=i;++j)
	gen_fd_coeffs(numeric(i),numeric(j),numeric(i)/2);
    gen_apply_periodic_fd();
    gen_apply_periodic_fd2();
    cout << "extern void periodic_finite_difference("<<endl
	 <<"int n, double *x, double *y, double h, int k, int m) {"
	 << endl <<"double coeff = pow(1.0/h,k), *c=NULL;"
	 << endl <<"  switch (k*1000+m) {"<<endl;
    for (int i=1;i<=n;++i)
      for (int j=1;j<=m&&j<=i;++j)
	cout <<"  case "<<j*1000+i
	     <<": c=fd_coeffs_"<<i<<"_"<<j<<"; break;"<<endl;
    cout << "  default:"<<endl
	 <<"    fprintf(stderr,"<<endl
	 <<"     \"Unsupported derivative/order combination: k,m=%d,%d\\n\","
	 <<endl<<"      k,m);"<<endl<<"  }"
	 <<endl<<"  if (c!=NULL) {"
	 <<endl<<"   if (k%2)"
	 <<endl<<"    apply_periodic_fd(n,x,y,coeff,m,c);"
	 <<endl<<"   else"
	 <<endl<<"    apply_periodic_fd2(n,x,y,coeff,m,c);"
	 <<endl<<"  }"
	 <<endl<<"}"<<endl;
    return 0;
}

static
int gen_fd_coeffs(const numeric &n,const numeric &m,const numeric &s) {
  symbol x("x");
  ex poly = pow(x,s)*pow(log(x),m);
  if (n.is_odd())
    poly = poly * pow(numeric(2),-m);
  //else
  //  poly = poly;
  poly = expand(series_to_poly(poly.series(x==1,n.to_int()+1)));
  Digits = 16;
  //cout << " /* poly=" << poly << " */"<<endl;
  cout <<"static double fd_coeffs_"<<n.to_int()<<"_"<<m.to_int()<<"[] = {"<<endl;
  if (n.is_even() && m.is_even())
    cout <<" "<< ex_to<numeric>(poly.coeff(x,n.to_int()/2).evalf())
	 <<","
      //<< " /* " << poly.coeff(x,n.to_int()/2) << " */"
	 <<endl;
  for (int i=n.to_int()/2+1,j=1; i<=n; ++i,++j) {
    cout << " "
	 << ex_to<numeric>(poly.coeff(x,i).evalf());
    if (i<n)
      cout <<",";
    //cout<< " /* " << poly.coeff(x,i) << " */";
    cout<<endl;
  }
  cout<<"};"<<endl;
}

static int gen_apply_periodic_fd(void) {
  cout <<"static void apply_periodic_fd"
       <<"(int n,double *x,double *y,double coeff,int m,double *c) {"<<endl
       <<" int i,k,l=(m+1)/2;"<<endl
       <<" if (m%2) {"<<endl
       <<"  for(i=m;i<n-m;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=m;k>0;k-=2)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]-x[i-k]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<"  for(i=0;i<m;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=m;k>0;k-=2)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]-x[(i+n-k)%n]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<"  for(i=n-m;i<n;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=m;k>0;k-=2)"<<endl
       <<"    d += (*(--c_ptr)) * (x[(i+k)%n]-x[i-k]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<" } else {"<<endl
       <<"  for(i=l;i<n-l;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=l;k>0;--k)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]-x[i-k]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<"  for(i=0;i<l;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=l;k>0;--k)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]-x[(i+n-k)%n]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<"  for(i=n-l;i<n;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=l;k>0;--k)"<<endl
       <<"    d += (*(--c_ptr)) * (x[(i+k)%n]-x[i-k]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<" }"<<endl
       <<"}"<<endl;
}

static int gen_apply_periodic_fd2(void) {
  cout <<"static void apply_periodic_fd2"
       <<"(int n,double *x,double *y,double coeff,int m,double *c) {"<<endl
       <<" int i,k,l=(m+1)/2;"<<endl
       <<" if (m%2) {"<<endl
       <<"  for(i=m;i<n-m;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=m;k>0;k-=2)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]+x[i-k]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<"  for(i=0;i<m;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=m;k>0;k-=2)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]+x[(i+n-k)%n]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<"  for(i=n-m;i<n;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l;"<<endl
       <<"   for(k=m;k>0;k-=2)"<<endl
       <<"    d += (*(--c_ptr)) * (x[(i+k)%n]+x[i-k]);"<<endl
       <<"   y[i] = coeff * d;"<<endl
       <<"  }"<<endl
       <<" } else {"<<endl
       <<"  for(i=l;i<n-l;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l + 1;"<<endl
       <<"   for(k=l;k>0;--k)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]+x[i-k]);"<<endl
       <<"   y[i] = coeff * (d+(*c)*x[i]);"<<endl
       <<"  }"<<endl
       <<"  for(i=0;i<l;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l + 1;"<<endl
       <<"   for(k=l;k>0;--k)"<<endl
       <<"    d += (*(--c_ptr)) * (x[i+k]+x[(i+n-k)%n]);"<<endl
       <<"   y[i] = coeff * (d+(*c)*x[i]);"<<endl
       <<"  }"<<endl
       <<"  for(i=n-l;i<n;++i) {"<<endl
       <<"   double d = 0.0, *c_ptr = c + l + 1;"<<endl
       <<"   for(k=l;k>0;--k)"<<endl
       <<"    d += (*(--c_ptr)) * (x[(i+k)%n]+x[i-k]);"<<endl
       <<"   y[i] = coeff * (d+(*c)*x[i]);"<<endl
       <<"  }"<<endl
       <<" }"<<endl
       <<"}"<<endl;
}
