"""
oofun (OpenOpt Function) usage for setting funcs <-> variables dependencies
to economy cputime during calculations
for example during finite-difference derivatives approximation

see also examples/oofun/*.py files for other oofun benefits

Requires openopt v > 0.18 (latest tarball or svn is ok).

Suppose we have NL problem
n = 150
x0 = ones(n)
f(x) = (x0-0) ^ 2 + (x1-1)^2 + ... + (x[n-1]-(n-1)) ^ 2 -> min
subjected to some non-lin constraints:
c0(x) = x0 ^ 4 + x1 ^ 4 <= 1
c1(x) = x1 ^ 4 + x2 ^ 4 <= 1
c2(x) = x2^10 + x3 ^ 10 <= 1

Classic openopt style assumes you provide either all derivatives (for all c[i]) or none.
Suppose we know derivatives of f and c0:
dc0 = lambda x: hstack((array((4*x[0]^3, 4*x[1]^3)), zeros(n-2)))
but we don't know derivatives of c1 and c2 (here could be much more complex expressions).
Still we know for sure c2 depends on x2 and x3 only.
Let's take benefits using openopt oofun
"""

from numpy import arange, ones, asarray, zeros, array, hstack
from scikits.openopt import NLP, oofun

n= 150

x0 = ones(n)
f = lambda x: ((x - arange(n)) ** 2).sum()
df = lambda x: 2 * (x-arange(n))

# derivative of oofun can be assigned via named argument "d"
c0 = oofun(lambda x: x[0] ** 4 + x[1] ** 4 - 1, d = lambda x:hstack((4*x[0]**3, 4*x[1]**3, zeros(n-2))))

# alternatively you can assign it directly:
#c0 = oofun(lambda x: x[0] ** 2 + x[1] ** 2 - 1)
#c0.d = ...

# Note that oofun can consist of several funcs (i.e. yield several output numbers).
# If you provide d for the case, it should calculate derivatives for all funcs from oofun.
# If you provide dep, it assumes all functions from the oofun depend on the indexes from dep.
# see also /tests/aggregate.py for another example related to oofun

# since we don't know anything about c1 except of function definition,
# it doesn't matter will we pass it as oofun:
# c1 = oofun(lambda x: x[1]**4 + x[2] ** 4 - 1)
# or as an ordinary function:
c1 = lambda x: x[1]**4 + x[2] ** 4 - 1

# now let's take benefits of c2 pattern.
# Getting 1st derivatives via finite-difference will omit unused x indexes.
c2 = oofun(lambda x: x[2]**10 + x[3] ** 10 - 1, dep = [2, 3])
# dep is "depends on indexes"
# of course you can use direct assignment:
# c2.dep = [2,3]

#now let's solve the NL problem:

p = NLP(f,  x0,  df=df, c=[c0, c1, c2], maxIter = 1e4, maxFunEvals = 1e10, iprint = 0)
r = p.solve('ralg')
# iter    objFunVal    log10(maxResidual)
#    0  1.092e+06               0.00
# 2800  5.426e+00            -100.00
#istop:  4 (|| F[k] - F[k-1] || < ftol)
#Solver:   Time Elapsed = 11.13     CPU Time Elapsed = 10.4
#objFunValue: 5.4258992 (feasible, max constraint =  0)


