# $Id$
#
# message.py -- Screen mixin to messages.
#
#    Copyright (C) 2001  Mike Meyer <mwm@mired.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"Message - a mixin to provide message on plwm screens."

import string
from types import StringType
from Xlib import X, Xutil
import event

class Message:
    "Holds and displays the message."

    def __init__(my, screen, fontname, draw_function,
                 foreground, background, bordercolor, borderwidth, seconds):
        "Initialize the message window, gc and font."

        my.wm = screen.wm
        my.seconds = seconds

        fg = screen.get_color(foreground)
        bg = screen.get_color(background)
        bc = screen.get_color(bordercolor)

        window = screen.root.create_window(0, 0, 1, 1, borderwidth,
                                           X.CopyFromParent, X.InputOutput,
                                           X.CopyFromParent,
                                           background_pixel = bg,
                                           border_pixel = bc,
                                           event_mask = (X.ExposureMask |
                                                         X.VisibilityChangeMask))

        my.font = my.wm.get_font(fontname, 'fixed')

        my.gc = window.create_gc(font = my.font, function = draw_function,
                                 foreground = fg, background = bg)
        my.window = screen.add_internal_window(window)
        my.window.dispatch.add_handler(X.VisibilityNotify, my.raisewindow)
        my.window.dispatch.add_handler(X.Expose, my.redraw)

    def setup(my, labels, align = 'left'):
        "Create the window."

        if type(labels) == StringType:
            labels = [labels]

        fontsize = my.font.query()
        high = (fontsize.font_ascent + fontsize.font_descent + 1)
        my.height = len(labels) * high

        width = 0
        lines = []
        for l in range(len(labels)):
            line = Line(labels[l], my.font,
                        l * high + fontsize.font_ascent + 1)
            w = line.width
            if w > width: width = w
            lines.append(line)
        my.width = width + 4

        for i in range(len(lines)):
            lines[i].setup(my.window.window, my.gc, my.width, align, i)
        my.lines = lines

        return my.width, my.height

    def display(my, x, y, timeout = None):
        "Start it up"

        x, y, width, height = my.window.keep_on_screen(x, y, my.width, my.height)
        my.window.configure(x = x, y = y, width = width, height = height)
        my.window.map()

        my.timeout = timeout or my.seconds * len(my.lines)

        if my.timeout:
            timer_id = event.new_event_type()
            my.timer = event.TimerEvent(timer_id, after = my.timeout)
            my.wm.events.add_timer(my.timer)
            my.wm.dispatch.add_handler(timer_id, my.close, handlerid = my)

    def close(my, event = None):
        my.window.destroy()
        if my.timeout:
            my.timer.cancel()

    def raisewindow(my, event = None):
        my.window.raisewindow()

    def redraw(my, event = None):
        "Redraw the window, with highlights"

        my.window.clear_area(width = my.width, height = my.height)
        for i in range(len(my.lines)):
            my.lines[i].redraw()


class Line:
    "A class for lines in a message."

    def __init__(my, label, font, where):
        "Figure out where to draw this string."

        my.name = label
        ext = font.query_text_extents(label)
        my.width = ext.overall_width
        my.y = where

    def setup(my, window, gc, width, align, count):
        "Save the drawing position."

        my.window, my.gc = window, gc
        if align == 'left':
            my.x = 0
        elif align == 'center':
            my.x = (width - my.width) / 2
        else:	# right
            my.x = width - my.width

    def redraw(my):
        "Draw myself."

        my.window.image_text(my.gc, my.x, my.y, my.name)

class screenMessage:
    """PLWM Screen mixin to provide messages..

    This mixin requires the color and font mixins be in the screen class."""

    message_font = "9x15Bold"
    message_foreground = "black"
    message_background = "white"
    message_bordercolor = "black"
    message_borderwidth = 3
    message_seconds = 5
    message_draw = X.GXset

    def message_make(my, labels, align = 'center'):
        """Create a message from the labels.

        Returns the width and height of the resulting menu."""

        my.message = Message(my, my.message_font, my.message_draw,
                             my.message_foreground, my.message_background,
                             my.message_bordercolor, my.message_borderwidth,
                             my.message_seconds)
        return my.message.setup(labels, align)

    def message_display(my, x, y):
        "Instantiate the menu, and return the label or None."

        my.message.display(x, y)
