# Copyright (c) 2006 Bea Lam. All rights reserved.
# 
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, sublicense, and/or sell copies of the Software,
# and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

# Mac-specific utility functions and constants.

import Foundation
import AppKit
import objc
import time
from PyObjCTools.AppHelper import NSDefaultRunLoopMode

import _IOBluetooth
import _lightbluecommon

# values of constants used in _IOBluetooth.framework 
kIOReturnSuccess = 0       # defined in <IOKit/IOReturn.h>
kIOBluetoothUserNotificationChannelDirectionIncoming = 1
        # defined in <IOBluetooth/IOBluetoothUserLib.h>
kBluetoothHCIErrorPageTimeout = 0x04   # <IOBluetooth/Bluetooth.h>

# defined in <IOBluetooth/IOBluetoothUserLib.h>
kIOBluetoothServiceBrowserControllerOptionsNone = 0L


# IOBluetoothSDPUUID objects for RFCOMM and OBEX protocol UUIDs
PROTO_UUIDS = {
    _lightbluecommon.RFCOMM: _IOBluetooth.IOBluetoothSDPUUID.uuid16_(0x0003),
    _lightbluecommon.OBEX: _IOBluetooth.IOBluetoothSDPUUID.uuid16_(0x0008)
}


def formatdevaddr(addr):
    """
    Returns address of a device in usual form e.g. "00:00:00:00:00:00"
    
    - addr: address as returned by device.getAddressString() on an
      IOBluetoothDevice
    """
    # make uppercase cos PyS60 & Linux seem to always return uppercase 
    # addresses
    # can safely encode to ascii cos BT addresses are only in hex (pyobjc
    # returns all strings in unicode)
    return addr.replace("-", ":").encode('ascii').upper()
    

def btaddrtochars(addr):   
    """
    Takes a bluetooth address and returns a tuple with the corresponding 
    char values. This can then be used to construct a 
    IOBluetoothDevice object, providing the signature of the withAddress: 
    selector has been set (as in _setpyobjcsignatures() in this module).
    
    For example:
        >>> chars = btaddrtochars("00:0e:0a:00:a2:00")
        >>> chars
        (0, 14, 10, 0, 162, 0)
        >>> device = _IOBluetooth.IOBluetoothDevice.withAddress_(chars)
        >>> type(device)
        <objective-c class IOBluetoothDevice at 0xa4024988>
        >>> device.getAddressString()
        u'00-0e-0a-00-a2-00'
    """
    if not _lightbluecommon._isbtaddr(addr):
        raise TypeError("address %s not valid bluetooth address" % str(addr))
    if addr.find(":") == -1:
        addr = addr.replace("-", ":")   # consider alternative addr separator
    
    # unhexlify gives binary value like '\x0e', then ord to get the char value.
    # unhexlify throws TypeError if value is not a hex pair.
    import binascii
    chars = [ord(binascii.unhexlify(part)) for part in addr.split(":")]
    return tuple(chars)

def looponce():
    app = AppKit.NSApplication.sharedApplication() 

    # to push the run loops I seem to have to do this twice
    # use NSEventTrackingRunLoopMode or NSDefaultRunLoopMode?
    for i in range(2):
        event = app.nextEventMatchingMask_untilDate_inMode_dequeue_(
            AppKit.NSAnyEventMask, Foundation.NSDate.dateWithTimeIntervalSinceNow_(0.02),
            NSDefaultRunLoopMode, False)

    # Note: if there are issues with sockets it might be worth changing the
    # time interval above. Sometimes recv() didn't get all the data if there was
    # a large amount of data (e.g. several hundred kilobytes) in the buffer,
    # and increasing the interval from 0.01 to 0.02 stopped the problem. 
    # So maybe there are problems if the run loop is driven too often?


def waitwhile(conditionfunc, timeout=None):
    """
    conditionfunc: callable
    timeout: float or int (timeout in seconds)
    
    Pauses the main thread while the given callback returns True.
    (Somehow it doesn't actually block the main thread -- i.e. the main thread
    waits, but asynchronous operations can still be performed while it's 
    waiting here -- and all without multithreading! I've tried it with 
    multithreading but it didn't work consistently, and of course it's bad 
    anyway since IOBluetooth doesn't like multithreading.)
    
    If timeout is given, returns False if this times out -- i.e. if this is
    still waiting (i.e. conditionfunc still returns True) after the given 
    length of time.
    
    Always returns True if no timeout is given.
    Always returns False if timeout=0.
    """
    app = AppKit.NSApplication.sharedApplication() 
     
    if timeout is None:
        while conditionfunc():   
            looponce()
        return True
    else:
        if not isinstance(timeout, (int, float)):
            raise TypeError(
                "timeout must be int or float, was %s" % type(timeout))
    
        starttime = time.time()
        timedout = lambda: time.time() - starttime > timeout
        while conditionfunc() and not timedout():
            looponce()
        if timedout():
            return False
        return True
        