/***********************************************************************************
* Run Command: Plasmoid to run commands without terminal.
* Copyright (C) 2008 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "plasma-runcommand.h"

#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <QRegExp>
#include <QFileInfo>
#include <QDir>
#include <QScriptEngine>

#include <plasma/theme.h>

#include <kworkspace/kworkspace.h>
#include <kworkspace/kdisplaymanager.h>
#include <kstandarddirs.h>
#include <KCompletion>
#include <KConfigDialog>
#include <KGlobalSettings>
#include <KService>
#include <KServiceTypeTrader>
#include <KShell>
#include <KToolInvocation>
#include <KUrl>
#include <KRun>
#include <KProtocolInfo>
#include <KLocale>
#include <KAuthorized>

RunCommandComboBox::RunCommandComboBox(QGraphicsWidget *parent) : QGraphicsProxyWidget(parent)
{
    comboBox = new KHistoryComboBox(false);

    lineEdit = new KLineEdit(comboBox);

    comboBox->setLineEdit(lineEdit);
    comboBox->setAttribute(Qt::WA_NoSystemBackground);

    setWidget(comboBox);
}


RunCommand::RunCommand(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args)
{
    setHasConfigurationInterface(true);
}

void RunCommand::init()
{
    QString command;
    KConfigGroup configuration = config();
    int width = configuration.readEntry("width", 200);

    setGlobalShortcut(KShortcut("Ctrl+Shift+R"));

    m_field = new RunCommandComboBox(this);
    m_field->lineEdit->setClearButtonShown(true);
    m_field->lineEdit->clear();

    m_completion = new KCompletion();
    m_completion->setOrder(KCompletion::Sorted);

    setMinimumWidth((formFactor() == Plasma::Horizontal)?width:100);

    if (formFactor() != Plasma::Horizontal)
    {
        setMaximumHeight(m_field->comboBox->height());
    }

    m_destroyAction = new KAction(KIcon("edit-delete"), i18n("Remove"), this);
    m_destroyAction->setEnabled(immutability() == Plasma::Mutable);

    resize(width, m_field->comboBox->height());

    setWidth(width);

    setHistory();

    connect(this, SIGNAL(activate()), this, SLOT(activated()));
    connect(m_field->lineEdit, SIGNAL(returnPressed()), this, SLOT(getCommand()));
    connect(m_field->comboBox, SIGNAL(cleared()), this, SLOT(clearHistory()));
    connect(m_field->comboBox, SIGNAL(activated(int)), this, SLOT(setActive(int)));
    connect(m_field->comboBox, SIGNAL(aboutToShowContextMenu(QMenu*)), this, SLOT(addContextMenuItems(QMenu*)));
    connect(m_field->comboBox, SIGNAL(editTextChanged(QString)), this, SLOT(textChanged(QString)));
    connect(m_destroyAction, SIGNAL(triggered()), this, SLOT(destroy()));
}

void RunCommand::createConfigurationInterface(KConfigDialog *parent)
{
    KConfigGroup configuration = config();
    QWidget *widget = new QWidget();

    m_ui.setupUi(widget);
    m_ui.width->setValue(configuration.readEntry("width", 200));

    configuration = globalConfig();

    m_ui.historyAmount->setValue(configuration.readEntry("historyAmount", 50));

    parent->setButtons(KDialog::Ok | KDialog::Cancel | KDialog::Apply);
    parent->addPage(widget, parent->windowTitle(), icon());

    connect(m_ui.width, SIGNAL(valueChanged(int)), this, SLOT(setWidth(int)));
    connect(parent, SIGNAL(applyClicked()), this, SLOT(configAccepted()));
    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
    connect(parent, SIGNAL(finished()), this, SLOT(configClosed()));
}

void RunCommand::configAccepted()
{
    KConfigGroup configuration = config();

    configuration.writeEntry("width", m_ui.width->value());

    emit configNeedsSaving();

    configuration = globalConfig();

    configuration.writeEntry("historyAmount", m_ui.historyAmount->value());

    emit configNeedsSaving();

    setHistory();
}

void RunCommand::configClosed()
{
    KConfigGroup configuration = config();

    setWidth(configuration.readEntry("width", 200));
}

void RunCommand::constraintsEvent(Plasma::Constraints constraints)
{
    setBackgroundHints(NoBackground);

    if (constraints & Plasma::SizeConstraint && formFactor() != Plasma::Horizontal)
    {
        KConfigGroup configuration = config();

        configuration.writeEntry("width", contentsRect().width());

        m_field->comboBox->resize((contentsRect().width() - ((formFactor() == Plasma::Horizontal)?5:0)), m_field->comboBox->height());

        emit configNeedsSaving();
    }

    if (constraints & Plasma::ImmutableConstraint)
    {
        m_destroyAction->setEnabled(immutability() == Plasma::Mutable);
    }
}

void RunCommand::activated()
{
    raise();

    m_field->lineEdit->setFocus(Qt::OtherFocusReason);
}

void RunCommand::addContextMenuItems(QMenu* menu)
{
    menu->addSeparator();
    menu->addAction(KIcon("configure"), i18n("Settings"), this, SLOT(showConfigurationInterface()));
    menu->addAction(m_destroyAction);
}

void RunCommand::setWidth(int width)
{
    if (formFactor() == Plasma::Horizontal)
    {
        setMinimumWidth(width);
    }

    resize(width, m_field->comboBox->height());

    m_field->comboBox->resize((contentsRect().width() - ((formFactor() == Plasma::Horizontal)?5:0)), m_field->comboBox->height());
}

void RunCommand::setActive(int index)
{
    Q_UNUSED(index);

    activated();
}

void RunCommand::setHistory()
{
    KConfigGroup configuration = globalConfig();

    m_commands = configuration.readEntry("commands", QStringList());

    m_completion->clear();
    m_completion->insertItems(m_commands);

    m_field->comboBox->setHistoryItems(m_commands);

    m_field->lineEdit->setCompletionObject(m_completion);
    m_field->lineEdit->setCompletionMode(KGlobalSettings::CompletionAuto);
}

void RunCommand::clearHistory()
{
    KConfigGroup configuration = globalConfig();

    configuration.deleteEntry("commands");

    emit configNeedsSaving();

    setHistory();
}

void RunCommand::textChanged(QString text)
{
    Q_UNUSED(text);

    m_field->comboBox->setStyleSheet("QLineEdit, QComboBox {background: auto;}");
}

void RunCommand::getCommand()
{
    if (m_field->lineEdit->text().isEmpty())
    {
        return;
    }

    runCommand(m_field->lineEdit->text());
}

void RunCommand::runCommand(QString command)
{
    KDisplayManager displayManager;
    KConfigGroup configuration = globalConfig();
    KConfig kuriconfig("kuriikwsfilterrc", KConfig::NoGlobals);
    KConfigGroup generalgroup(&kuriconfig, "General");
    QString delimiter = generalgroup.readPathEntry("KeywordDelimiter", QString(":"));
    QString path = QDir::cleanPath(KShell::tildeExpand(command));
    QFileInfo *location = new QFileInfo(path);
    QRegExp webAddress("^((ht|f)tps?://|www\\.).+");
    int history = configuration.readEntry("historyAmount", 50);
    int index;
    bool commandStatus = true;
    bool addToHistory = true;

    if (command.length() > 3 && command.contains(delimiter))
    {
        index = command.indexOf(delimiter);

        if (index != (command.length() - 1))
        {
            KService::List offers = KServiceTypeTrader::self()->query("SearchProvider", QString("'%1' in Keys").arg(command.left(index)));

            if (!offers.isEmpty())
            {
                QString query = offers.at(0)->property("Query").toString();

                command = query.replace("\\{@}", command.right(command.length() - index - 1));
            }
        }
    }

    KUrl url(command);

    if (command.startsWith("=") || command.startsWith("bin=") || command.startsWith("oct=") || command.startsWith("dec=") || command.startsWith("hex="))
    {
        int base = 10;
        QString result;

        addToHistory = false;

        if (!command.startsWith("="))
        {
            if (command.startsWith("bin="))
            {
                base = 2;
            }
            else if (command.startsWith("oct="))
            {
                base = 8;
            }
            else if (command.startsWith("hex="))
            {
                base = 16;
            }

            command.remove(0, 3);
        }

        command.remove(0, 1).replace(" ", "");

        if (command.contains(KGlobal::locale()->decimalSymbol(), Qt::CaseInsensitive))
        {
            command.replace(KGlobal::locale()->decimalSymbol(), ".", Qt::CaseInsensitive);
        }

        if (command.contains("0x"))
        {
            int position = 0;
            QString hex;

            while ((position = command.indexOf("0x")) > -1)
            {
                hex.clear();

                for (int i = (position + 2); i < command.size(); ++i)
                {
                    if (((command[i] <= '9') && (command[i] >= '0')) || ((command[i] <= 'f') && (command[i] >= 'a')) || ((command[i] <= 'F') && (command[i] >= 'A')))
                    {
                        hex.append(command[i]);
                    }
                    else
                    {
                        break;
                    }
                }

                command.replace("0x" + hex, QString::number(hex.toInt(NULL, 16)));
            }
        }

        command.replace(QRegExp("([a-zA-Z]+)"), "Math.\\1");

        QScriptEngine engine;
        QScriptValue value = engine.evaluate(command);

        if (!value.isError())
        {
            result = ((base == 16)?"0x":"") + QString::number(value.toString().toInt(), base).toUpper();

            m_field->lineEdit->setText(result);
        }
        else
        {
            commandStatus = false;
        }
    }
    else if (location->exists())
    {
        command = path;

        new KRun(path, NULL);
    }
    else if ((url.protocol() != "file" && KProtocolInfo::isKnownProtocol(url.protocol())) || webAddress.exactMatch(command))
    {
        if (url.protocol().isEmpty())
        {
            index = command.indexOf('/');

            url.clear();
            url.setHost(command.left(index));

            if (index != -1)
            {
                url.setPath(command.mid(index));
            }

            url.setProtocol("http");

            command.prepend("http://");
        }

        KToolInvocation::invokeBrowser(url.url());
    }
    else if (command.startsWith("mailto:"))
    {
        KToolInvocation::invokeMailer(command.remove(0, 7), NULL);
    }
    else if (command == "logout")
    {
        KWorkSpace::requestShutDown(KWorkSpace::ShutdownConfirmDefault, KWorkSpace::ShutdownTypeNone);
    }
    else if (command == "shutdown")
    {
        KWorkSpace::requestShutDown(KWorkSpace::ShutdownConfirmDefault, KWorkSpace::ShutdownTypeHalt);
    }
    else if (command == "restart" || command == "reboot")
    {
        KWorkSpace::requestShutDown(KWorkSpace::ShutdownConfirmDefault, KWorkSpace::ShutdownTypeReboot);
    }
    else if (command == "switch" && KAuthorized::authorizeKAction("start_new_session") && displayManager.isSwitchable() && displayManager.numReserve() > 0)
    {
        displayManager.startReserve();
    }
    else if (command == "lock")
    {
        KRun::runCommand("dbus-send --print-reply --dest=org.freedesktop.ScreenSaver /ScreenSaver org.freedesktop.ScreenSaver.Lock", NULL);
    }
    else
    {
        if (command.startsWith("$"))
        {
            KToolInvocation::invokeTerminal(command.remove(0, 1));
        }
        else
        {
            QString binaryName = KRun::binaryName(command, true);

            if (!QFile(binaryName).exists() && KStandardDirs::findExe(binaryName).isEmpty())
            {
                commandStatus = false;
            }
            else
            {
                commandStatus = KRun::runCommand(command, NULL);
            }
        }
    }

    if (commandStatus)
    {
        if (addToHistory)
        {
            m_commands = configuration.readEntry("commands", m_commands);

            m_field->lineEdit->clear();

            if (m_commands.contains(command))
            {
                m_commands.removeAt(m_commands.indexOf(command));
            }

            m_commands.prepend(command);

            while (m_commands.count() > history && !m_commands.isEmpty())
            {
                m_commands.removeLast();
            }

            configuration.writeEntry("commands", m_commands);

            emit configNeedsSaving();

            setHistory();
        }
    }
    else
    {
        m_field->comboBox->setStyleSheet("QLineEdit, QComboBox {background: pink;}");
    }
}

#include "plasma-runcommand.moc"
