#ifndef WORKSHEET_H
#define WORKSHEET_H

#include <qwidget.h>
#include <qpainter.h>
#include <qstring.h>
#include <qfont.h>
#include <qcolor.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <kprinter.h>
#include "MainWin.h"
#include "Plot.h"
#include "Label.h"
#include "Line.h"
#include "Rect.h"
#include "Ellipse.h"
#include "LImage.h"
#include "widgettype.h"
#include "scale.h"

#ifdef HAVE_UNDO
extern "C" {
#include "undo.h"
}
#endif

#define NR_PLOTS 16
#define NR_OBJECTS 100

class MainWin;
class Plot;
class Plot2DSimple;
class Plot2DSurface;
class Plot3D;
class PlotQWT3D;
class PlotPie;
class PlotPolar;
class PlotTernary;

class Worksheet: public QWidget
{
	Q_OBJECT
// this leads to loops on KDE 3.4
//	Q_PROPERTY(QString title READ Title WRITE setTitle)
	Q_PROPERTY(bool title_enabled READ TitleEnabled WRITE setTitleEnabled)
	Q_PROPERTY(bool timestamp_enabled READ TimeStampEnabled WRITE setTimeStampEnabled)
	Q_PROPERTY(QDateTime timestamp READ TimeStamp WRITE setTimeStamp)
	Q_PROPERTY(QColor background READ Background WRITE setBackground)
private:
	MainWin *mw;					// reference to the main window (for plotlist & messages)
	WidgetType widgettype;
	Plot *plot[NR_PLOTS];			// list of plots
	KPrinter *printer;				// printer for printing
	QPixmap *pixmap;				// pixmap for drawing
	QString title;					//!< title of Worksheet
	bool title_enabled;				//!< title enabled
	QDateTime timestamp;			//!< time stamp
	bool timestamp_enabled;		//!< timestamp enabled
	QColor background;
	
	Label *label[NR_OBJECTS];
	Line *line[NR_OBJECTS];
	Rect *rect[NR_OBJECTS];
	Ellipse *ellipse[NR_OBJECTS];
	LImage *image[NR_OBJECTS];
	
	int X, Y;						// window dimension
	int lx, ly;						// temp. legend position
	int ax, ay, anumber;				// temp. axis position and axis number
	int tx, ty;						// temp. title position
	int bx, by;						// temp. border position
	int tlx, tly;						// temp. tic label position
	int tmagx, tmagy,tmagx2,tmagy2;	// temp. maglens position
	int tmaskx, tmasky,tmaskx2,tmasky2;	// temp. mask position
	double tpanx, tpany;			// temp. pan zoom value (real data point)
	int tdatamodex;				// temp. x value for datamode
	int linesx,linesy,lineex,lineey,labelx,labely,rectx,recty;
	int ellipsex,ellipsey,imagex,imagey;
	bool moving_cornerF1, moving_cornerF2;
	bool moving_cornerB1, moving_cornerB2;
	bool moving_borderx1, moving_borderx2;
	bool moving_bordery1, moving_bordery2;
	bool moving_center;
	int object_index;	// index of moving object
	int tmp_object_index;	// index when defining object
	unsigned int api, nr_plots;	// active plot index and number of plots
public:
	Worksheet(QWidget *parent, MainWin *mw, const char *name);
	void closeNow();
	WidgetType getWidgetType() { return widgettype; }
	QStringList Info();
	MainWin *getMainWin() { return mw; }	// for project dialog
	Plot *getPlot(int i) { return plot[i]; }	//!< get plot number i
	void setPlot(int i,Plot *p) { plot[i] = p; }	//!< set plot number i
#ifdef HAVE_UNDO
	UNDO *getUndo();
	void setUndo(UNDO *u);
#endif
	void updatePixmap();
	void New();
	void save(QTextStream *t);
	void open(QTextStream *t,int version, PType oldtype);
	QDomElement saveXML(QDomDocument doc);
	void openXML(QDomNode node);

	// TODO : add to which plot ? -> check this
	void addGraph2D(Graph2D *g,PType type=P2D);
	void addGraph3D(Graph3D *g,PType type=P3D);
	void addGraphM(GraphM *g,PType type=PSURFACE);
	void addGraph4D(Graph4D *g);
	void addGraphIMAGE(GraphIMAGE *g);
	void addGraphL(GraphL *g,PType type=PPIE);
	
	void resetRanges();
	void checkRanges(TScale scale, double *min, double *max);
	double partRanges(TScale scale, double min, double max);
public slots:
	void Export(QString filename, QString format);								//!< export to filename with format
	void ExportPS(QString filename);											//!< export to filename as PS
	void ExportEPS(QString filename);											//!< export to filename as EPS
	void ExportPDF(QString filename);											//!< export to filename as PDF
	void ExportSVG(QString filename);											//!< export to filename as SVG
	void ExportPIC(QString filename);											//!< export to filename as PIC
	void Print(QString filename);												//!< print to filename

	// needed wrapper for QSA
	Plot2DSimple* get2DPlot(int i);  				//!< get 2d simple plot number i
	Plot2DSimple* get2DSimplePlot(int i) { return get2DPlot(i);}  	//!< get 2d simple plot number i
	Plot2DSurface* get2DSurfacePlot(int i);  			//!< get 2d surface plot number i
	Plot3D* get3DPlot(int i);  					//!< get 3d plot number i
	PlotQWT3D* getQWT3DPlot(int i);  				//!< get 3d qwt plot number i
	PlotPie* getPiePlot(int i);  					//!< get pie plot number i
	PlotPolar* getPolarPlot(int i);  				//!< get polar plot number i
	PlotTernary* getTernaryPlot(int i);  				//!< get ternary plot number i

	Plot* newPlot(PType type=P2D, bool insert=true);	// TODO, insert : insert new plot ?
	Plot* clonePlot(int item, bool insert=true);	//!< clone plot item, insert new plot?
	void deletePlot(int i);				//!< delete plot number i
	void deleteActivePlot();				// TODO
	
	int getX() { return X; }			//!< get window dimensions
	int getY() { return Y; }			//!< get window dimensions
	void redraw() {updatePixmap();}			//!< repaint the worksheet
	int API() { return api; }			//!< get active plot number
	unsigned int NrPlots() { return nr_plots; }	//!< get number of plots
	void setNrPlots(unsigned int i) { nr_plots=i; }	//!< set number of plots
	
	int NrObjects() { return NR_OBJECTS; }		//!< get number of drawing objects
	Line* getLine(int i) { return line[i]; }	//!< get line number i
	void setLine(int i, Line *l) { line[i]=l; }	//!< set line number i to l
	Label* getLabel(int i) { return label[i]; }	//!< get label number i
	void setLabel(int i, Label *l) { label[i]=l; }	//!< set label number i to l
	Rect* getRect(int i) { return rect[i]; }		//!< get rect number i
	void setRect(int i, Rect *r) { rect[i]=r; }	//!< set rect number i to rect
	Ellipse* getEllipse(int i) { return ellipse[i]; }	//!< get ellipse number i
	void setEllipse(int i, Ellipse* e) { ellipse[i]=e; }	//!< set ellipse i to e
	LImage* getImage(int i) { return image[i]; }		//!< get image number i
	void setImage(int i, LImage* im) { image[i]=im; }	//!< set image number i to im

	QString Title() const { return title; }										//!< get title
	void setTitle(QString t) { title = t.remove(QChar('&')); setCaption(title); updatePixmap(); }				//!< set title to t
	bool TitleEnabled() const { return title_enabled; }							//!< title enabled ?
	void enableTitle(bool e=true) { title_enabled = e; updatePixmap(); }			//!< enable title	
	void setTitleEnabled(bool e=true) { enableTitle(e); }
	QDateTime TimeStamp() const { return timestamp; }								//!< get timestamp 
	void setTimeStamp(QDateTime ts) { timestamp = ts; updatePixmap(); }			//!< set timestamp
	bool TimeStampEnabled() const { return timestamp_enabled; }				//!< timestamp enabled?
	void enableTimeStamp(bool e=true) { timestamp_enabled = e;updatePixmap(); }	//!< enable timestamp
	void setTimeStampEnabled(bool e=true) { enableTimeStamp(e); }
	QColor Background() const { return background; }							//!< get background color
	void setBackground(QColor c) { background = c; updatePixmap(); }				//!< set background color to c
	void setBackground(QString c) { setBackground(QColor(c)); }				//!< set background color to c
protected:
	void resizeEvent( QResizeEvent * );
	void paintEvent( QPaintEvent * );
	void mousePressEvent( QMouseEvent *);
	void mouseReleaseEvent( QMouseEvent *);
	void mouseDoubleClickEvent( QMouseEvent * );
	void mouseMoveEvent( QMouseEvent * );
	void keyPressEvent(QKeyEvent *);
	void closeEvent(QCloseEvent *);
	void Draw(QPainter *,int w, int h);
	void setupPrinter(KPrinter *pr, QString fn);
	double XCoordinate(double x, double xmin, double xmax);
	double YCoordinate(double y, double ymin, double ymax);
};

#endif //WORKSHEET
