/***************************************************************************
 *   Copyright (C) 2004 by Matthias Reif                                   *
 *   matthias.reif@informatik.tu-chemnitz.de                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "ktxpowergroup.h"

KTxPowerGroup::KTxPowerGroup( QString title, QWidget *parent, const char *name ) : QButtonGroup( title, parent, name )
{
	setInsideMargin( 25 );

	txPowerOffButton = new QRadioButton( i18n( "off" ),  this );
	txPowerAutoButton = new QRadioButton( i18n( "auto" ), this );
	txPowerFixedButton = new QRadioButton( i18n( "fixed" ), this );
	
	txPowerOffButton->adjustSize();
	txPowerAutoButton->adjustSize();
	txPowerFixedButton->adjustSize();

	txPowerOffButton->move( KWaveControl::SPACING, 25 );
	txPowerAutoButton->move( txPowerOffButton->x() + txPowerFixedButton->width() + KWaveControl::SPACING, txPowerOffButton->y() );
	txPowerFixedButton->move( txPowerAutoButton->x() + txPowerFixedButton->width() + KWaveControl::SPACING, txPowerAutoButton->y() );
	powerSelection = new KComboBox( this );
	powerSelection->move( txPowerOffButton->x(), txPowerOffButton->y() + txPowerOffButton->height() + KWaveControl::SPACING );

	powerSelection->adjustSize();
	adjustSize();

	connect( this, SIGNAL( clicked( int ) ), this, SLOT( changedSlot() ) );
	connect( powerSelection, SIGNAL( activated( int ) ), this, SLOT( changedSlot() ) );
}


KTxPowerGroup::~KTxPowerGroup()
{
}


/*!
    \fn KTxPowerGroup::insertTxPowers( WCard* card )
 */
bool KTxPowerGroup::insertTxPowers( WCard* card )
{
	powerSelection->clear();
	txPowers.clear();
	if( card->has.txPower )
	{
		if( ( card->range->num_txpower <= 0 ) || ( card->range->num_txpower > IW_MAX_TXPOWER ) )
		{
			return false;
		}
		int i;
		QString dbm, mw;
		TxPower txPower;
		for( i = 0; i < card->range->num_txpower; i++ )
		{
			if( card->range->txpower_capa & IW_TXPOW_RELATIVE )
			{
				powerSelection->insertItem( QString::number( (int)card->range->txpower[ i ] ) + " (" + i18n( "no unit" ) + ")" );
				txPower.unit = IW_TXPOW_RELATIVE;
			}
			else
			{
				
				if( card->range->txpower_capa & IW_TXPOW_MWATT )
				{
					dbm = QString::number( MW_TO_DBM(card->range->txpower[ i ]) );
					mw = QString::number( (int)card->range->txpower[ i ] );
					txPower.unit = IW_TXPOW_MWATT;
				}
				else
				{
					dbm = QString::number( (int)card->range->txpower[ i ] );
					mw = QString::number( DBM_TO_MW(card->range->txpower[ i ]) );
					txPower.unit = IW_TXPOW_DBM;
				}
				powerSelection->insertItem( dbm + " dBm (" + mw + "mW)" );
			}
			txPower.value = card->range->txpower[ i ];
			txPowers.push_back( txPower );
		}
	}
	else
	{
		return false;
	}
	return true;
}


/*!
    \fn KTxPowerGroup::setTxPower( bool disabled, bool fixed, int value, int unit )
 */
void KTxPowerGroup::setTxPower( bool disabled, bool fixed, int value, int unit )
{
	if( disabled )
	{
		txPowerOffButton->setChecked( true );
		txPowerFixedButton->setChecked( false );
		txPowerAutoButton->setChecked( false );
	}
	else
	{
		txPowerOffButton->setChecked( false );
		txPowerFixedButton->setChecked( fixed );
		txPowerAutoButton->setChecked( !fixed );
	}
	unsigned int i;
	for( i = 0; i < txPowers.size(); i++ )
	{
		if( txPowers[ i ].value == value )
			break;
	}
	if( i < txPowers.size() )
	{
		powerSelection->setCurrentItem( i );
	}
	else
	{
		TxPower txPower;
		if( unit == IW_TXPOW_RELATIVE )
		{
			powerSelection->insertItem( QString::number( value ) + " (" + i18n( "no unit" ) + ")" );
			txPower.unit = IW_TXPOW_RELATIVE;
		}
		else
		{
			QString dbm, mw;
			if(unit == IW_TXPOW_MWATT)
			{
				dbm = QString::number( MW_TO_DBM(value) );
				mw = QString::number( value );
				txPower.unit = IW_TXPOW_MWATT;
			}
			else
			{
				dbm = QString::number( value );
				mw = QString::number( DBM_TO_MW(value) );
				txPower.unit = IW_TXPOW_DBM;
			}
			powerSelection->insertItem( dbm + " dBm (" + mw + " mW)" );
		}
		txPower.value = value;
		txPowers.push_back( txPower );
	}
}


/*!
    \fn KTxPowerGroup::current()
 */
TxPower KTxPowerGroup::current()
{
	if( txPowers.size() > 0  )
		return txPowers[ powerSelection->currentItem() ];
	else
	{
		TxPower power;
		power.value = 100;
		power.unit = 0;
		return power;
	}
}

/*!
    \fn KTxPowerGroup::changedSlot()
 */
void KTxPowerGroup::changedSlot()
{
	TxPower power = current();
	changed( txPowerOffButton->isOn(), txPowerFixedButton->isOn(), power.value, power.unit );
}


/*!
    \fn KTxPowerGroup::clear()
 */
void KTxPowerGroup::clear()
{
	txPowerOffButton->setChecked( false );
	txPowerFixedButton->setChecked( false );
	txPowerAutoButton->setChecked( false );
	powerSelection->clear();
}


/*!
    \fn KTxPowerGroup::resizeEvent( QResizeEvent * event )
 */
void KTxPowerGroup::resizeEvent( QResizeEvent * event )
{
	QButtonGroup::resizeEvent( event );
	powerSelection->resize( this->width() - 2 * KWaveControl::SPACING, powerSelection->height() );
}


#include "ktxpowergroup.moc"
