/***************************************************************************
                          ktray.cpp  -  description
                             -------------------
    begin                : Die Nov 12 2002
    copyright            : (C) 2002 by Matthias Reif
    email                : matthias.reif@informatik.tu-chemnitz.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ktray.h"

#include "kwaveq0.xpm"
#include "kwaveq1.xpm"
#include "kwaveq2.xpm"
#include "kwaveq3.xpm"
#include "kwaveq4.xpm"
#include "kwaveq5.xpm"
#include "kwaveq6.xpm"
#include "kwaveq7.xpm"
#include "kwaveq8.xpm"
#include "kwaveq9.xpm"
#include "kwaveqx.xpm"
#include "hi16-app-kwavecontrol.xpm"

/*!
	\fn KTray::KTray( QString cmd, QWidget* parent, const char *name)
	@param cmd path and file of this executable, so argv[ 0 ]
	@param parent parent widget
	@param name name
*/
KTray::KTray( QString cmd, QWidget* parent, const char *name) : KSystemTray(parent, name)
{
	kwc = new KWaveControl( this );

	basicTimer = new QTimer( this );
	secondTimer = new QTimer( this );
	
	KGlobal::config()->setGroup("KWaveControl");
	
	// check if we have a device option from command line args
	KCmdLineArgs* args = KCmdLineArgs::parsedArgs();
	if( args->isSet( "device" ) )
	{
		// create card interface with device from command line arg
		card = new WCard( args->getOption( "device" ).data() );
	}
	else
	{
		// create card interface with device from last session
		card = new WCard( KGlobal::config()->readEntry("device", "").data() );
	}
	
	// the different tray icons for the differnet link states
	icons[0] = new QPixmap( kwaveq0_xpm );
	icons[1] = new QPixmap( kwaveq1_xpm );
	icons[2] = new QPixmap( kwaveq2_xpm );
	icons[3] = new QPixmap( kwaveq3_xpm );
	icons[4] = new QPixmap( kwaveq4_xpm );
	icons[5] = new QPixmap( kwaveq5_xpm );
	icons[6] = new QPixmap( kwaveq6_xpm );
	icons[7] = new QPixmap( kwaveq7_xpm );
	icons[8] = new QPixmap( kwaveq8_xpm );
	icons[9] = new QPixmap( kwaveq9_xpm );
	icons[10] = icons[9];
	icons[ NO_LINK_STATE ] = new QPixmap( kwaveqx_xpm ); // NO_LINK_STATE = 11
	
	widget = new WWidget( card, cmd );
	widget->setIcon( hi16_app_kwavecontrol_xpm );
	connect( widget, SIGNAL( pollingChanged( double ) ), this, SLOT( slotPolling( double ) ) );
	widget->hide();
	
	popup = contextMenu();
	
	popup->changeTitle( popup->idAt( 0 ), hi16_app_kwavecontrol_xpm, "KWaveControl" );
	
	KIconLoader iconLoader;
	
	openAction = new QAction(this, "open");
	openAction->setText( i18n ("Open") );
	openAction->setIconSet( iconLoader.loadIcon( "configure", KIcon::Small ) );
	openAction->addTo(popup);
	connect(openAction, SIGNAL(activated()), this, SLOT(slotOpen()));
	
	closeAction = new QAction(this, "close");
	closeAction->setText( i18n ("Close") );
	closeAction->setIconSet( iconLoader.loadIcon( "fileclose", KIcon::Small ) );
	closeAction->addTo(popup);
	connect(closeAction, SIGNAL(activated()), this, SLOT(slotClose()));
	
	smallAction = new QAction(this, "small");
	smallAction->setText( i18n ("Only Link") );
	smallAction->setIconSet( iconLoader.loadIcon( "info", KIcon::Small ) );
	smallAction->setToggleAction(true);
	smallAction->addTo(popup);
	connect(smallAction, SIGNAL(activated()), this, SLOT(slotSmall()));
	
	connect(basicTimer, SIGNAL(timeout()), this, SLOT(slotAct()));
	connect(basicTimer, SIGNAL(timeout()), widget, SLOT(basicTimerSlot()));
	connect(secondTimer, SIGNAL(timeout()), widget, SLOT(secondTimerSlot()));
	
	basicTimer->start(500, false);
	secondTimer->start(1000, false);
	
	// invoke a state update
	state = -1;
	setState( 0 );
	
	smallWidget = new SmallWidget( 0, "SmallWidget", WStyle_NoBorder | WStyle_StaysOnTop | WX11BypassWM );
	KGlobal::config()->setGroup( "small_widget" );
	smallWidget->setShown( KGlobal::config()->readBoolEntry( "shown", false ) );
	
	int weVersion = WCard::getWeVersion();
	if( ( weVersion > 0 ) && ( weVersion < 18 ) )
	{
		KMessageBox::information( widget, "You are using the wireless exensions version " + QString::number( WIRELESS_EXT ) + ".\nFor full capability, you should update to a newer version.\nhttp://www.hpl.hp.com/personal/Jean_Tourrilhes/Linux/Tools.html#wext", "old wireless extensions", "old_wireless_extensions" );
	}
}

/*!
	\fn KTray::~KTray()
	just clean up a little bit
*/
KTray::~KTray()
{
	int i;
	for( i=0; i<11; i++ )
		delete icons[i];
	delete kwc;
	delete widget;
	delete smallWidget;
	delete basicTimer;
	delete secondTimer;
	delete openAction;
	delete closeAction;
	delete smallAction;
	delete popup;
	delete card;
}

/*!
	\fn KTray::setState( int value )
	update state variable and tray icon if state changed
	@param value new state
*/
void KTray::setState( int value )
{
	if( value != state )
	{
		state = value;
		setPixmap( *icons[value] );
	}
}

/*!
	\fn KTray::slotAct()
	update stats, state and tooltip
*/
void KTray::slotAct()
{
	// reset
	link = -1;
	int newState = state;
	toolTipText = "";

	// update
	if( card->isWave() )
	{
		smallAction->setEnabled( true );
		if( card->has.stats )
		{
			card->getLinkQual( link, avg, signal, noise, maxQual, maxSignal, maxNoise );

			if( card->has.range && ( signal != 0 ) )
			{
				if( maxQual == 0)
				{
					toolTipText = QString::number( link ) + "/" + QString::number( maxQual );
					link = 0;
					newState = 0;
				}
				else
				{
					link = (100*link) / maxQual;
					toolTipText = QString::number( link ) + "%";
					if( link > 0 )
						newState = link / 10;
					else
						newState = NO_LINK_STATE;
				}
			}
			else
			{
				toolTipText = QString::number( link );
				newState = 0;
			}
		}
		
		int bitrate = card->getBitRate( );
		if( bitrate >= 0 )
		{
			if( ! toolTipText.isEmpty() )
			{
				toolTipText += "\n";
			}
			toolTipText += WWidget::bitrate2String( bitrate );
		}
		QToolTip::add( this, toolTipText );
		setState( newState );
	}
	else
	{
		setState( NO_LINK_STATE );
		smallAction->setEnabled( false );
	}


	// if "only link widget" visible, update it too
	if( smallWidget->isVisible() )
	{
		smallWidget->act( link );
		smallAction->setOn( true );
	}
	else
	{
		smallAction->setOn( false );
	}
}


/*!
	\fn KTray::slotOpen()
	show main widget
*/
void KTray::slotOpen()
{
	widget->setPlainCaption( kwc->getRandomCaption() );
	widget->show();
	widget->setFocus();
}


/*!
	\fn KTray::slotClose()
	hide main widget
*/
void KTray::slotClose()
{
	widget->hide();
}


/*!
	\fn KTray::mousePressEvent( QMouseEvent *ev )
	open/close main widget or show popup
*/
void KTray::mousePressEvent( QMouseEvent *ev )
{
	if( ev->button() == LeftButton )
	{
		if(widget->isVisible())
		{
			slotClose();
		}
		else
		{
			// compute good pos for main widget
			QPoint p;
			p.setX( ev->globalX() - ( widget->width() / 2 ) );
			p.setY( ev->globalY() - ( widget->height() + 50 ) );
			widget->move( p );
			slotOpen();
		}
	}
	if( ev->button() == RightButton )
		popup->exec( ev->globalPos() );
}


/*!
	\fn KTray::slotPolling( double value )
	restart the basic timer
	@param value new polling time in seconds
*/
void KTray::slotPolling( double value )
{
	KGlobal::config()->setGroup( "KWaveControl" );
	KGlobal::config()->writeEntry( "polling", value );
	basicTimer->changeInterval( (int)( value * 1000 ) );
}


/*!
	\fn KTray::slotSmall()
	toggle "only link widget" visible state
*/
void KTray::slotSmall()
{
	smallWidget->setShown( ! smallWidget->isVisible() );
	KGlobal::config()->setGroup( "small_widget" );
	KGlobal::config()->writeEntry( "shown", smallWidget->isVisible() );
}
