/*
 * Filename:	xstroke.c
 * Author:	David Ljung Madison <DaveSource.com>
 * See License:	http://MarginalHacks.com/License
 * Version:	1.5 beta
 * Description:	Xstroke is a utilty that will grab mouse button events and
 *		run commands in their place.
 */

#include "xstroke.h"

/* selects input from a tree of windows */
void SelectTree(w,level)
  Window w;
  int level;
{
  Window rootw;
  Window par;
  Window *childrn;
  int cnt;
  unsigned int nchild;

/*
  for (cnt=0;cnt<level;cnt++)
    fprintf(stderr,"\t");
  fprintf(stderr,"Window id: %d\n",(int)w);
*/

  /* if ((int)w > 100000) { */
    XSelectInput(dpy, w,KeyPressMask|StructureNotifyMask);
    XQueryTree(dpy, w, &rootw, &par, &childrn, &nchild);
  /* } */
  for(cnt=0;cnt<nchild;cnt++)
    SelectTree(childrn[cnt], level+1);

  if (childrn) XFree ((char *)childrn);
}


/* Button number is byte 1 */
/* Direction (BUP, BDOWN or BCLICK) is byte 2 */
/* Then the action is byte 3,4 */
#define BUTTON_BOTTOM_LEFT	0x100	/* bounding box clicks */
#define BUTTON_TOP_RIGHT	0x200
#define BUTTON_START		0x400	/* the first point */
#define BUTTON_END		0x800	/* the last point */
#define BUTTON_STRING		0x1000	/* Button string goes here */
/* Example action:  <2D:ST> - button 2 down at start point */
/*
 *  Create a format list from a button string.
 *  The .x field is the next character, if it is zero, then
 *  we look at the .y field for the 'action' (see defs above)
 *  - Remember, add_to_list pushes things backwards :(
 */
list_t *create_format(str)
  char *str;
{
  list_t *format=NULL;
  char buf[3];
  char ch;
  int action,button,direction;

  while(*str) {

    switch (*str) {
      case '\\':
      /* escaped characters */
        str++;
        switch (*str) {
          /* these are the only ones I could think of :) */
          case 'n':  ch='\n';  break;
          case 'r':  ch='\r';  break;
          case 't':  ch='\t';  break;
          case 'b':  ch='\b';  break;
          default:   ch=*str;  break;
        }
        add_to_list_end((int) ch,0,&format);
        str++;
        break;

      case '<':
      /* actions ( <type> format ) */
        action=0;
        if(!strncmp("<BSTR>",str,6)) {
          action=BUTTON_STRING;
          str+=6;
        } else {
          button=1; direction=BDOWN;
          switch (*(str+1)) {
            case '1': button=1; break;
            case '2': button=2; break;
            case '3': button=3; break;
            case '4': button=4; break;
            case '5': button=5; break;
            default:
              fprintf(stderr,"ERROR:  Don't know that button, using 1!\n");
              break;
          }
          switch (*(str+2)) {
            case 'U':
            case 'u': direction=BUP; break;
            case 'C':
            case 'c': direction=BCLICK; break;
            case 'D':
            case 'd': break;
            default:
              fprintf(stderr,"ERROR:  "
                      "Don't know that direction, using down!\n");
              break;
          }
          if ((*(str+3))!=':')
            fprintf(stderr,"ERROR:  malformed <action> string.\n");

          if(!strncmp("BL>",str+4,3)) action=BUTTON_BOTTOM_LEFT;
          else if(!strncmp("TR>",str+4,3)) action=BUTTON_TOP_RIGHT;
          else if(!strncmp("ST>",str+4,3)) action=BUTTON_START;
          else if(!strncmp("EN>",str+4,3)) action=BUTTON_END;

          if (action) {
            
            /* Put in the direction (shift 3 first) */
            action|=(direction<<4);

            /* Add in the button number */
            action|=button;

            /* Move to the end of the action */
            str+=7;
          }
        }  /* end parse button actions */

        if(action) {
          add_to_list_end(0,action,&format);
        } else {
          add_to_list_end((int) *str,0,&format);
          str++;
        }
        break;

      default:
      /* just a regular character */
        add_to_list_end((int) *str,0,&format);
        str++;
      break;
    }  /* end character switch */
  }  /* end while loop */
  return format;
}

void free_Win_l_t(ptr)
  Win_l_t *ptr;
{
  fprintf(stderr,"\nWindow has been destroyed - removing from watch\n");
  free(ptr->name);
  free_list(&(ptr->points));
  /* ZZZZZ ACK!  What if someone else is using this? */
  /* free_list(&(ptr->format)); */
  free(ptr);
}

/* take this out of our watch list.
 * ZZZZZZZ Maybe unselect the input and ungrab the button first???
 */
void remove_from_watch(win)
  Window win;
{
  Win_l_t *ptr,*ptr2;

  /* remove all the instances of this window in our watch list */
  ptr2=which_watch(win,TRUE);

  if(!ptr2) return;

  /* remove this window */
  fprintf(stderr,"Removing window: 0x%x (%s)\n",win,watch->name);
  if (ptr2==watch) {
    watch=watch->next;
    free_Win_l_t(ptr2);
  } else {
    ptr=watch;
    while(ptr->next != ptr2) ptr=ptr->next;
    ptr->next=ptr->next->next;
    free_Win_l_t(ptr2);
  }

  if(watch==NULL) {
    fprintf(stderr,"No more windows to watch - exiting\n");
    exit(0);
  }
}

/* Add this window to our watch list */
void add_to_watch(win,name,format,btn)
  Window win;
  char *name;
  list_t *format;
  int btn;
{
  Win_l_t *watch_this;

  if(watch_this=which_watch(win,FALSE)) {
    /* Make sure we aren't already watching this button */
    if(watch_this->formats[btn]) {
      fprintf(stderr,"ERROR:  I'm already watching button %d of window %s\n",btn,name);
      return;
    }
  } else {
    /* Make a new watch element */
    MK_WIN_L(watch_this);
    watch_this->formats[1]=NULL;
    watch_this->formats[2]=NULL;
    watch_this->formats[3]=NULL;
    watch_this->formats[4]=NULL;
    watch_this->formats[5]=NULL;
    watch_this->next=watch;
    watch_this->win=win;
    watch_this->name=strdup(name);
    watch_this->textwin=(int) NULL;
    watch_this->points=NULL;
    watch=watch_this;
  }
  watch_this->formats[btn]=format;
}

/*
 * Recursively search through the window hierarchy to find all
 * window with a specific name
 * format is a pointer to a button-string-output-format list (see create_format)
 */
void Select_Windows_by_name(top,name,format,btn)
  Window top;
  char *name;
  list_t *format;
  int btn;
{
  Window *children, dummy,w=0;
  unsigned int nchildren;
  int i;
  char *window_name;

  if (XFetchName(dpy, top, &window_name) && !strcmp(window_name, name))
    add_to_watch(top,window_name,format,btn);

  XFree((char *)window_name);

  if (!XQueryTree(dpy, top, &dummy, &dummy, &children, &nchildren))
    return;

  for (i=0; i<nchildren; i++)
    Select_Windows_by_name(children[i], name, format,btn);

  if (children) XFree ((char *)children);
}

/*
 * Routine to let user select a window using the mouse
 */
void Select_Window_by_click(format,btn)
  list_t *format;
  int btn;
{
  int status;
  Cursor cursor;
  XEvent event;
  Window target_win = None;
  int buttons = 0;
  char *name;

  fprintf(stderr,"\nClick on the window to watch for button %d...\n",btn);

  /* Make the target cursor */
  cursor = XCreateFontCursor(dpy, XC_crosshair);

  /* Grab the pointer using target cursor, letting it roam all over */
  status = XGrabPointer(dpy, root, False,
                        ButtonPressMask|ButtonReleaseMask, GrabModeSync,
                        GrabModeAsync, root, cursor, CurrentTime);
  if (status != GrabSuccess) {
    fprintf(stderr,"Can't grab the mouse.");
    exit(1);
  }

  /* Let the user select a window... */
  while ((target_win == None) || (buttons != 0)) {
    /* allow one more event */
    XAllowEvents(dpy, SyncPointer, CurrentTime);
    XWindowEvent(dpy, root, ButtonPressMask|ButtonReleaseMask, &event);
    switch (event.type) {
    case ButtonPress:
      if (target_win == None) {
        target_win = event.xbutton.subwindow; /* window selected */
        if (target_win == None) target_win = root;
      }
      buttons++;
      break;
    case ButtonRelease:
      if (buttons > 0) /* there may have been some down before we started */
        buttons--;
       break;
    }
  }

  XUngrabPointer(dpy, CurrentTime);      /* Done with pointer */

  if (!XFetchName(dpy, target_win, &name)) name="no name";
  add_to_watch(target_win,name,format,btn);
}

/* Find out if win is in the hierarchy of windows under top */
int find_child(win,top)
  Window win,top;
{
  Window rootw,parent,*children;
  unsigned int nchildren;
  int cnt,found=0;

  if (win==top) return 1;

  XQueryTree(dpy, top, &rootw, &parent, &children, &nchildren);
  for(cnt=0;cnt<nchildren;cnt++)
    if(find_child(win,children[cnt])) {
      found=1;
      break;
    }

  if (children) XFree ((char *)children);

  return found;
}

/* Which watch member is this window in? */
/* If btn=0 then just return the first one we hit */
Win_l_t *which_watch(win,check_children)
  Window win;
  int check_children;
{
  Win_l_t *ptr=watch;

  while(ptr) {
    if(win==ptr->win) return ptr;
    if(win==ptr->textwin) return ptr;
    if(check_children && find_child(win,ptr->win)) return ptr;
    ptr=ptr->next;
  }
  return NULL;
}


/* Print out a usage message */
#define USAGE	usage(argv[0])
void usage(name)
  char *name;
{
  fprintf(stderr,"\n\n");
  fprintf(stderr,"Usage:  %s [ -btn <button num> [ <window name> "
                 "[ <format string> ] ] ] ...\n",name);
  fprintf(stderr,"\n  Where arguments are:\n\n");
  fprintf(stderr,"button number:       button number to watch\n");
  fprintf(stderr,"window name:         name of window(s) to watch\n");
  fprintf(stderr,"format string:       format of the button string\n");
  fprintf(stderr,"\n  See the man page for details.\n");
  fprintf(stderr,"\n");
  exit(-1);
}

/*
 * Setup the system
 * We need to first choose a window - do this by mouse click now.
 * Then we make sure we get all the events we need.
 * We use XSelectInput to get most of our events, but unfortunately
 * this won't get us ButtonPresses - for that we use XGrabButton
 */
void SetupSystem(argc,argv,envp)
  int argc;
  char **argv,**envp;
{
  int scrn;
  Window window;
  char *name,*str;
  Win_l_t *tmp;
  list_t *formatptr;
  int argnum,mask,i;
  int btn = 2;

  fprintf(stdout, "\nXstroke: (c) copyright 1995, 2002, David Ljung Madison\n");

  if (NULL == (dpy = XOpenDisplay(NULL))) {
    fprintf(stderr,"Couldn't open display.\n");
    exit(0);
  }
 
  scrn=DefaultScreen(dpy);
  root=RootWindow(dpy,scrn);
  watch=NULL;

  if(argc>1) {
    argnum=1;
    while(argnum<argc) {
      if(strcmp("-btn",argv[argnum])) USAGE;

      /* If this is our only arg, stop */
      if(argnum+1>=argc) USAGE;

      /* Get button number */
      btn=atoi(argv[argnum+1]);
      if(btn<1 || btn>5) {
        fprintf(stderr,"Bad button number: %s\n\n",argv[argnum+1]);
        USAGE;
      }

      /* Is there a format string? */
      if(argnum+3>=argc) {
        fprintf(stderr,"Using default format string for button %d:  %s\n",
                       btn,DEFAULT_FORMAT_STRING);
        formatptr=create_format(DEFAULT_FORMAT_STRING);
      } else formatptr=create_format(argv[argnum+3]);

      /* If we have a name, use it.  Else select by clicking */
      if(argnum+2>=argc) {
        Select_Window_by_click(formatptr,btn);
      } else Select_Windows_by_name(root,argv[argnum+2],formatptr,btn);

      argnum+=4;
    } /* end parse args loop */
  } else {  /* no args */
    fprintf(stderr,"\nNo arguments on the command line.\n");
    fprintf(stderr,"Using defaults:  Button %d, Format string: %s\n",
                   DEFAULT_BUTTON,DEFAULT_FORMAT_STRING);

    formatptr=create_format(DEFAULT_FORMAT_STRING);
    Select_Window_by_click(formatptr,btn);
  }

  if(!watch) {
    fprintf(stderr,"Couldn't find any windows\n");
    exit(-1);
  }

  tmp=watch;
  while(tmp) {
    /* Do each button */
    for(i=1;i<=5;i++) {
      if(!tmp->formats[i]) continue;
      /* Figure out the mask */
      switch(i) {
        case 1:  mask=Button1Mask; break;
        case 2:  mask=Button2Mask; break;
        case 3:  mask=Button3Mask; break;
        case 4:  mask=Button4Mask; break;
        case 5:  mask=Button5Mask; break;
      }
      /* Grab the events for the windows */
      SelectTree(tmp->win,0);
      XGrabButton(dpy,i,0,tmp->win,False,
                  ButtonPress|ButtonReleaseMask|mask,
                  GrabModeAsync,GrabModeAsync,None,None);

      fprintf(stdout,"Watching:  Window 0x%x (%s), button %d\n",(int) tmp->win,tmp->name,i);

    } /* for each button */
    tmp=tmp->next;
  } /* for each (while) watch */
}

/* The keysyms that you can use in the DO_KEY macro can be found in
 * the header file:  /usr/include/X11/keysymdef.h
 * (alters the event variable)
 */
#define DO_KEY(keysym) \
		event->xkey.keycode=XKeysymToKeycode(dpy,keysym); \
		XSendEvent(dpy, event->xkey.window, 1, KeyPressMask, event); \
		XSendEvent(dpy, event->xkey.window, 1, KeyReleaseMask, event);

/*
 * Sends a key event
 */
void send_key(ch,window,x,y)
  int ch;
  Window window;
  int x,y;
{
  XKeyEvent kevent;
  int x_root,y_root;
  Window tmp;
  char str[3];
  KeySym keysym;

  /* find out the keycode..  here's an ugly way: */
  str[0]=ch;
  str[1]='\0';
  keysym=XStringToKeysym(str);
  /* ZZZZZZ  Don't know if this is necessary */
  /* THIS SUCKS!  Isn't there a better way? */
  if(ch=='\n') keysym=XK_Return;  /* special case */
  if(ch==' ') keysym=XK_space;  /* special case */
  kevent.keycode=XKeysymToKeycode(dpy,keysym);

  kevent.type=KeyPress;
  kevent.display=dpy;
  kevent.window=window;
  kevent.root=root;
  kevent.subwindow=0x0;  /* ?? Don't know why..  :) */
  kevent.time=CurrentTime;
  kevent.x=x;
  kevent.y=y;
  XTranslateCoordinates(dpy,window,root,x,y,&x_root,&y_root,&tmp);
  kevent.x_root=x_root;
  kevent.y_root=y_root;
  kevent.state=0x0; /* none of the modifiers/buttons are pressed */
  /* rudimentary case dealings... */
  if (ch >= 'A' && ch <= 'Z') kevent.state=0x1;
  kevent.same_screen=1;

  /* Send press and release events */
  XSendEvent(dpy, kevent.window, 1, KeyPressMask,(XEvent *) &kevent); \
  kevent.type=KeyRelease;
  XSendEvent(dpy, kevent.window, 1, KeyReleaseMask,(XEvent *) &kevent);
}

/*
 * Does 'button' in window
 * direction is either BUP, BDOWN or BCLICK  (BCLICK=BUP|BDOWN)
 */
void do_button(num,direction,window,x,y)
  int num;
  int direction;
  Window window;
  int x,y;
{
  XButtonEvent bevent;
  int x_root,y_root;
  Window tmp;

  bevent.display=dpy;
  /* This is weird - but this is the only way that works :( */
  bevent.window=window;
  bevent.root=root;
  bevent.subwindow=window;
  bevent.time=CurrentTime;
  bevent.x=x;
  bevent.y=y;
  XTranslateCoordinates(dpy,window,root,x,y,&x_root,&y_root,&tmp);
  bevent.x_root=x_root;
  bevent.y_root=y_root;
  bevent.button=num;
  bevent.same_screen=1;

  if(direction & BDOWN) {
    bevent.type=ButtonPress;
    bevent.state=0x0;
    XSendEvent(dpy, bevent.window, 1, ButtonPressMask,(XEvent *) &bevent);
  }
  if(direction & BUP) {
    bevent.type=ButtonRelease;
    bevent.state=num<<8;
    XSendEvent(dpy, bevent.window, 1, ButtonReleaseMask,(XEvent *) &bevent);
  }
}

#ifdef OBSOLETE
/* This is obsolete code that translated arrow keys to button clicks */

/* Percent across the window that we want to click (for the arrows) */
#define DIST 85

int tmp_global;

void handle_keypresses(event,watchptr)
  XEvent *event;
  Win_l_t *watchptr;
{
  KeySym keysym;
  char str[200];
  XWindowAttributes attrs;
  Window win;

  /* We want to use they window they typed into */
  win=event->xkey.window;

  /* Get the keysym */
  XLookupString(event->xkey, str, 200, &keysym, NULL);

  /* If we didn't create this event: */
  if(!event->xkey.send_event) {

    /* If it's an arrow key, then get the current drawing window size */
    /* (might have changed) */
    if(keysym==XK_Right || keysym==XK_Left ||
       keysym==XK_Up || keysym==XK_Down)
      XGetWindowAttributes(dpy,win,&attrs);

    /* convert the arrow keys to window clicks */

    /* You can use any other keysyms in here that you want as
     * long as it is a key that your application normally ignores
     * since all keystrokes still get passed through - this will change.
     * (see /usr/include/X11/keysymdef.h)
     */

    switch (keysym) {
      case XK_Right:

        /* XWarpPointer(dpy,None,window,0,0,0,0,20,attrs.height/2); */

        do_button(1,BCLICK,win,attrs.width*DIST/100,attrs.height/2);
        do_button(3,BCLICK,win,attrs.width*DIST/100,attrs.height/2);
        break;
      case XK_Up:
        do_button(1,BCLICK,win,attrs.width/2,attrs.height*(100-DIST)/100);
        do_button(3,BCLICK,win,attrs.width/2,attrs.height*(100-DIST)/100);
        break;
      case XK_Down:
        do_button(1,BCLICK,win,attrs.width/2,attrs.height*DIST/100);
        do_button(3,BCLICK,win,attrs.width/2,attrs.height*DIST/100);
        break;
      case XK_Left:
        do_button(1,BCLICK,win,attrs.width*(100-DIST)/100,attrs.height/2);
        do_button(3,BCLICK,win,attrs.width*(100-DIST)/100,attrs.height/2);
        break;
      case XK_Select:
        /* Select already does the button 1 click for us in some apps! */
        do_button(3,BCLICK,win,event->xkey.x,event->xkey.y);
        break;
    }  /* end keysym select */
  } /* end if not a send_event */
}
#endif

/* Add an element to our linked list type which holds two integers */
void add_to_list(x,y,list_h)
  int x,y;
  list_t **list_h;
{
  list_t *ptr;

  MK_PT_L(ptr);
  ptr->x=x;
  ptr->y=y;
  ptr->next=*list_h;
  *list_h=ptr;
}

void add_to_list_end(x,y,list_h)
  int x,y;
  list_t **list_h;
{
  list_t *ptr,*ptr2;

  MK_PT_L(ptr);
  ptr->x=x;
  ptr->y=y;
  ptr->next=NULL;

  if(*list_h==NULL)
    *list_h=ptr;
  else {
    ptr2=*list_h;
    while(ptr2->next) ptr2=ptr2->next;
    ptr2->next=ptr;
  }
}

/*
 * Send the stroke string to the text window
 * We'll use the watch->format type  (see create_format)
 */
void send_strokes(bstr,btn,watch,startx,starty,endx,endy,b,l,t,r)
  list_t *bstr;
  int btn;
  Win_l_t *watch;
  int startx,starty,endx,endy,b,l,t,r;
{
  list_t *ptr;
  list_t *format=watch->formats[btn];
  Window target;
  int direction,number;	/* for buttons */

  /* This shouldn't happen!  How'd we get the damn event! :) */
  if(!format) {
    fprintf(stderr,"ERROR:  I don't have a format string for button %d for that window (%s)\n"
                  ,btn,watch->name);
    return;
  }

  /* This is ugly...  we don't really know where the button should go */
  if (!(target=watch->textwin)) target=watch->eventwin;

  while(format) {
   if(format->x) 
     send_key(format->x,target,endx,endy);
   else {
     number= (format->y & 0xF); /* 1st byte */
     direction= (format->y & 0xF0)>>4; /* 2nd byte */
     switch (format->y & 0xFF00) {  /* 3rd and 4th byte */
       case BUTTON_BOTTOM_LEFT:
         do_button(number,direction,target,l,b);
         break;
       case BUTTON_TOP_RIGHT:
         do_button(number,direction,target,r,t);
         break;
       case BUTTON_START:
         do_button(number,direction,target,startx,starty);
         break;
       case BUTTON_END:
         do_button(number,direction,target,endx,endy);
         break;
       case BUTTON_STRING:
         ptr=bstr;
         while(ptr) {
         /* Send the character - (since it's an integer, do '0'+integer) */
           send_key('0'+ptr->x,target,endx,endy);
           ptr=ptr->next;
         }
         break;
     } /* end switch action (format->y) */
   } /* end else (is an action) */
   format=format->next;
  } /* end while format loop */

}

/* Add the sextant 'sex' to the buttonstring */
/* We might need to fill in sextants that we skipped over
 * either because we went too fast or we went around the shrunk
 * sextant.  For example, 13 should be 123
 * Some of these choices are arbitrary :(...  61 should be???
 * For these we'll just try to stick to the outside (61 -> 6321)
 */
void addsex(oldsex,sex,bstr_h)
  int oldsex,sex;
  list_t **bstr_h;
{
               /* sex   1,2,3,4,5,6,7,8,9 */
  int extrasex[9][9]= {{0,0,2,0,0,2,4,4,5},   /* oldsex=1 */
                       {0,0,0,1,0,3,1,5,3},   /* 2 */
                       {2,0,0,2,0,0,5,6,6},   /* 3 */
                       {0,1,1,0,0,5,0,7,7},   /* 4 */
                       {0,0,0,0,0,0,0,0,0},   /* 5 */
                       {3,3,0,5,0,0,9,9,0},   /* 6 */
                       {4,4,5,0,0,8,0,0,8},   /* 7 */
                       {7,5,9,7,0,9,0,0,0},   /* 8 */
                       {5,6,6,8,0,0,8,0,0}};  /* 9 */
  int extra,extra2;

  if(oldsex>0) {  /* At the first point, oldsex=-1 */
    if(extra=extrasex[oldsex-1][sex-1]) {
      add_to_list(extra,0,bstr_h);
      /* We might need up to one more point (Ex: 61 -> 6321) */
      if(extra2=extrasex[extra-1][sex-1])
        add_to_list(extra2,0,bstr_h);
    }
  }
  add_to_list(sex,0,bstr_h);
}

/* This is the percent that we shrink each sextant by to
 * make sure that a given point is *really* contained by the sextant
 */
#define SHRINK 1/8
/*
 * Our sextant now looks like:

 *  BZZZT!! We use quadrants now!  This has changed!
 *
 *   -------------------------  y5
 *   |     |  |     |  |     |
 *   |  7  |  |  8  |  |  9  |
 *   |_____|  |_____|  |_____|  y4
 *   |_____    _____    _____|  y3
 *   |     |  |     |  |     |
 *   |  4  |  |  5  |  |  6  |
 *   |_____|  |_____|  |_____|  y2
 *   |_____    _____    _____|  y1
 *   |     |  |     |  |     |
 *   |  1  |  |  2  |  |  3  |
 *   |     |  |     |  |     |
 *   -------------------------  y0
 *  x0    x1  x2   x3  x4   x5
 */

/* Analyze the list of points
 * Remember, the points are in reverse order!
 * To do:
 * IDEA:  Make boxes smaller, then fill in spaces (handles quick strokes too!)
 * Handle 2D strokes
 * Handle clicks as just a 5?
 * Handle quick strokes (ex:  9 1 -> 9 5 1)
 * Handle errors (esp. on diags) ->  (9851 should prob be 951)
 */
void analyze_strokes(btn,watch)
  int btn;
  Win_l_t *watch;
{
  list_t *ptr;
  int x0,y0,x1,y1,x2,y2,x3,y3;			/* quadrant bounds */
  int quad,oldquad;				/* Current quadrant, last quadrant */
  list_t *bstr,*tmp;    /* Button string (just ignore the .y field) */
  int i;
  int smallx=0,smally=0;	/* are the changes in x too small?  (or y) */
  int startx,starty,endx,endy;  /* send_strokes needs this */

  /* First find the bounding rectangle */
  /* Also set the end points (the first point in the list is the end!) */
  ptr=watch->points;
  endx=ptr->x;
  endy=ptr->y;
  x0=x3=endx;
  y0=y3=endy;
  ptr=ptr->next;

  while(ptr) {
    x0=MIN(x0,ptr->x);
    x3=MAX(x3,ptr->x);
    y0=MIN(y0,ptr->y);
    y3=MAX(y3,ptr->y);
    startx=ptr->x;
    starty=ptr->y;
/*fprintf(stderr,"POINT: %d %d\n",ptr->x,ptr->y);*/
    ptr=ptr->next;
  }

  /* fprintf(stderr,"BOUNDING BOX:  %d,%d to %d,%d\n",x0,y0,x3,y3); */

  /* Now figure out the bounds of all the quadrants */
  x1=x0+(x3-x0)/2;
  i=(x1-x0)*SHRINK;	/* this is how much we shrink each dimension */
  x2=x1+i;
  x1=x1-i;

  y1=y0+(y3-y0)/2;
  i=(y1-y0)*SHRINK;
  y2=y1+i;
  y1=y1-i;

  /* if this box is particularly thin (up or across) then we
   * just forget about all but either the top row or the right column
   * ("particularly thin" is a 7% ratio from one dimension to another,
   *  or less than 15 pixels)
   */
  if ( ((x3-x0)<15) || ((y3-y0)/(x3-x0))>=14) /* 100/7=14.28... */
    smallx=1;
  if ( ((y3-y0)<15) || ((x3-x0)/(y3-y0))>=14)
    smally=1;

/* For creating analyze.c
  SRGP_rectangleCoord(x0,y0,x1,y1);
  SRGP_rectangleCoord(x2,y0,x3,y1);
  SRGP_rectangleCoord(x0,y2,x1,y3);
  SRGP_rectangleCoord(x2,y2,x3,y3);
*/

  ptr=watch->points;
  oldquad=-1;
  bstr=NULL;

  while(ptr) {

    /* What quadrant is the point in? */
    /*  7 9  */
    /*  1 3  */
    if(!smallx) {
      if(ptr->x <= x1) quad=1;
      else if(ptr->x >= x2) quad=3;
      else quad=-100;  /* wasn't inside one of the shrunken quadrants */
    } else quad=2;

    if(!smally) {
      if(ptr->y <= y1) quad+=6;
      else if(ptr->y >= y2) quad+=0;
      else quad=-100;
    } else quad+=3;

    if(oldquad!=quad && quad>0) {
      add_to_list(quad,0,&bstr);
      oldquad=quad;
    }

    ptr=ptr->next;
  }

  if(!bstr) /* no button string, just a click */
    add_to_list(5,0,&bstr);

/*
  fprintf(stderr,"Button String: ");
  ptr=bstr;
  while(ptr) {
    fprintf(stderr,"%d ",ptr->x);
    ptr=ptr->next;
  }
  fprintf(stderr,"\n");
*/

  /* For analyze.c
  if(watch->textwin)
  */
  /* *sigh*  -- bottom is y3 and top is y0...  */
  send_strokes(bstr,btn,watch,startx,starty,endx,endy,y3,x0,y0,x3);

  free_list(&bstr);

}

void free_list(listh)
  list_t **listh;
{
  list_t *ptr1,*ptr2=NULL;

  ptr1=*listh;
  if(ptr1) ptr2=ptr1->next;

  while(ptr1) {
    free(ptr1);
    ptr1=ptr2;
    if(ptr1) ptr2=ptr1->next;
  }
  *listh=NULL;
}

/* Main */
int main(argc,argv,envp)
  int argc;
  char **argv,**envp;
{
  XEvent event;
  Win_l_t *watchptr;

  SetupSystem(argc,argv,envp);

  /* because of XGrabButton we need to allow the other events through */
  XAllowEvents(dpy,SyncPointer,CurrentTime);

  while (1) {

    /* Get the event */
    XNextEvent(dpy, &event);

    switch(event.type) {
      case KeyPress:

        if(!(watchptr=which_watch(event.xkey.window,FALSE))) {
          /* We don't know which watch text window this is */

          /* Find out what watch it is a child of */
          watchptr=which_watch(event.xkey.window,TRUE);
          if(watchptr) {
            if(watchptr->textwin)
              fprintf(stderr,"You have multiple text windows! :(\n");
            else {
              watchptr->textwin=event.xkey.window;
              fprintf(stderr,"Found text window (0x%x) for watching 0x%x (%s)\n",
                      watchptr->textwin,watchptr->win,watchptr->name);
            }
          } else
            fprintf(stderr,"ERROR:  "
                    "I don't know what window set this came from!\n");
        }

        watchptr->eventwin=event.xkey.window;
#ifdef OBSOLETE
/* See handle_keypresses comments */
        handle_keypresses(&event,watchptr);
#endif
        break;
      case ButtonPress:
        if(!(watchptr=which_watch(event.xbutton.window,FALSE)))
          watchptr=which_watch(event.xbutton.window,TRUE);
        if(!watchptr) {
          fprintf(stderr,"ERROR:  "
                  "I don't know what window set this came from!\n");
          break;
        }
        if(!watchptr->textwin) {
          fprintf(stderr,"\nERROR:  I don't know which window accepts text!\n"
                         "Please type something into the window before using button %d\n",
                         event.xbutton.button);
        }
        watchptr->eventwin=event.xbutton.window;
        if(watchptr->points) {
          fprintf(stderr,"ERROR:  Buttonpress in the middle of stroke??\n");
          break;
        }
        add_to_list(event.xbutton.x,event.xbutton.y,&watchptr->points);
        break;
      case MotionNotify:
        if(!watchptr) break;
        add_to_list(event.xmotion.x,event.xmotion.y,&watchptr->points);
        break;
      case ButtonRelease:
        if(!watchptr) break;
        add_to_list(event.xbutton.x,event.xbutton.y,&watchptr->points);
        analyze_strokes(event.xbutton.button,watchptr);
        free_list(&(watchptr->points));
        break;
      case DestroyNotify:
        remove_from_watch(event.xdestroywindow.window);
        break;
      default:
        /* fprintf(stderr,"\n[I don't know that event]\n"); */
        break;
    } /* end switch event type */
  } /* end while */
}

