/* virtualdirmode_copy.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2013 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <aguix/fieldlistviewdnd.h>
#include "virtualdirmode.hh"
#include "worker_locale.h"
#include "worker.h"
#include "nwcentryselectionstate.hh"
#include "datei.h"
#include "copyopwin.hh"
#include "copyorder.hh"
#include "copycore.hh"
#include "nmcopyopdir.hh"
#include "nwc_path.hh"

void VirtualDirMode::copy( struct copyorder *copyorder )
{
    bool cancel = false;
    CopyOpWin *cowin;
    char *textstr, *buttonstr;
    std::list<const FileEntry*>::iterator itfe1;
    VirtualDirMode *updatevdm = NULL;
    bool insertElementsIntoVDM = false;
    std::list< std::string > inserts;

    if ( copyorder == NULL ) return;

    if ( copyorder->destdir == NULL ) return;

    if ( copyorder->cowin == NULL ) return;

    if ( ! ce.get() ) return;

    copyorder->overwrite = copyorder->COPY_OVERWRITE_NORMAL;  // force to start in this mode

    finishsearchmode();
  
    if ( Datei::fileExistsExt( copyorder->destdir ) != Datei::D_FE_DIR ) {
        // no dir as destination
        textstr = (char*)_allocsafe( strlen( catalog.getLocale( 67 ) ) + strlen( copyorder->destdir ) + 1 );
        sprintf( textstr, catalog.getLocale( 67 ), copyorder->destdir );
        buttonstr = dupstring( catalog.getLocale(11) );
        Worker::getRequester()->request( catalog.getLocale( 125 ), textstr, buttonstr );
        _freesafe( textstr );
        _freesafe( buttonstr );
        return;
    }
  
    disableDirWatch();
  
    // clear the reclist so the slace doesn't block us because he is still reading
    // from disk
    ft_request_list_clear();
  
    if ( getCurrentDirectory() == copyorder->destdir ) {
        copyorder->do_rename = true;
        updatevdm = this;
    } else {
        Lister *olister = parentlister->getWorker()->getOtherLister( parentlister );
        if ( olister != NULL ) {
            ListerMode *lm = olister->getActiveMode();
            if ( lm != NULL ) {
                if ( lm->getCurrentDirectory() == copyorder->destdir ) {
                    updatevdm = dynamic_cast< VirtualDirMode * >( lm );
                }
            }
        }
    }

    if ( updatevdm != NULL && updatevdm != this ) {
        updatevdm->disableDirWatch();

        if ( ! updatevdm->currentDirIsReal() ) {
            insertElementsIntoVDM = true;
        }
    }
  
    // no sense for follow_symlinks when moving
    if ( copyorder->move == true ) copyorder->follow_symlinks = false;

    /* now presupposition checked
       next search for the files to copy
       therefore collect entries to copy in a list together with the LVC (when on root-level)
       then for each entry do recursive call to copy it
       after success deactivate LVC */

    /* first: create List contains NM_specialsource
       when source==COPY_SPECIAL then the given list but check for existing of the FEs
       else source==COPY_ONLYACTIVE => take the activefe (from ce) in this list
       else take all selected entries (except the "..")
    */

    cowin = copyorder->cowin;

    cowin->open();
    cowin->setmessage( catalog.getLocale( 122 ), 0 );
    if ( cowin->redraw() != 0 ) cancel = true;

    CopyCore *cc = new CopyCore( copyorder );
    int row_adjustment = 0;

    if ( cancel ) cc->setCancel( cancel );

    {
        std::list< NM_specialsourceExt > copylist;
        switch ( copyorder->source ) {
            case copyorder::COPY_SPECIAL:
                copySourceList( *( copyorder->sources ),
                                copylist );
                break;
            case copyorder::COPY_ONLYACTIVE:
                getSelFiles( copylist, LM_GETFILES_ONLYACTIVE, false );
                break;
            default:  // all selected entries
                getSelFiles( copylist, LM_GETFILES_SELORACT, false );
                break;
        }
        for ( auto &it : copylist ) {
            cc->registerFEToCopy( *it.entry(), it.getID() );
        }
    }

    if ( cc->buildCopyDatabase() != 0 ) cc->setCancel( true );

    // this is just the value from the last update, it might not be correct
    if ( parentlister->getWorker()->PS_readSpace( copyorder->destdir ) == 0 ) {
        loff_t free_blocks = parentlister->getWorker()->PS_getFreeSpace();
        loff_t block_size = parentlister->getWorker()->PS_getBlocksize();

        if ( block_size > 0 && ( ( cc->getBytesToCopy() ) / block_size ) + 1 >= free_blocks ) {
            // files will possibly not fit on target, ask to continue or cancel

            std::string buttons = catalog.getLocale( 629 );
            buttons += "|";
            buttons += catalog.getLocale( 8 );

            std::string freespaceh = parentlister->getWorker()->PS_getFreeSpaceH();

            std::string text = AGUIXUtils::formatStringToString( catalog.getLocale( 992 ),
                                                                 copyorder->destdir,
                                                                 freespaceh.c_str() );

            int erg;

            erg = cowin->request( catalog.getLocale( 123 ), text.c_str(), buttons.c_str() );

            if ( erg != 0 ) {
                cc->setCancel( true );
            }
        }
    }

    // now database ready, can start copy-process

    cc->setPreCopyCallback( [this,
                             &copyorder,
                             &row_adjustment]( const FileEntry &fe, int id, const NM_CopyOp_Dir *cod )
                            {
                                int row = getRowForCEPos( id, copyorder->move );
                                int actual_row = row - row_adjustment;
                                if ( actual_row >= 0 ) {
                                    m_lv->setVisMark( actual_row, true );
                                    m_lv->showRow( actual_row, false );

                                    // this is some optimization when copying a lot of small files
                                    // limit redrawing in this case
                                    if ( cod == NULL ) {
                                        loff_t size = 0;

                                        if ( fe.isLink && ! fe.isCorrupt ) {
                                            size = fe.dsize();
                                        } else {
                                            size = fe.size();
                                        }

                                        if ( size > 128 * 1024 ) {
                                            m_lv->redraw();
                                        } else {
                                            m_lv->timeLimitedRedraw();
                                        }
                                    } else {
                                        m_lv->redraw();
                                    }
                                }
                            } );

    std::vector< int > ids_to_remove;
    std::vector< std::tuple< int, int, std::string > > entries_to_delete;

    cc->setPostCopyCallback( [this,
                              &copyorder,
                              &entries_to_delete,
                              &cc,
                              &row_adjustment,
                              &ids_to_remove,
                              &insertElementsIntoVDM,
                              &inserts]( const FileEntry &fe, int id,
                                         CopyCore::nm_copy_t res, const NM_CopyOp_Dir *cod,
                                         const std::string &target_fullname )
                             {
                                 if ( copyorder->move == false ) {
                                     // just copy

                                     if ( ! cc->getCancel() &&
                                          ( res == CopyCore::NM_COPY_OK || res == CopyCore::NM_COPY_NEED_DELETE ) &&
                                          ( cod == NULL || cod->error_counter == 0 ) ) {
                                         // success

                                         int row = getRowForCEPos( id );

                                         auto es = ce->getEntry( id );

                                         if ( es ) {

                                             auto fse = es->getNWCEntry();

                                             if ( row >= 0 ) {
                                                 m_lv->showRow( row, false );
                                                 m_lv->setVisMark( row, false );
                                             }

                                             if ( fse && fse->getFullname() == fe.fullname ) {
                                                 deselectEntry( id );

                                                 if ( insertElementsIntoVDM && ! target_fullname.empty() ) {
                                                     inserts.push_back( target_fullname );
                                                 }
                                             }
                                         }
                                     } else {
                                         // failed
                                         int row = getRowForCEPos( id );

                                         if ( row >= 0 ) {
                                             m_lv->showRow( row, false );
                                             m_lv->setVisMark( row, false );
                                         }
                                     }
                                 } else {
                                     // move so remove entry if successful
                                     int row = getRowForCEPos( id, true );

                                     int actual_row = row - row_adjustment;

                                     auto es = ce->getEntry( id );

                                     if ( es ) {

                                         auto fse = es->getNWCEntry();

                                         m_lv->setVisMark( actual_row, false );

                                         if ( fse && fse->getFullname() == fe.fullname ) {
                                             if ( res == CopyCore::NM_COPY_OK &&
                                                  ( cod == NULL || cod->error_counter == 0 ) ) {
                                                 // success so remove entry
                                                 ce->hideEntry( id );

                                                 if ( !ce->isRealDir() ) {
                                                     ids_to_remove.push_back( id );
                                                 }

                                                 m_lv->deleteRow( actual_row );
                                                 m_lv->timeLimitedRedraw();

                                                 showCacheState();

                                                 row_adjustment++;

                                                 if ( insertElementsIntoVDM && ! target_fullname.empty() ) {
                                                     inserts.push_back( target_fullname );
                                                 }
                                             } else if ( res == CopyCore::NM_COPY_NEED_DELETE ) {
                                                 if ( actual_row >= 0 ) {
                                                     entries_to_delete.push_back( std::make_tuple( id, actual_row, fe.fullname ) );
                                                 }

                                                 if ( insertElementsIntoVDM && ! target_fullname.empty() ) {
                                                     inserts.push_back( target_fullname );
                                                 }
                                             } else if ( res == CopyCore::NM_COPY_CANCEL ) {
                                                 cc->setCancel( true );
                                             }
                                         }
                                     }
                                 }
                             } );

    if ( ! cc->getCancel() ) {
        cc->executeCopy();
    }

    m_lv->redraw();
  
    // now delete remaining files/dirs when moving
    if ( copyorder->move == true ) {

        auto iti2 = entries_to_delete.begin();
        row_adjustment = 0;

        cc->deleteCopiedButNotMoved( [&iti2,
                                      &entries_to_delete,
                                      &row_adjustment,
                                      &ids_to_remove,
                                      this]( const std::pair< std::string, bool > &removed_entry )
                                     {
                                         const NWCEntrySelectionState *es = NULL;
                                         const NWC::FSEntry *fse = NULL;

                                         if ( iti2 != entries_to_delete.end() ) {
                                             es = ce->getEntry( std::get< 0 >( *iti2 ) );

                                             if ( es ) {

                                                 fse = es->getNWCEntry();
                                             }
                                         }


                                         if ( fse && fse->getFullname() == removed_entry.first ) {
                                             m_lv->showRow( std::get< 1 > ( *iti2 ) );

                                             ce->hideEntry( std::get< 0 > ( *iti2 ) );

                                             if ( !ce->isRealDir() ) {
                                                 ids_to_remove.push_back( std::get< 0 > ( *iti2 ) );
                                             }

                                             m_lv->deleteRow( std::get< 1 >( *iti2 ) - row_adjustment );
                                             m_lv->timeLimitedRedraw();

                                             showCacheState();

                                             row_adjustment++;

                                             iti2++;
                                         }
                                     } );
    }


    delete cc;
    cc = NULL;
  
    if ( cowin != NULL ) cowin->close();

    if ( copyorder->move == true ) {
        if ( !ce->isRealDir() ) {
            ce->beginModification( DMCacheEntryNWC::MOD_REMOVAL );

            for ( auto id : ids_to_remove ) {
                // entry is removed from NWC::Dir, not from DMCacheEntryNWC
                // this will be done at end
                ce->removeEntry( id );
            }

            ce->endModification();
        } else {
            ce->reload();
        }

        // trigger alll the refresh stuff
        setCurrentCE( ce );
    } else {
        update();
    }

    if ( insertElementsIntoVDM && updatevdm ) {
        updatevdm->addEntries( inserts );
    }
  
    if ( updatevdm != NULL ) updatevdm->update();

    m_lv->showActive();
    aguix->Flush();

    enableDirWatch();
    if ( updatevdm != NULL && updatevdm != this ) {
        updatevdm->enableDirWatch();
    }
}
