/* copyop.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2006,2008,2011,2013 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "listermode.h"
#include "worker.h"
#include "copyop.h"
#include "copyopwin.hh"
#include "nmspecialsourceext.hh"
#include "worker_locale.h"
#include "datei.h"
#include "dnd.h"
#include <aguix/choosebutton.h>
#include <aguix/cyclebutton.h>
#include <aguix/button.h>
#include "copyorder.hh"
#include "virtualdirmode.hh"

const char *CopyOp::name="CopyOp";

CopyOp::CopyOp() : FunctionProto()
{
  follow_symlinks=false;
  move=false;
  do_rename=false;
  same_dir=false;
  request_dest=false;
  request_flags=false;
  overwrite=COPYOP_OVERWRITE_NORMAL;
  preserve_attr=true;
  hasConfigure = true;
    m_category = FunctionProto::CAT_FILEOPS;
}

CopyOp::~CopyOp()
{
}

CopyOp*
CopyOp::duplicate() const
{
  CopyOp *ta=new CopyOp();
  ta->follow_symlinks=follow_symlinks;
  ta->move=move;
  ta->do_rename=do_rename;
  ta->same_dir=same_dir;
  ta->request_dest=request_dest;
  ta->request_flags=request_flags;
  ta->overwrite=overwrite;
  ta->preserve_attr=preserve_attr;
  return ta;
}

bool
CopyOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *
CopyOp::getName()
{
  return name;
}

int
CopyOp::run( WPUContext *wpu, ActionMessage *msg )
{
  ListerMode *lm1;
  if(msg->mode!=msg->AM_MODE_DNDACTION) {
    Lister *l1 = msg->getWorker()->getActiveLister();
    if(l1!=NULL) {
      startlister=l1;
      endlister = msg->getWorker()->getOtherLister(startlister);
      lm1=startlister->getActiveMode();
      if(lm1!=NULL) {
          if(dynamic_cast< VirtualDirMode * >( lm1 ) ) {
              normalmodecopy( msg );
          } else {
              lm1->not_supported();
          }
      }
    }
  } else {
    normalmodecopy( msg );
  }
  return 0;
}

bool
CopyOp::save(Datei *fh)
{
  if ( fh == NULL ) return false;
  fh->configPutPairBool( "followsymlinks", follow_symlinks );
  fh->configPutPairBool( "move", move );
  fh->configPutPairBool( "rename", do_rename );
  fh->configPutPairBool( "samedir", same_dir );
  fh->configPutPairBool( "requestdest", request_dest );
  fh->configPutPairBool( "requestflags", request_flags );
  switch(overwrite) {
    case COPYOP_OVERWRITE_ALWAYS:
      fh->configPutPair( "overwrite", "always" );
      break;
    case COPYOP_OVERWRITE_NEVER:
      fh->configPutPair( "overwrite", "never" );
      break;
    default:
      fh->configPutPair( "overwrite", "normal" );
      break;
  }
  fh->configPutPairBool( "preserveattr", preserve_attr );
  return true;
}

const char *
CopyOp::getDescription()
{
  return catalog.getLocaleCom(8);
}

int
CopyOp::normalmodecopy( ActionMessage *am )
{
  struct copyorder copyorder;
  NM_specialsourceExt *specialsource=NULL;
  bool do_request,cont=true;
  std::string destination_directory;
  ListerMode *start_lm = NULL, *dest_lm = NULL;
  
  if(am->mode==am->AM_MODE_DNDACTION) {
      start_lm = am->dndmsg->getSourceMode();
      if ( start_lm == NULL ) return 1;

      dest_lm = am->dndmsg->getDestMode();
  } else {
      if ( startlister == NULL ) return 1;
      start_lm = startlister->getActiveMode();
      if ( start_lm == NULL ) return 1;

      if ( endlister != NULL ) {
          dest_lm = endlister->getActiveMode();
      }
  }
  
  if(request_flags==true) {
    if(doconfigure(1)!=0) cont=false;
  } else {
    // set values in t* variables
    tfollow_symlinks=follow_symlinks;
    tmove=move;
    trename=do_rename;
    tsame_dir=same_dir;
    trequest_dest=request_dest;
    toverwrite=overwrite;
    tpreserve_attr=preserve_attr;
  }
  
  if(cont==true) {
    memset( &copyorder, 0, sizeof( copyorder ) );
    copyorder.ignoreLosedAttr = false;
    // detect what to copy
    if(am->mode==am->AM_MODE_ONLYACTIVE)
      copyorder.source=copyorder.COPY_ONLYACTIVE;
    else if(am->mode==am->AM_MODE_DNDACTION) {
      // insert DND-element into list
      copyorder.source=copyorder.COPY_SPECIAL;
      copyorder.sources=new std::list<NM_specialsourceExt*>;
      specialsource = new NM_specialsourceExt( am->dndmsg->getFE() );
      copyorder.sources->push_back(specialsource);
    } else if(am->mode==am->AM_MODE_SPECIAL) {
      copyorder.source=copyorder.COPY_SPECIAL;
      copyorder.sources=new std::list<NM_specialsourceExt*>;
      specialsource = new NM_specialsourceExt( am->getFE() );
      copyorder.sources->push_back(specialsource);
    } else
      copyorder.source=copyorder.COPY_ALLENTRIES;

    /* now source defined
       next find destination
       
       Priority:
        1.Flag for requesting destination
        2.Flag for same dir
        3.When DNDAction destination of DND
        4.nonactive lister
       
       when no destination then also requesting
    */

    do_request = false;
    if ( trequest_dest == true ) {
        do_request = true;
    } else if ( tsame_dir == true ) {
        destination_directory = start_lm->getCurrentDirectory();
    } else if ( am->mode == am->AM_MODE_DNDACTION ) {
        std::string ddstr = am->dndmsg->getDestDir();
        if ( ! ddstr.empty() ) {
            destination_directory = ddstr;
        }
    } else {
        if ( dest_lm != NULL ) {
            destination_directory = dest_lm->getCurrentDirectory();

            if ( destination_directory.empty() ) {
                do_request = true;
            }

            if ( auto vdm = dynamic_cast< VirtualDirMode * >( dest_lm ) ) {
                if ( ! destination_directory.empty() && ! vdm->currentDirIsReal() ) {
                    std::string textstr = AGUIXUtils::formatStringToString( catalog.getLocale( 1002 ),
                                                                            destination_directory.c_str() );
                    std::string buttonstr = catalog.getLocale( 629 );
                    buttonstr += "|";
                    buttonstr += catalog.getLocale( 8 );

                    int erg = Worker::getRequester()->request( catalog.getLocale( 123 ),
                                                               textstr.c_str(),
                                                               buttonstr.c_str() );

                    if ( erg == 1 ) {
                        do_request = false;
                        destination_directory.clear();
                    }
                }
            }
        } else {
            do_request=true;
        }
    }

    if ( do_request == true ) {
        std::string default_dir;
        char *tstr = NULL;

        destination_directory.clear();
      
        if ( dest_lm != NULL ) {
            default_dir = dest_lm->getCurrentDirectory();

            if ( default_dir.empty() ) {
                default_dir = start_lm->getCurrentDirectory();
            }
        } else {
            default_dir = start_lm->getCurrentDirectory();
        }
      
        if ( requestdest( default_dir.c_str(), &tstr ) != 0 ) {
            destination_directory.clear();
        } else {
            destination_directory = tstr;
            _freesafe( tstr );
        }
    }
    /* if dest==null nothing to do
       otherwise: */
    if ( ! destination_directory.empty() ) {
        /* if dest starts with no slash then take it as local dir
           so add currentDir */
        if ( destination_directory[0] != '/' ) {
            std::string tstr = start_lm->getCurrentDirectory();

            destination_directory = tstr + "/" + destination_directory;
        }
        /* now also destination determined
           next describing what to do */
        copyorder.move=tmove;
        copyorder.follow_symlinks=tfollow_symlinks;
        copyorder.preserve_attr=tpreserve_attr;
        copyorder.do_rename=trename;
        switch(toverwrite) {
            case COPYOP_OVERWRITE_ALWAYS:
                copyorder.overwrite=copyorder.COPY_OVERWRITE_ALWAYS;
                break;
            case COPYOP_OVERWRITE_NEVER:
                copyorder.overwrite=copyorder.COPY_OVERWRITE_NEVER;
                break;
            default:
                copyorder.overwrite=copyorder.COPY_OVERWRITE_NORMAL;
        }
        copyorder.destdir = destination_directory.c_str();
        // now start copy process
        copyorder.cowin=new CopyOpWin( am->getWorker()->getAGUIX(),
                                       copyorder.move );

        if ( VirtualDirMode *vdm = dynamic_cast< VirtualDirMode * >( start_lm ) ) {
            vdm->copy( &copyorder );
        }

        delete copyorder.cowin;
    }

    if(copyorder.source==copyorder.COPY_SPECIAL) {
      if ( specialsource != NULL ) delete specialsource;
      delete copyorder.sources;
    }

  }
  return 0;
}

int
CopyOp::doconfigure(int mode)
{
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  ChooseButton *fscb,*mcb,*rcb,*sdcb,*rfcb=NULL,*rdcb,*pacb;
//  CycleButton *ocyb,*fccyb;
  AGMessage *msg;
  int endmode=-1;
  Requester *req;
  char *tstr;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  
  req=new Requester(aguix);

  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(8))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(8));
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr, AWindow::AWINDOW_DIALOG );
  win->create();
  _freesafe(tstr);

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 9 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocaleCom( 8 ), 1 ), 0, 0, cincwnr );

  fscb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( follow_symlinks == true ) ? 1 : 0,
						    catalog.getLocale( 298 ), LABEL_RIGHT, 1, 0 ), 0, 1, cincwnr );

  mcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( move == true ) ? 1 : 0,
						   catalog.getLocale( 299 ), LABEL_RIGHT, 1, 0 ), 0, 2, cincwnr );

  rcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( do_rename == true ) ? 1 : 0,
						   catalog.getLocale( 300 ), LABEL_RIGHT, 1, 0 ), 0, 3, cincwnr );

  sdcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( same_dir == true ) ? 1 : 0,
						    catalog.getLocale( 301 ), LABEL_RIGHT, 1, 0 ), 0, 4, cincwnr );

  rdcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( request_dest == true ) ? 1 : 0,
						    catalog.getLocale( 302 ), LABEL_RIGHT, 1, 0 ), 0, 5, cincwnr );

  pacb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( preserve_attr == true ) ? 1 : 0,
						    catalog.getLocale( 150 ), LABEL_RIGHT, 1, 0 ), 0, 6, cincwnr );

/*  ttext=(Text*)win->add(new Text(aguix,x,y,catalog.getLocale(303),1));
  x+=ttext->getWidth()+5;
  ocyb=(CycleButton*)win->add(new CycleButton(aguix,x,y,100,1,0,0));
  ocyb->addOption(catalog.getLocale(304));
  ocyb->addOption(catalog.getLocale(305));
  ocyb->addOption(catalog.getLocale(306));
  ocyb->resize(ocyb->getMaxSize(),ocyb->getHeight());
  switch(overwrite) {
    case COPYOP_OVERWRITE_ALWAYS:
      ocyb->setOption(1);
      break;
    case COPYOP_OVERWRITE_NEVER:
      ocyb->setOption(2);
      break;
    default:
      ocyb->setOption(0);
      break;
  }
  
  y+=ocyb->getHeight()+5;
  tw=x+ocyb->getWidth()+5;
  if(tw>w) w=tw;
  x=5;*/

  if(mode==0) {
    rfcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( request_flags == true ) ? 1 : 0,
						      catalog.getLocale( 294 ), LABEL_RIGHT, 1, 0 ), 0, 7, cincwnr );
  }

  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 8 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cb = (Button*)ac1_2->add( new Button( aguix,
						0,
						0,
						catalog.getLocale( 8 ),
						1,
						0,
						0 ), 1, 0, cfix );
  
  okb->takeFocus();
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) {
            endmode = 0;
          } else if(msg->button.button==cb) endmode=1;
          break;
        case AG_KEYPRESSED:
          if(win->isParent(msg->key.window,false)==true) {
            switch(msg->key.key) {
              case XK_1:
                fscb->setState((fscb->getState() == true ) ? false : true);
                break;
              case XK_2:
                mcb->setState((mcb->getState() == true ) ? false : true);
                break;
              case XK_3:
                rcb->setState((rcb->getState() == true ) ? false : true);
                break;
              case XK_4:
                sdcb->setState((sdcb->getState() == true ) ? false : true);
                break;
              case XK_5:
                rdcb->setState((rdcb->getState() == true ) ? false : true);
                break;
              case XK_6:
/*                i=ocyb->getSelectedOption()+1;
                if(i>=3) i=0;
                ocyb->setOption(i);
                break;
              case XK_7:*/
                break;
              case XK_Return:
                if ( cb->getHasFocus() == false ) {
                  endmode = 0;
                }
                break;
              case XK_Escape:
                endmode=1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    if(mode==1) {
      // store in t-variables
      tfollow_symlinks = fscb->getState();
      tmove = mcb->getState();
      trename = rcb->getState();
      tsame_dir = sdcb->getState();
      trequest_dest = rdcb->getState();
      tpreserve_attr = pacb->getState();
      toverwrite=COPYOP_OVERWRITE_NORMAL;
/*      switch(ocyb->getSelectedOption()) {
        case 1:
          toverwrite=COPYOP_OVERWRITE_ALWAYS;
          break;
        case 2:
          toverwrite=COPYOP_OVERWRITE_NEVER;
          break;
        default:
          toverwrite=COPYOP_OVERWRITE_NORMAL;
          break;
      }*/
    } else {
      // store in normal variables
      follow_symlinks = fscb->getState();
      move = mcb->getState();
      do_rename = rcb->getState();
      same_dir = sdcb->getState();
      request_dest = rdcb->getState();
      request_flags = rfcb->getState();
      preserve_attr = pacb->getState();
      overwrite=COPYOP_OVERWRITE_NORMAL;
/*      switch(ocyb->getSelectedOption()) {
        case 1:
          overwrite=COPYOP_OVERWRITE_ALWAYS;
          break;
        case 2:
          overwrite=COPYOP_OVERWRITE_NEVER;
          break;
        default:
          overwrite=COPYOP_OVERWRITE_NORMAL;
          break;
      }*/
    }
  }
  
  delete win;
  delete req;

  return endmode;
}

int
CopyOp::configure()
{
  return doconfigure(0);
}

int
CopyOp::requestdest( const char *defaultstr, char **dest )
{
  Requester *req=new Requester( Worker::getAGUIX());
  char *buttonstr;
  const char *textstr;
  int erg;
  
  textstr=catalog.getLocale(212);
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg = req->string_request( catalog.getLocale( 123 ),
                             textstr,
                             ( defaultstr != NULL ) ? defaultstr : "",
                             buttonstr,
                             dest,
                             Requester::REQUEST_SELECTALL );
  _freesafe(buttonstr);
  delete req;
  return erg;
}

void CopyOp::setFollowSymlinks(bool nv)
{
  follow_symlinks=nv;
}

void CopyOp::setMove(bool nv)
{
  move=nv;
}

void CopyOp::setRename(bool nv)
{
  do_rename=nv;
}

void CopyOp::setSameDir(bool nv)
{
  same_dir=nv;
}

void CopyOp::setRequestDest(bool nv)
{
  request_dest=nv;
}

void CopyOp::setRequestFlags(bool nv)
{
  request_flags=nv;
}

void CopyOp::setOverwrite(overwrite_t nv)
{
  overwrite=nv;
}

void CopyOp::setPreserveAttr(bool nv)
{
  preserve_attr=nv;
}
