/* aguix.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2014 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "aguix.h"
#include "awindow.h"
#include <locale.h>
#include "utf8.hh"
#include "drawablecont.hh"
#include "popupwindow.hh"
#include <algorithm>

#ifdef USE_AGUIXTIMER
#include <signal.h>
#endif

#ifdef HAVE_XINERAMA
#  include <X11/extensions/Xinerama.h>
#endif

#ifdef USE_XIM

#ifndef HAVE_XSETIMVALUES_PROTOTYPE
extern "C" {
  char *XSetIMValues( XIM, ... );
}
#endif

#include <X11/Xlocale.h>

/* ChosseBetterStyle is based on an example program I found somewhere
   But there's was no copyright and the orig codepiece also
   doesn't work...
   seems to be from the book "Xlib Programming Manual: For Version 11 of the X Window System"
*/
static XIMStyle ChooseBetterStyle( XIMStyle style1, XIMStyle style2 )
{
  XIMStyle s, t;
  XIMStyle preedit = XIMPreeditArea |
    XIMPreeditCallbacks |
    XIMPreeditPosition |
    XIMPreeditNothing |
    XIMPreeditNone;
  XIMStyle status = XIMStatusArea |
    XIMStatusCallbacks |
    XIMStatusNothing |
    XIMStatusNone;

  if ( style1 == 0 ) return style2;
  if ( style2 == 0 ) return style1;
  if ( ( style1 & ( preedit | status ) ) == ( style2 & ( preedit | status ) ) )
    return style1;

  s = style1 & preedit;
  t = style2 & preedit;
  if ( s != t ) {
    if ( ( s | t ) & XIMPreeditCallbacks ) {
      return ( s == XIMPreeditCallbacks ) ? style1 : style2;
    } else if ( ( s | t ) & XIMPreeditPosition )
      return ( s == XIMPreeditPosition ) ? style1 : style2;
    else if ( ( s | t ) & XIMPreeditArea )
      return ( s == XIMPreeditArea ) ? style1 : style2;
    else if ( ( s | t ) & XIMPreeditNothing )
      return ( s == XIMPreeditNothing ) ? style1 : style2;
  } else {
    s = style1 & status;
    t = style2 & status;
    if ( ( s | t ) & XIMStatusCallbacks)
      return ( s == XIMStatusCallbacks ) ? style1 : style2;
    else if ( ( s | t ) & XIMStatusArea )
      return ( s == XIMStatusArea ) ? style1 : style2;
    else if ( ( s | t ) & XIMStatusNothing )
      return ( s == XIMStatusNothing ) ? style1 : style2;
  }
  return 0;
}

#ifdef XIM_XREGISTER_OKAY
static void AGUIX_im_inst_callback( Display *calldsp, XPointer client_data, XPointer call_data )
{
  if ( client_data != NULL ) {
    ((AGUIX*)client_data)->IMInstCallback( calldsp );
  }
}

static void AGUIX_im_dest_callback( XIM callim, XPointer client_data, XPointer call_data )
{
  if ( client_data != NULL ) {
    ((AGUIX*)client_data)->IMDestCallback();
  }
}
#endif //XIM_XREGISTER_OKAY

#endif //USE_XIM

long AGUIX::timerEvent = 0;
std::map<std::string,int> AGUIX::_colors_for_widgets;

static void (*other_sighandler)( int ) = NULL;

#ifdef USE_AGUIXTIMER
static void aguix_timer( int s )
{
  if ( s == SIGALRM ) {
    AGUIX::timerEvent++;

    if ( other_sighandler != NULL ) {
        other_sighandler( s );
    }
  }
}
#endif

AGUIX::AGUIX( int argc, char **argv, const std::string &classname ) : m_argc( argc ),
                                                                      m_argv( argv ),
                                                                      m_classname( classname )
{
#ifdef USE_AGUIXTIMER
    struct sigaction sig_ac, old_sig_ac;
#endif

  initOK=False;
  mainfont=NULL;
  dsp=NULL;
  m_user_colors = 0;
  m_system_colors = 0;
  privatecmap=false;
  cutstart=NULL;
  pastestart = NULL;
  backpm = None;
  scr = -1;
#ifdef USE_XIM
  im_style = 0;
  inputmethod = NULL;
#endif
  keybuf_len = 10;
  keybuf = (char*)_allocsafe( keybuf_len );
  transientwindow = NULL;
  lastkeyrelease = None;
  lastmouserelease = 0;
  msgLockElement = NULL;
  timerEnabled = false;
  lastTimerEventNr = 0;

#ifdef USE_AGUIXTIMER
  memset( &sig_ac, 0, sizeof( sig_ac ) );
  sig_ac.sa_handler = aguix_timer;
  sigemptyset( &sig_ac.sa_mask );

  sigaction( SIGALRM, &sig_ac, &old_sig_ac );

  if ( old_sig_ac.sa_handler != SIG_ERR &&
       old_sig_ac.sa_handler != SIG_DFL &&
       old_sig_ac.sa_handler != SIG_IGN ) {
      other_sighandler = old_sig_ac.sa_handler;
  }
#endif

  m_open_popup_counter = 0;
  m_popup_ignore_button_release = false;

  m_current_xfocus_window = 0;

  m_last_selection_request_atom = None;
  m_last_selection_request_window = 0;

  m_apply_window_dialog_type = false;

  m_override_xim = false;
  m_filtered_key_events_in_a_row = 0;
  m_skip_filter_event = false;
}
  
int AGUIX::initX()
{
  debugmsg("Opening display...");

  if ( getOverrideXIM() ) {
      debugmsg("Overwrite XIM...");
      setenv( "XMODIFIERS", "@im=none", 1 );
  }
  
  char *displayname=getenv("DISPLAY");
  dsp=XOpenDisplay(displayname);
  if(dsp==NULL) {
    debugmsg("failed");
    return 5;
  }
  debugmsg("Ok");
  
  if ( setlocale( LC_ALL, "" ) == NULL ) {
    fprintf( stderr, "Worker Warning: locale not supported\n" );
  }
  if ( XSupportsLocale() == False ) {
    fprintf( stderr, "Worker Warning: X doesn't support current locale, setting locale to C\n" );
    setlocale( LC_ALL, "C" );
  }
  if ( XSetLocaleModifiers( "" ) == NULL ) {
    fprintf( stderr, "Worker Warning: Cannot set locale modifiers\n" );
  }

  UTF8::checkSupportedEncodings();

  scr=XDefaultScreen(dsp);
  cmap=DefaultColormap(dsp,scr);
  white=WhitePixel(dsp,scr);
  black=BlackPixel(dsp,scr);

  gc=XCreateGC(dsp,RootWindow(dsp,scr),0,0);
  debugmsg( "Loading font..." );
  if ( setFont( "fixed" ) != 0 ) {
    debugmsg( "Trying generic font..." );
    if ( setFont( "-misc-fixed-medium-r-semicondensed--13-*-*-*-c-*-*-*" ) != 0 ) {
      debugmsg( "Trying even more generic font..." );
      if ( setFont( "-misc-fixed-medium-r-semicondensed--*-*-*-*-c-*-*-*" ) != 0 ) {
        debugmsg( "Trying * as a last resort..." );
        if ( setFont( "*" ) != 0 ) {
          panic( "no fixed font found, exit" );
        }
      }
    }
  }
  debugmsg("Ok");

  unsigned long gc_valuemask=GCGraphicsExposures|GCForeground|GCBackground|GCLineWidth|GCCapStyle;
  XGCValues gc_values;
  gc_values.graphics_exposures=False;
  gc_values.foreground=black;
  gc_values.background=white;
  gc_values.line_width=0;
  gc_values.cap_style=CapButt;

  XChangeGC(dsp,gc,gc_valuemask,&gc_values);

  dotted_gc = XCreateGC( dsp, RootWindow( dsp, scr ), 0, 0 );
  XCopyGC( dsp, gc, ~0, dotted_gc );
  gc_values.line_style = LineOnOffDash;
  XChangeGC( dsp, dotted_gc, GCLineStyle, &gc_values );

  dashxor_gc = XCreateGC( dsp, RootWindow( dsp, scr ), 0, 0 );
  XCopyGC( dsp, gc, ~0, dashxor_gc );
  gc_values.line_style = LineOnOffDash;
  gc_values.function = GXxor;
  gc_values.plane_mask = ~0;
  gc_values.foreground = 0xffffffff;
  XChangeGC( dsp, dashxor_gc, GCLineStyle | GCFunction | GCPlaneMask | GCForeground, &gc_values );

  dashdouble_gc = XCreateGC( dsp, RootWindow( dsp, scr ), 0, 0 );
  XCopyGC( dsp, gc, ~0, dashdouble_gc );
  gc_values.line_style = LineDoubleDash;
  gc_values.foreground = black;
  gc_values.background = white;
  XChangeGC( dsp, dashdouble_gc, GCLineStyle | GCBackground | GCForeground, &gc_values );

  WM_delete_window=XInternAtom(dsp,"WM_DELETE_WINDOW",False);
  createGroupWin();

  cursors[WAIT_CURSOR]=XCreateFontCursor(dsp,XC_watch);
  cursors[SCROLLH_CURSOR]=XCreateFontCursor(dsp,XC_sb_h_double_arrow);
  cursors[SCROLLV_CURSOR]=XCreateFontCursor(dsp,XC_sb_v_double_arrow);

  unsigned int rd,rbw;
  int rx, ry;
  Window rootw;
  XGetGeometry( dsp, DefaultRootWindow( dsp ), &rootw, &rx, &ry, &rootWindowWidth, &rootWindowHeight, &rbw, &rd );
#ifdef USE_XIM
  openIM();
#endif

  XA_NET_WM_NAME = XInternAtom( dsp, "_NET_WM_NAME", False );
  XA_NET_WM_ICON_NAME = XInternAtom( dsp, "_NET_WM_ICON_NAME", False );
  XA_TARGETS = XInternAtom( dsp, "TARGETS", False );
  XA_NET_WM_WINDOW_TYPE = XInternAtom( dsp, "_NET_WM_WINDOW_TYPE", False );
  XA_NET_WM_WINDOW_TYPE_DIALOG = XInternAtom( dsp, "_NET_WM_WINDOW_TYPE_DIALOG", False );
  XA_NET_WM_WINDOW_TYPE_NORMAL = XInternAtom( dsp, "_NET_WM_WINDOW_TYPE_NORMAL", False );
  XA_UTF8_STRING = XInternAtom( dsp, "UTF8_STRING", False );

  initOK=True;
  return 0;
}

AGUIX::~AGUIX()
{
  AGMessage *agmsg;

  destroyBGHandlers();

  disableTimer();
  while((agmsg=getAGMsg())!=NULL) ReplyMessage(agmsg);
  if(checkX()==True) {
    if(dsp!=NULL) {
      closeX();
    }
  }
  cancelCut();
  _freesafe( keybuf );
}

void AGUIX::panic(const char *msg)
{
  fprintf(stderr,"%s\n",msg);
  exit(1);
}

void AGUIX::closeX()
{
    destroyBGHandlers();

  destroyGroupWin();

  while ( fonts.empty() == false ) {
      AGUIXFont *tf = fonts.front();
      fonts.pop_front();
      delete tf;
  }

  freeColors();
    
  if(privatecmap==true) {
    XFreeColormap(dsp,cmap);
  }
  XFreeCursor(dsp,cursors[WAIT_CURSOR]);
  XFreeCursor(dsp,cursors[SCROLLH_CURSOR]);
  XFreeCursor(dsp,cursors[SCROLLV_CURSOR]);
  XFreeGC( dsp, gc );
  XFreeGC( dsp, dotted_gc );
  XFreeGC( dsp, dashxor_gc );
  XFreeGC( dsp, dashdouble_gc );
  if ( backpm != None ) XFreePixmap( dsp, backpm );
#ifdef USE_XIM
  closeIM();
#endif

#ifdef DEBUG
  printf("Closing display\n");
#endif
  XCloseDisplay(dsp);
  dsp=NULL;
}

int AGUIX::checkX()
{
  return initOK;
}

int AGUIX::getDepth() const
{
  return DefaultDepth(dsp,scr);
}

Display *AGUIX::getDisplay() const
{
  return dsp;
}

int AGUIX::getScreen() const
{
  return scr;
}

int AGUIX::getCharHeight() const
{
  return CharHeight;
}

void AGUIX::setFG( const AGUIXColor &color )
{
    XSetForeground( dsp, gc, getPixel( color ) );
}

void AGUIX::setFG( GC tgc, const AGUIXColor &color )
{
    if ( tgc == 0 ) setFG( color );
    else {
        XSetForeground( dsp, tgc, getPixel( color ) );
    }
}

void AGUIX::setBG( const AGUIXColor &color )
{
    XSetBackground( dsp, gc, getPixel( color ) );
}

void AGUIX::setBG( GC tgc, const AGUIXColor &color )
{
    if ( tgc == 0 ) setBG( color );
    else {
        XSetBackground( dsp, tgc, getPixel( color ) );
    }
}

void AGUIX::FillRectangle(Drawable buffer,int x,int y,int w,int h)
{
  if ( w < 1 ) w = 1;
  if ( h < 1 ) h = 1;
  XFillRectangle(dsp,buffer,gc,x,y,w,h);
}

void AGUIX::FillRectangle(Drawable buffer,GC tgc,int x,int y,int w,int h)
{
  if ( w < 1 ) w = 1;
  if ( h < 1 ) h = 1;

  if(tgc==0) FillRectangle(buffer,x,y,w,h);
  else XFillRectangle(dsp,buffer,tgc,x,y,w,h);
}

void AGUIX::DrawText( DrawableCont &dc, const char *text, int x, int y, const AGUIXColor &color )
{
  DrawText( dc, mainfont, text, x, y, color );
}

void AGUIX::DrawText( DrawableCont &dc, AGUIXFont *tf, const char *text, int x, int y, const AGUIXColor &color )
{
  if ( tf == NULL ) {
    mainfont->drawText( dc, text, x, y, color );
  } else {
    tf->drawText( dc, text, x, y, color );
  }
}

int AGUIX::getFontBaseline() const
{
  return mainfont->getBaseline();
}

AGUIXColor AGUIX::AddColor( int red, int green, int blue, AGUIXColor::color_type_t type )
{
    AGUIXColor ret_col( -1, type );

    if ( type == AGUIXColor::SYSTEM_COLOR && m_system_colors >= 16 ) return ret_col;
    if ( type == AGUIXColor::USER_COLOR && m_user_colors >= 256 ) return ret_col;

#ifndef HAVE_XFT
    XColor col;
    col.flags = DoRed | DoGreen | DoBlue;
    col.red = red * 256;
    col.green = green * 256;
    col.blue = blue * 256;
    
    if ( !XAllocColor( dsp, cmap, &col ) ) {
        // error
        if ( privatecmap == true ) return -1;
        else {
            // now try to go to a private colormap
            changeColormap();
            return AddColor( red, green, blue, type );
        }
    }

    if ( type == AGUIXColor::SYSTEM_COLOR ) {
        ret_col = AGUIXColor( m_system_colors++, type );
        m_system_col_buf[ret_col.getColor()] = col.pixel;
    } else {
        ret_col = AGUIXColor( m_user_colors++, type );
        m_user_col_buf[ret_col.getColor()] = col.pixel;
    }
#else
    XRenderColor col;
    col.red = red * 256;
    col.green = green * 256;
    col.blue = blue * 256;
    col.alpha = 0xffff;

    if ( ! XftColorAllocValue ( dsp,
                                DefaultVisual( dsp, scr ),
                                cmap,
                                &col,
                                ( type == AGUIXColor::SYSTEM_COLOR ) ? &m_system_col_buf[m_system_colors] : &m_user_col_buf[m_user_colors] ) ) {
        // error
        if ( privatecmap == true ) return -1;
        else {
            // now try to go to a private colormap
            changeColormap();
            return AddColor( red, green, blue, type );
        }
    }

    if ( type == AGUIXColor::SYSTEM_COLOR ) {
        ret_col = AGUIXColor( m_system_colors++, type );
    } else {
        ret_col = AGUIXColor( m_user_colors++, type );
    }
#endif
    if ( ret_col.getColor() >= 0 ) {
        m_color_values[ret_col] = col_values_t( red, green, blue );
    }

    if ( type == AGUIXColor::USER_COLOR ) {
        updateSystemColors( m_user_colors - 1 );
    }
    return ret_col;
}

void AGUIX::freeColors()
{
    while ( m_user_colors > 0 ) {
#ifdef HAVE_XFT
        XftColorFree( dsp,
                      DefaultVisual( dsp, scr ),
                      cmap,
                      &m_user_col_buf[ m_user_colors - 1 ] );
#else
        XFreeColors( dsp, cmap, m_user_col_buf + m_user_colors - 1, 1, 0 );
#endif
        m_user_colors--;
    }

    while ( m_system_colors > 0 ) {
#ifdef HAVE_XFT
        XftColorFree( dsp,
                      DefaultVisual( dsp, scr ),
                      cmap,
                      &m_system_col_buf[ m_system_colors - 1 ] );
#else
        XFreeColors( dsp, cmap, m_system_col_buf + m_system_colors - 1, 1, 0 );
#endif
        m_system_colors--;
    }
}

int AGUIX::getNumberOfColorsForType( AGUIXColor::color_type_t type ) const
{
    if ( type == AGUIXColor::SYSTEM_COLOR ) {
        return m_system_colors;
    } else {
        return m_user_colors;
    }
}

std::string AGUIX::getClassname() const
{
    return m_classname;
}

unsigned long AGUIX::getPixel( const AGUIXColor &color ) const
{
    int col_pos;
    switch( color.getColorType() ) {
      case AGUIXColor::USER_COLOR:
          col_pos = color.getColor();
          if ( col_pos >= m_user_colors ) col_pos = m_user_colors - 1;
          if ( col_pos < 0 ) col_pos = 0;
          if ( col_pos >= m_user_colors ) return 0;
#ifdef HAVE_XFT
          return m_user_col_buf[col_pos].pixel;
#else
          return m_user_col_buf[col_pos];
#endif
          break;
      case AGUIXColor::SYSTEM_COLOR:
          col_pos = color.getColor();
          if ( col_pos >= m_system_colors ) col_pos = m_system_colors - 1;
          if ( col_pos < 0 ) col_pos = 0;
          if ( col_pos >= m_system_colors ) return 0;
#ifdef HAVE_XFT
          return m_system_col_buf[col_pos].pixel;
#else
          return m_system_col_buf[col_pos];
#endif
          break;
    }
    return 0;
}

#ifdef HAVE_XFT
XftColor *AGUIX::getColBufEntry( const AGUIXColor &color )
{
    switch ( color.getColorType() ) {
      case AGUIXColor::USER_COLOR:
          if ( color.getColor() < m_user_colors && color.getColor() >= 0 ) {
              return &m_user_col_buf[color.getColor()];
          }
          break;
      case AGUIXColor::SYSTEM_COLOR:
          if ( color.getColor() < m_system_colors && color.getColor() >= 0 ) {
              return &m_system_col_buf[color.getColor()];
          }
          break;
    }
    return NULL;
}
#else
unsigned long AGUIX::getColBufEntry( const AGUIXColor &color )
{
    switch ( color.getColorType() ) {
      case AGUIXColor::USER_COLOR:
          if ( color.getColor() < m_user_colors && color.getColor() >= 0 ) {
              return m_user_col_buf[color.getColor()];
          }
          break;
      case AGUIXColor::SYSTEM_COLOR:
          if ( color.getColor() < m_system_colors && color.getColor() >= 0 ) {
              return m_system_col_buf[color.getColor()];
          }
          break;
    }
    return 0;
}
#endif

Atom *AGUIX::getCloseAtom()
{
  return &WM_delete_window;
}

void AGUIX::insertWindow( AWindow *win, bool change_transient )
{
  if(win==NULL) return;
  wins.push_back( win );

  if ( win->isTopLevel() == true && change_transient ) {
      setTransientWindow( win );
  }
}

void AGUIX::removeWindow(AWindow *win)
{
  if(win==NULL) return;

  PopUpWindow *popwin = dynamic_cast<PopUpWindow*>( win );
  if ( popwin != NULL ) {
      popup_wins.remove( popwin );
  }

  wins.remove( win );
  wins_as_transient_for.remove( win );

  if ( win == transientwindow ) {
    AWindow *new_transient_win = NULL;

    if ( ! wins_as_transient_for.empty() ) {
        new_transient_win = wins_as_transient_for.back();
    }

    setTransientWindow( new_transient_win );
  }
}

Message *AGUIX::wait4mess( int mode )
{
  Message *newmsg;
  KeySym J;
  int count;
  AWindow *msgawin;
#ifdef USE_XIM
  Window msgwin;
  Status status;
#endif
  
  if(mode==MES_GET) {
    if(XEventsQueued(dsp,QueuedAfterFlush)==0) return NULL;
  }
  XNextEvent(dsp,&LastEvent);

#ifdef USE_XIM
  switch ( LastEvent.type ) {
    case KeyPress:
      msgwin = LastEvent.xkey.window;

      msgawin = NULL;
      for ( wins_cit_t it1 = wins.begin(); it1 != wins.end(); it1++ ) {
          msgawin = *it1;
          if ( msgawin->isTopLevel() == true ) {
              if ( msgawin->isParent( msgwin, false ) == true ) break;
          }
          msgawin = NULL;
      }
      break;
    default:
      msgawin = NULL;
      break;
  }
  if ( ! m_skip_filter_event &&
       XFilterEvent( &LastEvent, ( msgawin != NULL ) ? msgawin->getWindow() : None ) == True ) {
      if ( LastEvent.type == KeyPress ) {
          m_filtered_key_events_in_a_row++;
      }
      return NULL;
  }

  if ( LastEvent.type == KeyPress ) {
      m_filtered_key_events_in_a_row = 0;
  }
#else
  msgawin = NULL;
#endif

  newmsg = AGUIX_allocMessage();
  newmsg->type=LastEvent.type;
  newmsg->gadgettype=NON_GADGET;
  newmsg->gadget=NULL;
  newmsg->window=0;
  newmsg->time=0;

  newmsg->lockElement = msgLockElement;
  newmsg->ack = false;
  newmsg->loop = 0;

  switch(LastEvent.type) {
    case MapNotify:
    case UnmapNotify:
      newmsg->window = LastEvent.xmap.window;
      break;
    case MappingNotify:
      XRefreshKeyboardMapping((XMappingEvent*)&LastEvent);
      break;
    case KeyPress:
    case KeyRelease:
      if ( ( LastEvent.type == KeyPress ) && ( msgawin != NULL ) ) {
#ifdef USE_XIM
	if ( msgawin->getXIC() != NULL ) {
	  count = XmbLookupString( msgawin->getXIC(), &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, &status );
	  if ( status == XBufferOverflow ) {
	    _freesafe( keybuf );
	    keybuf_len = count + 1;
	    keybuf = (char*)_allocsafe( keybuf_len );
	    count = XmbLookupString( msgawin->getXIC(), &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, &status );
	  }
	  switch ( status ) {
	    case XLookupBoth:
	      keybuf[count] = '\0';
	      break;
	    case XLookupKeySym:
	      keybuf[0] = '\0';
	      break;
	    case XLookupChars:
	      J = XK_VoidSymbol;
	      keybuf[count] = '\0';
	      break;
	    default:
	      keybuf[0] = '\0';
	      J = XK_VoidSymbol;
	      break;
	  }
	  newmsg->key = J;
	} else {
	  count = XLookupString( &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, NULL );
	  keybuf[ count ] = '\0';
	  newmsg->key=J;
	}
#endif
      } else {
	count = XLookupString( &(LastEvent.xkey), keybuf, keybuf_len - 1, &J, NULL );
	keybuf[ count ] = '\0';
	newmsg->key=J;
      }
      newmsg->keystate=LastEvent.xkey.state;
      newmsg->window=LastEvent.xkey.window;
      newmsg->mousex=LastEvent.xkey.x;
      newmsg->mousey=LastEvent.xkey.y;
      newmsg->keybuf = dupstring( keybuf );
      newmsg->time = LastEvent.xkey.time;
      break;
    case ButtonPress:
    case ButtonRelease:
      newmsg->button = LastEvent.xbutton.button;
      newmsg->keystate = LastEvent.xbutton.state;
      newmsg->mousex = LastEvent.xbutton.x;
      newmsg->mousey = LastEvent.xbutton.y;
      newmsg->window = LastEvent.xbutton.window;
      newmsg->time = LastEvent.xbutton.time;
      break;
    case MotionNotify:
      /* I no longer use XQueryPointer here because of error if the window
         this message is for no longer exists and then XQueryPointer fails
	 Instead use queryPointer when get AG_MOUSEMOVE */
      newmsg->mousex = LastEvent.xmotion.x;
      newmsg->mousey = LastEvent.xmotion.y;
      newmsg->mouserx = LastEvent.xmotion.x_root;
      newmsg->mousery = LastEvent.xmotion.y_root;
      newmsg->window = LastEvent.xmotion.window;
      newmsg->keystate = LastEvent.xmotion.state;
      break;
    case Expose:
      newmsg->window = LastEvent.xexpose.window;
      newmsg->x = LastEvent.xexpose.x;
      newmsg->y = LastEvent.xexpose.y;
      newmsg->width = LastEvent.xexpose.width;
      newmsg->height = LastEvent.xexpose.height;
      break;
    case ConfigureNotify:
      // first try to find further configure notifies for
      // this window to improve handling of many resizes due to
      // solid resize used by many WMs
      while ( XCheckTypedWindowEvent( dsp,
				      LastEvent.xconfigure.window,
				      ConfigureNotify,
				      &LastEvent ) == True );
      newmsg->window=LastEvent.xconfigure.window;
      newmsg->mousex=LastEvent.xconfigure.x;
      newmsg->mousey=LastEvent.xconfigure.y;
      newmsg->width = LastEvent.xconfigure.width;
      newmsg->height = LastEvent.xconfigure.height;
      newmsg->x = LastEvent.xconfigure.x;
      newmsg->y = LastEvent.xconfigure.y;
      break;
    case ClientMessage:
      if(LastEvent.xclient.data.l[0]==(long)WM_delete_window) {
        newmsg->gadgettype=CLOSE_GADGET;
        newmsg->gadget=NULL;
        newmsg->window=LastEvent.xclient.window;
      }
      break;
    case EnterNotify:
    case LeaveNotify:
      newmsg->window=LastEvent.xcrossing.window;
      newmsg->mousex=LastEvent.xcrossing.x;
      newmsg->mousey=LastEvent.xcrossing.y;
      break;
    case SelectionClear:
      if(cutstart!=NULL) {
	if( ( LastEvent.xselectionclear.window==cutstart->getWindow() ) &&
	    ( LastEvent.xselection.selection==XA_PRIMARY ) ) {
	  cancelCut();
        } else if( LastEvent.xselectionclear.window == getGroupWin() &&
                   LastEvent.xselection.selection==XA_PRIMARY ) {
            clipboard_string = "";
	}
      }
      break;
    case SelectionNotify:
      if( (LastEvent.xselection.selection==XA_PRIMARY) &&
	  (LastEvent.xselection.property!=None) ) {
	Atom ret_type;
	int ret_format;
	unsigned long ret_len,ret_after;
	unsigned char *ret_prop;
	XGetWindowProperty(dsp,
			   LastEvent.xselection.requestor,
			   LastEvent.xselection.property,
			   0,
			   8192,
			   False,
			   LastEvent.xselection.target,
			   &ret_type,
			   &ret_format,
			   &ret_len,
			   &ret_after,
			   &ret_prop);
	if(ret_len>0) {
	  //	  ret_prop[ret_len-1]=0;
	  if(pastestart!=NULL) pastestart->paste(ret_prop);
	  XFree(ret_prop);
	  XDeleteProperty(dsp,LastEvent.xselection.requestor,LastEvent.xselection.property);
	} else {
	  if(pastestart!=NULL) pastestart->cancelpaste();
	  pastestart=NULL;
	}
      } else {
          if ( rerequestCut() ) {
          } else {
              if(pastestart!=NULL) pastestart->cancelpaste();
              pastestart=NULL;
          }
      }
      break;
    case SelectionRequest:
      XEvent ev;
      ev.type=SelectionNotify;
      ev.xselection.requestor=LastEvent.xselectionrequest.requestor;
      ev.xselection.selection=LastEvent.xselectionrequest.selection;
      ev.xselection.target=LastEvent.xselectionrequest.target;
      ev.xselection.time=LastEvent.xselectionrequest.time;
      ev.xselection.property=LastEvent.xselectionrequest.property;
      if( ( LastEvent.xselectionrequest.selection==XA_PRIMARY ) &&
	  ( LastEvent.xselectionrequest.target==XA_STRING ) ) {
	XChangeProperty(dsp,
			ev.xselection.requestor,
			ev.xselection.property,
			ev.xselection.target,
			8,
			PropModeReplace,
			(unsigned char*)clipboard_string.c_str(),
			clipboard_string.length());
#ifndef DISABLE_UTF8_SUPPORT
      } else if( LastEvent.xselectionrequest.selection == XA_PRIMARY &&
                 LastEvent.xselectionrequest.target == XA_UTF8_STRING &&
                 UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED &&
                 UTF8::getCurrentEncoding() == UTF8::ENCODING_UTF8 ) {
	XChangeProperty(dsp,
			ev.xselection.requestor,
			ev.xselection.property,
			ev.xselection.target,
			8,
			PropModeReplace,
			(unsigned char*)clipboard_string.c_str(),
			clipboard_string.length());
#endif
      } else if ( LastEvent.xselectionrequest.target == XA_TARGETS ) {
          // send the list of supported atoms
          Atom supported_targets[] = { XA_TARGETS, XA_STRING, XA_UTF8_STRING };
          XChangeProperty( dsp,
                           ev.xselection.requestor,
                           ev.xselection.property,
                           XA_ATOM,
                           8, PropModeReplace,
                           (unsigned char *)supported_targets,
                           sizeof( supported_targets ) );
      } else {
	ev.xselection.property=None;
      }
      XSendEvent(dsp,ev.xselection.requestor,False,0,&ev);
      break;
    case FocusIn:
      newmsg->window = LastEvent.xfocus.window;
      m_current_xfocus_window = LastEvent.xfocus.window;
      break;
    case FocusOut:
      newmsg->window = LastEvent.xfocus.window;
      m_current_xfocus_window = 0;
      break;
  }
  return newmsg;
}

void AGUIX::buildAGMessage( Message *msg )
{
  AGMessage *agmsg[2];

  agmsg[0] = AGUIX_allocAGMessage( msg );  // Hauptmessage
  agmsg[1] = NULL;           // eventl. Nebenmessage;
  switch( msg->type ) {
    case ButtonRelease:
      agmsg[1] = AGUIX_allocAGMessage();
      agmsg[1]->type = AG_MOUSECLICKED;
      agmsg[1]->mouse.window = msg->window;
      agmsg[1]->mouse.button = msg->button;
      agmsg[1]->mouse.x = msg->mousex;
      agmsg[1]->mouse.y = msg->mousey;
      agmsg[1]->mouse.time = msg->time;
  }
  // agmsg in FIFO speichern
  putAGMsg( agmsg[0] );
  if ( agmsg[1] != NULL ) putAGMsg( agmsg[1] );
}

int AGUIX::msgHandler( int mode, AWindow *parent, bool onlyExpose )
{
  Message *tmsg;
  int waitmode;
  long timerEventNr;
#ifndef USE_AGUIXTIMER
  struct timeval curTime;
  long t1;
#endif

  // do not limit to special window if there is a guielement
  // holding the lock
  if ( msgLockElement != NULL ) parent = NULL;

  for (;;) {
    waitmode = mode;

    // do not wait when timer is enabled
    if ( ( waitmode == MES_WAIT ) &&
	 ( msgLockElement != NULL ) &&
	 ( timerEnabled == true ) ) {
      waitmode = MES_GET;
    }

    if ( ( msgLockElement != NULL ) &&
	 ( timerEnabled == true ) ) {
      onlyExpose = false;
    }
    tmsg = wait4mess( waitmode );
    if ( tmsg != NULL ) {
      if ( ( onlyExpose == false ) || ( ( onlyExpose == true ) && ( tmsg->type == Expose ) ) ) {
	if ( ReactMessage( tmsg, parent ) == false ) {
          if ( msgLockElement == NULL ) {
            // only build msg for application when no element holds the lock
            //TODO: at the time of this writing (20031115) it would be okay
            //      to build the msg in any case, AGUIX can handle this
            //      but I keep this conforming to previous version
            buildAGMessage( tmsg );
          }
        }
      }
      if ( tmsg->type == ButtonRelease ) {
	lastmouserelease = tmsg->button;
      } else if ( tmsg->type == KeyRelease ) {
	lastkeyrelease = tmsg->key;
      }

      checkPopUpWindows( tmsg );
      
      AGUIX_freeMessage( tmsg );
    }

    if ( XEventsQueued( dsp, QueuedAfterFlush ) > 0 ) continue;
    // and check whether we hit a timer limit

    if ( ! getOverrideXIM() &&
         m_filtered_key_events_in_a_row == 10 ) {

        m_filtered_key_events_in_a_row++;

        tmsg = AGUIX_allocMessage();
        tmsg->type = ClientMessage;
        tmsg->specialType = Message::XIMSTUCK;
      
        buildAGMessage( tmsg );
        AGUIX_freeMessage( tmsg );
    }

    // break loop when timer isn't activated
    if ( ! ( ( msgLockElement != NULL ) &&
	     ( timerEnabled == true ) ) ) {
      break;
    }
    //TODO Because it's possible to break the loop
    //     the msgLockElement could be destroyed already
    //     ReactMessage will reset msgLockElement if it doesn't
    //     set ack, but when there's no msg, we can reach this
    //     code
    //     possible solution: check every window using contains()
    if ( msgLockElement->getAllowTimerEventBreak() == true ) {
      if ( msgLockElement->getNrOfLastEventBreak() != lastTimerEventNr ) {
	msgLockElement->setNrOfLastEventBreak( lastTimerEventNr );
	break;
      }
    }

    // now wait some time
#ifndef USE_AGUIXTIMER
    gettimeofday( &curTime, NULL );
    //TODO das koennte ein Precision-Problem sein, wenn zu lange im Timer-Mode
    //     verblieben wird
    t1 = ( ( lastTimerEventNr + 1 ) * 1000 / TIMER_TICKS ) - ldiffgtod_m( &curTime, &timerStart );
    if ( ( tmsg == NULL ) && ( t1 > 1 ) ) waittime( 5 );
#else
    if ( tmsg == NULL ) waittime( 5 );
#endif

#ifndef USE_AGUIXTIMER
    gettimeofday( &curTime, NULL );
    timerEventNr = ( ldiffgtod_m( &curTime, &timerStart ) * TIMER_TICKS ) / 1000;
#else
    timerEventNr = timerEvent;
#endif

    if ( timerEventNr != lastTimerEventNr ) {
      // send timer event msg
      tmsg = AGUIX_allocMessage();
      tmsg->type = ClientMessage;
      tmsg->specialType = Message::TIMEREVENT;
      tmsg->time = timerEventNr;
      tmsg->window = None;
      
      tmsg->lockElement = msgLockElement;
      tmsg->ack = false;
      tmsg->loop = 0;
      
      ReactMessage( tmsg, parent );
      AGUIX_freeMessage( tmsg );
      lastTimerEventNr = timerEventNr;
    }

    // test again for timer since the timer message could
    // have disabled the timer
    if ( ! ( ( msgLockElement != NULL ) &&
	     ( timerEnabled == true ) ) ) {
        break;
    }
  }
  return XEventsQueued( dsp, QueuedAfterFlush );
}

AGMessage *AGUIX::GetMessage(AWindow *parent)
{
  msgHandler( MES_GET, parent, false );
  return getAGMsg();
}

AGMessage *AGUIX::WaitMessage(AWindow *parent)
{
    while ( messages.empty() == true ) {
        msgHandler( MES_WAIT, parent, false );
    }
    return getAGMsg();
}

void AGUIX::copyArea(Drawable source,Drawable dest,int s_x,int s_y,int width,int height,int d_x,int d_y)
{
  if ( ( width < 1 ) || ( height < 1 ) ) return;  // do nothing when no valid width/height
  XCopyArea(dsp,source,dest,gc,s_x,s_y,width,height,d_x,d_y);
}

void AGUIX::DrawLine(Drawable buffer,int px1,int py1,int px2,int py2)
{
  XDrawLine(dsp,buffer,gc,px1,py1,px2,py2);
}

void AGUIX::DrawLine(Drawable buffer,GC tgc,int px1,int py1,int px2,int py2)
{
  if(tgc==0) DrawLine(buffer,px1,py1,px2,py2);
  else XDrawLine(dsp,buffer,tgc,px1,py1,px2,py2);
}

void AGUIX::DrawPoint(Drawable buffer,int px,int py)
{
  XDrawPoint(dsp,buffer,gc,px,py);
}

void AGUIX::DrawPoint(Drawable buffer,GC tgc,int px,int py)
{
  if(tgc==0) DrawPoint(buffer,px,py);
  else XDrawPoint(dsp,buffer,tgc,px,py);
}

bool AGUIX::ReactMessage(Message *msg,AWindow *parent)
{
  // als erstes das Fenster ausfindig machen
  // dann ReactMessage des Fenster aufrufen
  bool returnvalue=false;
  if ( ( msg->type == ClientMessage ) &&
       ( msg->specialType == Message::NONE ) ) {
    return returnvalue;
  }
  
  bool popup_opened = ( m_open_popup_counter > 0 );
  
  while ( msg->loop < 2 ) {
      for ( wins_cit_t it1 = wins.begin();
            ( it1 != wins.end() ) && ( returnvalue == false );
            it1++ ) {
          AWindow *win = *it1;

          if ( msg->type == Expose ) {
              returnvalue = win->handleMessage( &LastEvent, msg );
          } else if ( m_open_popup_counter > 0 ) {
              if ( dynamic_cast<PopUpWindow*>( win ) != NULL ) {
                  returnvalue = win->handleMessage( &LastEvent, msg );
              }
          } else if ( parent == NULL || parent->isParent( win->getWindow(), false ) == true ) {
              returnvalue = win->handleMessage( &LastEvent, msg );
          }
          // with lockElement don't quit this loop
          if ( msg->lockElement != NULL ) returnvalue = false;
      }

    if ( msg->lockElement != NULL ) {
      // there is a lock element
      // check whether the element accepted this msg
      // ack should be false only when the element doesn't exists anymore
      if ( msg->ack == true ) {
	// okay, element accepted msg
	break;
      } else {
	// lost element
	msgLockElement = NULL;
	msg->lockElement = NULL;
	fprintf( stderr, "Worker: msg lock element lost!\n" );
	// no repeat
      }
    } else {
      // no lock element => just quit loop
      break;
    }
    msg->loop++;
  }
  
  if ( popup_opened == true ) {
      // if there was (or is) a popup opened always
      // return true so we don't create AGMessages
      // (especially for key events)
      return true;
  }
  return returnvalue;
}

void AGUIX::Flush()
{
  XFlush(dsp);
}

int AGUIX::getargc() const
{
  return m_argc;
}

char **AGUIX::getargv() const
{
  return m_argv;
}

void AGUIX::putAGMsg(AGMessage *msg)
{
    messages.push_back( msg );
}

AGMessage *AGUIX::getAGMsg()
{
    if ( messages.empty() == true )
        return NULL;
    
    AGMessage *msg = messages.front();
    messages.pop_front();
    return msg;
}

void AGUIX::ReplyMessage(AGMessage *msg)
{
  AGUIX_freeAGMessage( msg );
}

void AGUIX::ClearWin(Window win)
{
  XClearWindow(dsp,win);
}

void AGUIX::SetWindowBG( Window win, const AGUIXColor &color )
{
    XSetWindowBackground( dsp, win, getPixel( color ) );
}

void AGUIX::WindowtoBack(Window win)
{
  XLowerWindow(dsp,win);
}

void AGUIX::WindowtoFront(Window win)
{
  XRaiseWindow(dsp,win);
}

int AGUIX::changeColor( const AGUIXColor &index2, int red, int green, int blue )
{
#ifdef DEBUG
    if ( index2.getColorType() == AGUIXColor::SYSTEM_COLOR ) {
        printf( "changeColor: system %d/%d\n", index2.getColor(), m_system_colors );
    } else {
        printf( "changeColor: user %d/%d\n", index2.getColor(), m_user_colors );
    }
#endif
    if ( index2.getColorType() == AGUIXColor::SYSTEM_COLOR ) {
        if ( index2.getColor() < 0 || index2.getColor() >= m_system_colors ) return 1;
    } else {
        if ( index2.getColor() < 0 || index2.getColor() >= m_user_colors ) return 1;
    }

#ifdef HAVE_XFT
    XftColor *buf_entry = NULL;

    if ( index2.getColorType() == AGUIXColor::SYSTEM_COLOR ) {
        buf_entry = &m_system_col_buf[index2.getColor()];
    } else {
        buf_entry = &m_user_col_buf[index2.getColor()];
    }
    XftColorFree( dsp,
                  DefaultVisual( dsp, scr ),
                  cmap,
                  buf_entry );
    
    XRenderColor col;
    col.red = red * 256;
    col.green = green * 256;
    col.blue = blue * 256;
    col.alpha = 0xffff;
    if ( ! XftColorAllocValue ( dsp,
                                DefaultVisual( dsp, scr ),
                                cmap,
                                &col,
                                buf_entry ) ) {
        return -1;
    }
#else
    XColor col;
    unsigned long *buf_entry = NULL;

    if ( index2.getColorType() == AGUIXColor::SYSTEM_COLOR ) {
        buf_entry = &m_system_col_buf[index2.getColor()];
    } else {
        buf_entry = &m_user_col_buf[index2.getColor()];
    }

    XFreeColors( dsp, cmap, buf_entry, 1, 0 );
    col.flags = DoRed | DoGreen | DoBlue;
    col.red = red * 256;
    col.green = green * 256;
    col.blue = blue * 256;
    if ( !XAllocColor( dsp, cmap, &col ) ) return -1;
    *buf_entry = col.pixel;
#endif
    
    m_color_values[index2] = col_values_t( red, green, blue );

    if ( index2.getColorType() == AGUIXColor::USER_COLOR ) {
        updateSystemColors( index2.getColor() );
    }
    return 0;
}

AGUIXFont *AGUIX::getFont( const char *name )
{
  AGUIXFont *tf = NULL;

  for ( aguixfont_list_cit_t it1 = fonts.begin();
        it1 != fonts.end();
        it1++ ) {
      tf = *it1;
      if ( strcmp( tf->getName(), name ) == 0 ) break;
      tf = NULL;
  }

  if ( tf == NULL ) {
    tf = new AGUIXFont( this );
    if ( tf->setFont( name ) != 0 ) {
      delete tf;
      return NULL;
    }
    fonts.push_back( tf );
  }
  return tf;
}

void AGUIX::ExposeHandler(Message *msg)
{
  if ( msg->window != 0 ) {
      for ( wins_cit_t it1 = wins.begin();
            it1 != wins.end();
            it1++ ) {
          AWindow *win = *it1;
          if ( msg->type == Expose ) {
              win->handleMessage( &LastEvent, msg );
          }
      }
  }
}

AWindow *AGUIX::findAWindow(Window win)
{
  AWindow *awin = NULL;

  for ( wins_cit_t it1 = wins.begin();
        it1 != wins.end();
        it1++ ) {
      awin = *it1;
      if ( awin->isParent( win, true ) == true ) break;
      awin = NULL;
  }
  
  return awin;
}

char *AGUIX::getNameOfKey(KeySym key,unsigned int mod) const
{
  char *tstr,*tstr2;
  tstr2=XKeysymToString(key); // Nicht freilassen
  if(tstr2==NULL) return NULL;
  tstr=dupstring(tstr2);
  if((mod&Mod1Mask)==Mod1Mask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Left Alt + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  if((mod&ControlMask)==ControlMask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Control + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  if((mod&ShiftMask)==ShiftMask) {
    tstr2=(char*)_allocsafe(11+strlen(tstr)+2);
    sprintf(tstr2,"Shift + %s",tstr);
    _freesafe(tstr);
    tstr=tstr2;
  }
  return tstr;
}

Window AGUIX::getGroupWin() const
{
  return groupwin;
}

void AGUIX::createGroupWin()
{
  XTextProperty windowname,iconname;
  Visual *vis=DefaultVisual(dsp,scr);
  unsigned long mask=CWEventMask;
  XSetWindowAttributes attr;
  char *tstr;
  attr.event_mask=0;
  groupwin=XCreateWindow(dsp,RootWindow(dsp,scr),0,0,10,10,0,getDepth(),InputOutput,vis,mask,&attr);
  if(!groupwin) {
    return;
  }
  XClassHint classhint;
  tstr = dupstring( getClassname().c_str() );
  classhint.res_name=tstr;
  classhint.res_class=tstr;
  XSetClassHint(dsp,groupwin,&classhint);
  XSetWMProtocols(dsp,groupwin,getCloseAtom(),1);
  XSizeHints *SizeHints;
  XWMHints *WMHints;
  SizeHints=XAllocSizeHints();
  WMHints=XAllocWMHints();
  SizeHints->flags=PSize;
  SizeHints->width=10;
  SizeHints->height=10;
  WMHints->input=True;

  WMHints->window_group=groupwin;

  WMHints->flags=InputHint|WindowGroupHint;
  XStringListToTextProperty(&tstr,1,&windowname);
  XStringListToTextProperty(&tstr,1,&iconname);
  XSetWMName(dsp,groupwin,&windowname);
  XSetWMIconName(dsp,groupwin,&iconname);
  XSetWMHints(dsp,groupwin,WMHints);
  XSetWMNormalHints(dsp,groupwin,SizeHints);
  XStoreName(dsp,groupwin,tstr);
  XFree(SizeHints);
  XFree(WMHints);
  _freesafe(tstr);
  XSetCommand(dsp,groupwin,getargv(),getargc());
  XFree( windowname.value );
  XFree( iconname.value );
}

void AGUIX::destroyGroupWin()
{
  XDestroyWindow(dsp,groupwin);
  groupwin=0;
}

int
AGUIX::queryPointer(Window win,int *x,int *y)
{
  Window usewin,root,child;
  int root_x,root_y;
  unsigned int keys_buttons;

  if ( ( x == NULL ) || ( y == NULL ) ) return -1;

  if(win==0) usewin=DefaultRootWindow(dsp);
  else usewin=win;
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,&keys_buttons);
  return 0;
}

int
AGUIX::queryPointer(Window win,int *x,int *y,unsigned int *buttons)
{
  Window usewin,root,child;
  int root_x,root_y;

  if ( ( x == NULL ) || ( y == NULL ) ) return -1;

  if(win==0) usewin=DefaultRootWindow(dsp);
  else usewin=win;
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,buttons);
  return 0;
}

int
AGUIX::queryRootPointer(int *x,int *y)
{
  Window usewin,root,child;
  int root_x,root_y;
  unsigned int keys_buttons;

  if ( ( x == NULL ) || ( y == NULL ) ) return -1;

  usewin=DefaultRootWindow(dsp);
  XQueryPointer(dsp,usewin,&root,&child,&root_x,&root_y,x,y,&keys_buttons);
  return 0;
}

int
AGUIX::setFont( const char *newfont )
{
  AGUIXFont *afont=getFont(newfont);
  if(afont==NULL) return 1;
  mainfont=afont;
  CharHeight=mainfont->getCharHeight();
  return 0;
}

void
AGUIX::changeColormap()
{
  Colormap newcmap;
  if(privatecmap==false) {
#ifdef DEBUG
    printf("go to private cmap\n");
#endif
    newcmap=XCopyColormapAndFree(dsp,cmap);
    cmap=newcmap;
    privatecmap=true;

    for ( wins_cit_t it1 = wins.begin();
          it1 != wins.end();
          it1++ ) {
        AWindow *win = *it1;
        XSetWindowColormap( dsp, win->getWindow(), cmap );
    }
  }
}

Colormap
AGUIX::getColormap() const
{
  return cmap;
}

void AGUIX::setCursor(Window win,int type)
{
  if((type>=WAIT_CURSOR)&&(type<MAXCURSORS)) {
    XDefineCursor(dsp,win,cursors[type]);
  }
}

void AGUIX::unsetCursor(Window win)
{
  XUndefineCursor(dsp,win);
}

int AGUIX::startCut(GUIElement *elem,const char *buffer)
{
  XSetSelectionOwner(dsp,XA_PRIMARY,elem->getWindow(),CurrentTime);
  if(elem->getWindow()==XGetSelectionOwner(dsp,XA_PRIMARY)) {
    cutstart=elem;
    clipboard_string = buffer;
  } else {
    cancelCut();
    return 1;
  }
  return 0;
}

void AGUIX::cancelCut()
{
  if(cutstart!=NULL) cutstart->cancelcut();
  cutstart=NULL;
  /*  if(cutbuffer!=NULL) _freesafe(cutbuffer);
      cutbuffer=NULL;*/

  m_last_selection_request_atom = None;
  m_last_selection_request_window = 0;
}

int AGUIX::startPaste(GUIElement *elem)
{
  pastestart=elem;
  return 0;
}

bool AGUIX::amiOwner() const
{
  if(cutstart!=NULL) return true;
  return false;
}

const char *AGUIX::getCutBuffer() const
{
    return clipboard_string.c_str();
}

void AGUIX::requestCut( Window win, bool fallback )
{
  Atom myproperty=XInternAtom(dsp,"Worker Paste Property",False);

  Atom string_atom = XA_STRING;
#ifndef DISABLE_UTF8_SUPPORT
  if ( ! fallback ) {
      if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED && UTF8::getCurrentEncoding() == UTF8::ENCODING_UTF8 ) {
          string_atom = XA_UTF8_STRING;
      }
  }
#endif

  XConvertSelection(dsp,
		    XA_PRIMARY,
		    string_atom,
		    myproperty,
		    win,
		    CurrentTime);

  m_last_selection_request_atom = string_atom;
  m_last_selection_request_window = win;
}

void AGUIX::DrawTriangleFilled(Drawable buffer, int px1, int py1, int px2, int py2, int px3, int py3)
{
  DrawTriangleFilled( buffer, 0, px1, py1, px2, py2, px3, py3 );
}

void AGUIX::DrawTriangleFilled(Drawable buffer, GC tgc, int px1, int py1, int px2, int py2, int px3, int py3)
{
  GC usegc;
  XPoint points[4];

  if( tgc == 0 )
    usegc = gc;
  else
    usegc = tgc;

  points[0].x = px1;
  points[0].y = py1;
  points[1].x = px2;
  points[1].y = py2;
  points[2].x = px3;
  points[2].y = py3;
  points[3].x = px1;
  points[3].y = py1;

  XFillPolygon( dsp, buffer, usegc, points, 4, Convex, CoordModeOrigin );
  XDrawLines( dsp, buffer, usegc, points ,4, CoordModeOrigin );
}

void AGUIX::cancelCutPaste(GUIElement *elem)
{
  if(cutstart==elem) {
    cancelCut();
  }
  if(pastestart==elem) {
    pastestart=NULL;
  }
}

bool AGUIX::isDoubleClick(struct timeval *t1,struct timeval *t2) const
{
  int dt=0;
  int s,us;

  s = abs( (int)( t1->tv_sec - t2->tv_sec ) );
  if ( s > 2 ) return false;
  us=t1->tv_usec-t2->tv_usec;
  if(us<0) {
    us+=1000000;
    s--;
  }
  dt=us+s*1000000;
  if(dt<350000) return true;
  return false;
}

bool AGUIX::isDoubleClick( Time t1, Time t2) const
{
  if ( abs( (int)( t2 - t1 ) ) < 350 ) return true;
  return false;
}

#define ADJUST_SPACE()				\
{						\
  w1 = 0;					\
  for ( i = 0; i < nr; i++ ) {			\
    w1 += elems[i]->getWidth();			\
  }						\
  						\
  sp = wantedWidth - w1 - borderwidth * 2;	\
  						\
  last = borderwidth;				\
  for ( i = 0; i < nr; i++ ) {			\
    widths[i] = elems[i]->getWidth();		\
    xpos[i] = last;				\
    if ( i < ( nr - 1 ) )			\
      tsp = sp / ( nr - 1 - i );		\
    else					\
      tsp = 0;					\
    sp -= tsp;					\
    last += widths[i] + tsp;			\
  }						\
}

#define ARRANGE_MINSPACE()			\
{						\
  last = borderwidth;				\
  for ( i = 0; i < nr; i++ ) {			\
    widths[i] = elems[i]->getWidth();		\
    xpos[i] = last;				\
    last += widths[i];				\
    if ( minSpace > 0 ) last += minSpace;	\
  }						\
}

/*
 * resize and pos all elems to fit given width
 *
 * args:
 *   wantedWidth: width for resize or <1 for any
 *   borderwidth: left and right border
 *                <0 means 0
 *   minSpace: minimal space between elems
 *             <0 means any
 *   maxSpace: maximal space between elems
 *             <0 means any
 *   allowShrink: true when shrinking is allowed
 *   allowStretch: true when stretching is allowed
 *   elems: Array of elements
 *   minWidths: minimal widths of the elements
 *   nr: Arraysize
 *
 * returnvalue:
 *   used width
 * used width = wantedWidth is NOT guaranteed
 *
 */
//TODO eigentlich obsolete
int AGUIX::scaleElementsW( int wantedWidth,
			   int borderwidth,
			   int minSpace,
			   int maxSpace,
			   bool allowShrink,
			   bool allowStretch,
			   GUIElement **elems,
			   int *minWidths,
			   int nr )
{
  //  GUIElement **elems;
  int cw, i, last, nw;
  int *widths, *xpos;
  int w1, w2, sp, tsp;
  int usedWidth = -1;

  // for ,... notation
  // nr = 0;
  // elems = &elem;
  // while ( elems[nr] != NULL )
  //   nr++;

  if ( elems == NULL ) return -1;
  if ( nr < 1 ) return -1;
  if ( ( minSpace >= 0 ) && ( maxSpace >= 0 ) && ( minSpace > maxSpace ) ) return -1;

  widths = new int[nr];
  xpos = new int[nr];

  if ( borderwidth < 0 ) borderwidth = 0;

  // calc the currently needed width
  cw = 2 * borderwidth;
  if ( nr > 1 )
    cw += ( nr - 1 ) * ( ( minSpace < 0 ) ? 0 : minSpace );
  
  for ( i = 0; i < nr; i++ ) {
    cw += elems[i]->getWidth();
  }
  
  if ( nr == 1 ) {
    // special case
    // center the only element
    // 2 possibilities (when cw<wantedWidth):
    //   1.:stretch never
    //   2.:stretch when allowStretch == true
    // currently I use the second one
    if ( wantedWidth < 1 ) {
      // align to border
      widths[0] = elems[0]->getWidth();
      xpos[0] = borderwidth;
    } else {
      if ( cw < wantedWidth ) {
	// stretching needed
	if ( allowStretch == false ) {
	  // place it in the center
	  widths[0] = elems[0]->getWidth();
	  xpos[0] = ( wantedWidth / 2 ) - ( widths[0] / 2 );
	  
	  // we cannot calc the used width later so set it here
	  usedWidth = wantedWidth;
	} else {
	  widths[0] = wantedWidth - 2 * borderwidth;
	  if ( widths[0] < 4 ) widths[0] = 4;
	  xpos[0] = borderwidth;
	}
      } else if ( cw > wantedWidth ) {
	// shrinking needed
	if ( allowShrink == false ) {
	  // not allowed
	  widths[0] = elems[0]->getWidth();
	  xpos[0] = borderwidth;
	} else {
	  widths[0] = wantedWidth - 2 * borderwidth;
	  if ( widths[0] < 4 ) widths[0] = 4;
	  if ( minWidths != NULL ) {
	    if ( widths[0] < minWidths[0] )
	      widths[0] = minWidths[0];
	  }
	  xpos[0] = borderwidth;
	}
      } else {
	// no changes needed
	widths[0] = elems[0]->getWidth();
	xpos[0] = borderwidth;
      }
    }
  } else {
    if ( wantedWidth < 1 ) {
      // means any width so just arrange them with minSpace
      ARRANGE_MINSPACE();
    } else {
      // try to get the elems to fit wantedWidth
      // prio: 1.change space between elems
      //       2.always stretch or shrink, not both
      
      if ( cw == wantedWidth ) {
	// nothing to do
	last = borderwidth;
	for ( i = 0; i < nr; i++ ) {
	  widths[i] = elems[i]->getWidth();
	  xpos[i] = last;
	  last += widths[i];
	  if ( minSpace > 0 ) last += minSpace;
	}
      } else if ( cw < wantedWidth ) {
	// stretching needed
	if ( maxSpace < 0 ) {
	  // no maxSpace set so just adjust space
	  ADJUST_SPACE();
	} else {
	  // first test if we need to stretch when using maxSpace
	  
	  w1 = 2 * borderwidth;
	  if ( nr > 1 )
	    w1 += ( nr - 1 ) * maxSpace;
	  
	  for ( i = 0; i < nr; i++ ) {
	    w1 += elems[i]->getWidth();
	  }
	  // w1 is now the width when using maxSpace
	  
	  if ( w1 > wantedWidth ) {
	    // we can adjust the space
	    ADJUST_SPACE();
	  } else {
	    // maxSpace isn't enough
	    // we have to stretch some elems
	    if ( allowStretch == false ) {
	      // not allowed
	      // we cannot give wantedWidth so use maxSpace
	      last = borderwidth;
	      for ( i = 0; i < nr; i++ ) {
		widths[i] = elems[i]->getWidth();
		xpos[i] = last;
		last += widths[i] + maxSpace;
	      }
	    } else {
	      bool *finished, change;
	      int tnr;

              finished = new bool[nr];
	      w1 = wantedWidth - ( nr - 1 ) * maxSpace;
	      w1 -= 2 * borderwidth;
	      tnr = nr;

	      for ( i = 0; i < nr; i++ ) {
		widths[i] = elems[i]->getWidth();
		finished[i] = false;
	      }
	      do {
		w2 = w1 / tnr;
		change = false;
		for ( i = 0; i < nr; i++ ) {
		  if ( finished[i] == false ) {
		    if ( widths[i] > w2 ) {
		      finished[i] = true;
		      w1 -= widths[i];
		      tnr--;
		      change = true;
		    }
		  }
		}
	      } while ( change == true );
	      for ( i = 0; i < nr; i++ ) {
		if ( finished[i] == false ) {
		  w2 = w1 / tnr;
		  widths[i] = w2;
		  w1 -= w2;
		  tnr--;
		}
	      }
	      // now calc the xpos
	      last = borderwidth;
	      for ( i = 0; i < nr; i++ ) {
		xpos[i] = last;
		last += widths[i] + maxSpace;
	      }
              delete [] finished;
	    }
	  }
	}
      } else {
	// shrinking needed
	// works different from stretch-case
	// because we calced the width need when using minSpace ( or 0 when <0)
	// so in any case we have to shrink the elems
	// maxSpace doesn't count at all
	if ( allowShrink == false ) {
	  // nothing we can do to fit wantedWidth
	  // just arrange them with minSpace
	  ARRANGE_MINSPACE();
	} else {
	  // again only shrink elems which are not smaller then the needed average

	  bool *finished, change;
	  int tnr;
          
          finished = new bool[nr];
	  w1 = wantedWidth - ( nr - 1 ) * ( minSpace < 0 ? 0 : minSpace );
	  w1 -= 2 * borderwidth;
	  
	  tnr = nr;
	  
	  for ( i = 0; i < nr; i++ ) {
	    widths[i] = elems[i]->getWidth();
	    finished[i] = false;
	  }

	  do {
	    w2 = w1 / tnr;
	    change = false;
	    for ( i = 0; i < nr; i++ ) {
	      if ( finished[i] == false ) {
		if ( widths[i] < w2 ) {
		  finished[i] = true;
		  w1 -= widths[i];
		  tnr--;
		  change = true;
		}
	      }
	    }
	  } while ( change == true );
	  for ( i = 0; i < nr; i++ ) {
	    if ( finished[i] == false ) {
	      w2 = w1 / tnr;
	      widths[i] = w2;
	      w1 -= w2;
	      tnr--;
	    }
	  }

	  if ( minWidths != NULL ) {
	    // now check for minWidths
	    // this algo is stupid, I know, but because we only have few elements
	    // I think it's okay

	    bool *finished2;
	    int *mws;
	    int nf, dis, tnf, tdis, t;
	    
            finished2 = new bool[nr];
            mws = new int[nr];
	    for ( i = 0; i < nr; i++ ) {
	      mws[i] = minWidths[i];
	      if ( mws[i] < 4 )
		mws[i] = 4;
	      finished2[i] = false;
	    }
	    
	    nf = nr;
	    do {
	      dis = 0;
	      for ( i = 0; i < nr; i++ ) {
		if ( widths[i] < mws[i] ) {
		  dis += mws[i] - widths[i];
		  widths[i] = mws[i];
		  finished2[i] = true;
		  nf--;
		}
	      }
	      if ( dis > 0 ) {
		// now distribute dis at all elements finished == false
		tdis = dis;
		tnf = nf;
		for ( i = 0; i < nr; i++ ) {
		  if ( finished2[i] == false ) {
		    t = tdis / tnf;
		    widths[i] -= t;
		    tnf--;
		    tdis -= t;
		  }
		}
		// now dis is distributed to all widths
		// but this means that some new elements could be too small
	      }
	    } while ( dis > 0 );
	    // dis will only be > 0 is there are too small elements
	    // because in each loop atleast one will be corrected this will terminated after n loops
            delete [] finished2;
            delete [] mws;
	  }

	  // now calc the xpos
	  last = borderwidth;
	  for ( i = 0; i < nr; i++ ) {
	    xpos[i] = last;
	    last += widths[i];
	    if ( minSpace > 0 ) last += minSpace;
	  }
          delete [] finished;
	}
      }
    } 
  }
  
  // now arrange the elements
  for ( i = 0; i < nr; i++ ) {
    elems[i]->move( xpos[i], elems[i]->getY() );
    elems[i]->resize( widths[i], elems[i]->getHeight() );
  }
  
  delete [] widths;
  delete [] xpos;

  if ( usedWidth >= 0 ) {
    return usedWidth;
  } else {
    nw = elems[nr - 1]->getX() + elems[nr - 1]->getWidth() + borderwidth;
    return nw;
  }
}

#undef ADJUST_SPACE
#undef ARRANGE_MINSPACE

int AGUIX::centerElementsY( GUIElement *element,
			    GUIElement *center2element )
{
  int h1,h2;

  if ( ( element == NULL ) || ( center2element == NULL ) )
    return -1;

  h1 = element->getHeight();
  h2 = center2element->getHeight();

  element->move( element->getX(),
		 center2element->getY() +
		 h2 / 2 -
		 h1 / 2 );
  return 0;
}

int AGUIX::getRootWindowWidth() const
{
  return rootWindowWidth;
}

int AGUIX::getRootWindowHeight() const
{
  return rootWindowHeight;
}

Pixmap AGUIX::createPixmap( Drawable d, int width, int height )
{
  if ( ( width < 1 ) || ( height < 1 ) || ( d == 0 ) ) return 0;
  
  return XCreatePixmap( dsp, d, width, height, DefaultDepth( dsp, scr ) );
}

void AGUIX::freePixmap( Pixmap p )
{
  if ( p == 0 ) return;
  XFreePixmap ( dsp, p );
}

void AGUIX::DrawDottedLine( Drawable buffer, int px1, int py1, int px2, int py2 )
{
  XDrawLine( dsp, buffer, dotted_gc, px1, py1, px2, py2 );
}

void AGUIX::DrawDottedRectangle( Drawable buffer, int x, int y, int w, int h )
{
  if ( ( w < 1 ) || ( h < 1 ) ) return;
  DrawDottedLine( buffer, x, y, x + w - 1, y );
  DrawDottedLine( buffer, x + w - 1, y, x + w - 1, y + h - 1 );
  DrawDottedLine( buffer, x + w - 1, y + h - 1, x, y + h - 1 );
  DrawDottedLine( buffer, x, y + h - 1, x, y );
}

void AGUIX::setDottedFG( const AGUIXColor &color )
{
  setFG( dotted_gc, color );
}

void AGUIX::DrawDashXorLine( Drawable buffer, int px1, int py1, int px2, int py2 )
{
  XDrawLine( dsp, buffer, dashxor_gc, px1, py1, px2, py2 );
}

void AGUIX::DrawDashXorRectangle( Drawable buffer, int x, int y, int w, int h )
{
  if ( ( w < 1 ) || ( h < 1 ) ) return;
  DrawDashXorLine( buffer, x, y, x + w - 1, y );
  DrawDashXorLine( buffer, x + w - 1, y, x + w - 1, y + h - 1 );
  DrawDashXorLine( buffer, x + w - 1, y + h - 1, x, y + h - 1 );
  DrawDashXorLine( buffer, x, y + h - 1, x, y );
}

void AGUIX::DrawDashDLine( Drawable buffer, int px1, int py1, int px2, int py2 )
{
  XDrawLine( dsp, buffer, dashdouble_gc, px1, py1, px2, py2 );
}

void AGUIX::DrawDashDRectangle( Drawable buffer, int x, int y, int w, int h )
{
  if ( ( w < 1 ) || ( h < 1 ) ) return;
  DrawDashDLine( buffer, x, y, x + w - 1, y );
  DrawDashDLine( buffer, x + w - 1, y, x + w - 1, y + h - 1 );
  DrawDashDLine( buffer, x + w - 1, y + h - 1, x, y + h - 1 );
  DrawDashDLine( buffer, x, y + h - 1, x, y );
}

void AGUIX::setDashDFG( const AGUIXColor &color )
{
  setFG( dashdouble_gc, color );
}

void AGUIX::setDashDBG( const AGUIXColor &color )
{
  setBG( dashdouble_gc, color );
}

bool AGUIX::isModifier( KeySym key )
{
  if ( IsModifierKey( key ) == true ) return true;

  // some additional modifiers not in IsModifierKey
  switch ( key ) {
  case XK_Multi_key:
    return true;
    break;
  }
  return false;
}

char *AGUIX::getStringForKeySym( KeySym key )
{
  char *tstr;

  tstr = XKeysymToString( key );
  if ( tstr != NULL ) {
    return dupstring( tstr );
  }
  return NULL;
}

KeySym AGUIX::getKeySymForString( const char *str1 )
{
  if ( str1 == NULL ) return NoSymbol;

  return XStringToKeysym( str1 );
}

void AGUIX::rebuildBackgroundPixmap()
{
  if ( backpm != None ) XFreePixmap( dsp, backpm );
  backpm = XCreatePixmap( dsp, DefaultRootWindow( dsp ), 2, 2, getDepth() );
  setFG( 0 );
  FillRectangle( backpm, 0, 0, 2, 2 );
  setFG( 2 );
  DrawPoint( backpm, 0, 1 );
  DrawPoint( backpm, 1, 0 );
}

void AGUIX::setWindowBackgroundPixmap( Window win )
{
  if ( backpm == None ) {
    rebuildBackgroundPixmap();
  }
  XSetWindowBackgroundPixmap( dsp, win, backpm );
}

void AGUIX::doXMsgs( AWindow *parent, bool onlyexpose )
{
  doXMsgs( MES_GET, parent, onlyexpose );
}

void AGUIX::doXMsgs( int mode, AWindow *parent, bool onlyexpose )
{
  if ( ( mode != MES_GET ) && ( mode != MES_WAIT ) ) mode = MES_GET;
  while ( msgHandler( mode, parent, onlyexpose ) > 0 );
}

bool AGUIX::noMoreMessages() const
{
    if ( messages.empty() == true ) return true;
    return false;
}

#ifdef USE_XIM
XIMStyle AGUIX::getXIMStyle() const
{
  return im_style;
}

XIM AGUIX::getXIM() const
{
  return inputmethod;
}

int AGUIX::openIM()
{
  XIMStyle supported_styles, style;
  XIMStyles *im_supported_styles;
  int i;
  
  inputmethod = XOpenIM( dsp, NULL, NULL, NULL );
  if ( inputmethod == NULL ) {
    fprintf( stderr, "Worker Warning: Cannot open input method\n" );
#ifdef XIM_XREGISTER_OKAY
    XRegisterIMInstantiateCallback( dsp, NULL, NULL, NULL, AGUIX_im_inst_callback, (XPointer)this );
#endif
  } else {
#ifdef XIM_XREGISTER_OKAY
    XIMCallback destCB;

    destCB.callback = AGUIX_im_dest_callback;
    destCB.client_data = (XPointer)this;
    if ( XSetIMValues( inputmethod, XNDestroyCallback, &destCB, (void*)NULL ) != NULL ) {
      fprintf( stderr, "Worker Warning: Couldn't set IM destroy callback\n" );
    }
#endif
    
    /* I could also "support" XIM...None but with these styles dead keys
       will be completly ignored (at least at my machine)
       But my code will fallback to normal XLookupString when my wanted styles
       are not supported and dead keys will be loockuped to their normal character
    */
    supported_styles = XIMPreeditNothing | XIMStatusNothing;
    XGetIMValues( inputmethod, XNQueryInputStyle, &im_supported_styles, (void*)NULL );
    im_style = 0;
    for ( i = 0; i < im_supported_styles->count_styles; i++ ) {
      style = im_supported_styles->supported_styles[i];
      if ( ( style & supported_styles ) == style ) {
	im_style = ChooseBetterStyle( style, im_style );
      }
    }
    
    if ( im_style == 0 ) {
#ifdef DEBUG
      fprintf( stderr, "Worker Warning: Wanted inputstyle not available\n");
#endif
    } else {
        for ( wins_cit_t it1 = wins.begin();
              it1 != wins.end();
              it1++ ) {
            (*it1)->createXIC();
        }
    }
    XFree( im_supported_styles );
  }
  return ( ( ( inputmethod != NULL ) && ( im_style != 0 ) ) ? 0 : 1 );
}

void AGUIX::closeIM()
{
  for ( wins_cit_t it1 = wins.begin();
        it1 != wins.end();
        it1++ ) {
      (*it1)->closeXIC();
  }
      
  im_style = 0;
  if ( inputmethod != NULL ) {
    XCloseIM( inputmethod );
    inputmethod = NULL;
  }
}

void AGUIX::IMInstCallback( Display *calldsp )
{
  if ( calldsp != dsp ) return; // not my display

#ifdef XIM_XREGISTER_OKAY
  XUnregisterIMInstantiateCallback( dsp, NULL, NULL, NULL, AGUIX_im_inst_callback, (XPointer)this );
#endif
  openIM();
}

void AGUIX::IMDestCallback()
{
    for ( wins_cit_t it1 = wins.begin();
          it1 != wins.end();
          it1++ ) {
        (*it1)->XICdestroyed();
    }
    
    im_style = 0;
    inputmethod = NULL;
}

#endif

/*
 * set transient window variable
 *
 * apply only when the window is registered
 * clear variable with NULL arg
 */
void AGUIX::setTransientWindow( AWindow *twin )
{
  if ( twin == NULL ) {
    transientwindow = NULL;
  } else {
      wins_cit_t it1 = std::find( wins.begin(), wins.end(), twin );
      if ( it1 != wins.end() ) {
          // window found
          transientwindow = twin;

          wins_as_transient_for.remove( twin );
          wins_as_transient_for.push_back( twin );
      }
  }
}

const AWindow* AGUIX::getTransientWindow() const
{
  return transientwindow;
}

int AGUIX::addDefaultColors()
{
    if ( AddColor( 180, 180, 180 ).getColor() < 0 ) return -1;
    if ( AddColor( 0, 0, 0 ).getColor() < 0 ) return -1;
    if ( AddColor( 255, 255, 255 ).getColor() < 0 ) return -1;
    if ( AddColor( 0, 85, 187).getColor() < 0 ) return -1;
    if ( AddColor( 204, 34, 0 ).getColor() < 0 ) return -1;
    if ( AddColor( 50, 180, 20 ).getColor() < 0 ) return -1;
    if ( AddColor( 119, 0, 119 ).getColor() < 0 ) return -1;
    if ( AddColor( 238, 170, 68 ).getColor() < 0 ) return -1;
    return 0;
}

/*
 * the following two functions returns the last key/mouse and reset
 * the variables
 * the variables are set by doXMsg currently
 */
KeySym AGUIX::getLastKeyRelease()
{
  KeySym k;

  k = lastkeyrelease;
  lastkeyrelease = None;
  return k;
}

unsigned int AGUIX::getLastMouseRelease()
{
  unsigned int m;

  m = lastmouserelease;
  lastmouserelease = 0;
  return m;
}

void AGUIX::msgLock( Widget *e )
{
  msgLockElement = e;
}

void AGUIX::msgUnlock( const Widget *e )
{
  if ( msgLockElement != e ) {
    fprintf( stderr, "Worker:wrong msgUnlock\n" );
  }
  msgLockElement = NULL;
}

bool AGUIX::msgHoldsLock( const Widget *e )
{
    if ( msgLockElement == e )
        return true;
    return false;
}

void AGUIX::enableTimer()
{
#ifdef USE_AGUIXTIMER
  struct itimerval itv;
#endif

  if ( timerEnabled == false ) {
    timerEnabled = true;
#ifndef USE_AGUIXTIMER
    gettimeofday( &timerStart, NULL );
    lastTimerEventNr = 0;
#else
    itv.it_value.tv_sec = itv.it_interval.tv_sec = 0;
    itv.it_value.tv_usec = itv.it_interval.tv_usec = 1000000 / TIMER_TICKS;
    setitimer( ITIMER_REAL, &itv, NULL );
    lastTimerEventNr = timerEvent;
#endif
  }
}

void AGUIX::disableTimer()
{
#ifdef USE_AGUIXTIMER
  struct itimerval itv;
#endif

  if ( timerEnabled == true ) {
#ifdef USE_AGUIXTIMER
    itv.it_value.tv_sec = itv.it_interval.tv_sec = 0;
    itv.it_value.tv_usec = itv.it_interval.tv_usec = 0;
    setitimer( ITIMER_REAL, &itv, NULL );
#endif
  }
  timerEnabled = false;
}

int AGUIX::getTextWidth( const char *str, AGUIXFont *font )
{
  return getTextWidth( str, font, -1 );
}

int AGUIX::getTextWidth( const char *str, AGUIXFont *font, int len )
{
  AGUIXFont *use_font;

  if ( str == NULL ) return 0;

  if ( font != NULL ) {
    use_font = font;
  } else {
    use_font = mainfont;
  }
  if ( use_font == NULL ) return 0;

  if ( len < 0 ) {
      len = strlen( str );
  } else {
      if ( AGUIXUtils::stringIsShorter( str, len ) == true ) {
          len = strlen( str );
      }
  }

  return use_font->textWidth( str, len );
}

int AGUIX::getStrlen4Width( const char *str, int width, int *return_width, AGUIXFont *font )
{
    return getStrlen4WidthMaxlen( str, -1, width, return_width, font );
}

int AGUIX::getStrlen4WidthMaxlen( const char *str, int maxlen,
                                  int width, int *return_width, AGUIXFont *font )
{
  int leftlen, leftw, rightlen, rightw, erglen, ergw, midlen, midw;
  AGUIXFont *use_font;

  if ( font != NULL ) {
    use_font = font;
  } else {
    use_font = mainfont;
  }
  if ( use_font == NULL ) return 0;

  if ( str == NULL || use_font == NULL ) {
    if ( return_width != NULL ) *return_width = 0;
    return 0;
  }

  std::vector<int> lookup_list;

  if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED ) {  
    if ( UTF8::buildCharacterLookupListMaxlen( str, maxlen, lookup_list ) != 0 ) {
      if ( return_width != NULL ) *return_width = 0;
      return 0;
    }
    
    if ( lookup_list.size() < 1 ) {
      if ( return_width != NULL ) *return_width = 0;
      return 0;
    }
  }

  leftlen = leftw = 0;

  if ( maxlen < 0 ) {
      rightlen = strlen( str );
  } else {
      if ( AGUIXUtils::stringIsShorter( str, maxlen ) == true ) {
          rightlen = strlen( str );
      } else {
          rightlen = maxlen;
      }
  }
  
  /* check if length is too large, some X libs have problems with too large strings
     a byte (or character) needs at least 2 pixel so width/2 is a good upper limit
  */

  if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED ) {
      int all_characters = lookup_list.size() - 1;
      if ( all_characters > ( width / 2 ) && width > 0 ) {
          // rightlen is supposed to be number of bytes
          rightlen = lookup_list[width / 2];
      }
  } else {
      if ( rightlen > ( width / 2 ) && width > 0 ) {
          rightlen = width / 2;
      }
  }
  rightw = use_font->textWidth( str, rightlen );

  if ( leftw >= width ) {
    erglen = leftlen;
    ergw = leftw;
  } else if ( rightw <= width ) {
    erglen = rightlen;
    ergw = rightw;
  } else {
    // from now on leftw is always less than width
    // and rightw always more than width

    // in UTF8 mode length means characters and not bytes!
    if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED ) {
      rightlen = lookup_list.size() - 1;
      
      // check again for pixel limit
      if ( rightlen > ( width / 2 ) && width > 0 ) {
          rightlen = width / 2;
      }
    }

    erglen = ergw = 0;
    for (;;) {
      midlen = ( leftlen + rightlen ) / 2;
      if ( ( midlen == leftlen ) || ( midlen == rightlen ) ) {
        erglen = leftlen;
	ergw = leftw;
	break;
      }

      if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED ) {
        midw = use_font->textWidth( str, lookup_list[midlen] );
      } else {
        midw = use_font->textWidth( str, midlen );
      }
      if ( midw == width ) {
	erglen = midlen;
	ergw = midw;
	break;
      }
      
      if ( midw < width ) {
	leftlen = midlen;
	leftw = midw;
      } else {
	rightlen = midlen;
	rightw = midw;
      }
    }
    if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED ) {
      if ( erglen > 0 ) {
        // erglen is the number of characters
        // the lookup list stores the number of bytes
        erglen = lookup_list[erglen];
      }
    }
  }

  if ( return_width != NULL ) *return_width = ergw;
  return erglen;
}

void AGUIX::setClip( AGUIXFont *font, DrawableCont *dc, int x, int y, int width, int height )
{
  if ( dc != NULL ) {
    dc->clip( x, y, width, height );
  }
  if ( font != NULL ) {
    font->clip( x, y, width, height );
  } else {
    mainfont->clip( x, y, width, height );
    
    XRectangle clip_rect;
    
    clip_rect.x = x;
    clip_rect.y = y;
    clip_rect.width = width;
    clip_rect.height = height;
    
    XSetClipRectangles( dsp, gc, 0, 0, &clip_rect, 1, Unsorted );
  }
}

void AGUIX::unclip( AGUIXFont *font, DrawableCont *dc )
{
  if ( dc != NULL ) {
    dc->unclip();
  }
  if ( font != NULL ) {
    font->unclip();
  } else {
    mainfont->unclip();
    XSetClipMask( dsp, gc, None );
  }
}

void AGUIX::drawBorder( Drawable buffer, bool pressed, int x, int y, int w, int h, int topright_space )
{
  drawBorder( buffer, gc, pressed, x, y, w, h, topright_space );
}

void AGUIX::drawBorder( Drawable buffer, GC usegc, bool pressed, int x, int y, int w, int h, int topright_space )
{
  XPoint wp[3], bp[3], gp[3];

  if ( w < 4 ) w = 4;
  if ( h < 4 ) h = 4;

  if ( topright_space > ( w - 2 ) ) topright_space = w - 2;
  if ( topright_space > ( h - 2 ) ) topright_space = h - 2;
  if ( topright_space < 0 ) topright_space = 0;

  if ( pressed == true ) {
    wp[0].x = x + 1;
    wp[0].y = y + h - 1;
    wp[1].x = x + w - 1;
    wp[1].y = y + h - 1;
    wp[2].x = x + w - 1;
    wp[2].y = y + topright_space + 1;
    
    bp[0].x = x;
    bp[0].y = y + h - 1;
    bp[1].x = x;
    bp[1].y = y;
    bp[2].x = x + w - 1 - topright_space;
    bp[2].y = y;
    
    gp[0].x = x + 1;
    gp[0].y = y + h - 2;
    gp[1].x = x + 1;
    gp[1].y = y + 1;
    gp[2].x = x + w - 2 - topright_space;
    gp[2].y = y + 1;
  } else {
    wp[0].x = x;
    wp[0].y = y + h - 2;
    wp[1].x = x;
    wp[1].y = y;
    wp[2].x = x + w - 2 - topright_space;
    wp[2].y = y;
    
    bp[0].x = x;
    bp[0].y = y + h - 1;
    bp[1].x = x + w - 1;
    bp[1].y = y + h - 1;
    bp[2].x = x + w - 1;
    bp[2].y = y + topright_space;
    
    gp[0].x = x + 1;
    gp[0].y = y + h - 2;
    gp[1].x = x + w - 2;
    gp[1].y = y + h - 2;
    gp[2].x = x + w - 2;
    gp[2].y = y + topright_space + 1;
  }
  
  setFG( usegc, 2 );
  DrawLine( buffer, usegc, wp[0].x, wp[0].y, wp[1].x, wp[1].y );
  DrawLine( buffer, usegc, wp[1].x, wp[1].y, wp[2].x, wp[2].y );

  setFG( usegc, 1 );
  DrawLine( buffer, usegc, bp[0].x, bp[0].y, bp[1].x, bp[1].y );
  DrawLine( buffer, usegc, bp[1].x, bp[1].y, bp[2].x, bp[2].y );

  // dark grey
  setFG( usegc, AGUIXColor( 1, AGUIXColor::SYSTEM_COLOR ) );
  DrawLine( buffer, usegc, gp[0].x, gp[0].y, gp[1].x, gp[1].y );
  DrawLine( buffer, usegc, gp[1].x, gp[1].y, gp[2].x, gp[2].y );
}

Atom AGUIX::getAtom( atom_name_t name ) const
{
    switch ( name ) {
        case CLOSE_ATOM:
            return WM_delete_window;
        case NET_WM_NAME_ATOM:
            return XA_NET_WM_NAME;
        case NET_WM_ICON_NAME_ATOM:
            return XA_NET_WM_ICON_NAME;
        case TARGETS_ATOM:
            return XA_TARGETS;
        case NET_WM_WINDOW_TYPE:
            return XA_NET_WM_WINDOW_TYPE;
        case NET_WM_WINDOW_TYPE_DIALOG:
            return XA_NET_WM_WINDOW_TYPE_DIALOG;
        case NET_WM_WINDOW_TYPE_NORMAL:
            return XA_NET_WM_WINDOW_TYPE_NORMAL;
        case UTF8_STRING:
            return XA_UTF8_STRING;
    }
    return None;
}

void AGUIX::setColorForName( const std::string &name, int col )
{
    _colors_for_widgets[name] = col;
}

int AGUIX::getColorForName( const std::string &name )
{
    if ( _colors_for_widgets.find( name ) == _colors_for_widgets.end() )
        return -1;
    return _colors_for_widgets[name];
}

void AGUIX::registerPopUpWindow( PopUpWindow *win )
{
    if ( win == NULL ) return;

    popup_wins.push_back( win );
}

void AGUIX::checkPopUpWindows( Message *msg )
{
    if ( msg == NULL ) return;

    if ( m_open_popup_counter < 1 ) return;
    if ( msg->type != ButtonPress && msg->type != ButtonRelease ) return;

    // check whether the event window is owned by some of the registered popups
    //  if there is a popup, get its groupid and close all other popups
    //  if not, close all popups

    if ( m_popup_ignore_button_release == true ) {
        m_popup_ignore_button_release = false;

        if ( msg->type == ButtonRelease ) {
            return;
        }
    }

    PopUpWindow *win_for_msg = NULL;

    for ( popup_wins_cit_t it1 = popup_wins.begin();
          it1 != popup_wins.end();
          it1++ ) {
        if ( (*it1)->isParent( msg->window ) == true ) {
            win_for_msg = *it1;
            break;
        }
    }

    if ( win_for_msg == NULL ) {
        // not a message for a popup window
        // so hide all popups
        for ( popup_wins_cit_t it1 = popup_wins.begin();
              it1 != popup_wins.end();
              it1++ ) {
            (*it1)->hide();
        }
    } else {
        // found a popup window for the message
        // so hide all popups which are not in the same group
        hideOtherPopUpWindows( win_for_msg->getGroupID() );
    }
}

void AGUIX::hidePopUpWindows( int group_id )
{
    for ( popup_wins_cit_t it1 = popup_wins.begin();
          it1 != popup_wins.end();
          it1++ ) {
        if ( (*it1)->getGroupID() == group_id ) {
            (*it1)->hide();
        }
    }
}

void AGUIX::hideOtherPopUpWindows( int group_id )
{
    for ( popup_wins_cit_t it1 = popup_wins.begin();
          it1 != popup_wins.end();
          it1++ ) {
        if ( (*it1)->getGroupID() != group_id ) {
            (*it1)->hide();
        }
    }
}

void AGUIX::xSync()
{
    XSync( dsp, False );
}

void AGUIX::popupOpened()
{
    m_open_popup_counter++;

    if ( m_open_popup_counter == 1 ) {
        // first popup opened, so ignore next button release
        m_popup_ignore_button_release = true;
    }
}

void AGUIX::popupClosed()
{
    m_open_popup_counter--;
    if ( m_open_popup_counter < 0 ) {
        fprintf( stderr, "Worker:bug in popup counter\n" );
        m_open_popup_counter = 0;
    }
}

int AGUIX::getWidgetRootPosition( Widget *widget, int *x, int *y )
{
    if ( widget == NULL || x == NULL || y == NULL ) return 1;
    
    int tx, tx2, ty, ty2;
    
    //TODO this is a rather ugly hack, better walk through window hierarchy with
    // XQueryTree and querying the corresponding postions
    queryPointer( widget->getWindow(), &tx, &ty );
    queryRootPointer( &tx2, &ty2 );
    
    tx2 -= tx;
    ty2 -= ty;
    
    *x = tx2;
    *y = ty2;
    
    return 0;
}

void AGUIX::updateSystemColors( int changed_user_color )
{
    if ( changed_user_color == 0 || changed_user_color == 1 ) {
        if ( m_color_values.count( AGUIXColor( 0, AGUIXColor::USER_COLOR ) ) > 0 ) {
            int r1 = m_color_values[AGUIXColor( 0, AGUIXColor::USER_COLOR )].red,
                g1 = m_color_values[AGUIXColor( 0, AGUIXColor::USER_COLOR )].green,
                b1 = m_color_values[AGUIXColor( 0, AGUIXColor::USER_COLOR )].blue;
            
            int r2 = 0, g2 = 0, b2 = 0;

            if ( m_color_values.count( AGUIXColor( 1, AGUIXColor::USER_COLOR ) ) > 0 ) {
                r2 = m_color_values[AGUIXColor( 1, AGUIXColor::USER_COLOR )].red,
                g2 = m_color_values[AGUIXColor( 1, AGUIXColor::USER_COLOR )].green,
                b2 = m_color_values[AGUIXColor( 1, AGUIXColor::USER_COLOR )].blue;
            }

            // let the first system color be 2/3 between user color 0 and
            // user color 1 (if available); usually this is some darker gray
            r2 = ( r1 * 2 + r2 ) / 3;
            g2 = ( g1 * 2 + g2 ) / 3;
            b2 = ( b1 * 2 + b2 ) / 3;

            if ( m_system_colors < 1 ) {
                AddColor( r2, g2, b2, AGUIXColor::SYSTEM_COLOR );
            } else {
                changeColor( AGUIXColor( 0, AGUIXColor::SYSTEM_COLOR ), r2, g2, b2 );
            }

            // let the second system color be 9/10 of user color 0
            r2 = ( r1 * 9 ) / 10;
            g2 = ( g1 * 9 ) / 10;
            b2 = ( b1 * 9 ) / 10;

            if ( m_system_colors < 2 ) {
                AddColor( r2, g2, b2, AGUIXColor::SYSTEM_COLOR );
            } else {
                changeColor( AGUIXColor( 1, AGUIXColor::SYSTEM_COLOR ), r2, g2, b2 );
            }
        }
    }
}

bool AGUIX::getLastTypedWindowEvent( Window win, int type, XEvent *return_event )
{
    if ( return_event == NULL ) return false;
    
    int count = 0;

    while ( XCheckTypedWindowEvent( dsp, win, type, return_event ) ) count++;

    if ( count == 0 ) return false;
    return true;
}

AWindow *AGUIX::getFocusedAWindow()
{
    if ( m_current_xfocus_window == 0 ) return NULL;

    return findAWindow( m_current_xfocus_window );
}

void AGUIX::registerBGHandler( AWindow *window, const RefCount< BackgroundMessageHandler > &handler )
{
    if ( ! window ) return;

    if ( m_bg_handlers.count( window ) > 0 ) {
        m_bg_handler_cleanup_list.push_back( m_bg_handlers[ window ] );
    }
    m_bg_handlers[ window ] = handler;
}

void AGUIX::unregisterBGHandler( AWindow *window )
{
    if ( ! window ) return;

    /* since the handler may remove it during its execution we still need a valid
     *  handle so we first move it to a cleanup list which gets clear after execution
     */
    if ( m_bg_handlers.count( window ) > 0 ) {
        m_bg_handler_cleanup_list.push_back( m_bg_handlers[ window ] );
        m_bg_handler_erase_handler.push_back( window );
    }
}

void AGUIX::executeBGHandlers( AGMessage &msg )
{
    std::map< AWindow *, RefCount< BackgroundMessageHandler > >::iterator it1;

    /* TODO check whether window still exists */
    for ( it1 = m_bg_handlers.begin();
          it1 != m_bg_handlers.end();
          it1++ ) {
        it1->second->handleAGUIXMessage( msg );
    }
    m_bg_handler_cleanup_list.clear();

    for ( std::list< AWindow *>::iterator it2 = m_bg_handler_erase_handler.begin();
	  it2 != m_bg_handler_erase_handler.end();
	  it2++ ) {
        m_bg_handlers.erase( *it2 );
    }
    m_bg_handler_erase_handler.clear();
}

/*
 * method to clean up BG handler
 * their destructor will be called so they hopefully release their resources
 */
void AGUIX::destroyBGHandlers()
{
    m_bg_handlers.clear();
    m_bg_handler_cleanup_list.clear();
}

void AGUIX::copyToClipboard( const std::string &s )
{
    clipboard_string = s;

    XStoreBytes( dsp, s.c_str(), s.length() );
    XSetSelectionOwner( dsp, XA_PRIMARY, getGroupWin(), CurrentTime );
    if ( getGroupWin() != XGetSelectionOwner( dsp, XA_PRIMARY ) ) {
        clipboard_string = "";
    }
}

void AGUIX::getLargestDimensionOfCurrentScreen( int *x, int *y,
                                                int *width, int *height )
{
#ifdef HAVE_XINERAMA
    XineramaScreenInfo *xinerama_info;
    int number_of_xinerama_screens = 0;
    int mx, my;
    bool found = false;

    queryRootPointer( &mx, &my );

    xinerama_info = XineramaQueryScreens( dsp,
                                          &number_of_xinerama_screens );

    if ( xinerama_info != NULL ) {
        int l_x = 0, l_y = 0, l_w = 0, l_h = 0;
        for ( int i = 0; i < number_of_xinerama_screens; i++ ) {
            if ( mx >= xinerama_info[i].x_org &&
                 my >= xinerama_info[i].y_org &&
                 mx < xinerama_info[i].x_org + xinerama_info[i].width &&
                 my < xinerama_info[i].y_org + xinerama_info[i].height ) {
                if ( xinerama_info[i].width > l_w &&
                     xinerama_info[i].height > l_h ) {
                    l_x = xinerama_info[i].x_org;
                    l_y = xinerama_info[i].y_org;
                    l_w = xinerama_info[i].width;
                    l_h = xinerama_info[i].height;
                    found = true;
                }
            }
        }

        XFree( xinerama_info );

        if ( found ) {
            if ( x ) *x = l_x;
            if ( y ) *y = l_y;
            if ( width ) *width = l_w;
            if ( height ) *height = l_h;
            return;
        }
    }
#endif

    if ( x ) *x = 0;
    if ( y ) *y = 0;
    if ( width ) *width = getRootWindowWidth();
    if ( height ) *height = getRootWindowHeight();
}

bool AGUIX::rerequestCut()
{
    if ( m_last_selection_request_atom != XA_STRING ) {
        // try again for fallback
        requestCut( m_last_selection_request_window, true );
        return true;
    }

    return false;
}

void AGUIX::setApplyWindowDialogType( bool nv )
{
    m_apply_window_dialog_type = nv;
}

bool AGUIX::getApplyWindowDialogType() const
{
    return m_apply_window_dialog_type;
}

void AGUIX::setOverrideXIM( bool nv )
{
    m_override_xim = nv;
}

bool AGUIX::getOverrideXIM() const
{
    return m_override_xim;
}

void AGUIX::setSkipFilterEvent( bool nv )
{
    m_skip_filter_event = true;
}
