/*
 * Copyright (c) 2011 Mathijs Henquet
 *
 * Marlin is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * Marlin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; see the file COPYING.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *      Mathijs Henquet <mathijs.henquet@gmail.com>
 *      ammonkey <am.monkeyd@gmail.com>
 *
 */

/* 
 * ButtonWithMenu 
 * - support long click / right click with depressed button states
 * - activate a GtkAction if any or popup a menu.
 * (used in history navigation buttons next/prev, appmenu) 
 */

using Gdk;
using Gtk;

namespace Varka.Widgets {

    public class ButtonWithMenu : ToggleButton 
    {
        /* delegate function used to populate menu */
        public delegate Gtk.Menu MenuFetcher();

        private int long_press_time = Gtk.Settings.get_default ().gtk_double_click_time * 2;
        public ulong toggled_sig_id;
        private int timeout = -1;
        private uint last_click_time = -1;

        private bool has_fetcher = false;
        private unowned MenuFetcher _fetcher;
        public MenuFetcher fetcher{
            set{
                _fetcher = value;
                has_fetcher = true;
            }
            get{
                return _fetcher;
            }
        }

        private Gtk.Menu _menu;
        public Gtk.Menu menu {
            get {
                return _menu;
            }
            set {
                if (has_fetcher) {
                    warning ("Don't set the menu property on a ToolMenuButton when there is allready a menu fetcher");
                } else {
                    _menu = value;
                    update_menu_properties();
                }
            }
        }

        public ButtonWithMenu.from_action (Gtk.Action action)
        {
            this ();
            //use_action_appearance = true;
            set_related_action (action);
            /* set_related_action doesn t do much on a Button compared to a ToolItem */
            if (action.stock_id != null) {
                /* to set an image we need a null label */
                label = null;
                var img = new Image.from_icon_name (action.stock_id, IconSize.MENU);
                img.set_tooltip_text (action.tooltip);
                set_image (img);
            }

            action.connect_proxy (this);
            related_action = action;
        }
        
        public ButtonWithMenu.from_params (Image _image, string? _label, string? _tooltip, Gtk.Menu? _menu)
        {
            this ();
            image = _image;
            label = _label;
            set_tooltip_text (_tooltip);

            menu = _menu;
        }

        public ButtonWithMenu ()
        {
            use_underline = true;
            can_focus = true;

            if (menu == null)
                menu = new Gtk.Menu ();

            mnemonic_activate.connect (on_mnemonic_activate);

            events |= EventMask.BUTTON_PRESS_MASK
                   |  EventMask.BUTTON_RELEASE_MASK;

            button_press_event.connect (on_button_press_event);
            button_release_event.connect (on_button_release_event);
        }

        private void update_menu_properties ()
        {
            menu.attach_to_widget (this, null);
            menu.deactivate.connect(() => {
                deactivate_menu ();
            });
            menu.deactivate.connect (popdown_menu);
        }

        public override void show_all (){
            menu.show_all ();
            base.show_all ();
        }

        private void deactivate_menu () {
            if (related_action != null)
                related_action.block_activate ();
            active = false;
            if (related_action != null)
                related_action.unblock_activate ();
        }

        private void popup_menu_and_depress_button (Gdk.EventButton ev) {
            if (related_action != null)
                related_action.block_activate ();
            active = true;
            if (related_action != null)
                related_action.unblock_activate ();
            popup_menu(ev);
        }

        private bool on_button_release_event (Gdk.EventButton ev)
        {
            if (ev.time - last_click_time < long_press_time) {
                if (related_action != null) {
                    related_action.activate ();
                } else {
                    active = true;
                    popup_menu (ev);
                }
            }

            if(timeout != -1){
                Source.remove((uint) timeout);
                timeout = -1;
            }

            return true;
        }

        private bool on_button_press_event (Gdk.EventButton ev)
        {
            if(timeout == -1 && ev.button == 1){
                last_click_time = ev.time;
                timeout = (int) Timeout.add (long_press_time, () => {
                    /* long click */
                    timeout = -1;
                    popup_menu_and_depress_button (ev);
                    return false;
                });
            }

            if(ev.button == 3){
                if (related_action != null)
                    popup_menu_and_depress_button (ev); 
                else {
                    /* Transmit the right click to the toolshell parent (ex: a toolbar) */
                    var par = get_parent ();
                    if (par is ToolItem) {
                        par = par.get_parent ();
                        if (par is ToolShell)
                            par.button_press_event (ev);
                    }
                }
            }

            return true;
        }

        private bool on_mnemonic_activate (bool group_cycling)
        {
            //stdout.printf ("on mnemonic activate\n");
            // ToggleButton always grabs focus away from the editor,
            // so reimplement Widget's version, which only grabs the
            // focus if we are group cycling.
            if (!group_cycling) {
                activate ();
            } else if (can_focus) {
                grab_focus ();
            }

            return true;
        }

        protected new void popup_menu (Gdk.EventButton? ev = null)
        {
            if(has_fetcher) fetch_menu ();

            try {
                menu.popup (null,
                            null,
                            get_menu_position,
                            (ev == null) ? 0 : ev.button,
                            (ev == null) ? get_current_event_time () : ev.time);
            } finally {
                menu.select_first (true);
            }
        }

        protected void popdown_menu ()
        {
            menu.popdown ();
        }

        private void fetch_menu(){
            _menu = fetcher ();
            update_menu_properties ();
        }

        private void get_menu_position (Gtk.Menu menu, out int x, out int y, out bool push_in)
        {
            Allocation menu_allocation;
            menu.get_allocation (out menu_allocation);
            if (menu.attach_widget == null ||
                menu.attach_widget.get_window () == null) {
                // Prevent null exception in weird cases
                x = 0;
                y = 0;
                push_in = true;
                return;
            }

            menu.attach_widget.get_window().get_origin (out x, out y);
            Allocation allocation;
            menu.attach_widget.get_allocation (out allocation);

            x += allocation.x;
            x -= menu_allocation.width/2;
            x += allocation.width/2;
            y += allocation.y;

            int width, height;
            menu.get_size_request (out width, out height);

            if (y + height >= menu.attach_widget.get_screen ().get_height ())
                y -= height;
            else
                y += allocation.height;

            push_in = true;
        }
    }
}

