#!/usr/bin/python
# -*- coding: utf-8 -*-

# main.py from the USBManager Package
#
# Copyright (c) 2008 Magnun Leno da Silva
#
# Author: Magnun Leno da Silva <magnun.leno@gmail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
# USAA.

import dbus
import gui
import usbdbus

__version__ = "1.0rc"
__authors__ = [ "Magnun Leno da Silva <magnun.leno@gmail.com>"]
__license__ = "LGPL"

RESPONSE_ACCEPT = -3
RESPONSE_CANCEL = -2

SUPPORTED_FS = ["FAT16", "FAT32", "NTFS"]

class Main:
    '''
        Main is the main class from the USBManager.
        It's responsible to concentrates all the functions connected to
        the DBus signals, stores the instances of TrayApplet, MainWindow
        and PropertiesWindow and controls the 'mode' of the USBManager
    '''
    def __init__(self, mode=1, arg=None):
        '''
            Start up the Manager
             @ mode - Can indicate if it will run as a single main
                      window (1), with a status (2) or only the
                      properties window (3). Th default value is 1.
             @ arg - arg indicates the block device and is only
                     necessary if the mode is set to 3.
        '''
        # Starts DBus
        print "Starting DBus..."
        self.bus = dbus.SystemBus()
        # Starts Devices
        self.devices = usbdbus.Devices(self)
        self.devices.connect('device-added', self.__on_new_device)
        self.devices.connect('device-changed', self.__on_device_changed)
        self.devices.connect('device-removed', self.__on_device_removed)
        self.devices.connect('notify-blocked-device', self.__on_notify_blocked_device)
        self.devices.connect('mounting-device-error', self.__on_mounting_device_error)
        self.devices.connect('umounting-device-error', self.__on_umounting_device_error)
        self.arg = arg

        # Windows 'controllers'
        self.main_window = None
        self.properties_window = None
        self.formating_window = None
        self.disk_usage_window = None
        self.tray = None
        self.mode = None
        if mode is 2:
            # Open SystemTray
            self.mode = 2
        elif mode is 3:
            # Open Properties Window
            self.mode = 3
        else:
            # Open Main Window (Default option)
            self.mode =1

    def run(self):
        if self.mode is 1:
            print 'Running Main Window (mode 1)'
            self.create_main_window()
            self.main_window.run()

        elif self.mode is 2:
            print 'Running SystemTray (mode 2)'
            self.create_system_tray()

        elif self.mode is 3:
            device = None
            for udi, dev in self.devices:
                if dev.block_device == self.arg:
                    device = dev
            if device is None:
                raise ValueError, " ** Error! Device not found. Quiting..."+ self.arg
                exit(-1)
            print 'Running Properties Window (mode 2)'
            self.create_properties_window(device)

    ######## Device Functions ########
    def __on_new_device(self, devices, new_device):
        if new_device is None:
            return
        if self.tray:
            self.tray.add_new_device(new_device)

        if self.main_window:
            self.main_window.add_new_device(new_device)

    def __on_device_removed(self, devices, removed_device):
        print ' * Device Removed:', removed_device.label, "("+removed_device.block_device+")"
        if self.tray:
            self.tray.remove_device(removed_device)

        if self.main_window:
            self.main_window.remove_device(removed_device)

        if self.properties_window and self.properties_window.udi == removed_device.udi:
            self.properties_window.destroy()

        if self.formating_window and self.formating_window.udi == removed_device.udi:
            self.formating_window.destroy()

    def __on_device_changed(self, devices, device):
        if device is None:
            return
        # Finish me
        if self.tray:
            self.tray.refresh_single_device(device.udi)

        if self.properties_window and self.properties_window.udi == device.udi:
            self.properties_window.update_fields()

    def __on_notify_blocked_device(self, devices, device):
        gui.messages.BlockedDevice()

    def __on_mounting_device_error(self, devices, device, msg):
        gui.messages.MountingError(device, msg)

    def __on_umounting_device_error(self, devices, device, msg):
        gui.messages.UmountingError(device, msg)

    def change_device_label(self, device, new_label):
        if usbdbus.CAN_SUDO is False:
            gui.messages.CantSudo()
            return False
        if device.is_renamable():
            if device.is_mounted:
                gui.messages.UmountingDevice(device)
                ret = device.umount()
                if ret is False:
                    return False

            gui.messages.BlockingDevice()
            ret, error = device.set_label(new_label)
            print 'change error',ret, error
            if not ret:
                gui.messages.ErrorDuringRename(device, error)
                return False
            return True
        gui.messages.DeviceNotRenamable(device)
        return False

    ######## Windows functions ########
    def create_system_tray(self):
        self.tray = gui.tray.TrayManager(self)
        self.tray.connect("destroying-window", self.__on_destroy_window)
        self.tray.run()

    def create_main_window(self):
        if self.main_window is None:
            self.main_window = gui.windows.MainWindow(self)
            self.main_window.connect('open-properties-window', self.__on_open_properties_window)
            self.main_window.connect('open-format-window', self.__on_open_format_window)
            self.main_window.connect('destroying-window', self.__on_destroy_window)
            self.main_window.connect('open-disk-usage-window', self.__on_open_disk_usage_window)
        else:
            gui.MainWindowAlreadyOpened()
            # TODO: Try to give focus if pressed

    def __on_open_properties_window(self, widget, device):
        self.properties_window = gui.windows.PropertiesWindow(device, self)
        while True:
            ret = self.properties_window.run()
            if ret is not RESPONSE_ACCEPT:
                break
            new_label = self.properties_window.get_new_label()
            if self.change_device_label(device, new_label):
                break
            self.properties_window.restore_orig_label()
        self.properties_window.destroy()

    def __on_open_format_window(self, widget, device):
        if device.blocked:
            gui.messages.BlockedDevice()
            return

        self.formating_window = gui.windows.FormatWindow(device, self, self.devices.get_supported_fs())
        ret = self.formating_window.run()

        if ret is RESPONSE_ACCEPT:
            if usbdbus.CAN_SUDO is False:
                gui.messages.CantSudo()
                self.formating_window.destroy()
                return
				
            new_label = self.formating_window.new_label
            new_fs = self.formating_window.fs_type
            fs_options = self.formating_window.fs_options
            self.formating_window.destroy()
            if device.is_mounted:
                gui.messages.UmountingDevice(device)
                ret = device.umount()
                if ret is False:
                    return
            if gui.messages.QuestionFormating() == -9:
                return

            bar = gui.windows.ProgressBar(self, device)
            bar.draw()
            bar.start_pulsing_thread()
            bar.write_text("Formating "+device.block_device+" to: "+new_fs+"\n")
            err = device.format(new_label, new_fs, fs_options, bar.write_text)
            if len(err) is 0:
                err = None
            else:
                err = '\n'.join(err)
            bar.enable_close(err)
            return

        self.formating_window.destroy()

    def __on_open_disk_usage_window(self, widget, device):
        self.disk_usage_window = gui.windows.DiskUsage(device, self)
        self.disk_usage_window.run()
        self.disk_usage_window.destroy()

    def __on_destroy_window(self, widget):
        if self.main_window:
            self.main_window.hide()
            self.main_window.destroy()

        if self.tray is None:
            self.main_window.gtk_quit()
            print 'Quiting...'
        else:
            self.main_window = None

    def destroy_format_progress_bar(self, device):
        gui.messages.BlockedDevice()
        device.blocked = True
