#!/usr/bin/env python
# -*- coding: utf-8 -*-

# gui.py
#
# Copyright (c) 2008 Magnun Leno da Silva
#
# Author: Magnun Leno da Silva <magnun.leno@gmail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
# USAA.

import gobject
import gtk
import pygtk
pygtk.require('2.0')
import locale
import gettext
import time
import os
from threading import Thread
import widgets

WINDOW_ICON = None
locale.setlocale(locale.LC_ALL, '')
if os.path.exists("data") and os.path.exists("locale"):
    WINDOW_ICON = "data/icons/64x64/apps/usbmanager.png"
    ABOUT_LOGO = "data/icons/128x128/apps/usbmanager.png"
    gettext.bindtextdomain('usbmanager', './locale')

else:
    WINDOW_ICON = "/usr/share/icons/hicolor/64x64/apps/usbmanager.png"
    ABOUT_LOGO = "/usr/share/icons/hicolor/128x128/apps/usbmanager.png"
    gettext.bindtextdomain('usbmanager', '/usr/share/locale')



gettext.textdomain('usbmanager')
_ = gettext.gettext

class MainWindow(gtk.Window):
    __gsignals__ = {
                    'open-properties-window' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)),
                    'open-format-window' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)),
                    'open-disk-usage-window' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)),
                    'destroying-window' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                   }
    def __init__(self, master):
        self.master = master
        self.devices = self.master.devices
        self.devices.connect('device-changed', self.__on_device_properties_changed)

        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        self.connect("destroy", self.__on_destroy)
        self.set_border_width(10)
        self.set_size_request(350,250)
        self.set_resizable(False)
        self.set_title(_('USB Volume Storage Manager'))
        self.set_icon_from_file(WINDOW_ICON)
        self.set_position(gtk.WIN_POS_CENTER)
        self.show()

        # Main VBox
        self.vbox = gtk.VBox(spacing=3)
        self.add(self.vbox)

        # Toolbar
        self.toolbar = widgets.ToolBar()
        self.toolbar.connect('properties-click', self.__on_properties_click)
        self.toolbar.connect('mount-umount-click', self.__on_mount_umount_click)
        self.toolbar.connect('disk-usage-click', self.__on_disk_usage_click)
        self.toolbar.connect('format-click', self.__on_format_click)
        self.toolbar.connect('about-click', self.__on_about_click)
        self.toolbar.connect('quit-click', self.__on_destroy)
        self.vbox.pack_start(self.toolbar, expand=False)

        # Devices List View
        self.devices_list_view = widgets.DevicesListView(self.devices)
        self.devices_list_view.connect('properties-click', self.__on_properties_click)
        self.devices_list_view.connect('device-selected', self.__on_device_selected)
        self.vbox.pack_start(self.devices_list_view)

        self.show_all()

    def __on_properties_click(self, widget):
        device = self.devices_list_view.get_selected_device()
        self.emit('open-properties-window', device)

    def __on_mount_umount_click(self, widget):
        device = self.devices_list_view.get_selected_device()
        if device.is_mounted:
            device.umount()
        else:
            device.mount()

    def __on_format_click(self, widget):
        device = self.devices_list_view.get_selected_device()
        self.emit('open-format-window', device)

    def __on_disk_usage_click(self, widget):
        device = self.devices_list_view.get_selected_device()
        self.emit('open-disk-usage-window', device)

    def __on_about_click(self, widget):
        about = USBManagerAbout(self)
        about.run()
        about.destroy()

    def __on_device_properties_changed(self, devices, device):
        self.devices_list_view.refresh_devices_view(device)
        self.__on_device_selected(None, device)

    def add_new_device(self, device):
        self.devices_list_view.add_new_device(device)

    def remove_device(self, device):
        self.devices_list_view.remove_device(device)
        self.toolbar.refresh_bar(None)

    def refresh_devices_view(self, device):
        self.devices_list_view.refresh_devices_view(device)
        self.refresh_device_toolbar(device)

    def __on_device_selected(self, widget, device):
        if device is None:
            self.toolbar.refresh_bar(None)
            return

        selected_device = self.devices_list_view.get_selected_device()
        if selected_device is None:
            self.toolbar.refresh_bar(None)
            return

        if selected_device.udi == device.udi:
            self.toolbar.refresh_bar(device)

    def __on_destroy(self, widget=None):
        self.emit('destroying-window')

    def gtk_quit(self):
        gtk.main_quit()

    def run(self):
        gtk.main()


class PropertiesWindow(gtk.Dialog):
    def __init__(self, device, master):
        self.master = master
        self.device = device
        self.udi = device.udi
        gtk.Dialog.__init__(self, _("%(device_label)s properties")%{"device_label":device.label},
                            self.master.main_window,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT)
        self.set_resizable(False)
        self.set_icon_from_file(WINDOW_ICON)
        self.set_border_width(10)
        self.vbox.set_spacing(10)

        self.apply = self.add_button(gtk.STOCK_APPLY, gtk.RESPONSE_ACCEPT)
        self.close = self.add_button(gtk.STOCK_CLOSE, gtk.RESPONSE_REJECT)
        self.apply.set_sensitive(False)


        self.field_labels = [_("Vendor"), _("Size"), _("Model"),
                             _("Is Mounted"), _("Read Only"),
                             _("Mount Point"), _("Category"),
                             _("Serial Number"), _("Type"),
                             _("Version"), _("Block Device")]

        max_len = max([len(label) for label in self.field_labels]) + 3
        self.field_labels = [label.rjust(max_len) + " :" for label in self.field_labels]

        # Add Category Image
        self.category_image = widgets.CategoryImage(self.device)
        self.vbox.pack_start(self.category_image, expand=False, fill=False)
        # Add device Description
        self.vbox.pack_start(widgets.AlignedDescription(self.device))
        # Add Separator
        self.vbox.pack_start(gtk.HSeparator())
        # Add Label & Entry
        label_entry = widgets.LabelEntryBox(self.device)
        self.device_label_entry = label_entry.entry
        self.vbox.pack_start(label_entry, expand=False, fill=False, padding=0)
        # Add Notebook
        self.__add_notebook()

        self.is_mounted_checkbtn.connect("toggled", self.__on_mount_check_toggled)
        self.device_label_entry.connect("key-release-event", self.__on_label_edit)

        self.show_all()

    def __add_notebook(self):
        self.notebook = gtk.Notebook()

        self.basic_page = widgets.BasicInfoPage(self.device, self.field_labels[:6])
        self.notebook.append_page(self.basic_page, widgets.MarkupLabel("<b>"+_("Basic Informations")+"</b>"))
        self.mount_point = self.basic_page.mount_point
        self.is_mounted_checkbtn = self.basic_page.is_mounted_checkbtn

        self.advanced_page = widgets.AdvancedInfoPage(self.device, self.field_labels[6:])
        self.notebook.append_page(self.advanced_page, widgets.MarkupLabel("<b>"+_("Advanced Informations")+"</b>"))

        self.notebook.set_property('homogeneous', True)
        self.vbox.pack_start(self.notebook)

    def update_fields(self):
        # Is mounted check btn
        self.is_mounted_checkbtn.set_active(self.device.is_mounted)
        # Mount point label
        self.mount_point.set_text(self.device.mount_point)
        # Volume label entry
        self.device_label_entry.set_text(self.device.label)

    def get_new_label(self):
        return self.device_label_entry.get_text()

    def restore_orig_label(self):
        self.device_label_entry.set_text(self.device.label)
        self.apply.set_sensitive(False)

    def __on_mount_check_toggled(self, toggle):
        if self.is_mounted_checkbtn.get_active():
            ret = self.device.mount()
            if ret is  False:
                self.is_mounted_checkbtn.set_active(False)
        else:
            ret = self.device.umount()
            if ret is  False:
                self.is_mounted_checkbtn.set_active(True)

    def __on_label_edit(self, widget, event):
        if self.device_label_entry.get_text() == self.device.label:
            self.apply.set_sensitive(False)
        else:
            self.apply.set_sensitive(True)


class FormatWindow(gtk.Dialog):
    def __init__(self, device, master, suported_fs):
        self.master = master
        self.device = device
        self.udi = self.device.udi
        suported_fs.sort()

        self.new_label = None
        self.fs_type = None
        self.fs_options = None
        self.fs_align = None

        gtk.Dialog.__init__(self, _("Formating %(device_label)s")%{"device_label":device.label},
                            self.master.main_window,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT)
        self.set_resizable(False)
        self.set_icon_from_file(WINDOW_ICON)
        self.set_border_width(10)
        self.vbox.set_spacing(10)

        self.field_labels = [_("Size"), _("Block Device"),
                             _("New Label"), _("Files System")]

        max_len = max([len(label) for label in self.field_labels])
        self.field_labels = [label.ljust(max_len) for label in self.field_labels]

        self.device_info = widgets.FormatingDeviceInfo(device, self.field_labels)
        self.vbox.pack_start(self.device_info)

        self.format_info = widgets.FormatingFormatInfo(device, self.field_labels, suported_fs)
        self.vbox.pack_start(self.format_info)

        self.format_info.fs_combo.connect("changed", self.__on_fs_selected)
        self.__on_fs_selected(None)

        btn = gtk.Button(_("Format"))
        btn.set_image(gtk.image_new_from_stock(gtk.STOCK_CLEAR, gtk.ICON_SIZE_BUTTON))
        self.add_action_widget(btn, gtk.RESPONSE_ACCEPT)
        self.close = self.add_button(gtk.STOCK_CLOSE, gtk.RESPONSE_REJECT)

        self.connect("response", self.__on_close)

        self.show_all()


    def __on_fs_selected(self, widget):
        if self.fs_align:
            self.vbox.remove(self.fs_align)
            self.fs_align.destroy()

        selected_fs = self.format_info.get_selected_fs().upper()
        self.fs_align = None
        if selected_fs in ("FAT16", "FAT32"):
            self.fs_align = widgets.FormatingVFatInfo()
        elif selected_fs == "NTFS":
            self.fs_align = widgets.FormatingNTFSInfo()
        elif selected_fs in ("EXT2", "EXT3", "EXT4"):
            self.fs_align = widgets.FormatingEXTNInfo()

        self.vbox.pack_start(self.fs_align)
        self.show_all()

    def __on_close(self, dialog, response_id):
        print 'closing'
        self.new_label = self.format_info.get_new_label()
        self.fs_type = self.format_info.get_selected_fs().upper()
        self.fs_options = self.fs_align.get_options()


class ProgressBarPulseThread(Thread):
    def __init__(self, bar):
        Thread.__init__(self)
        self.bar = bar

    def run(self):
        while self.bar.pulse():
            time.sleep(0.2)


class DiskUsage(gtk.Dialog):
    def __init__(self, device, master):
        self.device = device
        self.udi = self.device.udi

        gtk.Dialog.__init__(self, _("Disk Usage Analysis"), master.main_window, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT)
        #self.set_resizable(False)
        self.set_icon_from_file(WINDOW_ICON)
        self.set_border_width(10)
        self.vbox.set_spacing(10)

        title = widgets.MarkupLabel("<big><b>"+_("%(device_label)s disk usage")%{"device_label":device.label}+"</b></big>")
        self.vbox.pack_start(title)

        self.vbox.pack_start(gtk.HSeparator())

        self.device_analysis = widgets.DeviceAnalysisInfo(device)
        self.vbox.pack_start(self.device_analysis, expand=False, fill=False)

        self.vbox.pack_start(gtk.HSeparator())

        self.sw = gtk.ScrolledWindow()
        self.sw.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        self.size_list_view = widgets.SizeListView(device)
        self.size_list_view.set_size_request(-1, 300)
        self.size_list_view.connect('item-selected', self.__on_item_selected)
        self.sw.add(self.size_list_view)
        self.vbox.pack_start(self.sw)

        self.close = self.add_button(gtk.STOCK_CLOSE, gtk.RESPONSE_REJECT)
        self.device_analysis.update_data()
        self.show_all()

    def __on_item_selected(self, widget, item):
        if item is None:
            print 'none selected'
        else:
            print 'selected',item[0]


class ProgressBar:
    '''
            Simple class to create a progress bar to show progress
    '''
    def __init__(self, master, device):
        '''Initialize the class'''
        self.progress_bar = None
        self.window = None
        self.master = master
        self.device = device
        self.keep_pulsing = False
        self.thread = ProgressBarPulseThread(self)

    def draw(self):
        '''Draw the widgets'''
        # Simple window
        self.window = gtk.Window(gtk.WINDOW_POPUP)
        self.window.set_resizable(False)
        self.window.set_border_width(10)
        self.window.set_position(gtk.WIN_POS_CENTER)
        self.window.set_transient_for(self.master.main_window)
        self.window.set_destroy_with_parent(True)

        # Create a VBox
        vbox = gtk.VBox(False, 5)
        vbox.set_border_width(10)
        self.window.add(vbox)

        # Create the desired progress bar
        self.progress_bar = gtk.ProgressBar()
        self.progress_bar.set_size_request(500, -1)
        self.progress_bar.pulse()
        vbox.pack_start(self.progress_bar, False, False, 0)

        # An expander to hold the textview
        expander = gtk.Expander("<b>"+_("Details")+"</b>")
        expander.set_use_markup(True)
        expander.set_expanded(True)
        # Text buffer
        self.text_buffer = gtk.TextBuffer()
        # The text view
        self.text_view = gtk.TextView(self.text_buffer);
        self.text_view.modify_base(gtk.STATE_NORMAL, gtk.gdk.color_parse("black"))
        self.text_view.modify_text(gtk.STATE_NORMAL, gtk.gdk.color_parse("white"))
        self.text_view.set_editable(False)
        self.text_view.set_wrap_mode(gtk.WRAP_CHAR)
        self.text_view.set_justification(gtk.JUSTIFY_LEFT)
        self.text_view.set_left_margin(10)
        self.text_view.set_right_margin(10)
        self.text_view.set_size_request(-1, 150)
        # A scroller to the text view
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.text_view)
        expander.add(sw)
        vbox.pack_start(expander)

        # close button
        self.close_button = gtk.Button(stock=gtk.STOCK_CLOSE)
        self.close_button.set_sensitive(False)
        self.close_button.connect("clicked", self.__on_close_click)
        # HButtonBox
        btn_box = gtk.HButtonBox()
        btn_box.set_layout(gtk.BUTTONBOX_END)
        btn_box.pack_start(self.close_button)
        vbox.pack_start(btn_box)

        # Show all widgets
        self.window.show_all()

    def addTimer(self):
        self.timer = gobject.timeout_add(5, self.draw_progress, self.progress_bar)

    def write_text(self, text):
        '''Write the text to the label'''
        self.text_buffer.insert(self.text_buffer.get_end_iter(), text)
        self.text_view.scroll_mark_onscreen(self.text_buffer.get_insert())
        #self.pulse()
        while gtk.events_pending():
            gtk.main_iteration(False)

    def draw_end_progress(self, text, widget=None):
        '''Update the progress from the progressbar'''
        self.progress_bar.set_fraction(1)
        self.progress_bar.set_text(text)

    def enable_close(self, erro=None):
        self.keep_pulsing = False
        self.thread.join()
        if erro is None:
            self.draw_end_progress(_("Formating done"))
            self.write_text("\n\n"+_("Done")+"\n")
            print ' * Formationg done!'
        else:
            self.draw_end_progress(_("Error while formating"))
            self.write_text("\n\n*** "+_("Errors:")+"\n")
            self.write_text(erro)
            print ' * Error while formating!'
            print erro

        self.close_button.set_sensitive(True)

    def __on_close_click(self, widget=None):
        '''Close the progress bar'''
        print "Closing..."
        self.window.destroy()
        self.master.destroy_format_progress_bar(self.device)

    def pulse(self, widget=None):
        self.progress_bar.pulse()
        while gtk.events_pending():
            gtk.main_iteration(False)

        return self.keep_pulsing

    def start_pulsing_thread(self):
        self.keep_pulsing = True
        self.thread.start()

class USBManagerAbout(gtk.AboutDialog):
    def __init__(self, main_window):
        gtk.AboutDialog.__init__(self)
        self.set_name('USBManager')
        self.set_transient_for(main_window)
        self.set_destroy_with_parent(True)
        self.set_version('\n1.0.0')
        self.set_copyright('Copyright (c) 2010 Magnun Leno da Silva')
        self.set_license("Copyright (c) 2010 Magnun Leno da Silva\n\n"

                         "Author: Magnun Leno da Silva <magnun.leno@gmail.com>\n\n"

                         "This program is free software; you can redistribute it and/or\n"
                         "modify it under the terms of the GNU Lesser General Public License\n"
                         "as published by the Free Software Foundation; either version 2 of\n"
                         "the License, or (at your option) any later version.\n\n"

                         "This program is distributed in the hope that it will be useful,\n"
                         "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
                         "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
                         "GNU General Public License for more details.\n\n"

                         "You should have received a copy of the GNU Lesser General Public\n"
                         "License along with this program; if not, write to the Free Software\n"
                         "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307\n"
                         "USAA.")
        self.set_website('http://launchpad.net/usbmanager')
        #self.set_website_label('USBManager at Launchpad')
        self.set_authors(["Magnun Leno\n <magnun.leno@gmail.com>\n "+_("Lead Developer and Creator")+"\n"])
        self.set_translator_credits(_("Lauchpad Translators (These guys rocks!!):")+"\n\n"+\
                                    _(" * Arabic:")+"\n"+\
                                      "   - Nizar Kerkeni;\n\n"+\
                                    _(" * Brazilian Portuguese:")+"\n"+\
                                      "   - Magnun Leno;\n\n"+\
                                    _(" * Dutch:")+"\n"+\
                                      "   - cumulus007;\n"+\
                                      "   - Donkade;\n\n"+\
                                    _(" * French:")+"\n"+\
                                      "   - Guillaume Mazoyer;\n"+\
                                      "   - Nicolas Delvaux;\n"+\
                                      "   - Pascal Lemazurier;\n"+\
                                      "   - Sorkin;\n\n"+\
                                    _(" * German:")+"\n"+\
                                      "   - BBO;\n"+\
                                      "   - Fabian Affolter;\n"+\
                                      "   - Fred Seidemann;\n"+\
                                      "   - Georg Engelmann;\n"+\
                                      "   - mogli;\n\n"+\
                                    _(" * Russian:")+"\n"+\
                                      "   - Nikolay Edigaryev;\n"+\
                                      "   - Petron;\n"+\
                                      "   - Sergey Sedov;\n"+\
                                      "   - vsink;\n\n"+\
                                    _(" * Spanish:")+"\n"+\
                                      "   - Demuxer;\n"+\
                                      "   - Monkey;\n"+\
                                      "   - Paco Molinero;\n"+\
                                      "   - guillemsster;\n"+\
                                      "   - kaeltas;\n\n"+\
                                    _(" * Thai:")+"\n"+\
                                      "   - Krit Marukawisutthigul;\n\n"+\
                                    _(" * Turkish:")+"\n"+\
                                      "   - zeugma;")
        self.set_logo(gtk.gdk.pixbuf_new_from_file(ABOUT_LOGO))
