#!/usr/bin/env python
# -*- coding: utf-8 -*-

# gui.py
#
# Copyright (c) 2008 Magnun Leno da Silva
#
# Author: Magnun Leno da Silva <magnun.leno@gmail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
# USAA.

import gobject
import gtk
import pygtk
pygtk.require('2.0')
import locale
import gettext
import messages
import os
import cairo
import pango

BASE = None
locale.setlocale(locale.LC_ALL, '')

if os.path.exists('locale') and os.path.exists('data'):
    BASE = "data/icons/"
    gettext.bindtextdomain('usbmanager', './locale')
else:
    BASE = "/usr/share/icons/hicolor/"
    gettext.bindtextdomain('usbmanager', '/usr/share/locale')

gettext.textdomain('usbmanager')
_ = gettext.gettext

SMALL_DEVICE_ICONS = {
               "volume"                 : BASE+"24x24/apps/storage.png",
               "portable_audio_player"  : BASE+"24x24/apps/m_player.png",
              }

BIG_DEVICE_ICONS = {
               "volume"                 : BASE+"64x64/apps/storage.png",
               "portable_audio_player"  : BASE+"64x64/apps/m_player.png",
              }

DEFAULT_BIG_DEVICE_ICON = BASE+"64x64/apps/storage.png"
DEFAULT_SMALL_DEVICE_ICON = BASE+"24x24/apps/storage.png"
DISK_USAGE_BTN_ICON = BASE+"32x32/apps/diskusage.png"


class DevicesListView(gtk.TreeView):
    __gsignals__ = {
                    'properties-click'  : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                    'device-selected'   : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)),
                   }
    def __init__(self, devices):
        gtk.TreeView.__init__(self)
        self.devices = devices

        self.__add_pxbuf_column(_("Label"), 0)
        self.__add_text_column(_("Block Device"), 1)
        self.__add_toggle_column(_("Mounted"), 2)
        self.__add_text_column("UID", 3, False)

        # ListStore
        self.devicesList = gtk.ListStore(str, str, gobject.TYPE_BOOLEAN, str)
        self.set_model(self.devicesList)

        self.connect("row-activated", self.__on_row_double_click)
        self.connect("button-press-event", self.__on_row_click)
        self.connect("button-release-event", self.__on_row_click_leave)

        if len(self.devices) is not 0:
            for udi, device in self.devices:
                self.devicesList.append([device.label, device.block_device, device.is_mounted, udi])

    def __add_pxbuf_column(self, title, id):
        cellpb = gtk.CellRendererPixbuf()
        cell = gtk.CellRendererText()

        column = gtk.TreeViewColumn(title)
        column.set_sort_column_id(0)
        column.pack_start(cellpb, False)
        column.pack_start(cell, False)
        column.set_cell_data_func(cellpb, self.__device_pixbuf)
        column.set_cell_data_func(cell, self.__device_name)
        self.append_column(column)

    def __add_text_column(self, title, id, visible=True):
        column = gtk.TreeViewColumn(title, gtk.CellRendererText(), text=id, markup=0)
        column.set_resizable(True)
        column.set_sort_column_id(id)
        column.set_visible(visible)
        self.append_column(column)

    def __add_toggle_column(self, title, id):
        toggle = gtk.CellRendererToggle()
        toggle.set_property('activatable', True)
        toggle.connect('toggled', self.__on_mount_check)
        column = gtk.TreeViewColumn(title, toggle, active=id)
        self.append_column(column)

    def __device_pixbuf(self, column, cell, model, iter):
        device = self.devices.get_device_by_udi(model.get_value(iter, 3))
        pb = None
        icon_name = None

        if SMALL_DEVICE_ICONS.has_key(device.category):
            icon_name = SMALL_DEVICE_ICONS[device.category]
        else:
            icon_name = DEFAULT_SMALL_DEVICE_ICON

        pb = gtk.gdk.pixbuf_new_from_file(icon_name)

        cell.set_property('pixbuf', pb)
        cell.set_property('xalign', 0.5)

    def __device_name(self, column, cell, model, iter):
        cell.set_property('text', model.get_value(iter, 0).ljust(15))
        cell.set_property('xalign', 0)
        device = self.devices.get_device_by_udi(model.get_value(iter, 3))
        if device.blocked:
            cell.set_property("markup", '<span foreground="red">'+device.label+'</span>')

    def refresh_devices_view(self, device):
        current_devices = {}
        for properties in self.devicesList:
            current_devices[properties[3]] = properties

        # Existing Device
        if current_devices.has_key(device.udi):
            properties = current_devices[device.udi]
            properties[0] = device.label
            properties[1] = device.block_device
            properties[2] = device.is_mounted

    def add_new_device(self, device):
        current_devices = {}
        for properties in self.devicesList:
            current_devices[properties[3]] = properties

        if current_devices.has_key(device.udi):
            return

        self.devicesList.append([device.label, device.block_device, device.is_mounted, device.udi])

    def remove_device(self, device):
        for properties in self.devicesList:
            if properties[3] == device.udi:
                self.devicesList.remove(properties.iter)

    def get_selected_device(self):
        selection = self.get_selection()
        if selection is None:
            return
        model, selection_iter = selection.get_selected()
        if not selection_iter:
            return None
        # Get label
        return self.devices.get_device_by_udi(model.get_value(selection_iter,3))

    def __on_row_double_click(self, widget, column = None, pointer = None):
        device = self.get_selected_device()
        if device is None:
            return
        self.emit('properties-click')

    def __on_row_click(self, widget, event):
        path = widget.get_path_at_pos(int(event.x), int(event.y))
        selection = widget.get_selection()
        if path is None:
            selection.unselect_all()
            self.emit('device-selected', None)

    def __on_row_click_leave(self, widget, event):
        path = widget.get_path_at_pos(int(event.x), int(event.y))
        selection = widget.get_selection()
        model, selection_iter = selection.get_selected()
        if path:
            device = self.devices.get_device_by_udi(model.get_value(selection_iter,3))
            self.emit('device-selected', device)

    def __on_mount_check(self, cell, path):
        for udi, device in self.devices:
            if udi == self.devicesList[path][3]:
                if self.devicesList[path][2]:
                    device.umount()
                else:
                    device.mount()



class ToolBar(gtk.Toolbar):
    __gsignals__ = {
                    'properties-click'  : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                    'mount-umount-click': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                    'disk-usage-click'  : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                    'format-click'      : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                    'quit-click'        : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                    'about-click'       : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                   }
    def __init__(self):
        # Toolbar
        gtk.Toolbar.__init__(self)
        self.set_orientation(gtk.ORIENTATION_HORIZONTAL)
        self.set_style(gtk.TOOLBAR_ICONS)
        self.tooltips = gtk.Tooltips()
        self.sensitive_buttons = []

        # Properties
        self.properties = self.__add_button_to_toolbar(gtk.STOCK_PROPERTIES, _("Properties"), _("Show volume detailed properties"), self.__on_properties_click)
        self.sensitive_buttons.append(self.properties)

        # Mount/Umount
        self.mount_umount = self.__add_button_to_toolbar(gtk.STOCK_CONNECT, _("Mount/Umount"), _("Mount or Umount device"), self.__on_mount_umount_click)
        self.sensitive_buttons.append(self.mount_umount)

        # Format
        self.format = self.__add_button_to_toolbar(gtk.STOCK_CLEAR, _("Format"), _("Format devices"), self.__on_format_click)
        self.sensitive_buttons.append(self.format)

        # Disk Usage
        self.disk_usage = self.__add_button_to_toolbar(DISK_USAGE_BTN_ICON, _("Disk Usage"), _("Disk Usage Analysis"), self.__on_disk_usage_click)
        self.sensitive_buttons.append(self.disk_usage)

        #gtk.MenuToolButton
        #self.toolmenu = self.__add_menu_to_toolbar(gtk.STOCK_PREFERENCES, _("Option"), _("Option fot the selected device"), self.__on_menu_click)

        # Space
        self.__add_separator()

        # About
        self.__add_button_to_toolbar(gtk.STOCK_ABOUT, _("About"), _("About USBManager"), self.__on_about)

        # Quit
        self.__add_button_to_toolbar(gtk.STOCK_QUIT, _("Quit"), _("Quit USBManager"), self.__destroy)

        self.refresh_toolbar_sensitive(False)


    def __add_button_to_toolbar(self, image, label, tooltip, event_func):
        toolbutton = None
        if 'gtk-' in image:
            toolbutton = gtk.ToolButton(image)
        else:
            img = gtk.Image()
            img.set_from_file(image)
            toolbutton = gtk.ToolButton(img)
        #toolbutton.set_label(label)
        toolbutton.connect("clicked", event_func)
        self.tooltips.set_tip(toolbutton,tooltip, tooltip)
        self.insert(toolbutton, -1)
        return toolbutton

    def __add_menu_to_toolbar(self, image, label, tooltip, event_func):
        #TODO: Try to show menu when button clicked
        menubutton = gtk.MenuToolButton(gtk.STOCK_PREFERENCES)
        menubutton.connect("clicked", self.__on_menubutton_click)
        self.menu = gtk.Menu()
        menu_item = gtk.MenuItem('some menu item')
        self.menu.append(menu_item)
        self.menu.show_all()
        self.tooltips.set_tip(menubutton,tooltip, tooltip)
        menubutton.set_menu(self.menu)
        self.insert(menubutton, -1)
        return menubutton

    def __add_separator(self):
        sep = gtk.SeparatorToolItem()
        self.insert(sep, -1)

    def refresh_bar(self, device):
        if device is None:
            self.refresh_toolbar_sensitive(False)
            self.refresh_mount_baritem(None)
            return

        self.refresh_toolbar_sensitive(True)
        self.refresh_mount_baritem(device)

    def refresh_toolbar_sensitive(self, sensitive):
        for button in self.sensitive_buttons:
            button.set_sensitive(sensitive)

    def refresh_mount_baritem(self, device):
        if device is None:
            self.mount_umount.set_stock_id(gtk.STOCK_CONNECT)
            self.mount_umount.set_label(_("Mount/Umount"))
            self.tooltips.set_tip(self.mount_umount,_("Mount/Umount selected device"), _("Mount/Umount selected device"))
        elif device.is_mounted:
            self.mount_umount.set_stock_id(gtk.STOCK_CONNECT)
            self.mount_umount.set_label(_("Umount"))
            self.tooltips.set_tip(self.mount_umount,_("Umount selected device"), _("Umount selected device"))
            self.disk_usage.set_sensitive(True)
        else:
            self.mount_umount.set_stock_id(gtk.STOCK_DISCONNECT)
            self.mount_umount.set_label(_("Mount"))
            self.tooltips.set_tip(self.mount_umount,_("Mount selected device"), _("Mount selected device"))
            self.disk_usage.set_sensitive(False)

    def __on_about(self, widget):
        self.emit('about-click')

    def __on_properties_click(self, widget):
        self.emit('properties-click')

    def __on_disk_usage_click(self, widget):
        self.emit('disk-usage-click')

    def __on_format_click(self, widget, envet=None):
        self.emit('format-click')

    def __on_mount_umount_click(self, widget, event=None):
        self.emit('mount-umount-click')

    def __on_menubutton_click(self, widget):
        self.menu.popup(None, None, None, 1, 0)
        pass

    def __destroy(self, widget):
        self.emit('quit-click')


class CategoryImage(gtk.Image):
    def __init__(self, device):
        gtk.Image.__init__(self)
        icon_name = None

        if BIG_DEVICE_ICONS.has_key(device.category):
            icon_name = BIG_DEVICE_ICONS[device.category]
        else:
            icon_name = DEFAULT_BIG_DEVICE_ICON

        self.set_from_pixbuf(gtk.gdk.pixbuf_new_from_file(icon_name))

class MarkupLabel(gtk.Label):
    def __init__(self, text):
        gtk.Label.__init__(self)
        self.set_markup(text)

class AlignedDescription(gtk.Label):
    def __init__(self, device):
        gtk.Label.__init__(self, device.product + "\n("+device.vendor_full_name+")")
        self.set_alignment(xalign=0.5, yalign=0.5)
        self.set_property("wrap", True)
        self.set_justify(gtk.JUSTIFY_CENTER)

class LabelEntryBox(gtk.HBox):
    def __init__(self, device):
        gtk.HBox.__init__(self, spacing=0)
        self.label = MarkupLabel("<b> "+_("Label")+" : "+"</b>")
        self.label.set_alignment(xalign=1, yalign=0.5)
        self.entry = gtk.Entry(max=15)
        self.entry.set_text(device.label)

        if not device.is_renamable():
            self.entry.set_sensitive(False)
            messages.DeviceNotRenamable(device)

        self.pack_start(self.label, expand=True, fill=False)
        self.pack_start(self.entry, expand=True, fill=True)

class GenericPage(gtk.VBox):
    def __init__(self, device):
        gtk.VBox.__init__(self)
        self.set_spacing(10)
        self.device = device

    def add_framed_table(self, text, rows_count):
        frame = gtk.Frame()
        label = MarkupLabel("<b>"+text+"</b>")
        frame.set_shadow_type(gtk.SHADOW_NONE)
        frame.set_label_widget(label)
        self.pack_start(frame)

        align = gtk.Alignment(xalign=0.5, yalign=0.5, xscale=1.0, yscale=1.0)
        align.set_padding(padding_top=5, padding_bottom=5, padding_left=0, padding_right=0)
        frame.add(align)

        table = gtk.Table(rows=rows_count, columns=2, homogeneous=True)
        table.set_col_spacings(10)
        align.add(table)
        return table

    def add_label_to_table(self, table, text, index):
        label = gtk.Label(text)
        label.set_alignment(xalign=1, yalign=0.5)
        table.attach(label, 0, 1, index, index+1,
                     xoptions=gtk.EXPAND|gtk.FILL,
                     yoptions=gtk.EXPAND|gtk.FILL,
                     xpadding=0, ypadding=2)

    def add_text_value_to_table(self, table, text, index):
        label = gtk.Label(text)
        label.set_alignment(xalign=0, yalign=0.5)
        table.attach(label, 1, 2, index, index+1,
                     xoptions=gtk.EXPAND|gtk.FILL,
                     yoptions=gtk.EXPAND|gtk.FILL,
                     xpadding=0, ypadding=2)
        return label

    def add_checkbtn_to_table(self, table, value, index):
        check = gtk.CheckButton()
        check.set_active(value)
        check.set_alignment(xalign=0, yalign=0.5)
        table.attach(check, 1, 2, index, index+1,
                     xoptions=gtk.EXPAND|gtk.FILL,
                     yoptions=gtk.EXPAND|gtk.FILL,
                     xpadding=0, ypadding=0)
        return check

class BasicInfoPage(GenericPage):
    def __init__(self, device, labels):
        GenericPage.__init__(self, device)
        self.labels = labels

        ## First Table
        table = self.add_framed_table("<b>"+_("Storage Device")+"</b>", 3)
        # Vendor
        self.add_label_to_table(table, self.labels[0], 0)
        self.add_text_value_to_table(table, self.device.vendor, 0)
        # Size
        self.add_label_to_table(table, self.labels[1], 1)
        self.add_text_value_to_table(table, self.device.size, 1)
        # Model
        self.add_label_to_table(table, self.labels[2], 2)
        self.add_text_value_to_table(table, self.device.model, 2)

        ## Second Table
        table = self.add_framed_table("<b>"+_("Mount Options")+"</b>", 3)
        # Is mounted
        self.add_label_to_table(table, self.labels[3], 0)
        self.is_mounted_checkbtn = self.add_checkbtn_to_table(table, self.device.is_mounted, 0)
        # Read Only
        self.add_label_to_table(table, self.labels[4], 1)
        self.read_only_checkbtn = self.add_checkbtn_to_table(table, self.device.read_only, 1)
        self.read_only_checkbtn.set_sensitive(False)
        # Mount Point
        self.add_label_to_table(table, self.labels[5], 2)
        self.mount_point = self.add_text_value_to_table(table, self.device.mount_point, 2)

class AdvancedInfoPage(GenericPage):
    def __init__(self, device, labels):
        GenericPage.__init__(self, device)
        self.labels = labels

        ## First Table
        table = self.add_framed_table("<b>"+_("General")+"</b>", 2)
        # Category
        self.add_label_to_table(table, self.labels[0], 0)
        self.add_text_value_to_table(table, (' '.join(self.device.category.split('_'))).title(), 0)
        # Device Serial Number
        self.add_label_to_table(table, self.labels[1], 1)
        self.add_text_value_to_table(table, self.device.serial, 1)

        ## Second Table
        table = self.add_framed_table("<b>"+_("File System")+"</b>", 3)
        # File System
        self.add_label_to_table(table, self.labels[2], 0)
        self.add_text_value_to_table(table, self.device.fs_type, 0)
        # FS Version
        self.add_label_to_table(table, self.labels[3], 1)
        self.add_text_value_to_table(table, self.device.fs_version, 1)
        # Block Device
        self.add_label_to_table(table, self.labels[4], 2)
        self.add_text_value_to_table(table, self.device.block_device, 2)

class FormatingGenericInfo(gtk.Frame):
    def __init__(self, device, labels, text):
        self.device = device
        self.labels = labels
        gtk.Frame.__init__(self)
        label = MarkupLabel("<b>"+text+"</b>")
        self.set_shadow_type(gtk.SHADOW_NONE)
        self.set_label_widget(label)

        self.align = gtk.Alignment(xalign=0.5, yalign=0.5, xscale=1.0, yscale=1.0)
        self.align.set_padding(padding_top=5, padding_bottom=5, padding_left=10, padding_right=10)
        self.add(self.align)

        self.table = gtk.Table(rows=2, columns=3, homogeneous=False)
        self.table.set_col_spacings(10)
        self.align.add(self.table)


    def add_label_to_table(self, text, index):
        label = gtk.Label(text)
        label.set_alignment(xalign=0, yalign=0.5)
        self.table.attach(label, 0, 1, index, index+1,
                          xoptions=gtk.FILL,
                          yoptions=gtk.EXPAND|gtk.FILL,
                          xpadding=0, ypadding=2)
        self.add_point_sep(index)

    def add_entry(self, text, index):
        entry = gtk.Entry(15)
        entry.set_text(text)
        self.table.attach(entry, 2, 3, index, index+1,
                          xoptions=gtk.EXPAND|gtk.FILL,
                          yoptions=gtk.EXPAND|gtk.FILL,
                          xpadding=0, ypadding=2)
        return entry

    def add_combo_box(self, items, index):
        combo = gtk.combo_box_new_text()
        for item in items:
            combo.append_text(item)

        current_fs = None
        if self.device.fs_type == 'vfat':
            current_fs = self.device.fs_version
        else:
            current_fs = self.device.fs_type
        if self.device.fs_type == 'ntfs-3g':
            current_fs = 'NTFS'

        if items.count(current_fs.upper()) is 0:
            if items.count('FAT16') > 0:
                combo.set_active(items.index('FAT16'))
                print ' * Device current FS is unknown (%(fs)s). Assuming FAT16'%{"fs":current_fs}
            else:
                combo.set_active(0)
                print ' * Device current FS is unknown (%(fs)s). Assuming %(new_fs)s'%{"fs":current_fs, "new_fs":items[0]}
        else:
            print ' * Assuming device current FS:', current_fs
            combo.set_active(items.index(current_fs.upper()))

        self.table.attach(combo, 2, 3, index, index+1,
                          xoptions=gtk.EXPAND|gtk.FILL,
                          yoptions=gtk.EXPAND|gtk.FILL,
                          xpadding=0, ypadding=2)
        return combo

    def add_text_value_to_table(self, text, index):
        label = gtk.Label(text)
        label.set_alignment(xalign=0, yalign=0.5)
        self.table.attach(label, 2, 3, index, index+1,
                          xoptions=gtk.EXPAND|gtk.FILL,
                          yoptions=gtk.EXPAND|gtk.FILL,
                          xpadding=0, ypadding=2)
        return label

    def add_point_sep(self, index):
        label = gtk.Label(":")
        label.set_alignment(xalign=0.5, yalign=0.5)
        self.table.attach(label, 1, 2, index, index+1,
                          xoptions=gtk.FILL,
                          yoptions=gtk.EXPAND|gtk.FILL,
                          xpadding=0, ypadding=2)


class FormatingDeviceInfo(FormatingGenericInfo):
    def __init__(self, device, labels):
        FormatingGenericInfo.__init__(self, device, labels, _("Device"))

        # Size
        self.add_label_to_table(self.labels[0], 0)
        self.add_text_value_to_table(device.size, 0)
        # Block Device
        self.add_label_to_table(self.labels[1], 1)
        self.add_text_value_to_table(device.block_device, 1)

class FormatingFormatInfo(FormatingGenericInfo):
    def __init__(self, device, labels, suported_fs):
        FormatingGenericInfo.__init__(self, device, labels, _("Formating"))

        # Label
        self.add_label_to_table(self.labels[2], 0)
        self.label_entry = self.add_entry(device.label, 0)
        # FS Type
        self.add_label_to_table(self.labels[3], 1)
        self.fs_combo = self.add_combo_box(suported_fs, 1)

    def get_selected_fs(self):
        return self.fs_combo.get_active_text()

    def get_new_label(self):
        return self.label_entry.get_text()

class FormatingVFatInfo(gtk.Alignment):
    def __init__(self):
        gtk.Alignment.__init__(self, xalign=0.5, yalign=0.5, xscale=1.0, yscale=1.0)
        self.set_padding(padding_top=0, padding_bottom=5, padding_left=10, padding_right=10)

        self.check_bad_blocks = gtk.CheckButton(label=_("Check for Bad Blocks"))
        self.check_bad_blocks.set_active(False)

        self.vbox = gtk.VBox()
        self.add(self.vbox)
        self.vbox.pack_start(self.check_bad_blocks)

    def get_options(self):
        return [self.check_bad_blocks.get_active()]

class FormatingNTFSInfo(gtk.Alignment):
    def __init__(self):
        gtk.Alignment.__init__(self, xalign=0.5, yalign=0.5, xscale=1.0, yscale=1.0)
        self.set_padding(padding_top=0, padding_bottom=5, padding_left=10, padding_right=10)

        self.quick_format = gtk.CheckButton(label=_("Quick Format"))
        self.quick_format.set_active(True)

        self.compression = gtk.CheckButton(label=_("Enable Compression"))
        self.compression.set_active(False)

        self.vbox = gtk.VBox()
        self.add(self.vbox)
        self.vbox.pack_start(self.quick_format)
        self.vbox.pack_start(self.compression)

    def get_options(self):
        return [self.quick_format.get_active(),
                self.compression.get_active()]

class FormatingEXTNInfo(gtk.Alignment):
    def __init__(self):
        gtk.Alignment.__init__(self, xalign=0.5, yalign=0.5, xscale=1.0, yscale=1.0)
        self.set_padding(padding_top=0, padding_bottom=5, padding_left=10, padding_right=10)

        self.bad_blocks = gtk.CheckButton(label=_("Check for Bad Blocks"))
        self.bad_blocks.set_active(False)

        self.journaling = gtk.CheckButton(label=_("Use Journaling"))
        self.journaling.set_active(False)

        self.vbox = gtk.VBox()
        self.add(self.vbox)
        self.vbox.pack_start(self.bad_blocks)
        self.vbox.pack_start(self.journaling)

    def get_options(self):
        return [self.bad_blocks.get_active(),
                self.journaling.get_active()]


class CellRendererHeatBar(gtk.GenericCellRenderer):
    __gproperties__ = {
                        'value': (gobject.TYPE_DOUBLE,
                        'Value of the bar. Must be between 0 and 1',
                        'Value of the bar. Must be between 0 and 1',
                        -1,1,0,
                        gobject.PARAM_READWRITE)
                      }

    def __init__(self):
        gtk.GenericCellRenderer.__init__(self)
        self.__properties = {}
        self.draw_bg = True
        self.draw_bg_glow = True
        self.draw_border = False
        self.draw_colored_border = False
        self.draw_bar_glow = True
        self.round_edges = True
        self.single_color = True
        self.tones = None

    def on_get_size(self, widget, cell_area):
        if cell_area == None:
            return (0,0,0,0)
        x = cell_area.x
        y = cell_area.x
        w = cell_area.width
        h = cell_area.height

        return (x,y,w,h)

    def do_set_property(self, key, value):
        self.__properties[key] = value

    def do_get_property(self, key):
        return self.__properties[key]

    def on_render(self, window, widget, background_area, cell_area, expose_area, flags):
        cr = window.cairo_create()
        x0 = cell_area.x
        y0 = cell_area.y
        w = cell_area.width
        h = cell_area.height

        value = self.get_property('value')
        if value < 0:
            return
        if self.single_color:
            self.tones = self.get_heat_color(value)

        if self.draw_bg is True:
            self.__draw_bg(cr, x0, y0, w, h, value)

        if self.draw_bg_glow and self.draw_bg:
            self.__draw_bg_glow(cr, x0, y0, w, h, value)

        self.__draw_bar(cr, x0, y0, w, h, value)

        if self.draw_border:
            self.__draw_border_line(cr, x0, y0, w, h, value)
        if self.draw_bar_glow:
            self.__draw_glow(cr, x0, y0, w, h, value)

        self.__draw_percentage(cr, x0, y0, w, h, value)

    def __draw_bg(self, cr, x0, y0, w, h, value):
        if self.single_color:
            cr.set_source_rgba(self.tones[0], self.tones[1], self.tones[2], 0.75)
        else:
            pat = cairo.LinearGradient(x0, y0, x0+w, y0)
            pat.add_color_stop_rgb(0, 0, 0.8, 0)
            pat.add_color_stop_rgb(0.45, 0.8, 0.8, 0)
            pat.add_color_stop_rgb(0.55, 0.8, 0.8, 0)
            pat.add_color_stop_rgb(1, 0.8, 0, 0)
            cr.set_source(pat)

        x1 = x0 + w
        y1 = y0 + h
        radius = 3

        if self.round_edges:
            cr.move_to(x0, y0+radius)
            cr.curve_to(x0, y0+radius, x0, y0, x0 + radius, y0)

            cr.line_to(x1-radius, y0)
            cr.curve_to(x1-radius, y0, x1, y0, x1, y0+radius)
            cr.line_to(x1, y1-radius)
            cr.curve_to(x1, y1-radius, x1, y1, x1-radius, y1)

            cr.line_to(x0+radius, y1)
            cr.curve_to(x0+radius, y1, x0, y1, x0, y1-radius)
        else:
            cr.move_to(x0, y0)
            cr.line_to(x1, y0)
            cr.line_to(x1, y1)
            cr.line_to(x0, y1)

        cr.close_path()
        cr.fill()

    def __draw_bg_glow(self, cr, x0, y0, w, h, value):
        pat = cairo.LinearGradient(x0, y0, x0, y0+h)
        pat.add_color_stop_rgba(0, 1, 1, 1, 0.3)
        pat.add_color_stop_rgba(0.4, 1, 1, 1, 0.7)
        #pat.add_color_stop_rgba(0.5, 1, 1, 1, 0.6)
        pat.add_color_stop_rgba(0.6, 1, 1, 1, 0.7)
        pat.add_color_stop_rgba(0, 1, 1, 1, 0.3)
        cr.set_source(pat)

        x1 = x0 + w
        y1 = y0 + h
        radius = 3

        if self.round_edges:
            cr.move_to(x0, y0+radius)
            cr.curve_to(x0, y0+radius, x0, y0, x0 + radius, y0)

            cr.line_to(x1-radius, y0)
            cr.curve_to(x1-radius, y0, x1, y0, x1, y0+radius)
            cr.line_to(x1, y1-radius)
            cr.curve_to(x1, y1-radius, x1, y1, x1-radius, y1)

            cr.line_to(x0+radius, y1)
            cr.curve_to(x0+radius, y1, x0, y1, x0, y1-radius)
        else:
            cr.move_to(x0, y0)
            cr.line_to(x1, y0)
            cr.line_to(x1, y1)
            cr.line_to(x0, y1)

        cr.close_path()
        cr.fill()

    def __draw_bar(self, cr, x0, y0, w, h, value):
        rect_width = w*value
        if rect_width == 0:
            rect_width = w*0.005

        if self.single_color:
            cr.set_source_rgb(self.tones[0], self.tones[1], self.tones[2])
        else:
            pat = cairo.LinearGradient(x0, y0, x0+w, y0)
            pat.add_color_stop_rgb(0, 0, 0.8, 0)
            pat.add_color_stop_rgb(0.45, 0.8, 0.8, 0)
            pat.add_color_stop_rgb(0.55, 0.8, 0.8, 0)
            pat.add_color_stop_rgb(1, 0.8, 0, 0)
            cr.set_source(pat)

        x1 = x0 + rect_width
        y1 = y0 + h
        radius = 3

        if self.round_edges and rect_width < 2:
            cr.move_to(x0, y0+radius)
            cr.curve_to(x0, y0+radius, x0, y0, x0 + radius, y0)
            cr.line_to(x0+2, y0)
            cr.line_to(x0+2, y1)
            cr.curve_to(x0+radius, y1, x0, y1, x0, y1-radius)
            cr.close_path()
            cr.fill()
            return
        elif not self.round_edges and rect_width < 2:
            cr.move_to(x0, y0)
            cr.line_to(x0+1, y0)
            cr.line_to(x0+1, y1)
            cr.line_to(x0, y1)
            cr.close_path()
            cr.fill()
            return

        if self.round_edges:
            cr.move_to(x0, y0+radius)
            cr.curve_to(x0, y0+radius, x0, y0, x0 + radius, y0)
        else:
            cr.move_to(x0, y0)

        if value == 1.0 and self.round_edges:
            cr.line_to(x1-radius, y0)
            cr.curve_to(x1-radius, y0, x1, y0, x1, y0+radius)
            cr.line_to(x1, y1-radius)
            cr.curve_to(x1, y1-radius, x1, y1, x1-radius, y1)
        else:
            cr.line_to(x1, y0)
            cr.line_to(x1, y1)

        if self.round_edges:
            cr.line_to(x0+radius, y1)
            cr.curve_to(x0+radius, y1, x0, y1, x0, y1-radius)
        else:
            cr.line_to(x0, y1)

        cr.close_path()
        cr.fill()

    def __draw_glow(self, cr, x0, y0, w, rect_height, value):
        rect_width = w*value
        if rect_width == 0:
            rect_width = w*0.005

        if rect_width < 6:
            return
        glow_proportion = 0.5
        pat = cairo.LinearGradient(x0, y0, x0, y0 + (rect_height*glow_proportion))
        pat.add_color_stop_rgba(0,  1, 1, 1, 0.95)
        pat.add_color_stop_rgba(1, 1, 1, 1, 0)
        cr.set_source(pat)

        spacing = 1

        x0 += spacing
        y0 += spacing
        rect_width -= spacing*2
        rect_height -= spacing*2
        x1 = x0 + rect_width
        y1 = y0 + rect_height
        radius = 3

        if self.round_edges:
            cr.move_to(x0, y0+radius)
            cr.curve_to(x0, y0+radius, x0, y0, x0 + radius, y0)
        else:
            cr.move_to(x0, y0)

        if value == 1.0 and self.round_edges:
                cr.line_to(x1-radius, y0)
                cr.curve_to(x1-radius, y0, x1, y0, x1, y0+radius)
        else:
                cr.line_to(x1, y0)
        cr.line_to(x1, y0+(rect_height*glow_proportion))
        cr.line_to(x0, y0+(rect_height*glow_proportion))
        cr.close_path()
        cr.fill()

        shadow_proportion = 1-glow_proportion
        pat = cairo.LinearGradient(x0, y1, x0, y1 - (rect_height*shadow_proportion))
        pat.add_color_stop_rgba(0,  0, 0, 0, 0.3)
        pat.add_color_stop_rgba(1, 0, 0, 0, 0)
        cr.set_source(pat)

        if self.round_edges:
            cr.move_to(x0, y1-radius)
            cr.curve_to(x0, y1-radius, x0, y1, x0 + radius, y1)
        else:
            cr.move_to(x0, y1)

        if value == 1.0 and self.round_edges:
                cr.line_to(x1-radius, y1)
                cr.curve_to(x1-radius, y1, x1, y1, x1, y1-radius)
        else:
                cr.line_to(x1, y1)
        cr.line_to(x1, y1-(rect_height*shadow_proportion))
        cr.line_to(x0, y1-(rect_height*shadow_proportion))
        cr.close_path()
        cr.fill()

    def __draw_border_line(self, cr, x0, y0, w, h, value):
        if self.draw_colored_border:
            if self.single_color:
                cr.set_source_rgb(self.tones[0]-0.4, self.tones[1]-0.4, self.tones[2])
            else:
                pat = cairo.LinearGradient(x0, y0, x0+w, y0)
                pat.add_color_stop_rgb(0, 0, 0.7, 0)
                pat.add_color_stop_rgb(1, 0.7, 0, 0)
                cr.set_source(pat)
        else:
            cr.set_source_rgb(0,0,0)

        x1 = x0 + w
        y1 = y0 + h
        radius = 3

        def draw_line(x0, y0, x1, y1):
            cr.move_to(x0, y0)
            cr.line_to(x1, y0)
            cr.line_to(x1, y1)
            cr.line_to(x0, y1)
            cr.close_path()
            cr.fill()

        if self.round_edges:
            _x0, _y0 = x0+2, y0+2
            _x1, _y1 = x1-2, y1-2

            draw_line(_x0, y0, _x1, y0+1)
            draw_line(x1, _y0, x1-1, _y1)
            draw_line(_x1, y1, _x0, y1-1)
            draw_line(x0, _y1, x0+1, _y0)

            cr.move_to(x0, y0+radius)
            cr.curve_to(x0, y0+radius, x0, y0, x0+radius, y0)
            cr.line_to(x0+radius, y0+1)
            cr.curve_to(x0+radius, y0+1, x0+1, y0+1, x0+1, y0+radius)
            cr.close_path()
            cr.fill()

            cr.move_to(x1-radius, y0)
            cr.curve_to(x1-radius, y0, x1, y0, x1, y0+radius)
            cr.line_to(x1-1, y0+radius)
            cr.curve_to(x1-1, y0+radius, x1-1, y0+1, x1-radius, y0+1)
            cr.close_path()
            cr.fill()

            cr.move_to(x1, y1-radius)
            cr.curve_to(x1, y1-radius, x1, y1, x1-radius, y1)
            cr.line_to(x1-radius, y1-1)
            cr.curve_to(x1-radius, y1-1, x1-1, y1-1, x1-1, y1-radius)
            cr.close_path()
            cr.fill()

            cr.move_to(x0+radius, y1)
            cr.curve_to(x0+radius, y1, x0, y1, x0, y1-radius)
            cr.line_to(x0+1, y1-radius)
            cr.curve_to(x0+1, y1-radius, x0+1, y1-1, x0+radius, y1-1)
            cr.close_path()
            cr.fill()
        else:
            draw_line(x0, y0, x1, y0+1)
            draw_line(x1, y0, x1-1, y1)
            draw_line(x1, y1, x0, y1-1)
            draw_line(x0, y1, x0+1, y0)

    def __draw_percentage(self, cr, x0, y0, w, h, value):
        txt_value = '%.2f'%(round(value*100, 2))
        if txt_value[-1] == '0':
            txt_value = txt_value[:-1]
        txt_value += ' %'

        cr.move_to(x0, y0)

        cr.set_source_rgb(0.0, 0.0, 0.0)
        cr.select_font_face("Sans", cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_NORMAL)
        cr.set_font_size(10)
        x_bearing, y_bearing, width, height = cr.text_extents(txt_value)[:4]
        cr.move_to((x0+w/2)-(width/2), (y0+h/2)+(height/2))
        cr.show_text(txt_value)

    def get_heat_color(self, value):
        green_tone = 1
        red_tone = 0
        if value < 0.5:
            green_tone = 1
            red_tone = value/0.5
            #red_tone = (2**(value/0.5))-1
        elif value == 0.5:
            green_tone = 1
            red_tone = 1
        elif value > 0.5:
            green_tone = 2-(value/0.5)
            #green_tone = log(value, 0.5)
            red_tone = 1
        return [round(red_tone, 3)*0.8, round(green_tone, 3)*0.8, 0.0]

gobject.type_register(CellRendererHeatBar)


class DeviceUsageBar(gtk.DrawingArea):
    __gsignals__ = { "expose-event": "override" }
    def __init__(self):
        gtk.DrawingArea.__init__(self)
        self.real_size = None
        self.usage = None
        self.width = None
        self.height = None
        self.cr = None
        self.stop_point = None
        self.percentage = None
        self.radius = 7

    def set_args(self, real_size, usage):
        self.real_size = real_size
        self.usage = usage
        self.percentage = self.usage/float(self.real_size)

    def do_expose_event(self, event):
        if self.real_size is None and self.usage is NOne:
            pass
            return
        self.cr = self.window.cairo_create()
        self.cr.set_line_width(1)
        self.width, self.height = self.window.get_size()
        self.height = 20
        self.set_size_request(-1, self.height)

        self.stop_point = self.width*self.percentage

        self.__draw_bg()
        self.__draw_bg_glow()
        self.__draw_bar()
        self.__draw_glow()
        self.__draw_border_line()
        self.__draw_percentage()

    def __draw_bg(self):
        x0 = 0
        y0 = 0
        x1 = self.width
        y1 = self.height

        pat = cairo.LinearGradient(0, 0, self.width, 0)
        pat.add_color_stop_rgb(0, 0, 0.8, 0)
        pat.add_color_stop_rgb(0.45, 0.8, 0.8, 0)
        pat.add_color_stop_rgb(0.55, 0.8, 0.8, 0)
        pat.add_color_stop_rgb(1, 0.8, 0, 0)
        self.cr.set_source(pat)

        self.cr.move_to(x0, y0+self.radius)
        self.cr.curve_to(x0, y0+self.radius, x0, y0, x0+self.radius, y0)

        self.cr.line_to(x1-self.radius, y0)
        self.cr.curve_to(x1-self.radius, y0, x1, y0, x1, y0+self.radius)
        self.cr.line_to(x1, y1-self.radius)
        self.cr.curve_to(x1, y1-self.radius, x1, y1, x1-self.radius, y1)

        self.cr.line_to(x0+self.radius, y1)
        self.cr.curve_to(x0+self.radius, y1, x0, y1, x0, y1-self.radius)

        self.cr.close_path()
        self.cr.fill()

    def __draw_bg_glow(self):
        x0,y0 = 0,0
        x1,y1 = self.width, self.height

        pat = cairo.LinearGradient(0, 0, 0, self.height)
        pat.add_color_stop_rgba(0, 1, 1, 1, 0.7)
        pat.add_color_stop_rgba(0.5, 1, 1, 1, 0.8)
        pat.add_color_stop_rgba(1, 1, 1, 1, 0.7)
        self.cr.set_source(pat)

        self.cr.move_to(x0, y0+self.radius)
        self.cr.curve_to(x0, y0+self.radius, x0, y0, x0 + self.radius, y0)

        self.cr.line_to(x1-self.radius, y0)
        self.cr.curve_to(x1-self.radius, y0, x1, y0, x1, y0+self.radius)
        self.cr.line_to(x1, y1-self.radius)
        self.cr.curve_to(x1, y1-self.radius, x1, y1, x1-self.radius, y1)

        self.cr.line_to(x0+self.radius, y1)
        self.cr.curve_to(x0+self.radius, y1, x0, y1, x0, y1-self.radius)

        self.cr.close_path()
        self.cr.fill()

    def __draw_bar(self):
        x0,y0 = 0,0
        if self.stop_point == 0:
            self.stop_point = self.width*0.005

        #rect_height = h - (spacing*2)

        pat = cairo.LinearGradient(0, 0, self.width, 0)
        pat.add_color_stop_rgb(0, 0, 0.8, 0)
        pat.add_color_stop_rgb(0.45, 0.8, 0.8, 0)
        pat.add_color_stop_rgb(0.55, 0.8, 0.8, 0)
        pat.add_color_stop_rgb(1, 0.8, 0, 0)
        self.cr.set_source(pat)

        x1 = self.stop_point
        y1 = self.height

        if self.stop_point < self.radius:
            self.cr.move_to(x0, y0+self.radius)
            self.cr.curve_to(x0, y0+self.radius, x0, y0, x0 + self.radius, y0)
            self.cr.line_to(x0+3, y0)
            self.cr.line_to(x0+3, y1)
            self.cr.curve_to(x0+self.radius, y1, x0, y1, x0, y1-self.radius)
            self.cr.close_path()
            self.cr.fill()
            return

        self.cr.move_to(x0, y0+self.radius)
        self.cr.curve_to(x0, y0+self.radius, x0, y0, x0+self.radius, y0)

        if self.percentage == 1.0:
            self.cr.line_to(x1-self.radius, y0)
            self.cr.curve_to(x1-self.radius, y0, x1, y0, x1, y0+self.radius)
            self.cr.line_to(x1, y1-self.radius)
            self.cr.curve_to(x1, y1-self.radius, x1, y1, x1-self.radius, y1)
        else:
            self.cr.line_to(x1, y0)
            self.cr.line_to(x1, y1)

        self.cr.line_to(x0+self.radius, y1)
        self.cr.curve_to(x0+self.radius, y1, x0, y1, x0, y1-self.radius)

        self.cr.close_path()
        self.cr.fill()

    def __draw_glow(self):
        if self.stop_point < 6:
            return

        x0,y0 = 0,0

        glow_proportion = 0.5
        pat = cairo.LinearGradient(x0, y0, x0, self.height*glow_proportion)
        pat.add_color_stop_rgba(0,  1, 1, 1, 0.95)
        pat.add_color_stop_rgba(1, 1, 1, 1, 0)
        self.cr.set_source(pat)

        spacing = 1

        #x0 += spacing
        #x1 = self.stop_point-(spacing)
        x1 = self.stop_point-1
        y1 = self.height

        self.cr.move_to(x0, y0+self.radius)
        self.cr.curve_to(x0, y0+self.radius, x0, y0, x0+self.radius, y0)

        if self.percentage == 1.0:
            self.cr.line_to(x1+1-self.radius, y0)
            self.cr.curve_to(x1+1-self.radius, y0, x1+1, y0, x1+1, y0+self.radius)
            self.cr.line_to(x1+1, y0+(self.height*glow_proportion))
        else:
            self.cr.line_to(x1, y0)
            self.cr.line_to(x1, y0+(self.height*glow_proportion))
        self.cr.line_to(x0, y0+(self.height*glow_proportion))
        self.cr.close_path()
        self.cr.fill()

        shadow_proportion = 1-glow_proportion
        pat = cairo.LinearGradient(x0, y1, x0, y1 - (self.height*shadow_proportion))
        pat.add_color_stop_rgba(0,  0, 0, 0, 0.3)
        pat.add_color_stop_rgba(1, 0, 0, 0, 0)
        self.cr.set_source(pat)


        self.cr.move_to(x0, y1-self.radius)
        self.cr.curve_to(x0, y1-self.radius, x0, y1, x0+self.radius, y1)

        if self.percentage == 1.0:
                self.cr.line_to(x1-self.radius, y1)
                self.cr.curve_to(x1-self.radius, y1, x1, y1, x1, y1-self.radius)
        else:
                self.cr.line_to(x1, y1)
        self.cr.line_to(x1, y1-(self.height*shadow_proportion))
        self.cr.line_to(x0, y1-(self.height*shadow_proportion))
        self.cr.close_path()
        self.cr.fill()

    def __draw_border_line(self):
        x0,y0 = 0,0
        x1,y1 = self.width,self.height

        self.cr.set_source_rgb(0,0,0)

        self.cr.move_to(x0, y0+self.radius)
        self.cr.curve_to(x0, y0+self.radius, x0, y0, x0+self.radius, y0)

        self.cr.line_to(x1-self.radius, y0)
        self.cr.curve_to(x1-self.radius, y0, x1, y0, x1, y0+self.radius)
        self.cr.line_to(x1, y1-self.radius)
        self.cr.curve_to(x1, y1-self.radius, x1, y1, x1-self.radius, y1)

        self.cr.line_to(x0+self.radius, y1)
        self.cr.curve_to(x0+self.radius, y1, x0, y1, x0, y1-self.radius)

        self.cr.close_path()
        self.cr.stroke()

    def __draw_percentage(self):
        txt_value = '%.2f'%(round(self.percentage*100, 2))
        if txt_value[-1] == '0':
            txt_value = txt_value[:-1]
        txt_value += ' %'

        self.cr.set_source_rgb(0.0, 0.0, 0.0)
        self.cr.select_font_face("Sans", cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_NORMAL)
        self.cr.set_font_size(12)
        x_bearing, y_bearing, w, h = self.cr.text_extents(txt_value)[:4]
        self.cr.move_to((self.width/2)-(w/2), (self.height/2)+(h/2))
        self.cr.show_text(txt_value)


class DeviceAnalysisInfo(gtk.Table):
    def __init__(self, device):
        self.r=7
        self.c=5
        gtk.Table.__init__(self, rows=self.r, columns=self.c, homogeneous=False)
        self.set_col_spacings(10)
        self.set_row_spacings(2)

        self.device = device
        self.convert_bytes = device.convert_bytes

        self.files_count = 0
        self.folders_count = 0

        self.add_device_image_and_bar()
        self.add_files_stats()
        self.add_folders_stats()


    def add_device_image_and_bar(self):
        label = MarkupLabel("<b><big>"+_("%(device_label)s Disk Usage Analysis")%{'device_label':self.device.label}+"</big></b>")
        label.set_property('xalign', 0.5)
        #self.attach(label, 1, self.c, 0, 1, xoptions=gtk.EXPAND|gtk.FILL, ypadding=5)

        img = CategoryImage(self.device)
        self.attach(img, 0, 1, 0, self.r-1, xoptions=gtk.FILL, yoptions=gtk.FILL)

        self.bar = DeviceUsageBar()
        self.bar.set_args(real_size=self.device.real_size, usage=self.device.get_folder_size(self.device.mount_point))
        self.attach(self.bar, 0, self.c, self.r-1, self.r, xoptions=gtk.EXPAND|gtk.FILL, yoptions=gtk.FILL)


    def add_files_stats(self):
        label = MarkupLabel("<b>"+_("Files statistics")+"</b>")
        #label.set_property('xalign', 0.5)
        label.set_property('xalign', 0)
        self.attach(label, 1, 3, 1, 2)

        label = gtk.Label(_('Total of files')+':')
        #label.set_property('xalign', 1)
        label.set_property('xalign', 0)
        self.attach(label, 1, 2, 2, 3, xoptions=gtk.FILL, xpadding=10)

        self.total_files = gtk.Label(locale.format("%d",self.device.files_count,1))
        self.total_files.set_property('xalign', 0)
        self.attach(self.total_files, 2, 3, 2, 3, xoptions=gtk.EXPAND|gtk.FILL)

        label = gtk.Label(_('Average file size')+':')
        #label.set_property('xalign', 1)
        label.set_property('xalign', 0)
        self.attach(label, 1, 2, 3, 4, xoptions=gtk.FILL, xpadding=10)

        self.avg_file_size = gtk.Label('')
        self.avg_file_size.set_property('xalign', 0)
        self.attach(self.avg_file_size, 2, 3, 3, 4, xoptions=gtk.EXPAND|gtk.FILL)

    def add_folders_stats(self):
        label = MarkupLabel("<b>"+_("Folders statistics")+"</b>")
        #label.set_property('xalign', 0.5)
        label.set_property('xalign', 0)
        self.attach(label, 3, 5, 1, 2)

        label = gtk.Label(_('Total of folders')+':')
        #label.set_property('xalign', 1)
        label.set_property('xalign', 0)
        self.attach(label, 3, 4, 2, 3, xoptions=gtk.FILL, xpadding=10)

        self.total_folders = gtk.Label(locale.format("%d",self.device.folders_count,1))
        self.total_folders.set_property('xalign', 0)
        self.attach(self.total_folders, 4, 5, 2, 3, xoptions=gtk.EXPAND|gtk.FILL)

        label = gtk.Label(_('Average Folder Size')+':')
        #label.set_property('xalign', 1)
        label.set_property('xalign', 0)
        self.attach(label, 3, 4, 3, 4, xoptions=gtk.FILL, xpadding=10)

        self.avg_folder_size = gtk.Label(locale.format("%d",self.device.folders_count,1))
        self.avg_folder_size.set_property('xalign', 0)
        self.attach(self.avg_folder_size, 4, 5, 3, 4, xoptions=gtk.EXPAND|gtk.FILL)

    def update_data(self):
        self.total_files.set_text(locale.format("%d",self.device.files_count,1))
        if self.device.files_count == 0:
            self.avg_file_size.set_text('0 Bytes')
        else:
            self.avg_file_size.set_text(self.convert_bytes(self.device.real_size/float(self.device.files_count)))

        self.total_folders.set_text(locale.format("%d",self.device.folders_count,1))
        if self.device.folders_count == 0:
            self.avg_folder_size.set_text('0 Bytes')
        else:
            self.avg_folder_size.set_text(self.convert_bytes(self.device.real_size/float(self.device.folders_count)))



class SizeListView(gtk.TreeView):
    __gsignals__ = {
                    'item-selected' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, (gobject.TYPE_PYOBJECT,)),
                   }
    def __init__(self, device):
        gtk.TreeView.__init__(self)
        self.connect("button-press-event", self.__on_row_click)
        self.connect("button-release-event", self.__on_row_click_leave)
        self.connect("row-expanded", self.__on_row_expanded)

        self.device = device
        self.udi = device.udi

        self.convert_bytes = device.convert_bytes
        self.get_folder_size = device.get_folder_size
        self.get_file_size = device.get_file_size

        self.__add_pxbuf_column(_("Name"), 0)
        self.__add_bar_column(_("Occupation"), 1)
        self.__add_text_column(_("Size"), 2, sort_id=4)


        self.size_list_store = gtk.TreeStore(str, float, str, str, float, str, bool)
        self.set_model(self.size_list_store)

        tree = self.device.scan_folder()
        self.populate(tree)

        self.size_list_store.set_sort_column_id(1, gtk.SORT_DESCENDING)
        self.expand_row((0,), False)
        self.set_rules_hint(True)

    def __add_pxbuf_column(self, title, id):
        cellpb = gtk.CellRendererPixbuf()
        cell = gtk.CellRendererText()
        column = gtk.TreeViewColumn(title)
        column.set_sort_column_id(id)
        column.pack_start(cellpb, False)
        column.pack_start(cell, False)
        column.set_cell_data_func(cellpb, self.__set_pixbuf)
        column.set_cell_data_func(cell, self.__set_name)
        self.append_column(column)

    def __add_text_column(self, title, id, visible=True, sort_id=None):
        cell = gtk.CellRendererText()
        cell.set_property('xalign', 1)
        column = gtk.TreeViewColumn(title, cell, text=id, markup=0)
        column.set_resizable(True)
        if sort_id is None:
            column.set_sort_column_id(id)
        else:
            column.set_sort_column_id(sort_id)
        column.set_visible(visible)
        self.append_column(column)

    def __add_bar_column(self, title, id, visible=True):
        cellhb = CellRendererHeatBar()
        cellhb.draw_bg = False
        cellhb.single_color = True
        cellhb.round_edges = False
        cellhb.draw_border = True
        cellhb.draw_colored_border = False
        column = gtk.TreeViewColumn(title, cellhb, value=id)
        column.set_min_width(150)
        column.set_sort_column_id(id)
        self.append_column(column)

    def __set_pixbuf(self, column, cell, model, iter):
        stock = model.get_value(iter, 3)
        pb = self.render_icon(stock, gtk.ICON_SIZE_MENU, None)
        cell.set_property('pixbuf', pb)

    def __set_name(self, column, cell, model, iter):
        cell.set_property('text', model.get_value(iter, 0))
        cell.set_property('xalign', 0)

    def __on_row_click(self, widget, event):
        path = widget.get_path_at_pos(int(event.x), int(event.y))
        selection = widget.get_selection()
        if path is None:
            selection.unselect_all()
            self.emit('item-selected', None)

    def __on_row_click_leave(self, widget, event):
        path = widget.get_path_at_pos(int(event.x), int(event.y))
        selection = widget.get_selection()
        model, selection_iter = selection.get_selected()
        if path:
            name = None
            try:
                name = model.get_value(selection_iter, 0)
            except:
                return
            f_type = model.get_value(selection_iter, 3)
            full_path = model.get_value(selection_iter, 5)
            self.emit('item-selected', [name, f_type, full_path])

    def __on_row_expanded(self, widget, iter, path):
        parent_iter = iter
        model = self.get_model()
        if model.get_value(parent_iter, 6):
            return

        self.size_list_store.set_value(iter, 6, True)
        parent_size = model.get_value(parent_iter, 4)
        for n in range(model.iter_n_children(parent_iter)):
            child_iter = model.iter_nth_child(parent_iter, n)
            child_type = model.get_value(child_iter, 3)
            child_full_path = model.get_value(child_iter, 5)
            child_size = 0

            if 'gtk-file' == child_type:
                child_size = self.get_file_size(child_full_path)
            elif 'gtk-directory' == child_type:
                child_size = self.get_folder_size(child_full_path)

            child_allocation = 0
            if parent_size is not 0:
                child_allocation = (child_size/float(parent_size))

            model.set_value(child_iter, 4, child_size)
            model.set_value(child_iter, 2, self.convert_bytes(child_size))
            model.set_value(child_iter, 1, child_allocation)

    def populate(self, tree, parent_iter=None):
        if parent_iter is None:
            root_size = self.device.get_folder_size(self.device.mount_point)
            child_iter = self.size_list_store.append(None, [self.device.mount_point, -1, self.convert_bytes(root_size), gtk.STOCK_DIRECTORY, root_size, self.device.mount_point, False])
            self.populate(tree, parent_iter=child_iter)
            return

        for child in tree:
            if child[-1] is None:
                self.size_list_store.append(parent_iter, [child[0], 0, '0 bytes', gtk.STOCK_FILE, 0, child[1], True])
            else:
                child_iter = self.size_list_store.append(parent_iter, [child[0], 0, '0 bytes', gtk.STOCK_DIRECTORY, 0, child[1], False])
                self.populate(child[-1], parent_iter=child_iter)


#class DetailedUsage(gtk.VBox):
#    def __init__(self, item):
#        gtk.VBox.__init__(self)
