#!/usr/bin/python
# -*- coding: utf-8 -*-

# trayapplet.py
#
# Copyright (c) 2008 Magnun Leno da Silva
#
# Author: Magnun Leno da Silva <magnun.leno@gmail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
# USAA.

import gobject
import pygtk
pygtk.require('2.0')
import gtk
import locale
import gettext
import pynotify
import os

BASE = None
locale.setlocale(locale.LC_ALL, '')

if os.path.exists("data") and os.path.exists('locale'):
    BASE = "data/icons/"
    gettext.bindtextdomain('usbmanager', './locale')
else:
    BASE = "/usr/share/icons/hicolor/"
    gettext.bindtextdomain('usbmanager', '/usr/share/locale')


gettext.textdomain('usbmanager')
_ = gettext.gettext



SMALL_DEVICE_ICONS = {
               "volume"                 : BASE+"24x24/apps/storage.png",
               "portable_audio_player"  : BASE+"24x24/apps/m_player.png",
              }

DEFAULT_SMALL_DEVICE_ICON = BASE+"24x24/apps/storage.png"
TRAY_ICON = BASE+"24x24/apps/usbmanager.png"

class TrayManager(gobject.GObject):
    '''
        Class that creates the system tray applet.
    '''
    __gsignals__ = {
                    'destroying-window' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, ()),
                   }
    def __init__(self, master):
        '''
            Startup routine:
             - Create the StatusIcon using the small_icon in the data folder;
             - Create the popupmenu;
             - Connect the signals;
        '''
        gobject.GObject.__init__(self)
        self.master = master
        self.devices = self.master.devices
        self.items = {}
        self.notification_id = None

        # Icon
        self.statusIcon = gtk.StatusIcon()
        self.statusIcon.set_from_file(TRAY_ICON)
        self.statusIcon.set_visible(True)
        self.statusIcon.set_tooltip(_("USB Volume Storage Manager"))

        # Menu
        self.menu = gtk.Menu()
        self.add_open_main_window()
        self.add_separator()

        for udi, device in self.devices:
            self.add_device_button(udi, device)

        self.add_separator()
        self.add_quit_tray_button()

        # Connect signals
        self.statusIcon.connect('popup-menu', self.on_popup_menu, self.menu)
        self.statusIcon.connect("activate", self.on_open_main_window)

        # Notification
        pynotify.init(_("USB Volume Storage Manager"))

        self.statusIcon.set_visible(1)

    def notify(self, title, body, icon="dialog-information"):
        '''
            Displays a simple notification based in the arguments:
             @ title - A text displayed in bold
             @ body - The body of the message
             @ icon - An icon used to identify the 'type' of the notification
        '''
        # Create notification
        notification = pynotify.Notification(title, body, icon)
        notification.attach_to_status_icon(self.statusIcon)
        notification.set_urgency(pynotify.URGENCY_CRITICAL)
        # Timeout
        #notification.set_timeout(5000)
        # Show
        self.notification_id = notification.show()

    def notify_new_device(self, new_device):
        self.notify(_("New device"), _("A new USB device was added: %s")%(new_device.label))

    def notify_removed_device(self, removed_device):
        if removed_device.is_mounted:
            self.notify(_("Device removed incorrectly!"), _("The USB device '%s' was removed while mounted")%(removed_device.label), "dialog-warning")
        else:
            self.notify(_("Device removed"), _("The USB device '%s' was removed correctly")%(removed_device.label))

    def notify_changed_device(self, device, type):
        if type == "mount":
            self.notify(_("Device mounted!"), _("The USB device '%s' was mounted successfully")%(device.label))
        else:
            self.notify(_("Device umounted!"), _("The USB device '%s' was umounted successfully")%(device.label))

    def add_open_main_window(self):
        '''
            Add the 'Open USBManager' button to the context menu.
        '''
        menuItem = gtk.MenuItem(_('Open USBManager'))
        menuItem.connect('activate', self.on_open_main_window, self.statusIcon)
        self.menu.append(menuItem)

    def add_device_button(self, udi, device):
        '''
            Function made to help adding the MenuItems.
            It adds an item to the context menu based on the arguments
             @ udi - The unic device identifier used as a 'key'
             @ device - An isntance of the Device class defined in usbdbus.py
        '''
        # New menu item (Check menu)

        # Maybe implement a image and a label to use Tango Markup
        #label = gtk.Label()
        #label.set_markup('<span foreground="red">'+device.label+'</span>')
        #label.set_alignment(xalign=0, yalign=0.5)
        #
        #image = None
        #if SMALL_DEVICE_ICONS.has_key(device.category):
        #    image = gtk.image_new_from_file(SMALL_DEVICE_ICONS[device.category])
        #else:
        #    image = gtk.image_new_from_file(DEFAULT_SMALL_DEVICE_ICON)
        #image.set_alignment(xalign=0, yalign=0.5)
        #
        #box = gtk.HBox()
        #box.set_spacing(5)
        #box.pack_start(image, expand=False, fill=True)
        #box.pack_start(label, expand=True, fill=True)

        menuItem = gtk.CheckMenuItem(device.label)
        #menuItem.add(box)
        menuItem.set_active(device.is_mounted)
        menuItem.connect('activate', self.on_device_clicked, self.statusIcon, udi)
        # Append the menu item
        self.menu.append(menuItem)
        # Add the menu item in the items dictionary
        self.items[udi] = menuItem
        return menuItem

    def add_separator(self):
        '''
            A simple function that adds a separator in the context menu
        '''
        menuItem = gtk.SeparatorMenuItem()
        self.menu.append(menuItem)

    def add_quit_tray_button(self):
        '''
            Function that add the quit button with to the context menu
        '''
        menuItem = gtk.ImageMenuItem(gtk.STOCK_QUIT)
        menuItem.connect('activate', self.destroy, self.statusIcon)
        self.menu.append(menuItem)

    def refresh_single_device(self, udi):
        '''
            Refresh the menu item of a single device based in the udi.
        '''
        index = len(self.menu.get_children())-2
        if udi in self.items.keys():
            # Get menuItem, device and old index
            menuitem = self.items[udi]
            device = self.devices.get_device_by_udi(udi)
            index = self.menu.get_children().index(menuitem)
            # Destroy old menuItem
            menuitem.destroy()
        else:
            device = self.devices.get_device_by_udi(udi)

        # New menuItem
        menuitem = self.add_device_button(udi, device)
        # Move to the old index
        self.menu.reorder_child(menuitem, index)
        # TODO: Make it red for blocked devices

    def add_new_device(self, device):
        '''
            Called by the Master when a new device is detected
        '''
        # The device was already added??
        if self.items.has_key(device.udi):
            # Yes, so we do nothing
            return
        # Get the index from the bottom to add the new devide
        index = len(self.menu.get_children()) - 2
        # New Menu Item
        menuitem = self.add_device_button(device.udi, device)
        self.menu.reorder_child(menuitem, index)
        self.notify_new_device(device)

    def remove_device(self, removed_device):
        '''
            Called by the Master when a device is removed
        '''
        if self.items.has_key(removed_device.udi):
            menuitem = self.items.pop(removed_device.udi)
            menuitem.destroy()
            self.notify_removed_device(removed_device)

    ############### Envent Functions ###############
    def on_popup_menu(self, widget, button, time, data = None):
        '''
            Function that shows the context menu when the Status icon is
            clicked with the right mouse button
        '''
        if button == 3:
          if data:
            data.show_all()
            data.popup(None, None, gtk.status_icon_position_menu,
                       3, time, self.statusIcon)

    def on_open_main_window(self, widget, event=None):
        '''
            Called when the 'Open USBManager' button is pressed
        '''
        # This mimics the show/hide window functionality
        # If the StatusIcon was clicked with left mouse button event is None
        # If there is an showing istance of MainWindow
        if event is None and self.master.main_window:
            # Destroy the existing window
            self.emit('destroying-window')
            return

        # If was used the context menu item and there isn't an instance of
        # MainWindow currently showing
        self.master.create_main_window()

    def on_device_clicked(self, widget, event, udi):
        '''
            Called when the user click in the Mounted column.
            The function will mount or umount the device based in its
            current status
        '''
        device = self.devices.get_device_by_udi(udi)
        if device.blocked:
            self.refresh_single_device(udi)
        if widget.get_active():
            ret = device.mount()
            if ret:
                self.notify_changed_device(device, "mount")
        else:
            ret = device.umount()
            if ret:
                self.notify_changed_device(device, "umount")
        if ret is False:
            self.refresh_single_device(udi)

    ############### Basic Window functions ###############
    def destroy(self, widget, data = None):
        '''
            Quit the System Tray
        '''
        print 'Quiting...'
        self.emit('destroying-window')
        gtk.main_quit()

    def run(self):
        '''
            GTK main loop.
        '''
        gtk.main()
