<?php

/**
  * SquirrelMail Shared Calendar Plugin
  * Copyright (C) 2004-2005 Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under GPL. See COPYING for details
  *
  */



include_once(SM_PATH . 'functions/date.php');
include_once(SM_PATH . 'plugins/calendar/constants.php');
include_once(SM_PATH . 'plugins/calendar/classes/calendar.php');
include_once(SM_PATH . 'plugins/calendar/classes/event.php');
include_once(SM_PATH . 'plugins/calendar/classes/property.php');
include_once(SM_PATH . 'plugins/calendar/backend_functions.php');
include_once(SM_PATH . 'plugins/calendar/data/config.php');
include_once(SM_PATH . 'plugins/calendar/timezone_offsets.php');
include_once(SM_PATH . 'plugins/calendar/load_prefs.php');



// load prefs (can't use loading_prefs hook due to a chicken-and-egg
// problems with i18n'd strings in constants file (that is needed by
// config file)
//
cal_load_prefs_do();



global $externalCalendars;
if (!is_array($externalCalendars)) $externalCalendars = array();



/** 
  * Because there will always be someone who forgets
  * to enable a backend, check for one here.  Errors
  * otherwise are hard to diagnose
  *
  */
global $squirrelmail_plugin_hooks, $color;
if (empty($squirrelmail_plugin_hooks['get_calendar']))
{
   plain_error_message('ERROR: Please enable a calendar backend', $color);
   exit;
}



/**
  * Can we use JavaScript-enabled display elements?
  *
  * Ideally, this function would just be replaced with a
  * call to checkForJavascript() but we need newest version
  * of the Compatibility plugin for that, so we just do the
  * dirty work ourselves (it's not much trouble anyway)
  *
  * @return mixed TRUE (or non-zero) if JavaScript is allowable,
  *               FALSE (or zero) otherwise.
  *
  */
function cal_use_javascript()
{

   // only in 1.5.x
   //
   if (function_exists('checkForJavascript'))
      return checkForJavascript();


   // distilled from checkForJavascript()
   //
   if (sqGetGlobalVar('javascript_on', $javascript_on, SQ_SESSION))
      return $javascript_on;

   global $javascript_setting;
   if (!isset($javascript_setting))
      $javascript_setting = getPref($data_dir, $username, 'javascript_setting', SMPREF_JS_OFF);
   return $javascript_setting;

}



/**
  * Get list of all external calendar from user prefs
  *
  * @return array Returns a list of all external calendars 
  *
  */
function get_all_external_calendars()
{

   global $username, $data_dir;

   // get external calendars
   //
   $external_calendars = getPref($data_dir, $username, 'external_calendars', '');
   $externalCals = explode('||||', $external_calendars);
   $external_list = array();
   foreach ($externalCals as $externalCal)
   {
      if (empty($externalCal)) continue;
   
      list($id, $name, $uri) = explode('^^^^', $externalCal);
      $cal = loadExternalCalendar($uri);
//TODO: if just a temporary network outage, we might not want to just remove this... too harsh
// so figure out how to deal with that better
      if ($cal == FALSE)
      {
         removeExternalCalendarFromUserPrefs($id);
         continue;
      }
      $external_list[] = $cal;
   }
   
   return $external_list;

}



/**
  * Load calendar from user prefs
  *
  * @param string $calendarID The ID of the calendar to be loaded
  *
  * @return mixed Returns the Calendar object that represents the
  *               loaded calendar, or FALSE if the calendar could
  *               not be loaded
  *
  */
function loadExternalCalendarFromUserPrefs($calendarID)
{

   global $username, $data_dir;
   $external_calendars = getPref($data_dir, $username, 'external_calendars', '');

   $externalCals = explode('||||', $external_calendars);
   foreach ($externalCals as $externalCal)
   {
      if (empty($externalCal)) continue;

      list($id, $name, $uri) = explode('^^^^', $externalCal);

      if ($id == $calendarID)
         return loadExternalCalendar($uri);
   }

   return FALSE;

}



/**
  * Load event from external calendar
  *
  * @param string $calendarID The ID of the calendar from which to get the event
  * @param string $eventID The ID of the event to be loaded
  *
  * @return mixed Returns the Event object that represents the
  *               loaded event, or FALSE if the event could
  *               not be loaded
  *
  */
function loadExternalEvent($calendarID, $eventID)
{

   $cal = loadExternalCalendarFromUserPrefs($calendarID);
   if ($cal === FALSE) return FALSE;

   return $cal->getEvent($eventID);

}



/**
  * Load calendar from external URI
  *
  * @param string $URI The location of the calendar to be loaded
  *
  * @return mixed Returns the Calendar object that represents the
  *               loaded calendar, or FALSE if the calendar could
  *               not be loaded
  *
  */
function loadExternalCalendar($URI, $calID='')
{

   global $username, $data_dir, $external_calendar_clock_skew,
          $externalCalendars;
   sqgetGlobalVar('externalCalendars', $externalCalendars, SQ_SESSION);


   // since calendar IDs are date/time-encoded (based on creation
   // date), we need to manually override the ID to what is given
   // in user's prefs for when it was first linked (only if found
   // in user prefs, that is)
   //
   $correctID = '';
   $correctName = '';
   $external_calendars = getPref($data_dir, $username, 'external_calendars', '');

   $externalCals = explode('||||', $external_calendars);
   foreach ($externalCals as $externalCal)
   {
      if (empty($externalCal)) continue;

      list($id, $name, $calUri) = explode('^^^^', $externalCal);

      if ($calUri == $URI)
      {
         $correctID = $id;
         $correctName = $name;
         break;
      }
   }


   $headers = get_headers($URI, 1);
   if ($headers === FALSE) $headers = array();


   // look in cache first
   //
   if (!empty($correctID) && isset($externalCalendars[$correctID]))
   {
      $cal = unserialize($externalCalendars[$correctID]);

      // check if URI has since been modified...
      //
      if (!isset($headers['Last-Modified']))
         $mtime = $cal->createdOn();
      else
         $mtime = strtotime($headers['Last-Modified']);
// don't need the clock skew here, since mod time on the calendar object
// itself should have come from the original file's mtime
// and if not, we add it below
//      if ($mtime - ($external_calendar_clock_skew * 60) > $cal->lastUpdatedOn())
      if ($mtime > $cal->lastUpdatedOn())
      {
         unset($externalCalendars[$cal->getID()]);
         sqsession_register($externalCalendars, 'externalCalendars');
      }
      else
         return $cal;

   }


   $CAL = @fopen($URI, 'rb');
   if ($CAL === FALSE) return FALSE;

   $calContents = '';
   while (!feof($CAL)) $calContents .= fread($CAL, 4096);

   fclose($CAL);

   $calContentArray = explode("\r\n", $calContents);
   if (sizeof($calContentArray) == 1) $calContentArray = explode("\n", $calContents);

   $cal = Calendar::getCalendarFromICal($calContentArray);
   $cal->setExternal(TRUE);
   if (!empty($correctID)) $cal->setID($correctID);
   if (!empty($correctName)) $cal->setName($correctName);
   if (!isset($headers['Last-Modified']))
      $mtime = $cal->createdOn() + ($external_calendar_clock_skew * 60);
   else
      $mtime = strtotime($headers['Last-Modified']);
   $cal->setLastUpdateDate(gmdate('Ymd\THis\Z', $mtime));


   // cache it
   //
   $externalCalendars[$cal->getID()] = serialize($cal);
   sqsession_register($externalCalendars, 'externalCalendars');


   return $cal;

}



/**
  * Remove calendar from user prefs
  *
  * @param string $calendarID The ID of the calendar to be removed
  *
  */
function removeExternalCalendarFromUserPrefs($calendarID)
{

   global $username, $data_dir, $externalCalendars, $small_calendar_calID,
          $small_calendar_calID_default;

   $external_calendars = getPref($data_dir, $username, 'external_calendars', '');
   sqgetGlobalVar('externalCalendars', $externalCalendars, SQ_SESSION);
   $new_external_list = '';

   $first = TRUE;
   $externalCals = explode('||||', $external_calendars);
   foreach ($externalCals as $externalCal)
   {
      if (empty($externalCal)) continue;

      list($id, $name, $uri) = explode('^^^^', $externalCal);

      if ($id == $calendarID)
      {
         // remove from cache
         //
         if (isset($externalCalendars[$id]))
         {
            unset($externalCalendars[$id]);
            sqsession_register($externalCalendars, 'externalCalendars');
         }

         continue;
      }

      if (!$first) $new_external_list .= '||||';

      $new_external_list .= $externalCal;

      $first = FALSE;
   }
 
   setPref($data_dir, $username, 'external_calendars', $new_external_list);


   // if small calendar was using this calendar, just 
   // fall back to default calendar
   //
   if ($small_calendar_calID == $calendarID)
   {
      setPref($data_dir, $username, 'small_calendar_calID', $small_calendar_calID_default);
   }

}



/**
  * Get calendar ID for personal calendar
  *
  * Retrieves the calendar ID for the given user's 
  * personal calendar
  *
  * This function also checks to make sure that this personal
  * calendar actually exists and creates it if it does not.
  *
  * @param string $user The user whose personal calendar ID is being retrieved
  * @param string $domain The user's domain
  * @param boolean $dontCheckExists If TRUE, this function will not check
  *                                 to make sure the calendar exists (optional;
  *                                 default = FALSE)
  *
  * @return string The desired calendar ID
  *
  */
function get_personal_cal_id($user, $domain, $dontCheckExists=FALSE)
{

   global $useDomainInCalID;

   $calID = strtr('sm_cal_' . $user . ($useDomainInCalID ? '__' . $domain : ''), 
                  '@|_-.:/\ ', '________');


   // if calendar data file doesn't exist, create it
   //
   if (!$dontCheckExists && get_calendar($calID, TRUE) === FALSE)
   {

      bindtextdomain ('calendar', SM_PATH . 'locale');
      textdomain ('calendar');

      $calendar = new Calendar($calID, 0, $domain, '', SM_CAL_TYPE_PERSONAL,
                               sprintf(_("Personal Calendar for %s"), $user),
                               $user, gmdate('Ymd\THis\Z'), '', '', array($user));

      bindtextdomain ('squirrelmail', SM_PATH . 'locale');
      textdomain ('squirrelmail');

      create_calendar($calendar);

   }

   return $calID;

}



/** 
  * Build event start/end time string for display
  *
  * For example, "[Oct 14, 2004 10:00 - Oct 24, 2004 12:00]" 
  * or "[All Day]"
  *
  * @param object $event The event object for which to build
  *                      the start/end time string for display
  * @param int $year The year where this event is being displayed
  * @param int $month The month where this event is being displayed
  * @param int $day The day where this event is being displayed
  *
  * @return string The start/end time string, ready for display
  *
  */
function buildEventTimeDisplayText($event, $year, $month, $day)
{

   global $hour_format, $always_show_full_event_date_and_time;

   $text = ' [';

   $startsToday = $event->startsOnDay($year, $month, $day);
   $endsToday = $event->endsOnDay($year, $month, $day);

   // NOTE: should not need to switch gettext domain here, since
   //       this function should be being called from within an
   //       interface function that is already in the correct calendar
   //       gettext domain.... right?

   // all-day events render differently
   //
   if ($event->isAllDay())
   {

      if (!$startsToday && !$endsToday
       || ($always_show_full_event_date_and_time && 
        (!$startsToday || !$endsToday)))
         $text .= $event->formattedStartDate('', $year, $month, $day) . ' - '
                . $event->formattedEndDate('', $year, $month, $day);

      else if (!$startsToday)
         $text .= $event->formattedStartDate('', $year, $month, $day) . ' - '
                . _("Today");

      else if (!$endsToday)
         $text .= _("Today") . ' - '
                . $event->formattedEndDate('', $year, $month, $day);

      else
         $text .= _("All Day");

      $text .= "]";

   }


   // regular non-all-day events
   //
   else
   {

      if (!$startsToday || $always_show_full_event_date_and_time)
         $text .= $event->formattedStartDate('', $year, $month, $day) . ' ';

      $text .= $event->formattedStartTime($hour_format == SMPREF_TIME_24HR ? 'H:i' : 'g:ia') . ' - ';

      if (!$endsToday || $always_show_full_event_date_and_time)
         $text .= $event->formattedEndDate('', $year, $month, $day) . ' ';

      $text .= $event->formattedEndTime($hour_format == SMPREF_TIME_24HR ? 'H:i' : 'g:ia') . "]";

   }

   return $text;
}



/** 
  * Inserts link to calendar module in main menu bar
  *
  */
function cal_menu_link_do()
{

   bindtextdomain ('calendar', SM_PATH . 'locale');
   textdomain ('calendar');

   displayInternalLink("plugins/calendar/list_calendars.php", _("Calendar"));
   echo "&nbsp;&nbsp\n";

   bindtextdomain ('squirrelmail', SM_PATH . 'locale');
   textdomain ('squirrelmail');

}



/** 
  * Inserts link to calendar admin module and user settings
  * page in options page
  *
  */
function cal_options_block_do()
{

   global $username, $optpage_blocks, $cal_user_can_override_defaults;


   bindtextdomain ('calendar', SM_PATH . 'locale');
   textdomain ('calendar');

   if ($cal_user_can_override_defaults)
      $optpage_blocks[] = array(
         'name' => _("Calendar Preferences"),
         'url'  => SM_PATH . 'plugins/calendar/calendar_options.php',
         'desc' => _("Change the way your calendars behave and are displayed, including the miniature calendar beneath the folder list."),
         'js'   => false
      );


   $userType = check_cal_user_type($username);
 
   if (!($userType == SM_CAL_SUPERUSER || $userType == SM_CAL_LIMITED_ADMIN))
   {
      bindtextdomain ('squirrelmail', SM_PATH . 'locale');
      textdomain ('squirrelmail');
      return;
   }


   $optpage_blocks[] = array(
      'name' => _("Calendar Administration"),
      'url'  => SM_PATH . 'plugins/calendar/admin_options.php',
      'desc' => _("Create and maintain shared calendars. Edit holidays, assign calendar access by user, or create publicly available calendars."),
      'js'   => false
   );

   bindtextdomain ('squirrelmail', SM_PATH . 'locale');
   textdomain ('squirrelmail');

}



/**
  * Sort events by (start) time
  *
  * @param object $a Event A
  * @param object $b Event B
  *
  * @return -1 if event A comes before event B, 1 if event
  *         B comes before event A, or zero if the two are equal.
  *
  */
function sortEventsByTime($a, $b)
{

   return $a->compareTo($b);

}



//TODO -- do we need a version of this that doesn't work by reference???
/**
  * Fold text (by reference) for iCal data stream
  *
  * Text is folded after every 75 characters by using
  * a CRLF and one space.  These can be overridden
  * using the parameters for this function.
  *
  * @param string $text The text value to be folded
  * @param int $maxLineLength The maximum allowable 
  *                           length of any one line
  *                           (optional; default = 75)
  * @param string $foldDelimiter The character sequence
  *                              used to insert a fold
  *                              (optional; default = 
  *                              CRLF + SPACE)
  *
  */
function foldICalStreamByRef(&$text, $maxLineLength=75, $foldDelimiter='')
{

   if (empty($foldDelimiter)) $foldDelimiter = ICAL_LINE_DELIM . ' ';
   $newText = '';
   while (strlen($text) > 0)
   {

      $EOL = strpos($text, ICAL_LINE_DELIM);

      if ($EOL <= $maxLineLength)
         $cutoff = $EOL + strlen(ICAL_LINE_DELIM);
      else
         $cutoff = $maxLineLength;

      $newText .= substr($text, 0, $cutoff);
      $text = substr($text, $cutoff);

      if ($EOL > $maxLineLength)
         $newText .= $foldDelimiter;

   }
   $text = $newText;

}



//TODO -- do we need a version of this that doesn't work by reference???
/**
  * Unfold an iCal data stream (by reference)
  *
  * Text is unfolded based on the presence of CRLF followed
  * by one whitespace character (space or tab).  These (but
  * not the CRLF) can be overridden using this function's
  * parameters.
  *
  * @param array $iCalData The text value to be unfolded,
  *                        one text line in each array value.
  *                        NOTE that it is assumed that the
  *                        CRLF characters have already been
  *                        stripped from each line!
  * @param array $foldDelimiters An array of characters which,
  *                              when immediately following a 
  *                              CRLF, indicate a text fold
  *                              (optional; default = space 
  *                              and tab)
  *
  */
function unfoldICalStreamByRef(&$iCalData, $foldDelimiters='')
{

   if (empty($foldDelimiters) || !is_array($foldDelimiters)) 
      $foldDelimiters = array(' ', "\t");
   $newData = array();
   $x = -1;
   foreach ($iCalData as $line)
   {

      $fold = FALSE;
      foreach ($foldDelimiters as $delim)
         if (strpos($line, $delim) === 0)
         {
            $fold = TRUE;
            break;
         }
      if ($fold)
         $newData[$x] .= substr($line, 1);
      else
         $newData[++$x] = $line;

   }
   $iCalData = $newData;

}



/**
  * Utility function for sorting PERIOD iCal values
  * given as two arrays, each array containing two
  * elements: a starting timestamp and and ending 
  * timestamp.
  *
  * @param array $a First PERIOD array to compare
  * @param array $b Second PERIOD array to compare
  *
  * @return -1 if the first PERIOD comes first, 1 if 
  *         the second PERIOD comes first, or 0 if 
  *         the two are equal
  *
  */
function sortPeriods($a, $b)
{

   if ($a[0] < $b[0]) return -1;
   if ($a[0] > $b[0]) return 1;
   if ($a[1] < $b[1]) return -1;
   if ($a[1] > $b[1]) return 1;
   return 0;

}



/** 
  * Get timestamp in correct timezone
  *
  * @param int $time Original timestamp
  * @param string $zone Requested timezone
  *
  * @return int Modified timestamp, or same timestamp
  *             if timezone was unknown
  *
  */
function convertTimestampToTimezone($time, $zone)
{

   global $tz_array, $color;

   $isDST = date('I', $time);

   // try to guess when dash was used instead of forward slash:
   // US-Eastern --> US/Eastern
   //
   $dash = strpos($zone, '-');
   if (!array_key_exists($zone, $tz_array) && $dash !== FALSE)
      $zone = substr($zone, 0, $dash) . '/' . substr($zone, $dash + 1);

   if (array_key_exists($zone, $tz_array))
      $offset = $tz_array[$zone][$isDST];
   else
      $offset = '+0000';

   return getGMTSeconds($time, $offset);

}



/**
  * Determines what kind of calendar user the given username is; checks
  * the user's privelege level.
  *
  * @param string $user The username to check for admin permission
  * 
  * @return string Gives back one of the calendar user type constants:
  *                SM_CAL_SUPERUSER (user can edit any and all calendars),
  *                SM_CAL_LIMITED_ADMIN (user can edit calendars they own), 
  *                or SM_CAL_REGULAR_USER (no administrative permissions).
  *
  */
function check_cal_user_type($user) 
{

   global $cal_admins, $cal_debug;

   foreach ($cal_admins as $admin => $perms) 
   {

      if ($cal_debug) 
         echo "current username is \"$user\"; admin username is \"$admin\"; are they equal? " 
            . (strtolower($admin) == strtolower($user) ? 'YES' : 'NO') . '<br />';


      if (preg_match('/^' . str_replace(array('?', '*'), array('\w{1}', '.*?'),
                     strtoupper($admin)) . '$/', strtoupper($user)))
      {
         $perms = trim($perms);
         if (strtolower($perms) == 'yes')
            return SM_CAL_SUPERUSER;
         else
            return SM_CAL_LIMITED_ADMIN;
      }

   }

   return SM_CAL_REGULAR_USER;

}



/**
  * Determine the week of the month for a date
  *
  * The date in question may be given as a timestamp
  * or as separate day, month and year numbers
  * 
  * @param timestamp $timestamp The date in question
  * @param int $year The year of the date in question
  * @param int $month The month of the date in question
  * @param int $day The day of the date in question
  * 
  * @return int The week of the month that the given
  *             date falls within, or zero if error
  *
  */
function weekOfMonth($timestamp=0, $year=0, $month=0, $day=0)
{

   // bunch of junk to figure out parameters
   // skip to last line to see what you really
   // wanted... 
   //
   if ($month == 0 || $year == 0 || $day == 0)
   {
      if ($timestamp == 0)
         return 0;  // error
      else
         list($month, $year) = explode(',', date('m,Y', $timestamp));
   }
   else if ($month == 0 || $year == 0 || $day == 0)
      return 0;  // error
   else
      $timestamp = mktime(0, 0, 0, $month, $day, $year);


   // not much to it; just subtract week of year
   // for the first of the month from the week of
   // the year for the target timestamp
   //
   return date('W', $timestamp) 
        - date('W', mktime(0, 0, 0, $month, 1, $year));

}



// ==========================================================================
// Handy function to get a timestamp when all you have
// is a week number (of the year), a day (Monday - Sunday)
// and a year number -- taken from php.net
//
// TODO: what is not clear is if Sunday is considered day 7 or day 0???
//       if the former, we need to make some mods here, since PHP considers
//       Sunday as 0
//       After just a trivial look at the code, it looks like it considers
//       Sunday as 7, so I inserted code at the beginning of the function
//       to make the correct conversion.

/**
  * Get a date by providing a week number, day of week and a year.
  *
  * Be careful! There are different definitions for weeks. Here the European definition is used.
  * In Europe a week starts on Monday.
  * Also the start of the first week in a year is defined differently in different countries.
  * Here the ISO 8601 definition is used. This is the standard in Europe.
  *
  * I got the information from http://home.t-online.de/home/PeterJHaas/delphi.htm
  * There are many websites with information on week numbers.
  * An excellent site on this subject is http://www.pjh2.de/datetime/weeknumber/index.php
  *
  * This PHP source was based on the Delphi source code by Peter J. Haas
  *
  * 
  * //give me the date of Friday week 20 of the year 2004 (Should result in Friday May 14 2004)
  * $aWeek=20; $aDay=05; $aYear=2004;
  * $adate=datefromweeknr($aYear, $aWeek, $aDay);
  * echo 'The date (week='.$aWeek.' day='.$aDay.' year= '.$aYear.') is '.date('D d-m-Y',$adate).'<br>';
  *
  */
 function datefromweeknr($aYear, $aWeek, $aDay)
 {
  // correct for PHP Sundays
  if ($aDay == 0) $aDay = 7;

  $FirstDayOfWeek=1; //First day of week is Monday       
  $BaseDate=4; //We calculate from 4/1 which is always in week 1 
  $CJDDelta=2415019; //Based on start of Chronological Julian Day
  $StartDate = DelphiDate(mktime(1,0,0,01,$BaseDate,$aYear)); //The date to start with
  $Offset = ($aWeek-1) * 7 - mod(floor($StartDate) + $CJDDelta + 8 - $FirstDayOfWeek,7) + $aDay - 1;
  return PHPUnixTimeStamp($StartDate + $Offset);
 }

 #---------extra functions used----------

 function DelphiDate($aPHPTime)
 {
   # The Unix Timestamp holds the number of seconds after January 1 1970 01:00:00
   return div($aPHPTime,86400)+25569;
 }
 
 function PHPUnixTimeStamp($aDelphiDate)
 {
   # Delphi's TDate holds number of days after December 30 1899
   return ($aDelphiDate-25569)*86400-3600;
 }
 
 function mod($number, $div)
 {
   return $number - floor($number/$div)*$div;
 }

 function div($number, $div)
 {
   return floor($number/$div);
 }
// ==========================================================================



/** 
  * Determines what occurrence number the day of
  * the week is in its month for a given date.
  * That is, is this the 2nd Friday of the month?
  * Or the 1st Wednesday?
  *
  * The date in question may be given as a timestamp
  * or as separate day, month and year numbers
  * 
  * @param timestamp $timestamp The date in question
  * @param int $year The year of the date in question
  * @param int $month The month of the date in question
  * @param int $day The day of the date in question
  * 
  * @return int The occurrence number of the day in
  *             the month, or zero if error.
  *
  */
function dayOcurrenceInMonth($timestamp=0, $year=0, $month=0, $day=0)
{

   // bunch of junk to figure out parameters
   // skip to last line to see what you really
   // wanted... 
   //
   if ($month == 0 || $year == 0 || $day == 0)
   {
      if ($timestamp == 0)
         return 0;  // error
      else
         list($day, $month, $year) = explode(',', date('j,m,Y', $timestamp));
   }
   else if ($month == 0 || $year == 0 || $day == 0)
      return 0;  // error
   else
      $timestamp = mktime(0, 0, 0, $month, $day, $year);


   // manually count day occurrences
   //
   $count = 0;
   list($dayOfWeek, $daysThisMonth) = explode(',', date('w,t', $timestamp));
   for ($i = 1; $i < $daysThisMonth + 1 && $i <= $day; $i++)
   {
      if (date('w', mktime(0, 0, 0, $month, $i, $year)) == $dayOfWeek)
         $count++;
   }

   return $count;

}



/**
  * Checks if the given date is within any two 
  * timestamps (inclusive).
  *
  * Note that granularity only goes to DAYS.  Hours
  * and seconds of the start and end times are ignored.
  *
  * @param int $year The year of the day to check 
  * @param int $month The month of the day to check 
  * @param int $day The day to check 
  * @param timestamp $start Beginning of date range for check
  * @param timestamp $end End of date range for check
  * @param timestamp $theDay Timestamp corresponding to the
  *                          $year, $month and $day parameters
  *                          (optional, but gives performance
  *                          boost if provided)
  *
  * @return boolean TRUE if the target date is between the 
  *                 given date range (inclusive), FALSE otherwise.
  *
  */
function dayIsBetween($year, $month, $day, $start, $end, $theDay=0)
{

   // just verify that start day doesn't come after target day
   // and that end day doesn't come before target day


   // we need this to be as fast as possible, so check everything
   // we can before we start making dates (make dates one at a time,
   // as little as needed to just return outta here)
   //
   // the order below is important!  this is the fastest combination
   //
   $endYear = date('Y', $end);
   if ($endYear < $year)
      return FALSE;


   $startYear = date('Y', $start);
   if ($startYear > $year)
      return FALSE;


   if ($theDay == 0)
      $yearDay = date('z', mktime(0, 0, 0, $month, $day, $year));
   else
      $yearDay = date('z', $theDay);


   $endYearDay = date('z', $end);
   if ($endYear == $year && $endYearDay < $yearDay)
      return FALSE;


   $startYearDay = date('z', $start);
   if ($startYear == $year && $startYearDay > $yearDay)
      return FALSE;


   return TRUE;

}



/**
  * clean/encode a string for output in HTML pages by reference
  *
  * If the parameter is an array, any and all of the array's values that are
  * strings are cleaned/encoded, and if it contains any nested arrays, those
  * will be recursively scanned for string values to clean/encode.
  *
  * @param mixed $item The string (or array) to be encoded
  * @param boolean $convertNewlines If TRUE, all newlines are converted to <br /> tags;
  *                                 if FALSE, newlines are left as is (optional; default = TRUE)
  *
  */
function cal_encode_output_by_ref(&$item, $convertNewlines=TRUE)
{
   if (is_string($item)) $item = ($convertNewlines ? nl2br(htmlspecialchars($item, ENT_QUOTES))
                                                   : htmlspecialchars($item, ENT_QUOTES));
   else if (is_array($item))
      foreach ($item as $index => $value) cal_encode_output_by_ref($item[$index], $convertNewlines);
}



/**
  * clean/encode and return a string for output in HTML pages
  *
  * If the parameter is an array, any and all of the array's values that are
  * strings are cleaned/encoded, and if it contains any nested arrays, those
  * will be recursively scanned for string values to clean/encode.
  *
  * @param mixed $item The string (or array) to be encoded
  * @param boolean $convertNewlines If TRUE, all newlines are converted to <br /> tags;
  *                                 if FALSE, newlines are left as is (optional; default = TRUE)
  *
  */
function cal_encode_output($item, $convertNewlines=TRUE)
{
   if (is_string($item)) return ($convertNewlines ? nl2br(htmlspecialchars($item, ENT_QUOTES))
                                                  : htmlspecialchars($item, ENT_QUOTES));
   else if (is_array($item))
   {
      $newItem = array();
      foreach ($item as $index => $value)
         $newItem[$index] = cal_encode_output($value, $convertNewlines);
      return $newItem;
   }
   else return $item;
}



/**
  * Sorts a list of calendars
  *
  * Sorts by calendar name, using case insensitive comparisons
  *
  * @param object $a Calendar A 
  * @param object $b Calendar B
  *
  * @return -1 if calendar A comes before calendar B, 1 if calendar
  *         B comes before calendar A, or zero if the two are equal.
  *
  */
function calendar_sort($a, $b)
{

   return $a->compareTo($b);

}



/**
  * Sort and organize events for one day
  *
  * Takes a flat array of events and places them into
  * a nested array structure (based on start time) as such:
  *
  * Hour 
  *  |
  *  |--- Quarter Hour
  *  |      |
  *  |      |--- Event
  *  |      |--- Event
  *  |       
  *  |--- Quarter Hour
  *         |
  *         |--- Event
  *
  * For example, one event at 3:30pm and two at 8pm:
  *
  * Array
  * (
  *     [15] => Array
  *         (
  *             [30] => Array
  *                 (
  *                     [0] => event Object
  *                         (
  *                             ....
  *                         ) 
  *                 ) 
  *         ) 
  *     [20] => Array
  *         (
  *             [0] => Array
  *                 (
  *                     [0] => event Object
  *                         (
  *                             ....
  *                         ) 
  *                     [1] => event Object
  *                         (
  *                             ....
  *                         ) 
  *                 ) 
  *         ) 
  * ) 
  *
  * NOTE: events that start before this day are all
  *       placed at midnight ([0][0]) in the return array
  *       unless the $startHour parameter is given, in
  *       which case they are placed at minute zero for
  *       that hour.
  *
  * @param array $events The events that are to be organized
  * @param int $year The year of the day for which events are being organized.
  * @param int $month The month of the day for which events are being organized.
  * @param int $day The day for which events are being organized.
  * @param int $startHour The hour that the returned array should start
  *                       with - any events starting before that time are
  *                       pushed into the zero-minute slot for this hour
  *                       (optional; if not given, all hours of the day
  *                       (that have events) are returned)
  *
  * @return array Those same events, organized per above.
  *
  */
function organizeEvents($events, $year, $month, $day, $startHour=0) 
{

   if (!is_numeric($startHour) || $startHour < 0 || $startHour > 23)
   {
      global $color;
      plain_error_message('ERROR (organizeEvents): Bad start hour: ' . $startHour, $color);
      exit;
   }


   $returnArray = array();

   foreach ($events as $event)
   {

      $eventStartHour = $event->startHour($year, $month, $day);

      if ($startHour && $eventStartHour < $startHour) 
         $returnArray[$startHour][$event->startQuarterHour($year, $month, $day)][] = $event;

      else
         $returnArray[$eventStartHour][$event->startQuarterHour($year, $month, $day)][] = $event;

   }

   return $returnArray;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of years
  *
  * @param int $selected The year that should be preselected (optional)
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_year($selected='') 
{

   $html = '';

   for ($i = 1970; $i < 2038; $i++)
   {
      if (!empty($selected) && $i == $selected)
      {
         $html .= "<option value=\"$i\" selected>$i</option>\n";
      } 
      else 
      {
         $html .= "<option value=\"$i\">$i</option>\n";
      }
   }

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of months
  *
  * @param int $selected The month that should be preselected (optional)
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_month($selected='') 
{

   // NOTE: should not need to switch gettext domain here, since
   //       this function should be being called from within an
   //       interface function that is already in the correct calendar
   //       gettext domain.... right?

   $html = '';

   for ($i = 1; $i < 13; $i++)
   {
      $monthName = getMonthAbrv($i);

      if (!empty($selected) && $i == $selected)
      {
         $html .= "<option value=\"$i\" SELECTED>$monthName</option>\n";
      } 
      else 
      {
         $html .= "<option value=\"$i\">$monthName</option>\n";
      }
   }

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of days
  *
  * @param int $selected The day that should be preselected (optional)
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_day($selected='') 
{

   $html = '';

   for ($i = 1; $i < 32; $i++)
   {

      if (!empty($selected) && $i == $selected)
      {
         $html .= "<option value=\"$i\" SELECTED>$i</option>\n";
      } 
      else 
      {
         $html .= "<option value=\"$i\">$i</option>\n";
      }

   }

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of days of the week
  *
  * @param int $selected The day of the week that should 
  * be preselected (optional)
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_day_of_week($selected=0)
{

   // NOTE: should not need to switch gettext domain here, since
   //       this function should be being called from within an
   //       interface function that is already in the correct calendar
   //       gettext domain.... right?

   $html = '';


   $html .= '<option value="0"' . ($selected == 0 ? ' SELECTED' : '') 
         . '>' . _("Sunday") . "</option>\n";
   $html .= '<option value="1"' . ($selected == 1 ? ' SELECTED' : '') 
         . '>' . _("Monday") . "</option>\n";
   $html .= '<option value="2"' . ($selected == 2 ? ' SELECTED' : '') 
         . '>' . _("Tuesday") . "</option>\n";
   $html .= '<option value="3"' . ($selected == 3 ? ' SELECTED' : '') 
         . '>' . _("Wednesday") . "</option>\n";
   $html .= '<option value="4"' . ($selected == 4 ? ' SELECTED' : '') 
         . '>' . _("Thursday") . "</option>\n";
   $html .= '<option value="5"' . ($selected == 5 ? ' SELECTED' : '') 
         . '>' . _("Friday") . "</option>\n";
   $html .= '<option value="6"' . ($selected == 6 ? ' SELECTED' : '') 
         . '>' . _("Saturday") . "</option>\n";


   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of integers with the given range
  * 
  * @param int $startValue The integer from which option values should start
  * @param int $endValue The integer up to which option values should reach
  * @param int $selected The integer that should be preselected (optional)
  * 
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *                
  */
function select_option_integer($startValue, $endValue, $selected='')
{

   $html = '';
   
   for ($i = $startValue; $i < $endValue + 1; $i++)
   {
   
      if (!empty($selected) && $i == $selected)
      {
         $html .= "<option value=\"$i\" SELECTED>$i</option>\n";
      }  
      else
      {
         $html .= "<option value=\"$i\">$i</option>\n";
      }

   }

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of time lengths
  *
  * @param int $selected The length value that should be preselected (optional)
  *                      Note that this parameter need not match the exact
  *                      length minute value - the nearest length greater than
  *                      the given default value will be preselected if there 
  *                      is not an exact match.
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_length($selected='')
{

   // NOTE: should not need to switch gettext domain here, since
   //       this function should be being called from within an
   //       interface function that is already in the correct calendar
   //       gettext domain.... right?

   $html = '';

/* see below...
   $lengths = array(
      '0'     => '0 '   . _("min."),
      '15'    => '15 '  . _("min."),
      '30'    => '30 '  . _("min."),
      '45'    => '45 '  . _("min."),
      '60'    => '1 '   . _("hr."),
      '90'    => '1.5 ' . _("hr."),
      '120'   => '2 '   . _("hr."),
      '150'   => '2.5 ' . _("hr."),
      '180'   => '3 '   . _("hr."),
      '210'   => '3.5 ' . _("hr."),
      '240'   => '4 '   . _("hr."),
      '270'   => '4.5 ' . _("hr."),
      '300'   => '5 '   . _("hr."),
      '330'   => '5.5 ' . _("hr."),
      '360'   => '6 '   . _("hr."),
      '420'   => '7 '   . _("hr."),
      '480'   => '8 '   . _("hr."),
      '540'   => '9 '   . _("hr."),
      '600'   => '10 '  . _("hr."),
      '660'   => '11 '  . _("hr."),
      '720'   => '12 '  . _("hr."),
      '1440'  => '1 '   . _("day"),
      '2880'  => '2 '   . _("days"),
      '4320'  => '3 '   . _("days"),
      '5760'  => '4 '   . _("days"),
      '7200'  => '5 '   . _("days"),
      '8640'  => '6 '   . _("days"),
      '10080' => '1 '   . _("wk."),
      '20160' => '2 '   . _("wk."),
      '30240' => '3 '   . _("wk."),
      '40320' => '4 '   . _("wk."),
      '50400' => '5 '   . _("wk."),
      '60480' => '6 '   . _("wk."),
      '70560' => '7 '   . _("wk."),
      '80640' => '8 '   . _("wk."),
   );
*/
   $lengths=array();
   foreach (array(0,15,30,45,
                  60,90,120,150,180,210,240,270,300,330,360,420,480,540,
                  600,660,720,1440,2880,4320,5760,7200,8640,
                  10080,20160,30240,40320,50400,60480,70560,80640) as $length) 
   {
      if ($length < 60) 
      {
         // event length in minutes. Use abbreviation, that works with any numeric value
         //
         $lengths[$length] = sprintf(_("%d min."), $length);
      } 
      else if ($length < 1440) 
      {
         // event length in hours. Use abbreviation, that works with any numeric value
         // length is float number.
         // %s is used instead of %.1f in order to suppress zero padding.
         // if php implements strict variable format checking, all ($length/something) calls
         // used in sprintf, should be converted to strings. 
         // floating point delimiter depends on used squirrelmail version. Some SM versions
         // revert to C float delimiter (.) in order to solve issues with other plugins
         //
         $lengths[$length] = sprintf(_("%s hr."), ($length / 60));
      } 
      else if ($length < 10080) 
      {
         // event length in days.
         //
//TODO: wait until SM (and/or compatibility plugin) has ngettext wrapper
//         $lengths[$length] = sprintf(ngettext("%s day", "%s days", ($length / 1440)), ($length / 1440));
         if ($length < 2880)
            $lengths[$length] = sprintf(_("%s day"), ($length / 1440));
         else
            $lengths[$length] = sprintf(_("%s days"), ($length / 1440));
      } 
      else 
      {
         // event length in weeks. Use abbreviation, that works with any numeric value
         //
         $lengths[$length] = sprintf(_("%s wk."), ($length / 10080));
      }
   }


   $noDefaultYet = TRUE;
   foreach ($lengths as $min => $text) 
   {

      if (!empty($selected) && ($min == $selected || ($min > $selected && $noDefaultYet)))
      {
         $html .= "<option value=\"$min\" selected>$text</option>\n";
         $noDefaultYet = FALSE;
      }
      else
      {
         $html .= "<option value=\"$min\">$text</option>\n";
      }

   }

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of hours
  *
  * @param int $selected The hour that should be preselected (optional)
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_hour($selected='')
{

   // NOTE: should not need to switch gettext domain here, since
   //       this function should be being called from within an
   //       interface function that is already in the correct calendar
   //       gettext domain.... right?

   global $hour_format;

   $html = '';

   if ($hour_format == SMPREF_TIME_24HR)
      $hours = array(
         '0' => 0,
         '1' => 1,
         '2' => 2,
         '3' => 3,
         '4' => 4,
         '5' => 5,
         '6' => 6,
         '7' => 7,
         '8' => 8,
         '9' => 9,
         '10' => 10,
         '11' => 11,
         '12' => 12,
         '13' => 13,
         '14' => 14,
         '15' => 15,
         '16' => 16,
         '17' => 17,
         '18' => 18,
         '19' => 19,
         '20' => 20,
         '21' => 21,
         '22' => 22,
         '23' => 23,
      );

   else
      $hours = array(
         '0' => sprintf(_("%sam"), '12'),
         '1' => sprintf(_("%sam"), '1'),
         '2' => sprintf(_("%sam"), '2'),
         '3' => sprintf(_("%sam"), '3'),
         '4' => sprintf(_("%sam"), '4'),
         '5' => sprintf(_("%sam"), '5'),
         '6' => sprintf(_("%sam"), '6'),
         '7' => sprintf(_("%sam"), '7'),
         '8' => sprintf(_("%sam"), '8'),
         '9' => sprintf(_("%sam"), '9'),
         '10' => sprintf(_("%sam"), '10'),
         '11' => sprintf(_("%sam"), '11'),
         '12' => sprintf(_("%spm"), '12'),
         '13' => sprintf(_("%spm"), '1'),
         '14' => sprintf(_("%spm"), '2'),
         '15' => sprintf(_("%spm"), '3'),
         '16' => sprintf(_("%spm"), '4'),
         '17' => sprintf(_("%spm"), '5'),
         '18' => sprintf(_("%spm"), '6'),
         '19' => sprintf(_("%spm"), '7'),
         '20' => sprintf(_("%spm"), '8'),
         '21' => sprintf(_("%spm"), '9'),
         '22' => sprintf(_("%spm"), '10'),
         '23' => sprintf(_("%spm"), '11'),
      );


   foreach ($hours as $hour => $text)
   {

      if (!empty($selected) && $hour == $selected)
      {
         $html .= "<option value=\"$hour\" SELECTED>$text</option>\n";
      }
      else
      {
         $html .= "<option value=\"$hour\">$text</option>\n";
      }

   }

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of minutes
  *
  * @param int $selected The minute that should be preselected (optional)
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_minute($selected='')
{

   $html = '';

   $minutes = array();

   for ($i = 0; $i < 60; $i++) 
      $minutes[$i] = ($i < 10 ? '0' . $i : $i);

/*
   $minutes = array(
      '0'=>'00',
      '5'=>'05',
      '10'=>'10',
      '15'=>'15',
      '20'=>'20',
      '25'=>'25',
      '30'=>'30',
      '35'=>'35',
      '40'=>'40',
      '45'=>'45',
      '50'=>'50',
      '55'=>'55'
   );
*/


   foreach ($minutes as $min => $text)
   {

      if (!empty($selected) && $min == $selected)
      {
         $html .= "<option value=\"$min\" SELECTED>$text</option>\n";
      }
      else
      {
         $html .= "<option value=\"$min\">$text</option>\n";
      }

   }

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of priorities
  *
  * @param int $selected The priority that should be 
  *                      preselected (optional; default
  *                      is Normal priority)
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_priority($selected='')
{

   // NOTE: should not need to switch gettext domain here, since
   //       this function should be being called from within an
   //       interface function that is already in the correct calendar
   //       gettext domain.... right?

   $html = '';

   global $EVENT_PRIORITIES;


   foreach ($EVENT_PRIORITIES as $num => $text)
   {

      // skip unknown so we can put it at the end of the list,
      // where it is more intuiitive
      //
      if ($num == 0) continue;

      if (($selected !== '' && $num == $selected) 
       || ($selected === '' && $num == SM_CAL_EVENT_PRIORITY_NORMAL))
      {
         $html .= "<option value=\"$num\" SELECTED>$text</option>\n";
      }
      else
      {
         $html .= "<option value=\"$num\">$text</option>\n";
      }

   }
   if ($selected === 0)
      $html .= "<option SELECTED value=\"0\">" . _("Unknown") . "</option>\n";
   else
      $html .= "<option value=\"0\">" . _("Unknown") . "</option>\n";

   return $html;

}



/**
  * Generates generic set of <option> tags for inclusion
  * in a <select> tag for a list of recurrence types
  *
  * @param int $selected The recurrence type that should be 
  *                      preselected
  *
  * @return string The list of <option> tags, ready to be
  *                output to the HTML page
  *
  */
function select_option_recurrence_type($selected='')
{

   $html = '';

   global $RECURRENCE_TYPES;


   foreach ($RECURRENCE_TYPES as $code => $text)
   {

      if (!empty($selected) && $code == $selected)
      {
         $html .= "<option value=\"$code\" SELECTED>$text</option>\n";
      }
      else
      {
         $html .= "<option value=\"$code\">$text</option>\n";
      }

   }

   return $html;

}



/**
  * get_headers() implementation for php4
  *
  * Swiped from http://www.php.net/manual/function.get-headers.php
  *
  */
if (!function_exists('get_headers')) 
{
 
   /**
     * @return array
     * @param string $url
     * @param int $format
     * @desc Fetches all the headers
     * @author cpurruc fh-landshut de
     * @modified by dotpointer
     * @modified by aeontech
     */
   function get_headers($url,$format=0)
   {
       $url_info=parse_url($url);
       $port = isset($url_info['port']) ? $url_info['port'] : 80;
       $fp=fsockopen($url_info['host'], $port, $errno, $errstr, 30);
      
       if($fp)
       {
           $head = "HEAD ".@$url_info['path']."?".@$url_info['query']." HTTP/1.0\r\nHost: ".@$url_info['host']."\r\n\r\n";     
           fputs($fp, $head);     
           while(!feof($fp))
           {
               if($header=trim(fgets($fp, 1024)))
               {
                   if($format == 1)
                   {
                       $key = array_shift(explode(':',$header));
                       // the first element is the http header type, such as HTTP 200 OK,
                       // it doesn't have a separate name, so we have to check for it.
                       if($key == $header)
                       {
                           $headers[] = $header;
                       }
                       else
                       {
                           $headers[$key]=substr($header,strlen($key)+2);
                       }
                       unset($key);
                   }
                   else
                   {
                       $headers[] = $header;
                   }
               }
           }
           return $headers;
       }
       else
       {
           return false;
       }
   }

}



?>
