<?php

/**
  * SquirrelMail Shared Calendar Plugin
  * Copyright (C) 2004-2005 Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under GPL. See COPYING for details
  *
  */


/**
  * Returns a listing of all calendars that the given user 
  * owns.  If the user is a superuser, all calendars are 
  * returned (regardless of actual ownership); if the user 
  * is a limited administrator, the calendars owned by that
  * user are returned; and if the user is an unpriveleged
  * user, just that user's private calendars are returned.
  *
  * This function uses the "get_all_owned_calendars" hook 
  * to allow a pluggable backend to retrieve calendars as needed.
  * The handler for this hook can expect these parameters:
  *   - username
  *   - user type (using constants from calendar constants file)
  * The value returned from the hook handler should be the
  * array of calendar objects that will be passed back to the
  * original caller.
  *
  * @param string $user The user for which to retrieve calendars
  *
  * @return array An array of calendar objects
  *
  */
function get_all_owned_calendars($user) 
{

   $userType = check_cal_user_type($user);
   return do_hook_function('get_all_owned_calendars', array($user, $userType));

}



/**
  * Returns a listing of all shared and private (but not public)
  * calendars that the given user has read or write access to
  * EXCEPT the user's default private calendar.
  *
  * This function uses the "get_all_accessible_calendars" hook
  * to allow a pluggable backend to retrieve calendars as needed.
  * The handler for this hook can expect these parameters:
  *   - username
  *   - domain
  * The value returned from the hook handler should be the
  * array of calendar objects that will be passed back to the
  * original caller.
  *
  * @param string $user The user for which to retrieve calendars
  * @param string $domain The user's domain
  *
  * @return array An array of calendar objects
  *
  */
function get_all_accessible_calendars($user, $domain)
{

   return do_hook_function('get_all_accessible_calendars', array($user, $domain));

}



/**
  * Returns a listing of all public calendars
  *
  * This function uses the "get_all_public_calendars" hook 
  * to allow a pluggable backend retrieve calendars as needed.
  * The value returned from the hook handler should be the
  * array of calendar objects that will be passed back to the
  * original caller.
  *
  * @return array An array of calendar objects
  *
  */
function get_all_public_calendars() 
{

   return do_hook_function('get_all_public_calendars');

}



/**
  * Creates a new calendar
  *
  * Takes the given calendar object and inserts it into the
  * backend as a new calendar.  The ID contained in the given 
  * calendar is expected to be a correct unique ID value. 
  *
  * Note that this does not save any events contained inside
  * the calendar.
  *
  * This function uses the "create_calendar" hook
  * to allow a pluggable backend to create calendars as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar object
  *
  * @param object $calendar The new calendar object
  *
  */
function create_calendar($calendar)
{

   do_hook('create_calendar', $calendar);

}



/**
  * Delete calendar
  *
  * Removes the given calendar from the system.
  *
  * This function uses the "delete_calendar" hook
  * to allow a pluggable backend to delete calendars as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *
  * @param string $calendarID The ID of the calendar to be removed
  *
  */
function delete_calendar($calendarID)
{

   do_hook('delete_calendar', $calendarID);

}



/**
  * Updates a calendar
  *
  * Updates the given calendar by replacing it in the backend
  * with the given calendar object.
  *
  * This function uses the "update_calendar" hook
  * to allow a pluggable backend to update calendars as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar object
  *
  * @param object $calendar The updated calendar object
  *
  */
function update_calendar($calendar)
{

   do_hook('update_calendar', $calendar);

}



/**
  * Get calendar
  *
  * Retrieves the given calendar from the backend
  *
  * This function uses the "get_calendar" hook
  * to allow a pluggable backend to get calendars as needed
  * (unless the calendar is from an external URI, in which
  * case it is loaded using information from user prefs).
  *
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *   - quiet
  * The value returned from the hook handler should be the
  * calendar object that will be passed back to the original 
  * caller.
  *
  * @param string $calendarID The ID of the calendar to be retrieved
  * @param boolean $quiet When FALSE, if the requested calendar isn't
  *                       found, an error is shown and execution halts;
  *                       otherwise, FALSE is just returned quietly
  *                       (optional; default = FALSE)
  *
  * @return mixed A Calendar object corresponding to the desired
  *               calendar, or FALSE if the calendar is not found
  *               and the $quiet parameter is TRUE
  *
  */
function get_calendar($calendarID, $quiet=FALSE)
{

   if (strpos($calendarID, 'SM_EXTERNAL') === 0)
      return loadExternalCalendarFromUserPrefs($calendarID);


   // personal calendar: figure out correct ID first
   //
   if ($calendarID == 'personal')
   {
      global $username, $domain;
      $calendarID = get_personal_cal_id($username, $domain);
   }


   return do_hook_function('get_calendar', array($calendarID, $quiet));

}



/**
  * Retrieves all events, holidays, and other for the given calendar
  * for all time periods.
  *
  * This function uses the "get_all_events" hook
  * to allow a pluggable backend to get events as
  * needed.
  * The handler for this hook can expect these parameters:
  *   - calID
  *   - user
  * The value returned from the hook handler should be the
  * events array that will be passed back to the original
  * caller.
  *
  * @param string $calID The ID of the calendar for which to retrieve events
  * @param string $user The user for which events are being retrieved
  *
  * @return array An array of calendar events.  This array is keyed by
  *               event id, where associated values are Event objects
  *
  */
function get_all_events($calID, $user)
{

   return do_hook_function('get_all_events', array($calID, $user));

}



/**
  * Retrieves all one-time events for the given calendar
  * for the given month, including any that overlap
  * into previous/next months.
  *
  * Note that this function is intended to populate a
  * Calendar object and may return more events than needed,
  * the caller is responsible for extracting only the events
  * needed using other Calendar methods.
  *
  * This function uses the "get_events_for_month" hook
  * to allow a pluggable backend to get events as
  * needed.
  * The handler for this hook can expect these parameters:
  *   - calID
  *   - year
  *   - month
  *   - user
  * The value returned from the hook handler should be the
  * events array that will be passed back to the original
  * caller.
  *
  * @param string $calID The ID of the calendar for which to retrieve events
  * @param int $year The year of the month for which to retrieve events
  * @param int $month The month for which to retrieve events
  * @param string $user The user for which events are being retrieved
  *
  * @return array An array of calendar events.  This array is keyed by 
  *               event id, where associated values are Event objects
  *
  */
function get_events_for_month($calID, $year, $month, $user)
{

   return do_hook_function('get_events_for_month', array($calID, $year, $month, $user));

}



/**
  * Retrieves all one-time events for the given calendar
  * for the given day, including any that overlap
  * into previous/next days.
  *
  * Note that this function is intended to populate a
  * Calendar object and may return more events than needed,
  * the caller is responsible for extracting only the events
  * needed using other Calendar methods.
  *
  * This function uses the "get_events_for_day" hook
  * to allow a pluggable backend to get events as
  * needed.
  * The handler for this hook can expect these parameters:
  *   - calID
  *   - year
  *   - month
  *   - day
  *   - user
  * The value returned from the hook handler should be the
  * events array that will be passed back to the original
  * caller.
  *
  * @param string $calID The ID of the calendar for which to retrieve events
  * @param int $year The year of the day for which to retrieve events
  * @param int $month The month of the day for which to retrieve events
  * @param int $day The day for which to retrieve events
  * @param string $user The user for which events are being retrieved
  *
  * @return array An array of calendar events.  This array is keyed by 
  *               event id, where associated values are Event objects
  *
  */
function get_events_for_day($calID, $year, $month, $day, $user)
{

   return do_hook_function('get_events_for_day', array($calID, $year, $month, $day, $user));

}



/**
  * Retrieves all recurring events for the given calendar.
  *
  * This function uses the "get_recurring_events" hook
  * to allow a pluggable backend to get events as
  * needed.
  * The handler for this hook can expect these parameters:
  *   - calID
  *   - user
  * The value returned from the hook handler should be the
  * events array that will be passed back to the original
  * caller.
  *
  * @param string $calID The ID of the calendar for which to retrieve events
  * @param string $user The user for which events are being retrieved
  *
  * @return array An array of calendar events.  This array is keyed by
  *               event id, where associated values are Event objects
  *
  */
function get_recurring_events($calID, $user)
{

   return do_hook_function('get_recurring_events', array($calID, $user));

}



/**
  * Creates a new event
  *
  * Takes the given event object and inserts it into the
  * backend as a new event with the ID as given in the 
  * event object.
  *
  * This function uses the "create_event" hook
  * to allow a pluggable backend to create events as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *   - event object
  * The value returned from the hook handler should be the
  * event ID that will be passed back to the original
  * caller.
  *
  * @param string $calendarID The ID of the calendar having an event added
  * @param object $event The new event object
  *
  * @return string The ID of the newly created event
  *
  */
function create_event($calendarID, $event)
{

   return do_hook_function('create_event', array($calendarID, $event));

}



/**
  * Delete event
  *
  * Removes the given event from the given calendar.
  *
  * This function uses the "delete_event" hook
  * to allow a pluggable backend to delete events as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *   - event ID
  *
  * @param string $calendarID The ID of the calendar whose event is being removed
  * @param string $eventID The ID of the event to be removed
  *
  */
function delete_event($calendarID, $eventID)
{

   do_hook('delete_event', $calendarID, $eventID);

}



/**
  * Updates an event
  *
  * Updates the given event by replacing it in the backend
  * with the given event object.
  *
  * This function uses the "update_event" hook
  * to allow a pluggable backend to update events as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *   - event object
  *
  * @param string $calendarID The ID of the calendar whose event is being updated
  * @param object $event The updated event object
  *
  */
function update_event($calendarID, $event)
{

   do_hook('update_event', $calendarID, $event);

}



/**
  * Get event
  *
  * Retrieves the given event from the backend
  *
  * This function uses the "get_event" hook
  * to allow a pluggable backend to get events as needed
  * (unless the calendar is from an external URI, in which
  * case the event is loaded using information from user prefs).
  *
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *   - event ID
  * The value returned from the hook handler should be the
  * event object that will be passed back to the original
  * caller.
  *
  * @param string $calendarID The ID of the calendar whose event is to be retrieved
  * @param string $eventID The ID of the event to be retrieved
  *
  * @return object A Event object corresponding to the desired event or FALSE i
f not found
  *
  */
function get_event($calendarID, $eventID)
{

   if (strpos($calendarID, 'SM_EXTERNAL') === 0)
      return loadExternalEvent($calendarID, $eventID);

   return do_hook_function('get_event', array($calendarID, $eventID));

}



/**
  * Get all holidays for the given calendar
  *
  * Retrieves all the holidays for the given calendar from the backend
  *
  * This function uses the "get_calendar_holidays" hook
  * to allow a pluggable backend to get holidays as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *   - username
  * The value returned from the hook handler should be the
  * array of holiday objects that will be passed back to the 
  * original caller.
  *
  * @param string $calendarID The ID of the calendar whose holidays 
  *                           are being retrieved
  * @param string $user The user for which to retrieve holidays
  *
  * @return array An array of Holiday objects
  *
  */
function get_calendar_holidays($calendarID, $user)
{

   return do_hook_function('get_calendar_holidays', array($calendarID, $user));

}



//LEFT OFF HERE --- what functions below here are still relevant given that holidays are now no more than events?
//LEFT OFF HERE --- what functions below here are still relevant given that holidays are now no more than events?
//LEFT OFF HERE --- what functions below here are still relevant given that holidays are now no more than events?
/**
  * Get global holiday
  *
  * Retrieves the given global holiday from the backend
  *
  * This function uses the "get_global_holiday" hook
  * to allow a pluggable backend to get holidays as needed.
  * The handler for this hook can expect these parameters:
  *   - holiday ID
  * The value returned from the hook handler should be the
  * holiday object that will be passed back to the original
  * caller.
  *
  * @param string $holidayID The ID of the holiday to be retrieved
  *
  * @return object A Holiday object corresponding to the desired holiday
  *
  */
function get_global_holiday($holidayID)
{

   return do_hook_function('get_global_holiday', array($holidayID));

}



/**
  * Get holiday
  *
  * Retrieves the given holiday for the given calendar from the backend
  *
  * This function uses the "get_holiday" hook
  * to allow a pluggable backend to get holidays as needed.
  * The handler for this hook can expect these parameters:
  *   - calendar ID
  *   - holiday ID
  * The value returned from the hook handler should be the
  * holiday object that will be passed back to the original
  * caller.
  *
  * @param string $calendarID The ID of the calendar whose holiday is to be retrieved
  * @param string $holidayID The ID of the holiday to be retrieved
  *
  * @return object A Holiday object corresponding to the desired holiday
  *
  */
function get_holiday($calendarID, $holidayID)
{

   return do_hook_function('get_holiday', array($calendarID, $holidayID));

}



/**
  * Returns a listing of all global holidays
  *
  * This function uses the "get_all_global_holidays" hook
  * to allow a pluggable backend retrieve holidays as needed.
  * The value returned from the hook handler should be the
  * array of holiday objects that will be passed back to the
  * original caller.
  *
  * @return array An array of Holiday objects
  *
  */
function get_all_global_holidays()
{

   return do_hook_function('get_all_global_holidays');

}



?>
